package Zim::Utils;

use strict;
use File::BaseDir qw/xdg_data_files/;
use Zim::File;

our $VERSION = '0.23';

eval 'use Zim::OS::Win32' if $^O eq 'MSWin32';
die $@ if $@;

our @ISA = ($^O eq 'MSWin32') ? ('Zim::Utils::Win32')
                              : ('Zim::Utils::Unix' ) ;

our %Translation;

=head1 NAME

Zim::Utils - Some useful functions and objects for zim.

=head1 DESCRIPTION

This package contains various utilities needed by the zim modules.

=head1 EXPORT

By default the methods C<__()> and C<__actions()> are exported.

=head1 METHODS

=over 4

=item C<find_translation()>

Tries to load messages for the current locale.

=cut

sub find_translation {
	my ($file) = xdg_data_files('zim', 'lingua', $Zim::LANG.'.pl');
	unless ($file) {
		$Zim::LANG =~ /(\w+?)_(\w+)/; # e.g. nl_NL
		($file) = xdg_data_files('zim', 'lingua', $1.'.pl');
	}
	return () unless $file;
	warn "# Loading translation from: $file\n";
	eval { %Translation = %{do $file} };
	warn $@ if $@;
	#use Data::Dumper; warn Dumper \%Translation;
}

sub import {
	my ($caller) = caller;
	no strict 'refs';
	*{$caller.'::__'} = \&__
		unless $caller->can('__');
	*{$caller.'::__actions'} = \&__actions
		unless $caller->can('__actions');
}

=item C<__(STRING, PARAM => VALUE, ...)>

Return a translation for STRING. Parameters can be placed in
the string using C<{PARAM}> syntax.

=item C<__actions(BLOCK)>

Wrapper for C<__()> that can be used to translate actions.

=cut

sub __ {
	my ($string, %arg) = @_;
	#warn "Translating: >>$string<<\n";
	$string = $Translation{$string} || $string;
	#warn "\t>>$string<<\n";
	$string =~ s/\{(\w+)\}/$arg{$1}/g;
	return $string;
}

sub __actions {
	my $block = shift;
	my $new;
	for my $l (split "\n", $block) {
		my @col = split /\t+/, $l;
		$col[$_] ||= '' for 0 .. 4;
		my $key = __(qq{$col[2]|$col[4]}); # avoid quote for xgettext
		@col[2,4] = split /\|/, $key, 2;
		$new .= join("\t", @col)."\n";
	}
	return $new;
}

package Zim::Utils::Unix;

=item C<run(PROGRAM, ARG, ...)>

Run background process (fork + exec).

=cut

sub run {
	my ($self, @args) = @_;
	warn "Executing: @args\n";
	unless (fork) { # child process
		exec @args;
		exit 1; # just to be sure
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2007 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

