package Zim::GUI::EquationEditor;

use strict;
use Zim::Utils;
use Zim::GUI::Component;
use Zim::Template;

our @ISA = qw/Zim::GUI::Component/;

our $VERSION = '0.23';

$Zim::GUI::DEFAULTS{eqe_latex}  ||=
	'latex -no-shell-escape -halt-on-error -output-directory %2 %1';
$Zim::GUI::DEFAULTS{eqe_dvipng} ||=
	'dvipng -q -bg Transparent -T tight -o %2 %1';


=head1 NAME

Zim::GUI::EquationEditor - Equation editor for Zim

=head1 DESCRIPTION

This module provides the equation editor dialog for Zim.
It requires latex to generate the actual equation image.

=head1 METHODS

=over 4

=item C<show(OBJECT)>

Open the equation editor.

=cut

sub init {
	my $self = shift;
	# init settings for applications to use
	for (qw/eqe_latex eqe_dvipng/) {
		$self->{app}{settings}{$_} ||= $Zim::GUI::DEFAULTS{$_};
		$self->{$_} = $self->{app}{settings}{$_};
		$self->{$_} .= ' %1' unless $self->{$_} =~ /\%1/;
	}
}

sub show {
	my ($self, $file) = @_;

	my $dialog = Gtk2::Dialog->new(
		__("Equation Editor"), $self->{app}{window}, #. dialog title
	       	[qw/modal destroy-with-parent/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(1);
	$dialog->set_default_size(400, 0);
	$dialog->vbox->set_border_width(12);
	$self->{dialog} = $dialog;

	my $win1 = Gtk2::Viewport->new;
	my $color = Gtk2::Gdk::Color->parse('#FFF');
	$win1->modify_bg('normal', $color); # set white background
	$win1->set_border_width(12);
	my $win2 = Gtk2::ScrolledWindow->new;
	$win2->set_policy('automatic', 'automatic');
	$win2->set_shadow_type('in');
	$dialog->vbox->add($win1);
	$dialog->vbox->add($win2);

	$self->{image_window} = $win1;
	$win1->set_no_show_all(1);

	my $text = Gtk2::TextView->new;
	$self->{buffer} = $text->get_buffer;
	$win2->add($text);

	my $hbox = Gtk2::HBox->new;
	$dialog->vbox->pack_start($hbox, 0,1,0);

	my $pbutton = $self->{app}->new_button(
		'gtk-refresh', __('_Preview') ); #. button
	$pbutton->signal_connect(clicked => \&compile, $self);
	$pbutton->set_sensitive(0);
	$hbox->pack_start($pbutton, 0,1,0);
	$self->{buffer}->signal_connect('modified-changed' =>
		sub { $pbutton->set_sensitive($self->{buffer}->get_modified) } );

	my $lbutton = $self->{app}->new_button(
		'gtk-file', __('View _Log') ); #. button
	$lbutton->signal_connect(clicked => \&view_log, $self);
	$hbox->pack_start($lbutton, 0,1,0);
	$lbutton->set_sensitive(0);
	$self->{lbutton} = $lbutton;

	$dialog->show_all;
	$win1->set_no_show_all(0);
	$self->{compiled} = 0;
	if ($file) {
		$self->_load_image($file);
		$file =~ s/\.png$/.tex/;
		$file = Zim::File::CheckOnWrite->new($file);
		$self->{buffer}->set_text( $file->read || '' );
	}
	while (my $response = $dialog->run) {
		if ($response eq 'help') {
			$self->{app}->ShowHelp(
				':zim:usage:plugins:EquationEditor' );
			next;
		}
		elsif ($response eq 'ok') {
			last if ! $self->{compiled}
			     && ! $self->{buffer}->get_modified;
				# Not edited => cancel
			my ($tex, $img) = $self->save($file);
			next unless $tex && $img;
			warn "Written:\n\t$tex\n\t$img\n";
			$img .= '?type=equation';
			my $pageview = $self->{app}->PageView;
			$file ? $self->{app}->Reload
			      : $pageview->InsertImage($img, 'NO_ASK') ;
			last;
		}
		else { last } # cancel
	}
	delete @$self{qw/dialog image_window buffer lbutton/};
	$dialog->destroy;
}

=item C<save()>

Compile equation and save it to the namespace for the current page.
Returns file names for tex file and image.

=cut

sub save {
	my ($self, $file) = @_;
	unless ($file) {
		## Generate file name	
		my $dir = $self->{app}{page}->properties->{base};
		$dir .= '/' . $self->{app}{page}->basename;
		# FIXME FIXME need proper interface to determine this dir
	
		my $name = "equation_01";
		my $i = 1;
		while (
			-e "$dir/$name.tex" or
			-e "$dir/$name.png"
		) {
			$name = sprintf 'equation_%02d', ++$i;
		}
		$file = Zim::File::CheckOnWrite->new("$dir/$name.tex");
	}
	my $image = $file->path;
	$image =~ s/\.tex$/.png/;
	$image = Zim::File->new($image);

	## Compile
       	$self->compile if $self->{buffer}->get_modified;
	my $tmpdir = File::Spec->tmpdir;
	my $tmpimage = Zim::File->new(
		"$tmpdir/zim-EquationEditor-$ENV{USER}.png" );
	my $ok = $tmpimage->exists;
	unless ($ok) {
		my $r = $self->{app}->prompt_question(
			__('Equation invalid'), 'warning', #. dialog title
			__('The equation failed to compile. Do you want to save anyway?'),
			[cancel => 'gtk-cancel'],
			[save   => 'gtk-save'  ]  );
		return unless $r eq 'save';
	}

	## Save text
	my $text = $self->{buffer}->get_text(
		$self->{buffer}->get_bounds(), 0 );
	$file->write($text);

	## Copy image or delete if broken
	if   ($ok) { $tmpimage->copy($image)          }
	else       { $image->remove if $image->exists }
	
	return ($file, $image);
}

=item C<compile()>

Compiles the equation in the tmp dir and shows a preview in the dialog.

=cut

sub compile {
	my $self = pop;

	my $tmpdir = File::Spec->tmpdir;
	my $file = "$tmpdir/zim-EquationEditor-$ENV{USER}.tex";

	my $mask = umask 0077; # set umask to user only

	## Write file using template
	my $text = $self->{buffer}->get_text(
		$self->{buffer}->get_bounds(), 0 );

	$text =~ s/\n+/\n/g; # remove empty lines
	$text =~ s/^\n*|\n*$//g;
	my $data = {equation => $text};
	my $template = Zim::Formats->lookup_template('latex' => '_Equation');
	die "Could not find template latex/_Equation" unless $template;
	$template = Zim::Template->new($template);

	warn "Writing $file\n";
	$template->process($data => $file);

	## Generate Image
	$file =~ /^(.*)\/(.*?)\.tex$/ or die "BUG: \"$file\" failed regex";
	my ($dir, $base) = ($1, $2);
	my ($latex, $dvipng) = @$self{qw/eqe_latex eqe_dvipng/};
	$latex =~ s/\%1/$base.tex/g;
	$latex =~ s/\%2/$dir/g;
	$dvipng =~ s/\%1/$dir\/$base.dvi/g;
	$dvipng =~ s/\%2/$dir\/$base.png/g;

	## Run commands
	my $ok = (system($latex) == 0 and system($dvipng) == 0);
	warn "Generating equation failed: $!\n" if $?;
	
	umask $mask; # reset old umask
	
	$file =~ s/\.tex$/.png/;
	Zim::File->remove($file) if ! $ok and -f $file;

	$self->_load_image($file);
	$self->{compiled} = 1;
	$self->{buffer}->set_modified(0);
	$self->{lbutton}->set_sensitive(1);
	return $ok;
}

sub _load_image {
	## Load image in preview window
	my ($self, $file) = @_;

	my $win = $self->{image_window};
	$win->remove($_) for $win->get_children;
	my $image = Gtk2::Image->new_from_file($file);
		# uses missing image icon if file does not exist
	$win->add($image);
	$win->show_all;
}

=item C<view_log()>

Open a dialog with a TextView showing the latex log file.

=cut

sub view_log {
	my $self = shift;
	my $tmpdir = File::Spec->tmpdir;
	my $log = Zim::File->new(
		"$tmpdir/zim-EquationEditor-$ENV{USER}.log" );
	my $dialog = Gtk2::Dialog->new(
		__("Equation Editor Log"), $self->{dialog}, #. dialog title
	       	[qw/modal destroy-with-parent/],
		'gtk-close' => 'ok',
	);
	$dialog->set_resizable(1);
	$dialog->set_default_size(600, 300);
	my $win = Gtk2::ScrolledWindow->new;
	$win->set_policy('automatic', 'automatic');
	$dialog->vbox->add($win);
	my $text = Gtk2::TextView->new;
	$text->get_buffer->set_text($log->read || '');
	$text->set_editable(0);
	$win->add($text);
	$dialog->show_all;
	$dialog->run;
	$dialog->destroy;
}

=item C<populate_popup(MENU, OBJECT)>

Add menu items for editing equations.

=cut

sub populate_popup {
	my ($self, $menu, $pixbuf) = @_;
	my $file = $pixbuf->{image_file};
	
	# Separator
	my $seperator = Gtk2::MenuItem->new();
	$seperator->show;
	$menu->prepend($seperator);

	# Edit object
	my $edit_item = Gtk2::MenuItem->new(__('_Edit Equation')); #. context menu
	$edit_item->show;
	$edit_item->signal_connect_swapped(activate =>
		sub { $self->show($file) }  );
	$menu->prepend($edit_item);
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2007 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

