#!perl -w
#
# EvmsTab -- encapsulate evms_query output
#   Copyright (C) 2005  Erik van Konijnenburg, Marco Amadori, Mattia Dongili
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
#
use strict;
use warnings;
use Base;
use Conf;
use EvmsDev;
package EvmsTab;

my $evmsTab = undef;
my $evmsVersion = undef;


sub init () {
	if (defined ($evmsTab)) {
		return;
	}

	$evmsTab = [];
	
	#
	# foreach volume read extended info
	# 
	# - do these when initialised
	# my $evmsVolName;
	# my $evmsDevno;
	# my @evmsDisks;
	# my @evmsRegions;
	# my @evmsSegments;

	my ($rc, $lines) = Base::runCmd (
		missingOk => 1,
		cmd => ['/usr/sbin/evms_query', 'volumes']);
	if (! defined ($lines)) {
		return;
	}

	#
	# Every line of 'evms_query volumes' output is the pathname
	# of a device that is delivered by EVMS.
	#
	for my $volume (@{$lines}) {
		if (! -e $volume) {
			#
			# Except that not all these devices actually exist:
			# no block device at the given path name,
			# devno is 0:0.  This happens for compatibility
			# devices; it could be operator error,
			# or done on purpose.
			# Anyway, we'll simply ignore these for now.
			#
			# Note how regular files or directories are not
			# skipped here; this will produce an error when
			# finding devno; appropriate for such odd input.
			#
			next;
		}
		my $evmsDevno = Base::devno ($volume);

		#
		# Find the underling disks, only devno.
		# This is the whole device, ignoring any raid,
		# partitioning or lvm that may be going on;
		# EVMS will reconstruct all of that automatically.
		# Relevant lines look like so:
		#	Device Number: 34,0
		# and we store devices like "34:0".
		#
		my @disks = ();
		my ($rc, $lines) = Base::runCmd (
			missingOk => 1,
			cmd => ['/usr/sbin/evms_query','disks','-i', $volume]);
		for my $line (@{$lines}) {
			if ($line =~ /^Device Number: (\d+),(\d+)$/) {
				my $devno = "$1:$2";
				push @disks, $devno;
			}
		}

		# Put all infos in here now:
		my $descr = EvmsDev->new (
				path => $volume,
				devno => $evmsDevno,
				disks =>  [ @disks ],
				);
		push @{$evmsTab}, $descr;
	}

	
	# first invocation worked, so this one is not optional
	# any longer.
	($rc, $lines) = Base::runCmd (
		cmd => ['/usr/sbin/evms_query', 'info']);
	for my $line (@{$lines}) {
		$evmsVersion = $1 if ($line =~ /^EVMS Version: (.*)/);
	}
}

sub all () {
	init;
	return $evmsTab;
}

sub findByDevno ($) {
	my ($devno) = @_;
	for my $ed (@{all()}) {
		if ($ed->devno() eq $devno) {
			return $ed;
		}
	}
	return undef;
}

sub findVersion () {
        init;
	return $evmsVersion;
}


1;
