/*=============================================================================
XMOTO

This file is part of XMOTO.

XMOTO is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

XMOTO is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with XMOTO; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
=============================================================================*/

#include "CheckWwwThread.h"
#include "GameText.h"
#include "Game.h"
#include "states/StateManager.h"
#include "helpers/Log.h"
#include "VFileIO.h"
#include "WWW.h"

#define DEFAULT_WEBHIGHSCORES_FILENAME "webhighscores.xml"

CheckWwwThread::CheckWwwThread(bool forceUpdate)
  : XMThread()
{
  m_forceUpdate   = forceUpdate;
  m_pWebRoom      = new WebRoom(this);
  m_pWebLevels    = new WebLevels(this);
}

CheckWwwThread::~CheckWwwThread()
{
  delete m_pWebRoom;
  delete m_pWebLevels;
}

void CheckWwwThread::updateWebHighscores(unsigned int i_number)
{
  Logger::Log("WWW: Checking for new highscores...");

  setSafeKill(true);
  m_pWebRoom->update(i_number);
  setSafeKill(false);
}

void CheckWwwThread::upgradeWebHighscores(unsigned int i_number)
{
  try {
    m_pWebRoom->upgrade(i_number, m_pDb);
    StateManager::instance()->sendAsynchronousMessage("HIGHSCORES_UPDATED");
  } catch (Exception& e) {
    Logger::Log("** Warning ** : Failed to analyse web-highscores file");   
  }
}

void CheckWwwThread::updateWebLevels()
{
  Logger::Log("WWW: Checking for new or updated levels...");

  /* Try download levels list */
  setSafeKill(true);
  m_pWebLevels->update(m_pDb);
  setSafeKill(false);

  if(m_pWebLevels->nbLevelsToGet(m_pDb) != 0){
    StateManager::instance()->sendAsynchronousMessage("NEW_LEVELS_TO_DOWNLOAD");
  } else {
    StateManager::instance()->sendAsynchronousMessage("NO_NEW_LEVELS_TO_DOWNLOAD");
  }
}

bool CheckWwwThread::isNeeded() {
  return XMSession::instance()->checkNewHighscoresAtStartup() || XMSession::instance()->checkNewLevelsAtStartup();
}

int CheckWwwThread::realThreadFunction()
{
  std::string webRoomUrl;
  std::string webRoomName;
  std::string webLevelsUrl;
  ProxySettings* pProxySettings;

  if(XMSession::instance()->www() == true){
    pProxySettings = XMSession::instance()->proxySettings();

    /* check highscores */
    try {
      setThreadCurrentOperation(GAMETEXT_DLHIGHSCORES);
      setThreadProgress(0);

      for(unsigned int i=0; i<XMSession::instance()->nbRoomsEnabled(); i++) {
	if(m_forceUpdate == true
	   || XMSession::instance()->checkNewHighscoresAtStartup() == true){
	  webRoomUrl   = GameApp::instance()->getWebRoomURL(i, m_pDb);
	  webRoomName  = GameApp::instance()->getWebRoomName(i, m_pDb);
	  m_pWebRoom->setWebsiteInfos(webRoomName, webRoomUrl, pProxySettings);

	  setThreadCurrentOperation(GAMETEXT_DLHIGHSCORES + std::string(" (") + webRoomName + ")");
	  updateWebHighscores(i);
	  upgradeWebHighscores(i);
	  setThreadProgress((100 * (i+1)) / XMSession::instance()->nbRoomsEnabled());
	}
      }
    } catch (Exception& e) {
      Logger::Log("** Warning ** : Failed to update web-highscores [%s]",e.getMsg().c_str());
      m_msg = GAMETEXT_FAILEDDLHIGHSCORES + std::string("\n") + GAMETEXT_CHECK_YOUR_WWW;
      return 1;
    }
    setThreadProgress(100);

    /* check levels */
    webLevelsUrl = XMSession::instance()->webLevelsUrl();
    m_pWebLevels->setWebsiteInfos(webLevelsUrl, pProxySettings);
    if(m_forceUpdate == true
       || XMSession::instance()->checkNewLevelsAtStartup() == true){
      try {
	setThreadCurrentOperation(GAMETEXT_DLLEVELSCHECK);
	setThreadProgress(0);

	updateWebLevels();

      } catch (Exception& e){
	Logger::Log("** Warning ** : Failed to update web-levels [%s]",e.getMsg().c_str());
	m_msg = GAMETEXT_FAILEDDLHIGHSCORES + std::string("\n") + GAMETEXT_CHECK_YOUR_WWW;
	return 1;
      }
    }
  }

  setThreadProgress(100);

  return 0;
}

std::string CheckWwwThread::getMsg() const
{
  return m_msg;
}

void CheckWwwThread::setTaskProgress(float p_percent)
{
  setThreadProgress((int)p_percent);
}
