/*  $Id: governor-plugin.c 15 2007-07-17 18:59:25Z jf $
 *
 *  Copyright (c) 2007 Jean-François Wauthy
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <hal/libhal.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4panel/xfce-panel-plugin.h>

#define DBUS_HAL_INTERFACE          "org.freedesktop.Hal"
#define DBUS_HAL_INTERFACE_CPUFREQ  "org.freedesktop.Hal.Device.CPUFreq"

static gboolean read_values_from_system (gpointer data);

typedef struct
{
  GtkWidget *icon_image;

  GtkTooltips *tooltips;
  GtkWidget *tooltipbox;

  gint cpufreq;
  gchar *current_governor;

  GtkWidget *governor_combo;
} GovernorData;

static GovernorData *governor = NULL;
static DBusGProxy *proxy = NULL;
static guint timeout_id = 0;

/* xfce4-cpufreq-plugin */
static gchar *
cpufreq_get_human_readable_freq (guint freq)
{
  guint div;
  gchar *readable_freq = NULL, *freq_unit = NULL;

  if (freq > 999999) {
    div = (1000 * 1000);
    freq_unit = g_strdup ("GHz");
  }
  else {
    div = 1000;
    freq_unit = g_strdup ("MHz");
  }

  if ((freq % div) == 0 || div == 1000)
    readable_freq = g_strdup_printf ("%d %s", (freq / div), freq_unit);
  else
    readable_freq = g_strdup_printf ("%3.2f %s", ((gfloat) freq / div), freq_unit);

  g_free (freq_unit);
  return readable_freq;
}

static void
values_updated ()
{
  gchar *tmpstr = NULL;

  if (governor->cpufreq > 0) {
    gchar *freq = cpufreq_get_human_readable_freq (governor->cpufreq);
    tmpstr = g_strconcat (_("Governor:"), " ", governor->current_governor, "\n", _("Frequency:"), " ", freq, NULL);
    g_free (freq);
  } else
    tmpstr = g_strconcat (_("Governor:"), " ", governor->current_governor, NULL);

  gtk_tooltips_set_tip (governor->tooltips, governor->tooltipbox, tmpstr, NULL);
  g_free (tmpstr);
}

static void
governor_dialog_response (GtkDialog * dialog, gint response, gpointer data)
{
  gchar *selected_governor = gtk_combo_box_get_active_text (GTK_COMBO_BOX (governor->governor_combo));

  if (g_ascii_strcasecmp (governor->current_governor, selected_governor) != 0) {
    GError *error = NULL;

    if (!dbus_g_proxy_call (proxy, "SetCPUFreqGovernor", &error, G_TYPE_STRING, selected_governor, G_TYPE_INVALID,
                            G_TYPE_INVALID)) {
      if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
        g_printerr ("Caught remote method exception %s: %s", dbus_g_error_get_name (error), error->message);
      else
        g_printerr ("Error: %s\n", error->message);
      g_error_free (error);
    }
    else
      read_values_from_system (NULL);
  }
  
  g_free (selected_governor);

  g_object_set_data (G_OBJECT (governor->tooltipbox), "governor-dialog", NULL);
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static gboolean
show_governor_dialog (GtkWidget * widget, GdkEventButton * ev, gpointer data)
{
  GtkWidget *dialog = NULL;
  GtkWidget *table;
  GtkWidget *label;
  gchar *freq = NULL, *str_freq = NULL;

  GError *error;
  gchar **name_list;
  gchar **name_list_ptr;
  gint i = 0;

  if (ev->button != 1)
    return FALSE;

  dialog = g_object_get_data (G_OBJECT (governor->tooltipbox), "governor-dialog");
  if (dialog) {
    gtk_widget_show_all (dialog);
    gtk_window_present (GTK_WINDOW (dialog));
    return TRUE;
  }

  /* Create the dialog window */
  dialog = xfce_titled_dialog_new_with_buttons (_("Governor Information"), NULL, GTK_DIALOG_NO_SEPARATOR,
                                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
  xfce_titled_dialog_set_subtitle (XFCE_TITLED_DIALOG (dialog), _("You can change the governor used by the system"));
  g_object_set_data (G_OBJECT (governor->tooltipbox), "governor-dialog", dialog);

  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_icon_name (GTK_WINDOW (dialog), "cpufreq-applet");
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

  table = gtk_table_new (2, 2, FALSE);

  label = gtk_label_new (_("Governor:"));
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 0);
  gtk_table_set_row_spacings (GTK_TABLE (table), 10);

  governor->governor_combo = gtk_combo_box_new_text ();
  gtk_table_attach (GTK_TABLE (table), governor->governor_combo, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 10, 0);

  label = gtk_label_new (_("Frequency:"));
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 10, 0);
  label = gtk_label_new ("");
  freq = cpufreq_get_human_readable_freq (governor->cpufreq);
  str_freq = g_strdup_printf ("<span weight='bold'>%s</span>", freq);
  gtk_label_set_markup (GTK_LABEL (label), str_freq);
  g_free (freq);
  g_free (str_freq);
  gtk_widget_set_sensitive (label, FALSE);
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 10, 0);
  
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, FALSE, 4);

  /* Fill governor list */
  error = NULL;
  if (!dbus_g_proxy_call (proxy, "GetCPUFreqAvailableGovernors", &error, G_TYPE_INVALID,
                          G_TYPE_STRV, &name_list, G_TYPE_INVALID)) {
    if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
      g_printerr ("Caught remote method exception %s: %s", dbus_g_error_get_name (error), error->message);
    else
      g_printerr ("Error: %s\n", error->message);
    g_error_free (error);
    return FALSE;
  }

  for (name_list_ptr = name_list; *name_list_ptr; name_list_ptr++) {
    gtk_combo_box_append_text (GTK_COMBO_BOX (governor->governor_combo), *name_list_ptr);
    if (g_ascii_strcasecmp (*name_list_ptr, governor->current_governor) == 0)
      gtk_combo_box_set_active (GTK_COMBO_BOX (governor->governor_combo), i);
    i++;
  }
  g_strfreev (name_list);

  g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (governor_dialog_response), governor);

  gtk_widget_show_all (dialog);

  return TRUE;
}

static void
governor_free_data (XfcePanelPlugin * plugin, gpointer data)
{
  g_source_remove (timeout_id);

  g_free (governor->current_governor);
  g_free (governor);
  g_object_unref (proxy);

  governor = NULL;
}

static void
governor_configure (XfcePanelPlugin * plugin, gpointer data)
{
  DBG ("TODO: show configure dialog");
}

static gboolean
governor_size_changed (XfcePanelPlugin * plugin, gint size, gpointer data)
{
  GdkScreen *screen = NULL;
  GtkIconTheme *icon_theme = NULL;
  GdkPixbuf *icon = NULL;

  screen = gtk_widget_get_screen (GTK_WIDGET (plugin));
  icon_theme = gtk_icon_theme_get_for_screen (screen);
  icon = gtk_icon_theme_load_icon (icon_theme, "cpufreq-applet", size, 0, NULL);

  if (G_UNLIKELY (icon == NULL))
    return FALSE;

  gtk_image_set_from_pixbuf (GTK_IMAGE (governor->icon_image), icon);
  g_object_unref (G_OBJECT (icon));

  return TRUE;
}

static void
governor_construct (XfcePanelPlugin * plugin)
{
  GdkScreen *screen = NULL;
  GtkIconTheme *icon_theme = NULL;
  GdkPixbuf *icon = NULL;

  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

  governor = g_new0 (GovernorData, 1);

  screen = gtk_widget_get_screen (GTK_WIDGET (plugin));
  icon_theme = gtk_icon_theme_get_for_screen (screen);
  icon = gtk_icon_theme_load_icon (icon_theme, "cpufreq-applet", xfce_panel_plugin_get_size (plugin), 0, NULL);

  governor->icon_image = gtk_image_new_from_pixbuf (icon);
  gtk_misc_set_alignment (GTK_MISC (governor->icon_image), 0.5, 1);
  gtk_widget_show (governor->icon_image);

  if (G_LIKELY (icon))
    g_object_unref (G_OBJECT (icon));

  governor->tooltips = gtk_tooltips_new ();
  g_object_ref (governor->tooltips);
  gtk_object_sink (GTK_OBJECT (governor->tooltips));

  governor->tooltipbox = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (governor->tooltipbox), governor->icon_image);
  gtk_widget_show_all (governor->tooltipbox);
  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (governor->tooltipbox), GTK_NO_WINDOW);

  xfce_panel_plugin_add_action_widget (plugin, governor->tooltipbox);
  gtk_container_add (GTK_CONTAINER (plugin), governor->tooltipbox);

  xfce_panel_plugin_menu_show_configure (plugin);

  read_values_from_system (NULL);       /* TODO timeout */
  timeout_id = g_timeout_add (1100, read_values_from_system, NULL);

  g_signal_connect (G_OBJECT (plugin), "free-data", G_CALLBACK (governor_free_data), governor);
  g_signal_connect (G_OBJECT (plugin), "configure-plugin", G_CALLBACK (governor_configure), governor);
  g_signal_connect (G_OBJECT (plugin), "size-changed", G_CALLBACK (governor_size_changed), governor);

  g_signal_connect (G_OBJECT (governor->tooltipbox), "button-press-event", G_CALLBACK (show_governor_dialog), governor);
}

static gboolean
read_values_from_system (gpointer data)
{
  FILE *freq_file = NULL;
  GError *error = NULL;
  gint cur_freq = -1;
  gchar *current_governor = NULL;
  gboolean updated = FALSE;

  if (governor == NULL)
    return FALSE;

#ifdef __linux__
  freq_file = fopen ("/sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq", "r");
  if (freq_file) {
    fscanf (freq_file, "%d", &cur_freq);
    fclose (freq_file);
  }
  if (cur_freq != governor->cpufreq) {
    governor->cpufreq = cur_freq;
    updated = TRUE;
  }
#else
  governor->cpufreq = -1;
#endif

  /* retrieve current governor name */
  if (!dbus_g_proxy_call (proxy, "GetCPUFreqGovernor", &error, G_TYPE_INVALID,
                          G_TYPE_STRING, &current_governor, G_TYPE_INVALID)) {
    if (error->domain == DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION)
      g_printerr ("Caught remote method exception %s: %s", dbus_g_error_get_name (error), error->message);
    else
      g_printerr ("Error: %s\n", error->message);
    g_error_free (error);
    exit (1);
  }
  else {
    if (governor->current_governor == NULL || g_ascii_strcasecmp (governor->current_governor, current_governor) != 0) {
      g_free (governor->current_governor);
      governor->current_governor = current_governor;
      updated = TRUE;
    }
    else
      g_free (current_governor);
  }

  /* TODO retrieve current CPU frequency */

  if (updated)
    values_updated ();

  return TRUE;
}

static gboolean
governor_check (GdkScreen * screen)
{
  DBusGConnection *connection;
  GError *error;

  LibHalContext *hal_ctx = NULL;
  char **devices = NULL;
  int n_devices = 0;
  gchar *device = NULL;

#ifdef __linux__
  if (!g_file_test ("/sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq", G_FILE_TEST_EXISTS))
    return FALSE;
#else
#warning showing the frequency of the CPU is only supported under Linux so far
#endif

  error = NULL;
  connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
  if (connection == NULL) {
    g_printerr ("Failed to open connection to bus: %s\n", error->message);
    g_error_free (error);
    return FALSE;
  }

  hal_ctx = libhal_ctx_new ();
  libhal_ctx_set_dbus_connection (hal_ctx, dbus_g_connection_get_connection (connection));
  devices = libhal_find_device_by_capability (hal_ctx, "cpufreq_control", &n_devices, NULL);

  if (n_devices == 0) {
    g_printerr ("Could not find a device able to handle cpufreq control");
    return FALSE;
  }

  /* Let's take the first device */
  device = g_strdup (devices[0]);
  libhal_free_string_array (devices);

  proxy = dbus_g_proxy_new_for_name (connection, DBUS_HAL_INTERFACE, device, DBUS_HAL_INTERFACE_CPUFREQ);
  g_free (device);

  return TRUE;
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL_WITH_CHECK (governor_construct, governor_check);
