/*
 * Copyright (C) 1999 John Meacham
 * released under the GPL version 2, see the file COPYING for details.
 * wmscope release 3
 * graphical sound display dockapp.
 * john@foo.net
 */

/* this is required for Solaris */
#define __EXTENSIONS__

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

/* for sound support either SOUND_OSS or SOUND_SOLARIS
 * must be defined. */

#if defined(SOUND_OSS)
#	if defined(__Linux__)
#		include <linux/soundcard.h>
#	elif defined(__FreeBSD__)
#		include <machine/soundcard.h>
#	else
#		include <sys/soundcard.h>
#	endif
#elif defined(SOUND_SOLARIS)
#	include <sys/audioio.h>
#elif !defined(SOUND_NONE)
#	warning no audio device setup.
#	define SOUND_NONE 1
#endif

#if defined(SOUND_SOLARIS) + defined(SOUND_OSS) + defined(SOUND_NONE) != 1
#error exactly one of SOUND_OSS,SOUND_SOLARIS, or SOUND_NONE should be defined.
#endif

#ifndef M_PI
#define M_PI 3.14159
#endif


#define SAMPLING_SPEED 8000

/* should be in stdbool.h but not all systems have that. */
typedef enum {false = 0, true} bool;



#include"tile.xpm"
#include"cover.xpm"

static Pixmap cover;
static Pixmap buffer;

static bool wmaker = false;
static bool ushape = false;
static const char *audio_device = "/dev/audio";
static const char *fg_color = "green";
static const char *bg_color = "#282828";
static float amp_factor = 1.0;

static bool sound_on = false;
static bool use_stdin = false;
static int audio_fd;

static Atom WM_DELETE_WINDOW;
static Display *disp;
static Window Win[2];
static Window root;
static GC gc;
static int activeWin;

static int sine[256];


static void createWin(Window *win) {
	XClassHint classHint;
	*win = XCreateSimpleWindow(disp, root, 10, 10, 64, 64,0,0,0);
	classHint.res_name = "wmscope";
	classHint.res_class = "WMScope";
	XSetClassHint(disp, *win, &classHint);
}

static unsigned long getColor(const char *colorName, float dim) {
	XColor Color;
	XWindowAttributes Attributes;
	XGetWindowAttributes(disp, root, &Attributes);
	Color.pixel = 0;
	XParseColor(disp, Attributes.colormap, colorName, &Color);
	Color.red   = (unsigned short)(Color.red   / dim);
	Color.blue  = (unsigned short)(Color.blue  / dim);
	Color.green = (unsigned short)(Color.green / dim);
	Color.flags = DoRed | DoGreen | DoBlue;
	XAllocColor(disp, Attributes.colormap, &Color);
	return Color.pixel;
}

#ifdef SOUND_NONE
static void open_audio(void) { sound_on = false; }
#endif

#ifdef SOUND_OSS

static void open_audio(void) {
	int arg = 0x00020008;
	int speed = SAMPLING_SPEED;
	int format = AFMT_U8;
	audio_fd = open(audio_device,O_RDONLY);
	if(audio_fd==-1) {
		perror(audio_device);
		sound_on = false;
		return;
	}
	if (ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &arg)) {
		perror(audio_device);
		sound_on = false;
		return;
	}
	ioctl(audio_fd, SNDCTL_DSP_SPEED, &speed);
	fprintf(stderr, "Sampling Rate: %i\n", speed);
	if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format) == -1) {
		perror("SNDCTL_DSP_SETFMT");
		sound_on = false;
		return;
	}
	if(format != AFMT_U8)
		fprintf(stderr,
			"Could not set %s to unsigned 8 bit mode\n"
			"attempting to ignore error.\n", audio_device);
	else
		fprintf(stderr, "Mode: 8 bit unsigned.\n");
}

#endif

#ifdef SOUND_SOLARIS

static void open_audio(void) {
	audio_info_t info;
	audio_fd = open(audio_device, O_RDONLY);
	if(audio_fd == -1) { perror(audio_device); sound_on = false; return; }
	ioctl(audio_fd, AUDIO_GETINFO, &info);
	info.play.sample_rate = info.record.sample_rate = SAMPLING_SPEED;
	info.play.channels = info.record.channels = 1;
	info.play.precision = info.record.precision = 16;
	info.play.encoding = info.record.encoding = AUDIO_ENCODING_LINEAR;
	ioctl(audio_fd, AUDIO_SETINFO, &info);
	ioctl(audio_fd, AUDIO_GETINFO, &info);
	printf("speed %i channels %i prec %i enc %i\n",
	       info.record.sample_rate, info.record.channels,
	       info.record.precision, info.record.encoding);

}

#endif


static void pressEvent(XButtonEvent *xev) {
	sound_on = !sound_on;
	if(use_stdin)
		return;
	if(!sound_on ) 
		close(audio_fd);
	else
		open_audio();
}

static void sinus_update(void) {
	int i;
	static int loc=0;
	static int loc1=0;
	XPoint pts[52];
	XCopyArea(disp, cover, buffer, gc, 5, 5, 54, 54, 5, 5);
	for(i = 0; i < 52; i++) {
		int y = sine[((loc+i))&0xFF];
		y += sine[((y+i))&0xFF];
		y += sine[((loc1+i))&0xFF];
		loc1++;
		y+=26;
		pts[i].x =i+6;
		pts[i].y =y+6;
	}
	XDrawLines(disp,buffer,gc,pts,52,CoordModeOrigin);
	loc++;
	XCopyArea(disp,buffer,Win[activeWin],gc,0,0,64,64,0,0);
}

#ifdef SOUND_SOLARIS

static void sound_update(void) {
	int i = 0;
	XPoint pts[52];
	signed short buf[52];
	XCopyArea(disp, cover, buffer, gc, 5, 5, 54, 54, 5, 5);
	while(i < 52 * 2)
		i += read(audio_fd, buf + (i/2) , 52 * 2 - i);
	for(i = 0; i < 52; i++)  {
		int y = ((int)buf[i]*((int)(26 * amp_factor)) / 32768) + 26;
		pts[i].x = i + 6;
		pts[i].y = y + 6;
	}
	XDrawLines(disp, buffer, gc, pts, 52, CoordModeOrigin);
	XCopyArea(disp, buffer, Win[activeWin], gc, 0, 0, 64, 64, 0, 0);
}

#endif

#ifdef SOUND_OSS
static void sound_update(void) {
	int i = 0;
	XPoint pts[52];
	unsigned char buf[52];
	XCopyArea(disp, cover, buffer, gc, 5, 5, 54, 54, 5, 5);
	while (i < 52)
	i += read(audio_fd, buf + i, 52 - i);
	for(i = 0;i<52;i++)  {
		int y = ((int)(buf[i]-128)*((int)(26*amp_factor))/128)+26;
		pts[i].x =i+6;
		pts[i].y =y+6;
	}
	XDrawLines(disp, buffer, gc, pts, 52, CoordModeOrigin);
	XCopyArea(disp, buffer, Win[activeWin], gc, 0, 0, 64, 64, 0, 0);
}

#endif

#ifdef SOUND_NONE
static void sound_update(void) { }
#endif


static void
createPixmap(char *data[], Pixmap *image,
	     Pixmap *mask, int *width, int *height) {
	XpmAttributes pixatt;
	XpmColorSymbol fg_colors[2]={
		{"fg_color_low", NULL, 0},
		{"bg_color",  NULL, 0},
	};
	fg_colors[0].pixel = getColor(fg_color,2.60);
	fg_colors[1].pixel = getColor(bg_color,1.00);
	pixatt.numsymbols = 2;
	pixatt.colorsymbols = fg_colors;
	pixatt.exactColors = false;
	pixatt.closeness = 40000;
	pixatt.valuemask = XpmColorSymbols | XpmExactColors
		| XpmCloseness | XpmSize;
	XpmCreatePixmapFromData(disp, root, data, image, mask, &pixatt);
	if(width!=NULL)
		*width=pixatt.width;
	if(height!=NULL)
		*height=pixatt.height;
}

static void xwin_init(int argc, char *argv[]) {
	XWMHints hints;
	XSizeHints shints;

	disp = XOpenDisplay(NULL);
	if(!disp) {
		fprintf(stderr, "Cannot open X display.\n");
		exit(1);
	}
	WM_DELETE_WINDOW = XInternAtom(disp, "WM_DELETE_WINDOW", False);
	root = DefaultRootWindow(disp);
	createWin(&Win[0]);
	hints.window_group = Win[0];
	shints.min_width=64;
	shints.min_height=64;
	shints.max_width=64;
	shints.max_height=64;
	shints.x=0;
	shints.y=0;
	if(wmaker) {
		createWin(&Win[1]);
		hints.initial_state = WithdrawnState;
		hints.icon_window = Win[1];
		hints.flags = WindowGroupHint | StateHint | IconWindowHint;
		shints.flags = PMinSize | PMaxSize | PPosition;
		activeWin=1;
	} else {
		hints.initial_state = NormalState;
		hints.flags = WindowGroupHint | StateHint;
		shints.flags = PMinSize | PMaxSize;
		activeWin=0;
	}
	XSetWMHints(disp, Win[0], &hints);
	XSetWMNormalHints(disp, Win[0], &shints);
	XSetCommand(disp, Win[0], argv, argc);
	XStoreName(disp, Win[0], "wmscope");
	XSetIconName(disp, Win[0], "wmscope");
	XSetWMProtocols(disp, Win[activeWin], &WM_DELETE_WINDOW, 1);
}

void xwin_fini(void) {
	XCloseDisplay(disp);
}

static void useage(char *progname) {
	fprintf(stderr,
	"wmscope release 3 (C) 1999 John Meacham (john@foo.net).\n"
	"distributed under the GPL.\n\n"
	"usage:\n\n   %s [options]\n\noptions:\n\n"
	"  -h               display this help screen\n"
	"  -w               WindowMaker dockapp mode\n"
	"  -s               shaped window mode\n"
	"  -f fg_color	    set foreground color\n"
	"  -b bg_color      set background color\n"
	"  -a amp_factor    amplification factor (may cause distortion)\n"
	"  -d audio_dev     use specified audio device\n"
	"  -c               read sound data from stdin. (PCM 8bit unsigned)\n"
	, progname);
	exit(0);
}

static void parse_args(int argc, char *argv[]) {
	int i;
	for(i = 1; i < argc; i++) {
		if(argv[i][0] != '-' || argv[i][2] != '\0')
			useage(argv[0]);
		switch(argv[i][1]) {
			case 'w': wmaker = true; break;
			case 's': ushape = true; break;
			case 'c': use_stdin = true;
				  audio_fd = STDIN_FILENO;
				  break;
			case 'd': if(++i < argc)
					  audio_device = argv[i];
				  else
					  useage(argv[0]);
				  break;
			case 'a': if(++i < argc)
					  amp_factor = atof(argv[i]);
				  else
					  useage(argv[0]);
				  break;
			case 'f': if(++i < argc)
					  fg_color = argv[i];
				  else
					  useage(argv[0]);
				  break;
			case 'b': if(++i < argc)
					  bg_color = argv[i];
				  else
					  useage(argv[0]);
				  break;
			default: useage(argv[0]); break;
		}
	}
}

int main(int argc, char *argv[]) {
	int i;
	bool finished = false;
	XGCValues gcv;
	unsigned long gcm;
	XEvent event;
	Pixmap dmsk;

	parse_args(argc, argv);
	xwin_init(argc, argv);

	for(i = 0; i < 256; i++)
		sine[i] = (int)(sin(5.0 * 2.0 * M_PI * i / 256.0) * 10.0);

	createPixmap(cover_xpm, &cover, &dmsk, NULL, NULL);
	buffer = XCreatePixmap(disp, root, 64, 64,
			       DefaultDepth(disp,DefaultScreen(disp)));
	gcm = GCForeground | GCBackground | GCGraphicsExposures;
	gcv.graphics_exposures = False;
	gc = XCreateGC(disp, root, gcm, &gcv);
	XSetForeground(disp, gc, getColor(fg_color,1.00));
	XSetClipOrigin(disp, gc, 0, 0);
	if(!(wmaker || ushape)) {
		Pixmap tile;
		createPixmap(tile_xpm, &tile, NULL, NULL, NULL);
		XSetWindowBackgroundPixmap(disp, Win[activeWin], tile);
		XSetClipMask(disp, gc, dmsk);
	} else {
		XShapeCombineMask(disp, Win[activeWin],
				  ShapeBounding, 0, 0, dmsk, ShapeSet);
		XSetClipMask(disp, gc, None);
	}

	sinus_update();
	XSelectInput(disp, Win[activeWin], ButtonPress);
	XMapWindow(disp, Win[0]);

	while(!finished) {
		while(XPending(disp)) {
			XNextEvent(disp,&event);
			if(event.type == ButtonPress)
				pressEvent(&event.xbutton);
			else if(event.type == ClientMessage
				&& (Atom)event.xclient.data.l[0]
				== WM_DELETE_WINDOW)
				finished = true;
		}
		if(!sound_on) {
			sinus_update();
			XFlush(disp);
			usleep(10);
		} else {
			sound_update();
			XFlush(disp);
		}
	}
	xwin_fini();
	return 0;
}
