/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef _WF_METAPORT_H
#define _WF_METAPORT_H

#include <list>
#include <string>

#include "wfport_base.h"
#include "wfport.h"
#include "wfport_range.h"

/**
 * metaport
 */
class wf_metaport;

class wf_metaport_elem {
 public:
  wf_metaport_elem();
  wf_metaport_elem(const wf_metaport& metaport);
  wf_metaport_elem(const wf_port& port);
  wf_metaport_elem(const wf_port_range& port_range);
  virtual ~wf_metaport_elem();
  wf_metaport_elem(const wf_metaport_elem&);
  wf_metaport_elem& operator=(const wf_metaport_elem&);

  bool set(const wf_metaport& metaport);
  bool set(const wf_port& port);
  bool set(const wf_port_range& port_range);

  const wf_metaport* get_metaport() const;
  const wf_port* get_port() const;
  const wf_port_range* get_port_range() const;
  wf_metaport* get_metaport();
  wf_port* get_port();
  wf_port_range* get_port_range();

  ostream& print(ostream& os) const;
  ostream& output_xml(ostream& os, unsigned int indent_level = 0) const;
  ostream& debugprint(ostream& os) const;

  enum metaport_elem_type gettype() const {
    return (port_base != NULL) ? port_base->gettype() : WF_METAPORT_NONE;
  }

  friend ostream& operator<<(ostream& os, const wf_metaport_elem& elem);
  friend bool operator==(const wf_metaport_elem& elem1,
			 const wf_metaport_elem& elem2);

 private:
  wf_port_base* port_base;
};

class wf_metaport : public wf_port_base {
 public:
  wf_metaport() :
    name(),
    comment(),
    elems() {}
  virtual ~wf_metaport() {}
  // wf_metaport(const wf_metaport& metaport) {} implicit copy constructor
  virtual wf_port_base* VClone() const;

  bool add(const wf_metaport& metaport);
  bool add(const wf_metaport_elem& elem);
  void del(const wf_metaport_elem& elem);
  void clear();

  void substitute(const string& substname, const wf_metaport& metaport);

  bool isreference() const;
  unsigned int count_ports() const;
  bool empty() const { return elems.empty(); }

  virtual enum metaport_elem_type gettype() const { return WF_METAPORT_METAPORT; }
  virtual ostream& print(ostream& os) const;
  virtual ostream& print_value(ostream& os) const;
  virtual ostream& output_xml(ostream& os, unsigned int indent_level = 0) const;
  ostream& debugprint(ostream& os) const;

  friend ostream& operator<<(ostream& os, const wf_metaport& metaport);

 public:
  string name;
  string comment;
  list<wf_metaport_elem> elems;
};


inline wf_port_base*
wf_metaport::VClone() const {
  return new wf_metaport(*this);
}

inline const wf_metaport*
wf_metaport_elem::get_metaport() const {
  return (wf_metaport*)port_base;
}

inline const wf_port*
wf_metaport_elem::get_port() const {
  return (wf_port*)port_base;
}

inline const wf_port_range*
wf_metaport_elem::get_port_range() const {
  return (wf_port_range*)port_base;
}

inline wf_metaport*
wf_metaport_elem::get_metaport() {
  return (wf_metaport*)port_base;
}

inline wf_port*
wf_metaport_elem::get_port() {
  return (wf_port*)port_base;
}

inline wf_port_range*
wf_metaport_elem::get_port_range() {
  return (wf_port_range*)port_base;
}

#endif
