#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>

#ifdef HAVE_IMLIB
# include <Imlib.h>
#endif	/* HAVE_IMLIB */

#include <GL/gl.h>
#include <GL/glu.h>

#include <gtk/gtk.h>   
#include <gtkgl/gtkglarea.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"
#include "../include/tga.h"

#include "msglist.h"
#include "v3dtex.h"
#include "guiutils.h"
#include "cdialog.h"

#include "editor.h"

#include "texbrowser.h"
#include "texbrowsercb.h"
#include "texbrowserpv.h"

#include "vma.h"
#include "vmautils.h"

#include "config.h"
#include "messages.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_zoom_in_20x20.xpm"
#include "images/icon_zoom_onetoone_20x20.xpm"
#include "images/icon_zoom_out_20x20.xpm"
#include "images/icon_zoom_tofit_20x20.xpm"
#include "images/icon_properties_20x20.xpm"


static void TexBrowserPreviewCreateCursors(ma_texture_browser_struct *tb);
GtkWidget *TexBrowserPreviewCreate(ma_texture_browser_struct *tb);
void TexBrowserPreviewDestroy(ma_texture_browser_struct *tb);

void TexBrowserPreviewDraw(ma_texture_browser_struct *tb);
void TexBrowserPreviewLoad(
	ma_texture_browser_struct *tb, 	gint i
);

void TexBrowserPreviewUnload(ma_texture_browser_struct *tb);

void TexBrowserPreviewZoomRect(
	ma_texture_browser_struct *tb,
	gint x0w, gint y0w,
	gint x1w, gint y1w
);


static gint gl_attributes_list[] = {
	GDK_GL_RGBA,
	GDK_GL_DOUBLEBUFFER,
	GDK_GL_DEPTH_SIZE,      1,
	GDK_GL_NONE 
};


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Creates cursors for the preview window on the texture browser.
 *
 *	This function should be called from TexBrowserPreviewCreate().
 */
static void TexBrowserPreviewCreateCursors(ma_texture_browser_struct *tb)
{
	/* Remember that bytes are swapped. */
	static guint8 select_cd[] = {
0x00, 0x00,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x3e, 0x7c,
0x3e, 0x7c,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x00, 0x00 
	};
	static guint8 select_mask_cd[] = {   
0x80, 0x01,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xfe, 0x7f,
0x7f, 0xfe,
0x7f, 0xfe,
0xfe, 0x7f,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0x80, 0x01 
};
	gint select_cd_w = 16, select_cd_h = 16;

	static guint8 translate_cd[] = {
0x00, 0x00,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x84, 0x21,
0xfe, 0x7f,
0xfe, 0x7f,
0x84, 0x21,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x00, 0x00 
	}; 
	static guint8 translate_mask_cd[] = {
0x80, 0x01,
0xc0, 0x03,
0xe0, 0x07,
0xf0, 0x0f,
0xc8, 0x13,
0xcc, 0x33,
0xfe, 0x7f,
0xff, 0xff,
0xff, 0xff,
0xfe, 0x7f,
0xcc, 0x33,
0xc8, 0x13,
0xf0, 0x0f,
0xe0, 0x07,
0xc0, 0x03,
0x80, 0x01 
	}; 
	gint translate_cd_w = 16, translate_cd_h = 16;

	static guint8 zoom_cd[] = {
0x00, 0x00,
0xfe, 0x7f,
0x06, 0x60,
0x0a, 0x50,
0x12, 0x48,
0xe2, 0x47,
0x22, 0x44,
0x22, 0x44,
0x22, 0x44,
0x22, 0x44,
0xe2, 0x47,
0x12, 0x48,
0x0a, 0x50,
0x06, 0x60,
0xfe, 0x7f,
0x00, 0x00 
	}; 
	static guint8 zoom_mask_cd[] = { 
0xff, 0xff,
0xff, 0xff,
0xff, 0xff,
0x3f, 0xfc,
0xff, 0xff,
0xff, 0xff,
0xf7, 0xef,
0x77, 0xee,
0x77, 0xee,   
0xf7, 0xef,
0xff, 0xff,
0xff, 0xff,
0x3f, 0xfc,
0xff, 0xff,
0xff, 0xff,
0xff, 0xff 
	};
	gint zoom_cd_w = 16, zoom_cd_h = 16;

	guint8 *cd, *mask_cd;
	gint cd_w, cd_h;
	gdouble cd_x, cd_y;
	GdkCursor *cursor;
	GdkPixmap *source, *mask;
	GdkColor fg = {
	    0,
	    (u_int16_t)(-1),
	    (u_int16_t)(-1),
	    (u_int16_t)(-1)
	};
	GdkColor bg = {
	    0, 0, 0, 0
	};

	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return;

	pv = &tb->preview;

#define DO_CREATE_CURSOR        \
{ \
 source = gdk_bitmap_create_from_data( \
  NULL, cd, cd_w, cd_h \
 ); \
 mask = gdk_bitmap_create_from_data( \
  NULL, mask_cd, cd_w, cd_h \
 ); \
 cursor = gdk_cursor_new_from_pixmap( \
  source, mask, &fg, &bg, \
  (gint)(cd_x * cd_w), (gint)(cd_y * cd_h) \
 ); \
 gdk_pixmap_unref(source); \
 gdk_pixmap_unref(mask); \
}

	cd = translate_cd;
	mask_cd = translate_mask_cd;
	cd_w = translate_cd_w;
	cd_h = translate_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	pv->translate_cur = cursor;

	cd = zoom_cd;
	mask_cd = zoom_mask_cd;
	cd_w = zoom_cd_w;
	cd_h = zoom_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	pv->zoom_cur = cursor;

	cd = select_cd;
	mask_cd = select_mask_cd;
	cd_w = select_cd_w;
	cd_h = select_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	pv->zoom_rect_cur = cursor;

#undef DO_CREATE_CURSOR

	return;
}

/*
 *	Creates a new preview window on the given texture
 *	browser, returns toplevel (table) of the preview window.
 *
 *	This function should be called from TexBrowserCreate().
 */
GtkWidget *TexBrowserPreviewCreate(ma_texture_browser_struct *tb)
{
	const gchar *msglist[] = VMA_MSGLIST_TEXBROWSER_TOOLTIPS;
	GtkWidget *w, *fw, *parent, *parent2, *parent3, *menu;
	GtkAdjustment *adj;
	gint accel_key;
	gpointer accel_group;
	guint accel_mods;
	guint8 **icon;
	const gchar *label = NULL;
	gpointer mclient_data;
	void (*func_cb)(GtkWidget *w, gpointer);
	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return(NULL);

	pv = &tb->preview;


	/* Reset values. */
	pv->initialized = TRUE;
	pv->realized = FALSE;

	/* Create toplevel table. */
	w = gtk_table_new(2, 1, FALSE);
	pv->toplevel = w;
	parent = w;

	/* Frame to hold table containing zoom widgets. */
	w = gtk_frame_new(NULL);
	gtk_table_attach(GTK_TABLE(parent), w,
	    0, 1,
	    0, 1,
	    GTK_EXPAND | GTK_FILL,
	    0,  
	    0, 0
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_show(w);
	parent2 = w;

	/* Table holding zoom widgets. */
	w = gtk_table_new(1, 6, FALSE);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Frame to hold zoom label. */
	w = gtk_frame_new(NULL);
	gtk_table_attach(GTK_TABLE(parent2), w,
	    0, 1,
	    0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    0,
	    2, 2
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(w), 1);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("");
	pv->zoom_label = w;
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);

	/* Zoom buttons. */
	w = GUIButtonPixmap((guint8 **)icon_zoom_in_20x20_xpm);
	pv->zoom_in_btn = w;
	gtk_table_attach(GTK_TABLE(parent2), w,
	    1, 2,
	    0, 1,
	    0,
	    0,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(TexBrowserZoomInPressedCB),
	    tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(TexBrowserZoomInReleasedCB),
	    tb
	);
	gtk_widget_set_sensitive(w, FALSE);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_TEXBROWSER_PV_ZOOM_IN
	    )
	);
	gtk_widget_show(w);

	w = GUIButtonPixmap((guint8 **)icon_zoom_out_20x20_xpm);
	pv->zoom_out_btn = w;
	gtk_table_attach(GTK_TABLE(parent2), w,
	    2, 3,
	    0, 1,
	    0,
	    0,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",     
	    GTK_SIGNAL_FUNC(TexBrowserZoomOutPressedCB),
	    tb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(TexBrowserZoomOutReleasedCB),
	    tb
	);
	gtk_widget_set_sensitive(w, FALSE);
	GUISetWidgetTip(
	    w,   
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_TEXBROWSER_PV_ZOOM_OUT
	    )
	);
	gtk_widget_show(w);

	/* Separator. */
	w = gtk_label_new("");
	gtk_table_attach(GTK_TABLE(parent2), w,
	    3, 4,
	    0, 1,
	    0,
	    0,
	    0, 0
	);
	gtk_widget_set_usize(w, 2, 2);
	gtk_widget_show(w);

	w = GUIButtonPixmap((guint8 **)icon_zoom_tofit_20x20_xpm); 
	pv->zoom_to_fit_btn = w;
	gtk_table_attach(GTK_TABLE(parent2), w,
	    4, 5,
	    0, 1,
	    0,
	    0,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(TexBrowserZoomToFitCB),
	    tb
	);
	gtk_widget_set_sensitive(w, FALSE);
	GUISetWidgetTip(
	    w,   
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_TEXBROWSER_PV_ZOOM_TO_FIT
	    ) 
	);
	gtk_widget_show(w);

	w = GUIButtonPixmap((guint8 **)icon_zoom_onetoone_20x20_xpm);
	pv->zoom_one_to_one_btn = w;
	gtk_table_attach(GTK_TABLE(parent2), w,
	    5, 6,
	    0, 1,
	    0,
	    0,
	    0, 0
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(TexBrowserZoomOneToOneCB),
	    tb
	);
	gtk_widget_set_sensitive(w, FALSE);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_TEXBROWSER_PV_ZOOM_ONE_TO_ONE
	    )
	);
	gtk_widget_show(w);


	/* Table to hold preview. */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_attach(GTK_TABLE(parent), w,
	    0, 1,
	    1, 2,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    0, 0
	);
	gtk_widget_show(w);
	parent2 = w;

	/* Frame widget to hold preview glarea. */
	w = gtk_frame_new(NULL);
	gtk_table_attach(GTK_TABLE(parent2), w,
	    0, 1,
	    0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    0, 0
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_widget_show(w);
	parent3 = w;

	/* Preview glarea widget. */
	w = gtk_gl_area_new(gl_attributes_list);
	pv->preview = w;
	if(w != NULL)
	{
	    gtk_widget_add_events(
		w,
		GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK |
		GDK_KEY_RELEASE_MASK | GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(TexBrowserPreviewEventCB),
		tb
	    );
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_show(w);
	}

	/* Scroll bar adjustments. */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0,		/* Current value. */
	    0, 0,	/* Lower, upper. */
	    10, 50, 50	/* Step inc, page inc, and page size. */
	);
	pv->preview_vadj = adj;
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(TexBrowserPreviewScrollCB), tb
	);

	adj = (GtkAdjustment *)gtk_adjustment_new( 
	    0,          /* Current value. */
	    0, 0,       /* Lower, upper. */
	    10, 50, 50  /* Step inc, page inc, and page size. */
	);
	pv->preview_hadj = adj;
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(TexBrowserPreviewScrollCB), tb
	);

	/* Vertical scroll bar. */
	w = gtk_vscrollbar_new(pv->preview_vadj);
	pv->preview_vsb = w;
	if(w != NULL)
	{
	    gtk_table_attach(GTK_TABLE(parent2), w,
		1, 2,
		0, 1,
		0,
		GTK_FILL,
		0, 0
	    );
	    gtk_widget_show(w);
	}
	/* Horizontal scroll bar. */
	w = gtk_hscrollbar_new(pv->preview_hadj);
	pv->preview_hsb = w;
	if(w != NULL)
	{
	    gtk_table_attach(GTK_TABLE(parent2), w,
		0, 1,
		1, 2,
		GTK_FILL,
		0,
		0, 0
	    );
	    gtk_widget_show(w);
	}


	/* Right click menu for preview glarea. */
	menu = (GtkWidget *)GUIMenuCreate();
	pv->preview_menu = menu;
	accel_group = NULL;
	mclient_data = tb;

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}

#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

	icon = (guint8 **)icon_zoom_tofit_20x20_xpm;
	label = "Zoom To Fit";
	accel_key = 0;
	accel_mods = 0;
	func_cb = TexBrowserZoomToFitCB;
	DO_ADD_MENU_ITEM_LABEL
	pv->preview_zoom_to_fit_mi = w;

	icon = (guint8 **)icon_zoom_onetoone_20x20_xpm;
	label = "One To One"; 
	accel_key = 0;
	accel_mods = 0;
	func_cb = TexBrowserZoomOneToOneCB;
	DO_ADD_MENU_ITEM_LABEL
	pv->preview_zoom_one_to_one_mi = w;

	DO_ADD_MENU_SEP

	icon = (guint8 **)icon_properties_20x20_xpm;
	label = "Properties...";
	accel_key = 0;
	accel_mods = 0;
	func_cb = TexBrowserEditPropertiesCB;
	DO_ADD_MENU_ITEM_LABEL
	pv->preview_properties_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL 
#undef DO_ADD_MENU_SEP


	/* Create cursors. */
	TexBrowserPreviewCreateCursors(tb);

	pv->flags = 0;

	pv->zoom_coeff_wtod = 1.0;
	pv->zoom_min_wtod = 0.05;
	pv->zoom_max_wtod = 16.0;
	pv->xw = 0;
	pv->yw = 0;
	pv->ww = 0;
	pv->hw = 0;

	pv->drag_state = MA_TEXBROWSER_DRAG_NONE;

	pv->tex = NULL;
	pv->tex_width = 0;
	pv->tex_height = 0;

	return(pv->toplevel);
}


/*
 *	Destroys the preview window on the texture browser.
 */
void TexBrowserPreviewDestroy(ma_texture_browser_struct *tb) 
{
	GtkWidget *w;
	GdkCursor *cur;
	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return;

	pv = &tb->preview;

#define DO_DESTROY_WIDGET	\
{ \
 if(w != NULL) \
  gtk_widget_destroy(w); \
}
#define DO_DESTROY_CURSOR       \
{ \
 if(cur != NULL) \
  gdk_cursor_destroy(cur); \
}

	if(pv->initialized)
	{
	    /* Unload any data. */
	    TexBrowserPreviewUnload(tb);

	    w = pv->preview;
	    DO_DESTROY_WIDGET

	    cur = pv->translate_cur;
	    DO_DESTROY_CURSOR

	    cur = pv->zoom_cur;
	    DO_DESTROY_CURSOR

	    cur = pv->zoom_rect_cur;
	    DO_DESTROY_CURSOR

	    w = pv->toplevel;
	    DO_DESTROY_WIDGET
	}

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR

	memset(pv, 0x00, sizeof(ma_texbrowser_preview_struct));

	return;
}

/*
 *	Redraws the preview glarea widget on the texture browser.
 */
void TexBrowserPreviewDraw(ma_texture_browser_struct *tb)
{
	static gbool reenterant = FALSE;
	GtkWidget *w, *label;
	gint ww, wh, svalue, slower, supper, spage;
	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return;

	if(!tb->map_state)
	    return;

	pv = &tb->preview;
	if(!pv->realized)
	    return;

	w = pv->preview;
	if(w == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	label = pv->zoom_label;


	/* Sanitize scroll positions. */
	svalue = pv->xw * pv->zoom_coeff_wtod;
	slower = 0;
	supper = pv->tex_width * pv->zoom_coeff_wtod;
	spage = pv->ww;
	if((svalue > (supper - spage)) &&
	   ((supper - slower) > spage)
	)
	{
	    svalue = supper - spage;
	    if(pv->zoom_coeff_wtod > 0.0)
		pv->xw = svalue / pv->zoom_coeff_wtod;
	}

	svalue = pv->yw * pv->zoom_coeff_wtod;
	slower = 0;
	supper = pv->tex_height * pv->zoom_coeff_wtod;
	spage = pv->hw;
	if((svalue > (supper - spage)) &&
	   ((supper - slower) > spage)
	)
	{
	    svalue = supper - spage;
	    if(pv->zoom_coeff_wtod > 0.0)
		pv->yw = svalue / pv->zoom_coeff_wtod;
	}


	/* Make the glarea widget as the current OpenGL rendering
	 * context.
	 */
	if(TexBrowserPreviewEnableContext(tb))
	{
	    reenterant = FALSE;
	    return;
	}

	/* Get size of view in pixels. */
	ww = w->allocation.width;
	wh = w->allocation.height;


	/* Set up camera viewport. */
	glViewport(0, 0, ww, wh);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	/* Go to 2D and make units the same as the size of the window. */
	gluOrtho2D(0, ww, wh, 0);
	glMatrixMode(GL_MODELVIEW);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	/* Update states. */
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glDisable(GL_LIGHT0);
	glDisable(GL_TEXTURE_1D);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_COLOR_MATERIAL);
	glDisable(GL_TEXTURE_3D);

	/* Clear background. */
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glClearDepth(1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* Draw texture. */
	if((pv->tex != NULL) && (pv->tex_width > 0) &&
	   (pv->tex_height > 0)
	)
	{
	    gdouble xd = pv->xw * pv->zoom_coeff_wtod;
	    gdouble yd = pv->yw * pv->zoom_coeff_wtod;
	    gdouble wd = pv->tex_width * pv->zoom_coeff_wtod;
	    gdouble hd = pv->tex_height * pv->zoom_coeff_wtod;

	    /* Center if image data displayed is less than viewport
	     * dimension.
	     */
	    if(wd < ww)
		xd = -((ww - wd) / 2);
	    if(hd < wh)
		yd = -((wh - hd) / 2);

	    V3DTextureSelect(pv->tex);
	    glColor4f(1.0, 1.0, 1.0, 1.0);
	    glBegin(GL_QUADS);
	    {
		glTexCoord2f(0.0, 0.0);
		glVertex2f((GLfloat)(-xd), (GLfloat)(-yd));
		glTexCoord2f(0.0, 1.0);
		glVertex2f((GLfloat)(-xd), (GLfloat)(-yd + wd));
		glTexCoord2f(1.0, 1.0);
		glVertex2f((GLfloat)(-xd + wd), (GLfloat)(-yd + hd));
		glTexCoord2f(1.0, 0.0);
		glVertex2f((GLfloat)(-xd + wd), (GLfloat)(-yd));
	    }
	    glEnd();
	}

	/* If dragging zoom rect, then draw rectangular outline. */
	if((pv->drag_state == MA_TEXBROWSER_DRAG_ZOOM_RECT) &&
	   (pv->flags & (MA_TEXBROWSER_FLAG_BUTTON1 |
			 MA_TEXBROWSER_FLAG_BUTTON2 |
			 MA_TEXBROWSER_FLAG_BUTTON3)
	   )
	)
	{
	    glDisable(GL_TEXTURE_2D);
/*
	    glEnable(GL_BLEND);
	    glBlendFunc(???);
 */
	    glColor4f(1.0, 1.0, 1.0, 1.0);

	    glBegin(GL_LINE_LOOP);
	    {
		glVertex2f(
		    (GLfloat)(pv->zoom_rect_start_xw),
		    (GLfloat)(pv->zoom_rect_start_yw)
		);
		glVertex2f(
		    (GLfloat)(pv->zoom_rect_start_xw),
		    (GLfloat)(pv->last_yw)
		);
		glVertex2f(
		    (GLfloat)(pv->last_xw),
		    (GLfloat)(pv->last_yw)
		);
		glVertex2f(
		    (GLfloat)(pv->last_xw),
		    (GLfloat)(pv->zoom_rect_start_yw)
		);
	    }
	    glEnd();

	    glDisable(GL_BLEND);
	}


	/* Make glarea rendered buffer active. */
	gtk_gl_area_swapbuffers(GTK_GL_AREA(w));

	/* Check for GL errors. */
	if(1)
	{
	    GLenum error_code = glGetError();
	    if(error_code != GL_NO_ERROR)
		VMAReportGLError(NULL, (int)error_code);
	}


	/* Draw zoom label. */
	if(label != NULL)
	{
	    char text[256];

	    if(pv->tex == NULL)
		*text = '\0';
	    else
		sprintf(text, "Zoom %.0f%%",
		    (double)(pv->zoom_coeff_wtod * 100)
		);

	    gtk_label_set_text(GTK_LABEL(label), text);
	}

	reenterant = FALSE;
}


/*
 *	Loads image data for the specified texture i which corresponds to
 *	a row number on the texture browser's textures list.
 *
 *	Any previously loaded data will be unloaded first.
 *	If i is -1 or reffers to a texture that cannot be loaded then
 *	no texture data will be loaded into the preview.
 */
void TexBrowserPreviewLoad(
	ma_texture_browser_struct *tb,
	gint i		/* Row number on texture browser's texture list. */
)
{
	gint tex_num;
	GtkWidget *w;
	gchar *base_dir, *path, *strptr;
	guint8 *tex_data = NULL;
	gint tex_width = 0, tex_height = 0;
	ma_editor_struct *editor;
	v3d_texture_ref_struct *t;
	ma_texbrowser_preview_struct *pv;
	gchar tmp_path[PATH_MAX + NAME_MAX];
	struct stat stat_buf;


	if(tb == NULL)
	    return;

	pv = &tb->preview;

	/* Unload any loaded preview texture data on the texture
	 * browser.
	 */
	TexBrowserPreviewUnload(tb);

	/* Preview widget must be realized. */
	if(!pv->realized)
	    return;

	/* Begin fetching path to new texture image file. */

	/* Get pointer to editor structure. */
	editor = (ma_editor_struct *)tb->editor_ptr;
	if(editor == NULL)
	    return;

	/* Get base dir. */
	w = tb->base_dir_entry;
	if(w == NULL)
	    return;

	base_dir = gtk_entry_get_text(GTK_ENTRY(w));
	if(base_dir == NULL)
	    return;

	/* Get textures clist. */
	w = tb->textures_list;
	if(w == NULL)
	    return;

	if((i < 0) || (i >= GTK_CLIST(w)->rows))
	    return;

	/* Match actually loaded texture structure on editor. */
	tex_num = (gint)gtk_clist_get_row_data(
	    GTK_CLIST(w), i
	);

	if((tex_num < 0) || (tex_num >= editor->total_textures))
	    return;

	t = editor->texture[tex_num];
	if(t == NULL)
	    return;

	/* Get path from texture. */
	path = t->filename;
	if(path == NULL)
	    return;

	strncpy(tmp_path, path, PATH_MAX + NAME_MAX);
	tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

	/* Check incase its not absolute (it should always be absolute). */
	if(!ISPATHABSOLUTE(tmp_path))
	{
	    strptr = PrefixPaths(base_dir, tmp_path);
	    if(strptr != NULL)
	    {
		strncpy(tmp_path, strptr, PATH_MAX + NAME_MAX);
		tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
	    }
	}

	/* Image file actually exists? */
	if(stat(tmp_path, &stat_buf))
	    return;


	TexBrowserSetBusy(tb);


#ifdef HAVE_IMLIB
/*
Disable this for now.
	while((imlib_handle != NULL) &&
	      (tex_data == NULL)
	)
 */
	while(0)
	{
	    gint len;
	    ImlibImage *imlib_image;


	    imlib_image = Imlib_load_image(imlib_handle, tmp_path);
	    if(imlib_image == NULL)
		break;

	    /* Need to re-realize Imlib image just in case it changed
	     * on file.
	     */
	    Imlib_changed_image(imlib_handle, imlib_image);

	    tex_width = imlib_image->rgb_width;
	    tex_height = imlib_image->rgb_height;

	    /* Calculate length of image data in bytes, since in RGB
	     * that's (w * h * 3) bytes.
	     */
	    len = tex_width * tex_height * 3;

	    /* Allocate and copy image data. */
	    if(len > 0)
		tex_data = (guint8 *)malloc(len);
	    if(tex_data != NULL)
		memcpy(
		    tex_data,
		    imlib_image->rgb_data,
		    len
		);

	    /* Unref and cache the Imlib image. */
	    Imlib_destroy_image(imlib_handle, imlib_image);
	    imlib_image = NULL;

	    break;
	}
#endif	/* HAVE_IMLIB */

	/* Didn't successfully load texture image yet? */
	while(tex_data == NULL)
	{
	    /* Attempt to load as tga image using internal tga
	     * loading functions.
	     */
	    u_int32_t	*ptr32_src, *ptr32_src_end;
	    guint8	*ptr8_tar, *ptr8_tar_end;
	    guint8 a, r, g, b;
	    gint status, len;
	    tga_data_struct td;

	    memset(&td, 0x00, sizeof(tga_data_struct));

	    status = TgaReadFromFile(
		tmp_path,
		&td,
		32
	    );
	    if(status != TgaSuccess)
	    {
		TgaDestroyData(&td);
		break;
	    }

	    tex_width = td.width;
	    tex_height = td.height;

	    /* Allocate target buffer for RGB, that's (w * h * 3). */
	    len = tex_width * tex_height * 3;
	    if(len > 0)
		ptr8_tar = (guint8 *)malloc(len);
	    else
		ptr8_tar = NULL;

	    /* Set pointer to target buffer. */
	    tex_data = ptr8_tar;
	    ptr8_tar_end = ptr8_tar + len;

	    /* Get pointer to source buffer. */
	    ptr32_src = (u_int32_t *)td.data;
	    ptr32_src_end = ptr32_src + (tex_width * tex_height);

	    if((ptr32_src != NULL) && (ptr8_tar != NULL))
	    {
		while(ptr32_src < ptr32_src_end)
		{
		    a = (guint8)(((*ptr32_src) & 0xff000000) >> 24);
		    r = (guint8)(((*ptr32_src) & 0x00ff0000) >> 16);
		    g = (guint8)(((*ptr32_src) & 0x0000ff00) >> 8);
		    b = (guint8)(((*ptr32_src) & 0x000000ff) >> 0);

		    *ptr8_tar++ = r;
		    *ptr8_tar++ = g;
		    *ptr8_tar++ = b;

		    ptr32_src++;
		}
	    }

	    TgaDestroyData(&td);

	    break;
	}


	/* Create texture. */
	if(tex_data != NULL)
	{
	    w = pv->preview;
	    if(w != NULL)
	    {
		/* Set preview glarea widget into gl context and
		 * load texture.
		 */
		if(!TexBrowserPreviewEnableContext(tb))
		{
		    pv->tex = V3DTextureLoadFromData2D(
			tex_data,
			"preview",	/* Not needed afterwards but just give something. */
			tex_width, tex_height,
			24,		/* Bits per pixel. */
			V3D_TEX_FORMAT_RGB,	/* Destination format. */
			NULL, NULL
		    );
		    if(pv->tex != NULL)
		    {
			pv->tex_width = tex_width;
			pv->tex_height = tex_height;
		    }
		}
	    }

	    /* Free texture image data, no longer needed. */
	    free(tex_data);
	    tex_data = NULL;

	    tex_width = 0;
	    tex_height = 0;
	}

	/* Call zoom to fit to fit texture all snuggly and redraw. */
	TexBrowserZoomToFitCB(NULL, tb);


	/* Recreate preview_label_table on texture browser. */
	if((tb->preview_label_vbox != NULL) &&
	   (tb->preview_label_table == NULL)
	)
	{
	    GtkWidget	*parent = tb->preview_label_vbox,
			*parent2,
			*parent3;
	    gint attach_x, attach_y;
	    gchar text[256];


	    /* Create table. */
	    w = gtk_table_new(6, 2, FALSE);
	    tb->preview_label_table = w;
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 2);
	    gtk_widget_show(w);
	    parent2 = w;


	    /* Begin creating labels. */
	    attach_y = 0;

	    /* Referance name. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("Referance Name:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    *text = '\0';
	    strncat(
		text,
		((t->name == NULL) ? "(null)" : t->name),
		80
	    );
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++;


	    /* File name. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("File Name:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    *text = '\0';
	    strptr = strrchr(
		((t->filename == NULL) ? "" : t->filename),
	        DIR_DELIMINATOR
	    );
	    if(strptr == NULL)
	    {
		strncat(
		    text,
		    ((t->filename == NULL) ? "" : t->filename),
		    80
		);
	    } else {
		strcat(text, "...");
		strncat(text, strptr, 80);
	    }
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++;


	    /* Frames. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("Frames:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    sprintf(text, "%i", t->total_frames);
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++;


	    /* Geometry. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,   
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("Geometry:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2   
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    sprintf(text, "%ix%i pixels", t->width, t->height);
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++;


	    /* Dimensions. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("Dimensions:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    sprintf(text, "%i", t->dimensions);
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++; 


	    /* Priority. */
	    attach_x = 0;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(  
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_label_new("Priority:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_widget_show(w);


	    attach_x++;
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach(
		GTK_TABLE(parent2), w,
		attach_x, attach_x + 1,
		attach_y, attach_y + 1,
		GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		0,
		2, 2
	    );
	    gtk_widget_show(w);
	    parent3 = w;

	    sprintf(text, "%f", t->priority);
	    w = gtk_label_new(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_widget_show(w);
	    attach_y++;
	}


	TexBrowserSetReady(tb);

	return;
}

/*
 *	Unlaods any data for the preview texture on the texture browser.
 */
void TexBrowserPreviewUnload(ma_texture_browser_struct *tb)
{
	GtkWidget *w;
	GtkAdjustment *adj;
	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return;

	pv = &tb->preview;

	/* Need to make preview glarea into context before destroying
	 * texture used on it.
	 */
	TexBrowserPreviewEnableContext(tb);
	V3DTextureDestroy(pv->tex);
	pv->tex = NULL;

	pv->tex_width = 0;
	pv->tex_height = 0;

	pv->zoom_coeff_wtod = 1.0;
	pv->xw = 0;
	pv->yw = 0;

	pv->drag_state = MA_TEXBROWSER_DRAG_NONE;


	/* Destroy preview_label_table on texture browser. */
	w = tb->preview_label_table;
	if(w != NULL)
	    gtk_widget_destroy(w);
	tb->preview_label_table = NULL;


	/* Reset scroll adjustments. */
	adj = pv->preview_vadj;
	w = pv->preview_vsb;
	if((adj != NULL) && (w != NULL))
	{
	    adj->lower = adj->upper = 0;
	    if((gint)(adj->upper - adj->lower) <= (gint)(adj->page_size))
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	}

	adj = pv->preview_hadj;
	w = pv->preview_hsb;
	if((adj != NULL) && (w != NULL))
	{
	    adj->lower = adj->upper = 0;
	    if((gint)(adj->upper - adj->lower) <= (gint)(adj->page_size))
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	}
}


/*
 *	Zooms the preview to the specified rectangular area in window
 *	coordinates and redraws.
 */
void TexBrowserPreviewZoomRect(
	ma_texture_browser_struct *tb,
	gint x0w, gint y0w,
	gint x1w, gint y1w
)
{
	gdouble	nx0d, ny0d, nwd, nhd, tex_wd, tex_hd, txd, tyd;
	gint	nww, nhw;

	GtkWidget *w;
	GtkAdjustment *adj;
	gdouble coeff_wtod;
	ma_texbrowser_preview_struct *pv;


	if(tb == NULL)
	    return;

	pv = &tb->preview;

	if(x0w > x1w)
	{
	    gint tmp = x0w;

	    x0w = x1w;
	    x1w = tmp;
	}
	if(y0w > y1w)
	{
	    gint tmp = y0w;

	    y0w = y1w;
	    y1w = tmp;
	}

	coeff_wtod = pv->zoom_coeff_wtod;

	nx0d = x0w;	/* New x/y offset in data units. */
	ny0d = y0w;
	nww = x1w - x0w;		/* New w/h in window units. */
	nhw = y1w - y0w;
	nwd = nww * coeff_wtod;		/* New w/h in data units. */
	nhd = nhw * coeff_wtod;
	tex_wd = pv->tex_width * coeff_wtod;
	tex_hd = pv->tex_height * coeff_wtod;
	txd = pv->xw * coeff_wtod;
	tyd = pv->yw * coeff_wtod;

	/* No zoomed in area defined? */
	if((nww <= 0) ||
	   (nhw <= 0)
	)
	    return;

	/* Adjust txw and tyw if data is zoomed smaller than window. */
	if(tex_wd < pv->ww)
	    txd = -((pv->ww - tex_wd) / 2);
	if(tex_hd < pv->hw)
	    tyd = -((pv->hw - tex_hd) / 2);

	/* Adjust zoom coeff from longest dimension. */
	if(nww > nhw)
	    coeff_wtod *= (double)pv->ww / (double)nww;
	else
	    coeff_wtod *= (double)pv->hw / (double)nhw;

	/* Update adjustments. */
	adj = pv->preview_hadj;
	w = pv->preview_hsb;
	if((adj != NULL) && (w != NULL) && (coeff_wtod > 0.0))
	{
	    gdouble old_upper = adj->upper;
	    gdouble zoom_change_coeff;

	    adj->upper = (double)pv->tex_width * coeff_wtod;
	    if(old_upper > 0.0)
		zoom_change_coeff = adj->upper / old_upper;
	    else
		zoom_change_coeff = 1.0;

	    adj->value = (txd + nx0d) * zoom_change_coeff;

	    if((gint)adj->value > (gint)(adj->upper - adj->page_size))
		adj->value = adj->upper - adj->page_size;
	    if((gint)adj->value < 0)
		adj->value = 0.0;
	    pv->xw = (gint)(adj->value / coeff_wtod);

	    if((gint)(adj->upper - adj->lower) <= (gint)(adj->page_size))
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	}

	adj = pv->preview_vadj;
	w = pv->preview_vsb;
	if((adj != NULL) && (w != NULL) && (coeff_wtod > 0.0))
	{
	    gdouble old_upper = adj->upper;
	    gdouble zoom_change_coeff;

	    adj->upper = (double)pv->tex_height * coeff_wtod;
	    if(old_upper > 0.0)
		zoom_change_coeff = adj->upper / old_upper;
	    else
		zoom_change_coeff = 1.0;
	    adj->value = (tyd + ny0d) * zoom_change_coeff;

	    if((gint)adj->value > (gint)(adj->upper - adj->page_size))
		adj->value = adj->upper - adj->page_size;
	    if((gint)adj->value < 0)
		adj->value = 0.0;
	    pv->yw = (gint)(adj->value / coeff_wtod);

	    if((gint)(adj->upper - adj->lower) <= (gint)(adj->page_size))
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	}

	/* Update zoom. */
	pv->zoom_coeff_wtod = coeff_wtod;

	/* Redraw preview. */
	TexBrowserPreviewDraw(tb);
}

