/* valafield.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valafield.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valanamedargument.h>
#include <vala/valastringliteral.h>




struct _ValaFieldPrivate {
	MemberBinding _binding;
	gboolean _is_volatile;
	gboolean _no_array_length;
	char* cname;
	gboolean lock_used;
	ValaExpression* _initializer;
	ValaDataType* _data_type;
};

#define VALA_FIELD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FIELD, ValaFieldPrivate))
enum  {
	VALA_FIELD_DUMMY_PROPERTY,
	VALA_FIELD_FIELD_TYPE,
	VALA_FIELD_INITIALIZER,
	VALA_FIELD_BINDING,
	VALA_FIELD_IS_VOLATILE,
	VALA_FIELD_NO_ARRAY_LENGTH
};
static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a);
static gboolean vala_field_real_get_lock_used (ValaLockable* base);
static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used);
static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_field_parent_class = NULL;
static ValaLockableIface* vala_field_vala_lockable_parent_iface = NULL;
static void vala_field_finalize (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new field.
 *
 * @param name   field name
 * @param type   field type
 * @param init   initializer expression
 * @param source reference to source code
 * @return       newly created field
 */
ValaField* vala_field_new (const char* name, ValaDataType* field_type, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaField * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (field_type), NULL);
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_FIELD, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_field_set_field_type (self, field_type);
	vala_field_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_field_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_member (visitor, VALA_MEMBER (self));
	vala_code_visitor_visit_field (visitor, self);
}


static void vala_field_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_field_get_field_type (self)), visitor);
	if (vala_field_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_field_get_initializer (self)), visitor);
	}
}


/**
 * Returns the name of this field as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_field_get_cname (ValaField* self) {
	const char* _tmp1;
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_field_get_default_cname (self), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
}


/**
 * Sets the name of this field as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_field_set_cname (ValaField* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_FIELD (self));
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


/**
 * Returns the default name of this field as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_field_get_default_cname (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	if (self->priv->_binding == MEMBER_BINDING_STATIC) {
		char* _tmp0;
		char* _tmp1;
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = g_strconcat ((_tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self)), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	} else {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = vala_symbol_get_name (VALA_SYMBOL (self)), (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	}
}


static void vala_field_process_ccode_attribute (ValaField* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_FIELD (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_field_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
				{
					vala_member_add_cheader_filename (VALA_MEMBER (self), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) (g_free))), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_field_process_attributes (ValaField* self) {
	g_return_if_fail (VALA_IS_FIELD (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) (a_it->data)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_field_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "NoArrayLength") == 0) {
						vala_field_set_no_array_length (self, TRUE);
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_field_real_get_lock_used (ValaLockable* base) {
	ValaField * self;
	self = VALA_FIELD (base);
	return self->priv->lock_used;
}


static void vala_field_real_set_lock_used (ValaLockable* base, gboolean used) {
	ValaField * self;
	self = VALA_FIELD (base);
	self->priv->lock_used = used;
}


static void vala_field_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaField * self;
	self = VALA_FIELD (base);
	g_return_if_fail (VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (VALA_IS_DATA_TYPE (new_type));
	if (vala_field_get_field_type (self) == old_type) {
		vala_field_set_field_type (self, new_type);
	}
}


char* vala_field_get_ctype (ValaField* self) {
	ValaAttribute* attr;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	attr = vala_code_node_get_attribute (VALA_CODE_NODE (self), "CCode");
	if (attr == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = NULL, (attr == NULL ? NULL : (attr = (g_object_unref (attr), NULL))), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = vala_attribute_get_string (attr, "type"), (attr == NULL ? NULL : (attr = (g_object_unref (attr), NULL))), _tmp1);
}


void vala_field_set_ctype (ValaField* self, const char* ctype) {
	ValaAttribute* attr;
	ValaNamedArgument* _tmp4;
	ValaStringLiteral* _tmp3;
	char* _tmp2;
	g_return_if_fail (VALA_IS_FIELD (self));
	g_return_if_fail (ctype != NULL);
	attr = vala_code_node_get_attribute (VALA_CODE_NODE (self), "CCode");
	if (attr == NULL) {
		ValaAttribute* _tmp0;
		ValaAttribute* _tmp1;
		_tmp0 = NULL;
		attr = (_tmp0 = vala_attribute_new ("CCode", NULL), (attr == NULL ? NULL : (attr = (g_object_unref (attr), NULL))), _tmp0);
		_tmp1 = NULL;
		VALA_CODE_NODE (self)->attributes = g_list_append (VALA_CODE_NODE (self)->attributes, (_tmp1 = attr, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))));
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp2 = NULL;
	vala_attribute_add_argument (attr, (_tmp4 = vala_named_argument_new ("type", VALA_EXPRESSION ((_tmp3 = vala_string_literal_new ((_tmp2 = g_strdup_printf ("\"%s\"", ctype)), NULL))), NULL)));
	(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
	(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
	_tmp2 = (g_free (_tmp2), NULL);
	(attr == NULL ? NULL : (attr = (g_object_unref (attr), NULL)));
}


ValaDataType* vala_field_get_field_type (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	return self->priv->_data_type;
}


void vala_field_set_field_type (ValaField* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_FIELD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
	g_object_notify (((GObject *) (self)), "field-type");
}


ValaExpression* vala_field_get_initializer (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), NULL);
	return self->priv->_initializer;
}


void vala_field_set_initializer (ValaField* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_FIELD (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_initializer), VALA_CODE_NODE (self));
	}
	g_object_notify (((GObject *) (self)), "initializer");
}


MemberBinding vala_field_get_binding (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), 0);
	return self->priv->_binding;
}


void vala_field_set_binding (ValaField* self, MemberBinding value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_binding = value;
	g_object_notify (((GObject *) (self)), "binding");
}


gboolean vala_field_get_is_volatile (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), FALSE);
	return self->priv->_is_volatile;
}


void vala_field_set_is_volatile (ValaField* self, gboolean value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_is_volatile = value;
	g_object_notify (((GObject *) (self)), "is-volatile");
}


gboolean vala_field_get_no_array_length (ValaField* self) {
	g_return_val_if_fail (VALA_IS_FIELD (self), FALSE);
	return self->priv->_no_array_length;
}


void vala_field_set_no_array_length (ValaField* self, gboolean value) {
	g_return_if_fail (VALA_IS_FIELD (self));
	self->priv->_no_array_length = value;
	g_object_notify (((GObject *) (self)), "no-array-length");
}


static void vala_field_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaField * self;
	self = VALA_FIELD (object);
	switch (property_id) {
		case VALA_FIELD_FIELD_TYPE:
		g_value_set_object (value, vala_field_get_field_type (self));
		break;
		case VALA_FIELD_INITIALIZER:
		g_value_set_object (value, vala_field_get_initializer (self));
		break;
		case VALA_FIELD_BINDING:
		g_value_set_enum (value, vala_field_get_binding (self));
		break;
		case VALA_FIELD_IS_VOLATILE:
		g_value_set_boolean (value, vala_field_get_is_volatile (self));
		break;
		case VALA_FIELD_NO_ARRAY_LENGTH:
		g_value_set_boolean (value, vala_field_get_no_array_length (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_field_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaField * self;
	self = VALA_FIELD (object);
	switch (property_id) {
		case VALA_FIELD_FIELD_TYPE:
		vala_field_set_field_type (self, g_value_get_object (value));
		break;
		case VALA_FIELD_INITIALIZER:
		vala_field_set_initializer (self, g_value_get_object (value));
		break;
		case VALA_FIELD_BINDING:
		vala_field_set_binding (self, g_value_get_enum (value));
		break;
		case VALA_FIELD_IS_VOLATILE:
		vala_field_set_is_volatile (self, g_value_get_boolean (value));
		break;
		case VALA_FIELD_NO_ARRAY_LENGTH:
		vala_field_set_no_array_length (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_field_class_init (ValaFieldClass * klass) {
	vala_field_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaFieldPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_field_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_field_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_field_finalize;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_field_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_field_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_field_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_FIELD_TYPE, g_param_spec_object ("field-type", "field-type", "field-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_BINDING, g_param_spec_enum ("binding", "binding", "binding", TYPE_MEMBER_BINDING, MEMBER_BINDING_INSTANCE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_IS_VOLATILE, g_param_spec_boolean ("is-volatile", "is-volatile", "is-volatile", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_FIELD_NO_ARRAY_LENGTH, g_param_spec_boolean ("no-array-length", "no-array-length", "no-array-length", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_field_vala_lockable_interface_init (ValaLockableIface * iface) {
	vala_field_vala_lockable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_lock_used = vala_field_real_get_lock_used;
	iface->set_lock_used = vala_field_real_set_lock_used;
}


static void vala_field_instance_init (ValaField * self) {
	self->priv = VALA_FIELD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->lock_used = FALSE;
}


static void vala_field_finalize (GObject * obj) {
	ValaField * self;
	self = VALA_FIELD (obj);
	self->priv->cname = (g_free (self->priv->cname), NULL);
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_field_parent_class)->finalize (obj);
}


GType vala_field_get_type (void) {
	static GType vala_field_type_id = 0;
	if (vala_field_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFieldClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_field_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaField), 0, (GInstanceInitFunc) vala_field_instance_init };
		static const GInterfaceInfo vala_lockable_info = { (GInterfaceInitFunc) vala_field_vala_lockable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_field_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaField", &g_define_type_info, 0);
		g_type_add_interface_static (vala_field_type_id, VALA_TYPE_LOCKABLE, &vala_lockable_info);
	}
	return vala_field_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




