/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-properties.h"

#include "tpa-properties-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_PROPERTIES

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    PROPERTIES_CHANGED,
    PROPERTIES_FLAGS_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIPropertiesPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_properties_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIProperties),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIProperties",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_properties_init (TpaIProperties *iface,
                     gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->get_properties = NULL;
    iface->set_properties = NULL;
    iface->list_properties = NULL;

    /* Interface signals */
    signals[PROPERTIES_CHANGED] =
        g_signal_new ("properties-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_collection ("GPtrArray", (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_VALUE, G_TYPE_INVALID)))));

    signals[PROPERTIES_FLAGS_CHANGED] =
        g_signal_new ("property-flags-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_collection ("GPtrArray", (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID)))));
    iface->priv = g_new0 (TpaIPropertiesPrivate, 1);
    VERBOSE ("private members %p", iface->priv);
}

void
tpa_properties_finalize (GObject *obj)
{
    TpaIProperties *iface = TPA_IPROPERTIES (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_properties_get_properties
 *
 * Implements DBus method GetProperties
 * on interface org.freedesktop.Telepathy.Properties
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_properties_get_properties (GObject *obj,
                               const GArray *properties,
                               GPtrArray **ret,
                               GError **error)
{
    TpaIProperties *iface = TPA_IPROPERTIES (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p, %p)", obj, properties, ret);

    g_return_error_val_if_fail (iface->get_properties != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_properties (obj, properties, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_properties_list_properties
 *
 * Implements DBus method ListProperties
 * on interface org.freedesktop.Telepathy.Properties
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_properties_list_properties (GObject *obj,
                                GPtrArray **ret,
                                GError **error)
{
    TpaIProperties *iface = TPA_IPROPERTIES (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->list_properties != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->list_properties (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_properties_set_properties
 *
 * Implements DBus method SetProperties
 * on interface org.freedesktop.Telepathy.Properties
 *
 * @context: The DBUS invocation context to use to return values
 *           or throw an error.
 */
gboolean
tpa_properties_set_properties (GObject *obj,
                               const GPtrArray *properties,
                               DBusGMethodInvocation *context)
{
    TpaIProperties *iface = TPA_IPROPERTIES (obj);
    TpaError error_code = TPA_ERROR_NONE;
    GError *error = NULL;

    g_return_context_error_val_if_fail (iface != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, properties);

    g_return_context_error_val_if_fail (iface->set_properties != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->set_properties (obj, properties, context);

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_properties_signal_properties_changed
 *
 * Implements DBus signal PropertiesChanged
 * on interface org.freedesktop.Telepathy.Properties
 */
void
tpa_properties_signal_properties_changed (GObject *obj,
                                          GPtrArray *properties)
{
    g_assert (TPA_IS_IPROPERTIES (obj));

    VERBOSE ("(%p, %p)", obj, properties);

    g_signal_emit (obj, signals[PROPERTIES_CHANGED], 0, properties);
}

/**
 * tpa_properties_signal_properties_flags_changed
 *
 * Implements DBus signal PropertiesFlagsChanged
 * on interface org.freedesktop.Telepathy.Properties
 */
void
tpa_properties_signal_properties_flags_changed (GObject *obj,
                                                GPtrArray *properties)
{
    g_assert (TPA_IS_IPROPERTIES (obj));

    VERBOSE ("(%p, %p)", obj, properties);

    g_signal_emit (obj, signals[PROPERTIES_FLAGS_CHANGED], 0, properties);
}
