/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-connection.h"
#include "tpa-session.h"
#include "tpa-ifaces.h"

#include "tpa-connection-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-enums.h>
#include <tapioca/base/tpa-errors.h>

#define _DBUS_PATH "/org/freedesktop/Telepathy/Connection"

/* signal enum */
enum
{
    NEW_CHANNEL,
    STATUS_CHANGED,
    DISCONNECTED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* we need to define the get_type function */
GType
tpa_connection_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIConnection),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIConnection",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_connection_init (TpaIConnection *iface,
                     gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->connect = NULL;
    iface->disconnect = NULL;
    iface->get_self_handle = NULL;
    iface->get_interfaces = NULL;
    iface->hold_handles = NULL;
    iface->inspect_handles = NULL;
    iface->release_handles = NULL;
    iface->request_handles = NULL;
    iface->handles = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, g_free);
    iface->sessions = g_ptr_array_new ();
    iface->protocol = NULL;
    iface->handle = NULL;
    iface->status = TPA_CONNECTION_STATUS_DISCONNECTED;

    /* Interface signals */
    signals[NEW_CHANNEL] =
        g_signal_new ("new-channel",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__STRING_STRING_UINT_UINT_BOOLEAN,
                      G_TYPE_NONE, 5, DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_BOOLEAN);

    signals[STATUS_CHANGED] =
        g_signal_new ("status-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_UINT,
                      G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);

    signals[DISCONNECTED] =
        g_signal_new ("disconnected",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);
}

void
tpa_connection_finalize (GObject *obj)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);

    VERBOSE ("(%p)", obj);

    /* Destroy handles table */
    if (iface->handles)
        g_hash_table_destroy (iface->handles);

    if (iface->protocol)
        g_free (iface->protocol);

    if (iface->handle)
        g_free (iface->handle);
}

/**
 * tpa_connection_connect
 *
 * Implements DBus method Connect
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_connect (GObject *obj,
                        GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->connect != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    iface->status = TPA_CONNECTION_STATUS_CONNECTING;

    error_code = iface->connect (obj);

    if (error_code == TPA_ERROR_NONE)
        iface->status = TPA_CONNECTION_STATUS_DISCONNECTED;

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    iface->status = TPA_CONNECTION_STATUS_CONNECTED;

    return TRUE;
}

/**
 * tpa_connection_disconnect
 *
 * Implements DBus method Disconnect
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_disconnect (GObject *obj,
                           GError  **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->status == TPA_CONNECTION_STATUS_CONNECTED,
        error, TPA_ERROR_DISCONNECTED);

    g_return_error_val_if_fail (iface->disconnect != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->disconnect (obj);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    iface->status = TPA_CONNECTION_STATUS_DISCONNECTED;

    return TRUE;
}

/**
 * tpa_connection_get_interfaces
 *
 * Implements DBus method GetInterfaces
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_get_interfaces (GObject *obj,
                               gchar ***ret,
                               GError **error)
{
    const gchar *interfaces[] = { NULL };
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (iface->get_interfaces)
        error_code = iface->get_interfaces (obj, ret);
    else if (ret) {
        *ret = g_strdupv ((gchar **) interfaces);
    }

    return TRUE;
}

/**
 * tpa_connection_get_protocol
 *
 * Implements DBus method GetProtocol
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_get_protocol (GObject *obj,
                             gchar **ret,
                             GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (ret && iface->protocol)
       *ret = g_strdup (iface->protocol);

    return TRUE;
}

/**
 * tpa_connection_get_obj_handle
 *
 * Implements DBus method GetSelfHandle
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_get_self_handle (GObject *obj,
                                guint *ret,
                                GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    TpaHandle *handle = NULL;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    /* CM provide own handles implementation */
    if (iface->get_self_handle)
        error_code = iface->get_self_handle (obj, ret);
    else {
        if (iface->handle && ret)
            *ret = handle->id;
        else
            error_code = TPA_ERROR_DISCONNECTED;
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_connection_get_status
 *
 * Implements DBus method GetStatus
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_get_status (GObject *obj,
                           TpaConnectionStatus *ret,
                           GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    *ret = iface->status;

    return TRUE;
}

/**
 * tpa_connection_hold_handles
 *
 * Implements DBus method HoldHandle
 * on interface org.freedesktop.Telepathy.Account
 *
 * @context: The DBUS invocation context to use to return values
 *           or throw an error.
 */
gboolean
tpa_connection_hold_handles (GObject *obj,
                             TpaConnectionHandleType handle_type,
                             const GArray *handles,
                             DBusGMethodInvocation *context)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    guint index = 0;
    GError *error = NULL;

    g_return_context_error_val_if_fail (iface != NULL, context, error,
        TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d, %p)", obj, handle_type, handles);

    /* CM provide own handles implementation */
    if (iface->hold_handles)
        error_code = iface->hold_handles (obj, handle_type, handles);
    else {
        if (handles) {
            for (index = 0; index < handles->len; index++) {
                TpaHandle *hdl = NULL;
                guint handle = 0;

                handle = g_array_index (handles, guint, index);
                hdl = g_hash_table_lookup (iface->handles, GINT_TO_POINTER(handle));

                if (hdl)
                    hdl->hold = TRUE;
                else {
                    error_code = TPA_ERROR_INVALID_HANDLE;
                    break;
                }
            }
        }
    }

    g_return_context_error_val_if_fail (error_code != TPA_ERROR_NONE, context, error,
        TPA_ERROR_NOT_IMPLEMENTED);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_connection_inspect_handles
 *
 * Implements DBus method InspectHandle
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_inspect_handles (GObject *obj,
                                TpaConnectionHandleType handle_type,
                                const GArray *handles,
                                gchar **ret,
                                GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    guint index = 0;

    VERBOSE ("(%p, %d, %p, %p)", obj, handle_type, handles, ret);

    /* CM provide own handles implementation */
    if (iface->inspect_handles)
        error_code = iface->inspect_handles (obj, handle_type, handles, ret);
    else {
        if (ret) {
            *ret = (gchar *) g_new (gchar *, handles->len + 1);

            if (handles) {
                for (index = 0; index < handles->len; index++) {
                    TpaHandle *hdl = NULL;
                    guint handle = 0;

                    handle = g_array_index (handles, guint, index);
                    hdl = g_hash_table_lookup (iface->handles, GINT_TO_POINTER(handle));

                    if (hdl && hdl->string) {
                        VERBOSE ("handle %d is %s", hdl->id, hdl->string);
                        ret[index] = g_strdup (hdl->string);
                    }
                    else {
                        VERBOSE ("handle %d invalid", handle);
                        error_code = TPA_ERROR_INVALID_HANDLE;
                        break;
                    }
                }
            }

            ret[index] = NULL;
        }
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_connection_list_channels
 *
 * Implements DBus method ListChannels
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_list_channels (GObject *obj,
                              GPtrArray **ret,
                              GError **error)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaSession *session = NULL;
    guint c = 0;

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->status == TPA_CONNECTION_STATUS_CONNECTED,
        error, TPA_ERROR_DISCONNECTED);

    if (ret && iface->sessions) {
        *ret = g_ptr_array_new ();
        for (c = 0; c < iface->sessions->len; c++) {
            session = TPA_SESSION (g_ptr_array_index (iface->sessions, c));
            tpa_session_list (session, *ret);
        }
    }

    return TRUE;
}

/**
 * tpa_connection_release_handle
 *
 * Implements DBus method ReleaseHandle
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @context: The DBUS invocation context to use to return values
 *           or throw an error.
 */
gboolean
tpa_connection_release_handles (GObject *obj,
                                TpaConnectionHandleType handle_type,
                                const GArray *handles,
                                DBusGMethodInvocation *context)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    guint index = 0;
    GError *error = NULL;

    VERBOSE ("(%p, %d, %p)", obj, handle_type, handles);

    /* CM provide own handles implementation */
    if (iface->release_handles)
        error_code = iface->release_handles (obj, handle_type, handles);
    else if (handles) {
        for (index = 0; index < handles->len; index++) {
            TpaHandle *hdl = NULL;
            guint handle = 0;

            handle = g_array_index (handles, guint, index);
            hdl = g_hash_table_lookup (iface->handles, GINT_TO_POINTER(handle));

            if (hdl && !hdl->hold) {
                g_hash_table_remove (iface->handles, GINT_TO_POINTER (handle));
                VERBOSE ("handle %d released", handle);
            }
            else {
                VERBOSE ("invalid handle %d", handle);
                error_code = TPA_ERROR_INVALID_HANDLE;
                break;
            }
        }
    }
    else
        error_code = TPA_ERROR_INVALID_HANDLE;

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_connection_request_channel
 *
 * Implements DBus method RequestChannel
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @context: The DBUS invocation context to use to return values
 *           or throw an error.
 */
gboolean
tpa_connection_request_channel (GObject *obj,
                                const gchar *type,
                                TpaConnectionHandleType handle_type,
                                guint handle,
                                gboolean suppress_handler,
                                DBusGMethodInvocation *context)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    GError *error = NULL;
    TpaHandle *hdl = NULL;

    VERBOSE ("(%p, %s, %d, %d, %d)", obj, type, handle_type, handle, suppress_handler);

    g_return_context_error_val_if_fail (iface->request_session != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    hdl = g_hash_table_lookup (iface->handles, GINT_TO_POINTER(handle));

    g_return_context_error_val_if_fail (hdl != NULL, context, error, TPA_ERROR_INVALID_HANDLE);

    error_code = iface->request_session (obj, type, hdl->string, suppress_handler);

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_connection_request_handle
 *
 * Implements DBus method RequestHandle
 * on interface org.freedesktop.Telepathy.Connection
 *
 * @context: The DBUS invocation context to use to return values
 *           or throw an error.
 */
gboolean
tpa_connection_request_handles (GObject *obj,
                                TpaConnectionHandleType handle_type,
                                const gchar **names,
                                DBusGMethodInvocation *context)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaError error_code = TPA_ERROR_NONE;
    TpaHandle *handle = NULL;
    GError *error = NULL;
    GArray *handles = NULL;
    gint c = 0;
    gint fail = 0;

    VERBOSE ("(%p, %d, %p)", obj, handle_type, names);

    /* Check if CM provides its own implementation */
    if (iface->request_handles)
        error_code = iface->request_handles (obj, handle_type, names);
    else {
        if (names) {
            gchar *string_formated = NULL;
            handles = g_array_new (FALSE, FALSE, sizeof(guint));
            for (c = 0; names[c]; c++) {
                if (iface->format_string) {
                    error_code = iface->format_string (obj, names[c], &string_formated);
                    handle = tpa_connection_get_handle (obj, handle_type, string_formated);
                    g_free (string_formated);
                }
                else
                    handle = tpa_connection_get_handle (obj, handle_type, names[c]);
                /* Invalid string create handle 0 */
                if (handle)
                    g_array_append_val (handles, handle->id);
                else
                    g_array_append_val (handles, fail);
            }
        }
    }

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context, handles);

    return TRUE;
}

/**
 * tpa_connection_get_handle
 *
 * Get handle for given string, if it is not available yet create one.
 * Given string must be normalized to protocol.
 */
TpaHandle*
tpa_connection_get_handle (GObject *obj,
                           TpaConnectionHandleType type,
                           const gchar *string)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaHandle *handle = NULL;

    g_assert (TPA_IS_ICONNECTION (obj));

    VERBOSE ("(%p, %s)", obj, string);

    if (iface->handles && string) {
        gpointer key = GINT_TO_POINTER (g_quark_from_string (string));
        if (!(handle = g_hash_table_lookup (iface->handles, key))) {
            handle = g_new0 (TpaHandle, 1);
            handle->id = GPOINTER_TO_INT (key);
            handle->string = g_strdup (string);
            handle->type = type;
            handle->hold = FALSE;
            g_hash_table_insert (iface->handles, key, handle);
            VERBOSE ("new handle %d of \"%s\" created", handle->id, handle->string);
        }
        else
            VERBOSE ("handle %d of \"%s\" already created", handle->id, handle->string);
    }

    return handle;
}

/**
 * tpa_connection_get_string
 * 
 * Returned string must be freed.
 */
gchar*
tpa_connection_get_string (GObject *obj,
                           guint handle)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);
    TpaHandle *hdl = NULL;

    g_assert (TPA_IS_ICONNECTION (obj));

    hdl = g_hash_table_lookup (iface->handles, GINT_TO_POINTER (handle));
    if (hdl)
        return g_strdup (hdl->string);

    return NULL;
}

/**
 * tpa_connection_signal_new_channel
 *
 * Implements DBus signal NewChannel
 * on interface org.freedesktop.Telepathy.Connection
 */
void
tpa_connection_signal_new_channel (GObject *obj,
                                   const gchar *obj_path,
                                   TpaChannelType type,
                                   const gchar *string,
                                   gboolean suppress_handler)
{
    TpaHandle *handle = NULL;
    gchar *channel_type = NULL;

    g_assert (TPA_IS_ICONNECTION (obj));

    handle = tpa_connection_get_contact_handle (obj, string);

    VERBOSE ("(%p, %s, %d, %s, %d)", obj, obj_path, type, string, suppress_handler);

    switch (type)
    {
        case TPA_CHANNEL_TYPE_CONTACTLIST:
            channel_type = g_strdup (DBUS_TYPE_CONTACTLIST_IFACE);
            break;
        case TPA_CHANNEL_TYPE_CONTACTSEARCH:
            channel_type = g_strdup (DBUS_TYPE_CONTACTSEARCH_IFACE);
            break;
        case TPA_CHANNEL_TYPE_STREAMEDMEDIA:
            channel_type = g_strdup (DBUS_TYPE_STREAMEDMEDIA_IFACE);
            break;
        case TPA_CHANNEL_TYPE_ROOMLIST:
            channel_type = g_strdup (DBUS_TYPE_ROOMLIST_IFACE);
            break;
        case TPA_CHANNEL_TYPE_TEXT:
            channel_type = g_strdup (DBUS_TYPE_TEXT_IFACE);
            break;
    }

    if (handle && channel_type) {
        g_signal_emit (obj, signals[NEW_CHANNEL], 0, obj_path, channel_type,
            handle->type, handle->id, suppress_handler);
        g_free (channel_type);
    }
}

/**
 * tpa_connection_signal_status_changed
 *
 * Implements DBus signal StatusChanged
 * on interface org.freedesktop.Telepathy.Connection
 */
void
tpa_connection_signal_status_changed (GObject *obj,
                                      TpaConnectionStatus status,
                                      TpaConnectionStatusReason reason)
{
    TpaIConnection *iface = TPA_ICONNECTION (obj);

    g_assert (TPA_IS_ICONNECTION (obj));

    VERBOSE ("(%p, %d, %d)", obj, status, reason);

    iface->status = status;
    iface->reason = reason;

    g_signal_emit (obj, signals[STATUS_CHANGED], 0, status, reason);
}
