/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <time.h>

#include "tpa-text-channel-priv.h"
#include "tpa-handle.h"
#include "tpa-channel-target-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#define LOG_SIZE 200

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaTextChannelPrivate {
    DBusGProxy *proxy;
    DBusGProxy *chat_state_proxy;
    GPtrArray *log;
    guint log_size;
    TpaChatState current_chat_state;
    gboolean disposed;
};

enum {
    MESSAGE_RECEIVED,
    MESSAGE_SENT,
    MESSAGE_ERROR,
    CHAT_STATE_CHANGED,
    LAST_SIGNAL
};

static guint tpa_text_channel_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE(TpaTextChannel, tpa_text_channel, TPA_TYPE_CHANNEL)

static void
proxy_sent_signal (DBusGProxy *proxy,
                   guint timestamp,
                   guint type,
                   const gchar *contents,
                   gpointer user_data)
{
    TpaTextChannel *self = TPA_TEXT_CHANNEL (user_data);
    TpaTextMessage *message;
    TpaChannelTarget *target;
    const gchar *uri;

    g_assert (self);

    target = TPA_CHANNEL_TARGET (tpa_channel_get_owner (TPA_CHANNEL (self)));
    uri = tpa_channel_target_get_uri (target);
    /* Log any type of message */
    message = tpa_text_message_new (0, uri, contents, timestamp, type);
    g_ptr_array_add (self->priv->log, message);
    INFO ("[message-sent %s]: (%d - %d) %s", uri, timestamp, type, contents);
    g_signal_emit (self, tpa_text_channel_signals[MESSAGE_SENT], 0, message);
}

static void
proxy_received_signal (DBusGProxy *proxy,
                       guint id,
                       guint timestamp,
                       guint sender,
                       guint type,
                       guint flags,
                       const gchar* contents,
                       gpointer user_data)
{
    TpaTextChannel *self = TPA_TEXT_CHANNEL (user_data);
    TpaChannelTarget *target;
    TpaTextMessage *message;
    const gchar *uri;

    g_assert (self);

    target = tpa_channel_get_target (TPA_CHANNEL (self));
    uri = tpa_channel_target_get_uri (target);
    /* Log any type of message */
    message = tpa_text_message_new (id, uri, contents, timestamp, type);
    g_ptr_array_add (self->priv->log, message);
    INFO ("[message-received %s] (%d - %d) %s", uri, timestamp, type, contents);
    g_signal_emit (self, tpa_text_channel_signals[MESSAGE_RECEIVED], 0, message);
}

static void
chat_state_proxy_changed_signal (DBusGProxy *proxy,
                                 guint contact,
                                 guint state,
                                 gpointer user_data)
{
    TpaTextChannel *self = TPA_TEXT_CHANNEL (user_data);
    TpaHandle *handle = NULL;
    TpaChannelTarget *target = NULL;
    TpaUserContact *user = NULL;

    g_assert (self);

    user = tpa_channel_get_owner (TPA_CHANNEL (self));


    if (user) {
        target = TPA_CHANNEL_TARGET (TPA_CONTACT_BASE (user));
        handle = tpa_channel_target_get_handle (target);

        if (tpa_handle_get_id (handle) == contact)
            self->priv->current_chat_state = state;
    }

    INFO ("[chat-state-changed] (NewState:%d)", state);

    g_signal_emit (self, tpa_text_channel_signals[CHAT_STATE_CHANGED], 0, state);
}

static GObject*
tpa_text_channel_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaTextChannel *self;

    object = G_OBJECT_CLASS (tpa_text_channel_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_TEXT_CHANNEL (object);

    self->priv->proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_TEXT);

    if (!self->priv->proxy) {
        tpa_channel_close (TPA_CHANNEL (self));
        return object;
    }

    /* Connect all DBus signals */
    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_TEXT,
                               "Received",
                               G_CALLBACK (proxy_received_signal),
                               self);

    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_TEXT,
                               "Sent",
                               G_CALLBACK (proxy_sent_signal),
                               self);

    self->priv->chat_state_proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CHATSTATE);

    if (self->priv->chat_state_proxy)
        tpa_object_connect_signal (TPA_OBJECT (self),
                                   TPA_INTERFACE_CHATSTATE,
                                   "ChatStateChanged",
                                   G_CALLBACK (chat_state_proxy_changed_signal),
                                   self);

    return object;
}

static void
tpa_text_channel_dispose (GObject *object)
{
    TpaTextChannel *self = TPA_TEXT_CHANNEL (object);
    guint i;

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    /* free logged messages */
    if (self->priv->log) {
        for (i = 0; i < self->priv->log->len; i++)
            g_object_unref (g_ptr_array_index (self->priv->log, i));
        g_ptr_array_free (self->priv->log, TRUE);
    }

    g_object_unref (self->priv->proxy);

    if (self->priv->chat_state_proxy)
        g_object_unref (self->priv->chat_state_proxy);

    G_OBJECT_CLASS (tpa_text_channel_parent_class)->dispose (object);
}

static void
tpa_text_channel_class_init (TpaTextChannelClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_text_channel_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaTextChannelPrivate));

    gobject_class->dispose = tpa_text_channel_dispose;
    gobject_class->constructor = tpa_text_channel_constructor;

    tpa_text_channel_signals[MESSAGE_SENT] = g_signal_new ("message-sent",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE,
        1,
        TPA_TYPE_TEXT_MESSAGE);

    tpa_text_channel_signals[MESSAGE_RECEIVED] = g_signal_new ("message-received",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE,
        1,
        TPA_TYPE_TEXT_MESSAGE);

    tpa_text_channel_signals[MESSAGE_ERROR] = g_signal_new ("message-error",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE,
        0);
}

static void
tpa_text_channel_init (TpaTextChannel *self)
{
    self->priv = TPA_TEXT_CHANNEL_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->log = g_ptr_array_sized_new (LOG_SIZE);
    self->priv->log_size = LOG_SIZE;
    self->priv->current_chat_state = TPA_CHAT_STATE_INACTIVE;
}

TpaTextChannel *
tpa_text_channel_new (DBusGProxy *proxy,
                      TpaUserContact *user,
                      TpaChannelTarget *target)
{
    TpaTextChannel *self;

    g_return_val_if_fail (proxy != NULL, NULL);
    g_return_val_if_fail (user != NULL, NULL);
    g_return_val_if_fail (target != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_TEXT_CHANNEL (g_object_new (TPA_TYPE_TEXT_CHANNEL,
	                                       "proxy", proxy,
                                           "target", target,
                                           "user", user,
                                           "type", TPA_CHANNEL_TYPE_TEXT,
                                           NULL));

    return self;
}

/**
 * tpa_text_channel_send:
 * @self: #TpaTextChannel instance
 * @contents: Text string to be sent.
 * @type: contents message type.
 *
 * Send raw text string.
 */
void
tpa_text_channel_send (TpaTextChannel *self,
                       const gchar *contents,
                       TpaTextMessageType type)
{
    TpaTextMessage *message = NULL;

    VERBOSE ("(%p, %s, %d)", self, contents, type);

    message = tpa_text_message_new (0, NULL, contents, 0, type);

    tpa_text_channel_send_message (self, message);
    g_object_unref (message);
    VERBOSE ("return");
}

/**
 * tpa_text_channel_send_message:
 * @self: #TpaTextChannel instance
 * @message: #TpaTextMessage instance.
 *
 * Send message.
 */
void
tpa_text_channel_send_message (TpaTextChannel *self,
                               TpaTextMessage *message)
{
    GError *error = NULL;
    const gchar *contents;
    TpaTextMessageType type;

    VERBOSE ("(%p, %p)", self, message);

    g_assert (self);
    g_return_if_fail (message != NULL);

    contents = tpa_text_message_get_contents (message);
    type = tpa_text_message_type (message);
    /* telepathy send message*/
    if (!org_freedesktop_Telepathy_Channel_Type_Text_send (self->priv->proxy, type, contents, &error) || error)
    {
        if(error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }
    VERBOSE ("return");
}

/**
 * tpa_text_channel_get_messages:
 * @self: #TpaTextChannel instance
 * @messages: number of messages required.
 * @type: #TpaTextMessageType required.
 * @returns: #GPtrArray containing #TpaTextMessage messages.
 *
 * Return a number messages logged with the given type.
 */
GPtrArray *
tpa_text_channel_get_messages (TpaTextChannel *self,
                               guint messages,
                               TpaTextMessageType type)
{
    guint index;
    GPtrArray *ret;
    TpaTextMessageType mtype;

    VERBOSE ("(%p, %d, %d)", self, messages, type);

    g_assert (self);

    ret = g_ptr_array_new ();

    if (messages <= 0)
        messages = self->priv->log->len;

    for (index = 0; index < self->priv->log->len && index < messages; index++) {
        TpaTextMessage *message = g_ptr_array_index (self->priv->log, index);
        mtype = tpa_text_message_type (message);
        if (mtype == type)
            g_ptr_array_add (ret, message);
    }

    VERBOSE ("return %p", ret);
    return ret;
}

/**
 * tpa_text_channel_get_pending:
 * @self: #TpaTextChannel instance.
 * @acknowledge: Acknowledge messages.
 */
GPtrArray *
tpa_text_channel_get_pending (TpaTextChannel *self,
                              gboolean acknowledge)
{
    GPtrArray *messages;
    GPtrArray *ret;
    const gchar *uri;
    GError *error = NULL;

    if (!org_freedesktop_Telepathy_Channel_Type_Text_list_pending_messages (self->priv->proxy, TRUE, &ret, &error)) {
        ERROR ("%s", error->message);
        g_error_free (error);
        return NULL;
    }
    else {
        guint i;
        TpaChannelTarget *target = tpa_channel_get_target (TPA_CHANNEL (self));

        messages = g_ptr_array_new ();
        for (i = 0; i < ret->len; i++) {
            GValueArray *msg = g_ptr_array_index (ret, i);
            TpaTextMessage *message;

            uri = tpa_channel_target_get_uri (target);
            /* Log any type of message */
            message = tpa_text_message_new (g_value_get_uint (&(msg->values[0])),
                                            uri,
                                            g_value_get_string (&(msg->values[5])),
                                            g_value_get_uint (&(msg->values[1])),
                                            g_value_get_uint (&(msg->values[2])));
            g_ptr_array_add (messages, message);

            if (acknowledge) {
                tpa_text_channel_acknowledge_message (self, message);
                g_ptr_array_add (self->priv->log, message);
            }
        }
        g_ptr_array_free (ret, TRUE);
    }

    return messages;
}

/**
 * tpa_text_channel_set_log_size:
 * @self: #TpaTextChannel instance
 * @log_size: new log size.
 *
 * Set number of logged messages available.
 */
void
tpa_text_channel_set_log_size (TpaTextChannel *self,
                               guint log_size)
{
    VERBOSE ("(%p, %d)", self, log_size);

    g_assert (self);

    self->priv->log_size = log_size;
    VERBOSE ("return");
}

/**
 * tpa_text_channel_set_log_size:
 * @self: #TpaTextChannel instance.
 * @returns: log size of self channel.
 *
 * Get number of logged messages available.
 */
guint
tpa_text_channel_get_log_size (TpaTextChannel *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %d", self->priv->log_size);
    return self->priv->log_size;
}

/**
 * tpa_text_channel_acknowledge_message:
 * @self: #TpaTextChannel instance.
 * @message: #TpaTextChannel instance.
 */
void
tpa_text_channel_acknowledge_message (TpaTextChannel *self,
                                      TpaTextMessage *message)
{
    GError *error = NULL;
    GArray *ids;
    gint id;

    VERBOSE ("(%p, %p)", self, message);
    g_assert (self);
    g_return_if_fail (message != NULL);

    if ((id = tpa_text_message_get_id (message))) {
        ids = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (ids, id);
        if (!org_freedesktop_Telepathy_Channel_Type_Text_acknowledge_pending_messages (self->priv->proxy, ids, &error)
           || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
            return;
        }
    }
}

/**
 * tpa_text_channel_set_local_chat_state:
 * @self: #TpaTextChannel instance.
 * @state: #TpaChatState value.
 * @returns: TRUE if the state was updated.
 *
 * Update chat state.
 */
gboolean
tpa_text_channel_set_local_chat_state (TpaTextChannel *self,
                                       TpaChatState state)
{
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, state);
    g_assert (self);

    if (!self->priv->chat_state_proxy)
        return FALSE;

    if (self->priv->current_chat_state != state)
        if (!org_freedesktop_Telepathy_Channel_Interface_ChatState_set_chat_state (self->priv->chat_state_proxy, state, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
            return FALSE;
        }

    return TRUE;
}

/**
 * tpa_text_channel_get_local_chat_state:
 * @self: #TpaTextChannel instance.
 * @returns: The current chat state.
 */
TpaChatState
tpa_text_channel_get_local_chat_state (TpaTextChannel *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    return self->priv->current_chat_state;
}
