/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-contact-list-priv.h"
#include "tpa-contact-base-priv.h"
#include "tpa-contact-priv.h"
#include "tpa-handle-factory-priv.h"
#include "tpa-channel-target-priv.h"
#include "tpa-channel.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-bindings.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

/* Signals */
enum {
    SUBSCRIPTION_ACCEPTED,
    AUTHORIZATION_REQUESTED,
    LOADED,
    LAST_SIGNAL
};

/* Args */
enum
{
    ARG_0,
    ARG_LOADED
};

/* Lists */
typedef enum
{
    LIST_SUBSCRIBE,
    LIST_PUBLISH,
    LIST_HIDE,
    LIST_DENY
} TpaContactListType;

struct _TpaContactListPrivate {
    GHashTable *contacts;
    DBusGProxy *proxy_subscribe;
    DBusGProxy *proxy_publish;
    DBusGProxy *proxy_hide;
    DBusGProxy *proxy_deny;
    TpaHandleFactory *handle_factory;
    gboolean loaded;
    gboolean fetched;
    gboolean local;
    gboolean disposed;
};

typedef struct {
    gpointer key;
    gpointer value;
} TpaHashEntry;

static guint tpa_contact_list_signals[LAST_SIGNAL] = { 0 };
static const gchar *lists[5] = { "subscribe", "publish", "hide", "deny", NULL };

static void         tpa_contact_list_load_contacts      (TpaContactList *self,
                                                         TpaContactListType type,
                                                         GArray *contacts,
                                                         TpaSubscriptionStatus subscription,
                                                         TpaAuthorizationStatus authorization,
                                                         gboolean hide,
                                                         gboolean block);

G_DEFINE_TYPE(TpaContactList, tpa_contact_list, TPA_TYPE_OBJECT)

/**
 * Helper function for _hash_table_to_array()
 */
static void
_hash_table_to_array_iterator (gpointer key, gpointer value, GArray* array)
{
    TpaHashEntry hash_entry;

    hash_entry.key = key;
    hash_entry.value = value;

    g_array_append_val (array, hash_entry);
}

/**
 * Hashtable to Array
 */
static GArray*
_hash_table_to_array (GHashTable* hash_table)
{
    GArray* array;

    g_assert (hash_table != NULL);

    array = g_array_sized_new (FALSE, FALSE, sizeof (TpaHashEntry), g_hash_table_size (hash_table));

    g_hash_table_foreach (hash_table, (GHFunc) _hash_table_to_array_iterator, array);

    return array;
}

/**
 * Value to String
 */
static gchar*
_value_to_string (const GValue* value)
{
    gchar* string = NULL;

    switch (G_VALUE_TYPE (value))
    {
        case G_TYPE_UINT:
            string = g_strdup_printf ("%u", g_value_get_uint (value));
            break;

        case G_TYPE_INT:
            string = g_strdup_printf ("%u", g_value_get_int (value));
            break;

        case G_TYPE_STRING:
            string = g_value_dup_string (value);
            break;

        case G_TYPE_BOOLEAN:
            if (g_value_get_boolean (value) == TRUE)
            {
                string = g_strdup ("True");
            }
            else
            {
                string = g_strdup ("False");
            }
            break;

        default:
            g_critical ("_value_to_string: Unknown/unhandled GValue type.");
            string = g_strdup ("_value_to_string: Unknown/unhandled GValue type.");
    }

    return string;
}

static gchar*
_presence_get_message (GHashTable *opt_params)
{
    GArray* opt_params_array = NULL;
    TpaHashEntry hash_entry;
    const gchar* opt_param_name;
    gchar* opt_param_value = NULL;
    gchar *msg_str = NULL;
    guint i;

    opt_params_array = _hash_table_to_array (opt_params);

    for (i = 0; i < opt_params_array->len; i++)
    {
        hash_entry = g_array_index (opt_params_array, TpaHashEntry, i);
        opt_param_name = (gchar*) hash_entry.key;
        opt_param_value = _value_to_string ((GValue*) hash_entry.value);

        if (g_str_equal(opt_param_name, "message"))
        {
            msg_str = g_strdup(opt_param_value);
            goto CLEANUP;
        }
        g_free (opt_param_value);
    }

CLEANUP:
    g_free (opt_param_value);
    g_array_free (opt_params_array, TRUE);
    return msg_str;
}

static void
proxy_avatar_updated_signal (DBusGProxy *proxy,
                             guint id,
                             gchar *new_avatar_token,
                             gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);
    TpaContactBase *contact;

    VERBOSE ("(%p, %d, %p, %p)", proxy, id, new_avatar_token, user_data);
    g_assert (self);

    if ((contact = g_hash_table_lookup (self->priv->contacts, GUINT_TO_POINTER (id))))
        tpa_contact_base_avatar_updated (contact, new_avatar_token);

    VERBOSE ("return");
}

static void
proxy_aliases_changed_signal (DBusGProxy *proxy,
                              GPtrArray* aliases,
                              gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);
    guint i;

    VERBOSE ("(%p, %p, %p)", proxy, aliases, user_data);
    g_assert (self);

    for (i=0; i < aliases->len; i++)
    {
        TpaContactBase *contact;
        GValueArray *alias_change = g_ptr_array_index (aliases, i);
        guint id = g_value_get_uint (&(alias_change->values[0]));
        gchar *alias = g_value_dup_string (&(alias_change->values[1]));

        if ((contact = g_hash_table_lookup (self->priv->contacts, GUINT_TO_POINTER (id))))
            tpa_contact_base_alias_changed (contact, alias);

        g_free (alias);
    }
    VERBOSE ("return");
}

static void
proxy_presence_update_signal (DBusGProxy *proxy,
                              GHashTable* presence_update,
                              gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);
    GArray* presence_update_array = NULL;
    GArray* statuses_array = NULL;
    GValueArray* contact_presence;
    GHashTable* statuses;
    GHashTable* opt_params;
    GValue* gvalue;
    const gchar* status_id;
    guint timestamp;
    guint i;

    VERBOSE ("(%p, %p, %p)", proxy, presence_update, user_data);
    g_assert (self);

    presence_update_array = _hash_table_to_array (presence_update);

    for (i = 0; i < presence_update_array->len; i++)
    {
        TpaContactBase *contact;
        TpaHashEntry hash_entry = g_array_index (presence_update_array, TpaHashEntry, i);
        guint id;

        id = GPOINTER_TO_UINT (hash_entry.key);

        if ((contact = g_hash_table_lookup (self->priv->contacts, GUINT_TO_POINTER (id)))) {
            contact_presence = (GValueArray*) (hash_entry.value);

            g_assert (contact_presence->n_values == 2);

            gvalue = g_value_array_get_nth (contact_presence, 0);
            timestamp = g_value_get_uint (gvalue);

            gvalue = g_value_array_get_nth (contact_presence, 1);
            statuses = (GHashTable*) g_value_get_boxed (gvalue);
            g_assert (statuses != NULL);

            statuses_array = _hash_table_to_array (statuses);

            hash_entry = g_array_index (statuses_array, TpaHashEntry, 0);
            status_id = (gchar*) (hash_entry.key);
            opt_params = (GHashTable*) (hash_entry.value);

            tpa_contact_base_presence_updated (contact,
                                               status_id,
                                               _presence_get_message (opt_params));
            /* TODO: if we have to handle more than one presence status, we need the code bellow */
            /*
            if (statuses_array->len > 1)
            {
                for (i = 1; i < statuses_array->len; i++)
                {
                    hash_entry = g_array_index (statuses_array, TpaHashEntry, i);
                    status_id = (gchar*) (hash_entry.key);
                    opt_params = (GHashTable*) (hash_entry.value);
                    message = _presence_get_message(opt_params);
                }
            }
            */
        }
    }

    g_array_free (presence_update_array, TRUE);
    if (statuses_array) g_array_free (statuses_array, TRUE);
    VERBOSE ("return");
}

static void
proxy_capabilities_changed_signal (DBusGProxy *proxy,
                                   GPtrArray *caps,
                                   gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);
    guint i;

    VERBOSE ("(%p, %p)", caps, user_data);

    for (i = 0; i < caps->len; i++) {
        TpaContactBase *contact;
        TpaCapability capability = TPA_CAPABILITY_NONE;
        GValueArray *caps_change = g_ptr_array_index (caps, i);
        guint id = g_value_get_uint (&(caps_change->values[0]));
        gchar *type = g_value_dup_string (&(caps_change->values[1]));
        guint specific_caps = g_value_get_uint (&(caps_change->values[5]));

        if ((contact = g_hash_table_lookup (self->priv->contacts, GUINT_TO_POINTER (id)))) {
            if (g_str_equal (type, TPA_INTERFACE_TEXT))
                capability |= TPA_CAPABILITY_TEXT;
            if (g_str_equal (type, TPA_INTERFACE_STREAMED_MEDIA)) {
                capability |= TPA_CAPABILITY_AUDIO;
                if (specific_caps == 2)
                    capability |= TPA_CAPABILITY_VIDEO;
            }
            tpa_contact_base_capabilities_changed (contact, capability);
        }
        g_free (type);
    }

    VERBOSE ("return");
}

/**
 * Subscribe list Members Changed
 */
static void
proxy_subscribe_members_changed (gpointer proxy,
                                 const gchar *message,
                                 GArray *added,
                                 GArray *removed,
                                 GArray *local_pending,
                                 GArray *remote_pending,
                                 guint actor,
                                 guint reason,
                                 gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);

    VERBOSE ("(%s, %p, %p, %p, %p, %d, %d, %p)", message, added, removed, local_pending,
            remote_pending, actor, reason, user_data);
    g_assert (self);

    /* load added members */
    tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, added,
                                    TPA_SUBSCRIPTION_STATUS_CURRENT, 0, 0, 0);

    /* load removed members */
    tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, removed,
                                    TPA_SUBSCRIPTION_STATUS_NONE, 0, 0, 0);

    /* load local pending members */
    tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, local_pending,
                                    TPA_SUBSCRIPTION_STATUS_LOCAL_PENDING, 0, 0, 0);

    /* load remote pending members */
    tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, remote_pending,
                                    TPA_SUBSCRIPTION_STATUS_REMOTE_PENDING, 0, 0, 0);
}

/**
 * Subscribe list Group Flags Changed
 */
static void
proxy_subscribe_group_flags_changed (gpointer proxy,
                                     guint added,
                                     guint removed,
                                     gpointer user_data)
{
// TODO:    TpaContact* self = TPA_CONTACT (user_data);
    VERBOSE ("(%d, %d, %p)", added, removed, user_data);
}

/**
 * Publish list Members Changed
 */
static void
proxy_publish_members_changed (gpointer proxy,
                               const gchar *message,
                               GArray *added,
                               GArray *removed,
                               GArray *local_pending,
                               GArray *remote_pending,
                               guint actor,
                               guint reason,
                               gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);

    VERBOSE ("(%s, %p, %p, %p, %p, %d, %d, %p)", message, added, removed, local_pending,
            remote_pending, actor, reason, user_data);
    g_assert (self);

    /* load added members */
    tpa_contact_list_load_contacts (self, LIST_PUBLISH, added, 0,
                                    TPA_AUTHORIZATION_STATUS_AUTHORIZED, 0, 0);

    /* load removed members */
    tpa_contact_list_load_contacts (self, LIST_PUBLISH, removed, 0,
                                    TPA_AUTHORIZATION_STATUS_NON_EXISTENT, 0, 0);

    /* load local pending members */
    tpa_contact_list_load_contacts (self, LIST_PUBLISH, local_pending, 0,
                                    TPA_AUTHORIZATION_STATUS_LOCAL_PENDING, 0, 0);
}

/**
 * Publish list Group Flags Changed
 */
static void
proxy_publish_group_flags_changed (gpointer proxy,
                                   guint added,
                                   guint removed,
                                   gpointer user_data)
{
// TODO:    TpaContact* self = TPA_CONTACT (user_data);
    VERBOSE ("(%d, %d, %p)", added, removed, user_data);
}

/**
 * Hide list Members Changed
 */
static void
proxy_hide_members_changed (gpointer proxy,
                            const gchar *message,
                            GArray *added,
                            GArray *removed,
                            GArray *local_pending,
                            GArray *remote_pending,
                            guint actor,
                            guint reason,
                            gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);

    VERBOSE ("(%s, %p, %p, %p, %p, %d, %d, %p)", message, added, removed, local_pending,
            remote_pending, actor, reason, user_data);
    g_assert (self);

    /* load added members */
    tpa_contact_list_load_contacts (self, LIST_HIDE, added, 0, 0, TRUE, 0);

    /* load removed members */
    tpa_contact_list_load_contacts (self, LIST_HIDE, removed, 0, 0, FALSE, 0);
}

/**
 * Hide list Group Flags Changed
 */
static void
proxy_hide_group_flags_changed (gpointer proxy,
                                guint added,
                                guint removed,
                                gpointer user_data)
{
// TODO:    TpaContact* self = TPA_CONTACT (user_data);
    VERBOSE ("(%d, %d, %p)", added, removed, user_data);
}

/**
 * Deny list Members Changed
 */
static void
proxy_deny_members_changed (gpointer proxy,
                            const gchar *message,
                            GArray *added,
                            GArray *removed,
                            GArray *local_pending,
                            GArray *remote_pending,
                            guint actor,
                            guint reason,
                            gpointer user_data)
{
    TpaContactList* self = TPA_CONTACT_LIST (user_data);

    VERBOSE ("(%s, %p, %p, %p, %p, %d, %d, %p)", message, added, removed, local_pending,
            remote_pending, actor, reason, user_data);
    g_assert (self);

    /* load added members */
    tpa_contact_list_load_contacts (self, LIST_DENY, added, 0, 0, 0, TRUE);

    /* load removed members */
    tpa_contact_list_load_contacts (self, LIST_DENY, removed, 0, 0, 0, FALSE);
}

/**
 * Deny list Group Flags Changed
 */
static void
proxy_deny_group_flags_changed (gpointer proxy,
                                guint added,
                                guint removed,
                                gpointer user_data)
{
// TODO:    TpaContact* self = TPA_CONTACT (user_data);
    VERBOSE ("(%d, %d, %p)", added, removed, user_data);
}

static void
tpa_contact_list_connect_signals (TpaContactList *self,
                                  TpaContactListType type,
                                  DBusGProxy *proxy)
{
    VERBOSE ("(%p, %d, %p)", self, type, proxy);
    g_assert (self);

    switch (type) {
        case LIST_SUBSCRIBE:
            tpa_object_add_proxy_with_name (TPA_OBJECT (self), TPA_INTERFACE_LIST_SUBSCRIBE, proxy);
            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_SUBSCRIBE,
                                       "MembersChanged",
                                       G_CALLBACK (proxy_subscribe_members_changed),
                                       self);

            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_SUBSCRIBE,
                                       "GroupFlagsChanged",
                                       G_CALLBACK (proxy_subscribe_group_flags_changed),
                                       self);

            VERBOSE ("subscribe channel signals connected.");
            break;
        case LIST_PUBLISH:
            tpa_object_add_proxy_with_name (TPA_OBJECT (self), TPA_INTERFACE_LIST_PUBLISH, proxy);
            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_PUBLISH,
                                       "MembersChanged",
                                       G_CALLBACK (proxy_publish_members_changed),
                                       self);

            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_PUBLISH,
                                       "GroupFlagsChanged",
                                       G_CALLBACK (proxy_publish_group_flags_changed),
                                       self);

            VERBOSE ("publish channel signals connected.");
            break;
        case LIST_HIDE:
            tpa_object_add_proxy_with_name (TPA_OBJECT (self), TPA_INTERFACE_LIST_HIDE, proxy);
            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_HIDE,
                                       "MembersChanged",
                                       G_CALLBACK (proxy_hide_members_changed),
                                       self);

            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_HIDE,
                                       "GroupFlagsChanged",
                                       G_CALLBACK (proxy_hide_group_flags_changed),
                                       self);

            VERBOSE ("hide channel signals connected.");
            break;
        case LIST_DENY:
            tpa_object_add_proxy_with_name (TPA_OBJECT (self), TPA_INTERFACE_LIST_DENY, proxy);
            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_DENY,
                                       "MembersChanged",
                                       G_CALLBACK (proxy_deny_members_changed),
                                       self);

            tpa_object_connect_signal (TPA_OBJECT (self),
                                       TPA_INTERFACE_LIST_DENY,
                                       "GroupFlagsChanged",
                                       G_CALLBACK (proxy_deny_group_flags_changed),
                                       self);

            VERBOSE ("block channel signals connected.");
            break;
    }

   VERBOSE ("return");
}

static GObject*
tpa_contact_list_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaContactList *self;
    DBusGProxy *proxy;
    guint i;


    object = G_OBJECT_CLASS (tpa_contact_list_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_CONTACT_LIST (object);

    self->priv->handle_factory = tpa_handle_factory_new (TPA_OBJECT (self));
    proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CONNECTION);

    for (i = 0; lists[i]; i++) {
        gchar *object_path = NULL;
        GError *error = NULL;
        TpaHandle *handle;

        handle = tpa_handle_factory_create_handle (self->priv->handle_factory,
                                                   TPA_HANDLE_TYPE_LIST, lists[i]);
        if (!handle) continue;
        else if (!org_freedesktop_Telepathy_Connection_request_channel (proxy,
            TPA_INTERFACE_CONTACT_LIST, TPA_HANDLE_TYPE_LIST, tpa_handle_get_id (handle), FALSE, &object_path, &error)
            || error) {
            VERBOSE ("%s", error->message);
            g_error_free (error);
        }
        else {
            VERBOSE ("new contact list channel on %s", object_path);
            if (i == LIST_SUBSCRIBE &&
                (self->priv->proxy_subscribe = dbus_g_proxy_new_from_proxy (proxy, TPA_INTERFACE_GROUP, object_path))) {
                tpa_contact_list_connect_signals (self, i, self->priv->proxy_subscribe);
            }
            else if (i == LIST_PUBLISH &&
                (self->priv->proxy_publish = dbus_g_proxy_new_from_proxy (proxy, TPA_INTERFACE_GROUP, object_path))) {
                tpa_contact_list_connect_signals (self, i, self->priv->proxy_publish);
            }
            else if (i == LIST_HIDE &&
                (self->priv->proxy_hide = dbus_g_proxy_new_from_proxy (proxy, TPA_INTERFACE_GROUP, object_path))) {
                tpa_contact_list_connect_signals (self, i, self->priv->proxy_hide);
            }
            else if (i == LIST_DENY &&
                (self->priv->proxy_deny = dbus_g_proxy_new_from_proxy (proxy, TPA_INTERFACE_GROUP, object_path))) {
                tpa_contact_list_connect_signals (self, i, self->priv->proxy_deny);
            }
        }
    }
    g_object_unref (proxy);

    if (tpa_object_has_proxy (TPA_OBJECT (self), TPA_INTERFACE_AVATARS)) {
       tpa_object_connect_signal (TPA_OBJECT (self),
                                  TPA_INTERFACE_AVATARS,
                                  "AvatarUpdated",
                                  G_CALLBACK (proxy_avatar_updated_signal),
                                  self);
    }
    if (tpa_object_has_proxy (TPA_OBJECT (self), TPA_INTERFACE_ALIASING)) {
        tpa_object_connect_signal (TPA_OBJECT (self),
                                   TPA_INTERFACE_ALIASING,
                                   "AliasesChanged",
                                   G_CALLBACK (proxy_aliases_changed_signal),
                                   self);
    }
    if (tpa_object_has_proxy (TPA_OBJECT (self), TPA_INTERFACE_PRESENCE)) {
        tpa_object_connect_signal (TPA_OBJECT (self),
                                   TPA_INTERFACE_PRESENCE,
                                   "PresenceUpdate",
                                   G_CALLBACK (proxy_presence_update_signal),
                                   self);
    }
    if (tpa_object_has_proxy (TPA_OBJECT (self), TPA_INTERFACE_CAPABILITIES)) {
        tpa_object_connect_signal (TPA_OBJECT (self),
                                   TPA_INTERFACE_CAPABILITIES,
                                   "CapabilitiesChanged",
                                   G_CALLBACK (proxy_capabilities_changed_signal),
                                   self);
    }
    return object;
}

static void
tpa_contact_list_dispose (GObject *object)
{
    TpaContactList *self = TPA_CONTACT_LIST (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->handle_factory)
        g_object_unref (self->priv->handle_factory);

    if (self->priv->proxy_subscribe)
        g_object_unref (self->priv->proxy_subscribe);
    if (self->priv->proxy_publish)
        g_object_unref (self->priv->proxy_publish);
    if (self->priv->proxy_hide)
        g_object_unref (self->priv->proxy_hide);
    if (self->priv->proxy_deny)
        g_object_unref (self->priv->proxy_deny);

    if (self->priv->contacts)
       g_hash_table_destroy (self->priv->contacts);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_contact_list_parent_class)->dispose (object);
}

static void
tpa_contact_list_set_property (GObject *object,
                               guint prop_id,
                               const GValue *value,
                               GParamSpec *pspec)
{
    TpaContactList *self = TPA_CONTACT_LIST (object);

    switch (prop_id) {
        case ARG_LOADED:
            self->priv->loaded = g_value_get_boolean (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_contact_list_class_init (TpaContactListClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_contact_list_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaContactListPrivate));

    gobject_class->dispose = tpa_contact_list_dispose;
    gobject_class->constructor = tpa_contact_list_constructor;
    gobject_class->set_property = tpa_contact_list_set_property;

    g_object_class_install_property (gobject_class,
                                     ARG_LOADED,
                                     g_param_spec_boolean ("loaded",
                                     "loaded",
                                     "loaded",
                                     FALSE,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_WRITABLE));

    /* Let's create our signals */
    tpa_contact_list_signals[SUBSCRIPTION_ACCEPTED] = g_signal_new ("subscription-accepted",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE,
        1,
        TPA_TYPE_CONTACT);

    tpa_contact_list_signals[AUTHORIZATION_REQUESTED] = g_signal_new ("authorization-requested",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE,
        1,
        TPA_TYPE_CONTACT);

    tpa_contact_list_signals[LOADED] = g_signal_new ("loaded",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE,
        0);
}

static void
tpa_contact_list_init (TpaContactList *self)
{
    self->priv = TPA_CONTACT_LIST_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->local = FALSE;
    self->priv->loaded = FALSE;
    self->priv->fetched = FALSE;
    self->priv->contacts =
        g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, g_object_unref);
    self->priv->handle_factory = NULL;
}

static void
_get_all_values (gpointer key,
                 gpointer value,
                 gpointer user_data)
{
    g_ptr_array_add (user_data, g_object_ref (value));
}

static gboolean
tpa_contact_list_load_presence (TpaContactList *self,
				guint id)
{
    GError* error = NULL;
    DBusGProxy *proxy_presence;
    GArray *members;

    VERBOSE ("(%p)", self);
    g_assert (self);

    proxy_presence = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_PRESENCE);
    members = g_array_new (FALSE, FALSE, sizeof (guint));
    g_array_append_val (members, id);

    if (!org_freedesktop_Telepathy_Connection_Interface_Presence_request_presence (proxy_presence, members, &error)
        || error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    g_object_unref (proxy_presence);
    g_array_free (members, TRUE);
    VERBOSE ("return false");
    return FALSE;
}

static void
tpa_contact_list_load (TpaContactList *self)
{
    GError* error = NULL;
    GArray *members;
    GArray *local_pending;
    GArray *remote_pending;

    VERBOSE ("(%p)", self);
    g_assert (self);

    /* Load contacts from subscribe list */
    if (self->priv->proxy_subscribe) {
        if (!org_freedesktop_Telepathy_Channel_Interface_Group_get_all_members (self->priv->proxy_subscribe, &members, &local_pending, &remote_pending, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else {
            tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, members,
                                            TPA_SUBSCRIPTION_STATUS_CURRENT, 0, 0, 0);

            tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, local_pending,
                                            TPA_SUBSCRIPTION_STATUS_LOCAL_PENDING, 0, 0, 0);

            tpa_contact_list_load_contacts (self, LIST_SUBSCRIBE, remote_pending,
                                            TPA_SUBSCRIPTION_STATUS_REMOTE_PENDING, 0, 0, 0);
            g_array_free (members, TRUE);
            g_array_free (local_pending, TRUE);
            g_array_free (remote_pending, TRUE);
        }
    }

    /* Load local pending contacts from publish list */
    if (self->priv->proxy_publish) {
        if (!org_freedesktop_Telepathy_Channel_Interface_Group_get_all_members (self->priv->proxy_publish, &members, &local_pending, &remote_pending, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else {
            tpa_contact_list_load_contacts (self, LIST_PUBLISH, members,
                                            0, TPA_AUTHORIZATION_STATUS_AUTHORIZED, 0, 0);

            tpa_contact_list_load_contacts (self, LIST_PUBLISH, local_pending,
                                            0, TPA_AUTHORIZATION_STATUS_LOCAL_PENDING, 0, 0);
            g_array_free (members, TRUE);
            g_array_free (local_pending, TRUE);
            g_array_free (remote_pending, TRUE);
       }
    }

    VERBOSE ("return");
}

static void
tpa_contact_list_load_contacts (TpaContactList *self,
                                TpaContactListType type,
                                GArray *contacts,
                                TpaSubscriptionStatus subscription,
                                TpaAuthorizationStatus authorization,
                                gboolean hide,
                                gboolean block)
{
    guint i;

    VERBOSE ("(%p, %d, %p, %d, %d, %s, %s)", self, type, contacts, subscription,
            authorization, hide ? "true" : "false", block ? "true" : "false");
    g_assert (self);

    for (i = 0; i < contacts->len; i++) {
        guint id = g_array_index (contacts, guint, i);
        TpaContact *contact =
            g_hash_table_lookup (self->priv->contacts, GUINT_TO_POINTER (id));

        if (!contact) {
            /* Do not create contacts if they arent in any list */
            if (subscription || authorization || hide || block) {
                TpaHandle *handle =
                    tpa_handle_factory_create_handle_by_id (self->priv->handle_factory,
                                                            TPA_HANDLE_TYPE_CONTACT,
                                                            id);
                contact = tpa_contact_list_add_with_handle (self, handle);
            }
        }
        if (contact) {
            const gchar *uri;
            switch (type) {
                case LIST_SUBSCRIBE:
                    tpa_contact_set_subscription_status (contact, subscription);
                    if (subscription == TPA_SUBSCRIPTION_STATUS_CURRENT){
                       /* Request presence only when it is loaded */
                        if (self->priv->loaded)
                                tpa_contact_list_load_presence (self, id);

                        if (self->priv->fetched) {
                            g_signal_emit (self, tpa_contact_list_signals[SUBSCRIPTION_ACCEPTED], 0, contact);
                            if (DEBUGGING) {
                                uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET(contact));
                                INFO ("[subscription-accepted %s]", uri);
                            }
                        }
                    }
                    break;
                case LIST_PUBLISH:
                    tpa_contact_set_authorization_status (contact, authorization);
                    if (authorization == TPA_AUTHORIZATION_STATUS_LOCAL_PENDING) {
                        if (DEBUGGING) {
                            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET(contact));
                            INFO ("[authorization-requested %s]", uri);
                        }

                        if (self->priv->fetched) {
                            g_signal_emit (self, tpa_contact_list_signals[AUTHORIZATION_REQUESTED], 0, contact);
                            if (DEBUGGING) {
                                uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET(contact));
                                INFO ("[authorization-requested %s]", uri);
                            }
                        }
                   }
                    break;
                case LIST_HIDE:
                    tpa_contact_set_hide (contact, hide);
                    break;
                case LIST_DENY:
                    tpa_contact_set_block (contact, block);
                    break;
            }
        }
    }
}

/**
 * tpa_contact_list_new:
 * @object: TpaObject instance.
 * @returns: #TpaContactList new instance.
 *
 * Create a new instance of #TpaContactList.
 */
TpaContactList *
tpa_contact_list_new (TpaObject *object,
                      gboolean loaded)
{
    TpaContactList *self;

    VERBOSE ("(%p)", object);
    g_return_val_if_fail (object != NULL, NULL);

    self = TPA_CONTACT_LIST (g_object_new (TPA_TYPE_CONTACT_LIST,
                             "object", object,
                             "loaded", loaded,
                             NULL));

    return self;
}

TpaContact *
tpa_contact_list_add_with_handle (TpaContactList *self,
                                  TpaHandle *handle)
{
    TpaContact *contact;
    guint id;

    VERBOSE ("(%p, %p)", self, handle);
    g_assert (self);
    g_return_val_if_fail (handle != NULL, NULL);

    contact = tpa_contact_new (handle, TPA_OBJECT (self));
    id = tpa_handle_get_id (handle);
    g_hash_table_insert (self->priv->contacts, GUINT_TO_POINTER (id), contact);

    VERBOSE ("return %p", contact);
    return contact;
}

/**
 * tpa_contact_list_add:
 * @self: #TpaContactList instance.
 * @uri: Contact uri to be added.
 * @returns: #TpaContact contact added.
 *
 * Add contact to contact list by its string uri.
 */
const TpaContact *
tpa_contact_list_add (TpaContactList *self,
                      const gchar *uri)
{
    TpaContact *contact;
    TpaHandle *handle;

    VERBOSE ("(%p, %s)", self, uri);
    g_assert (self);
    g_return_val_if_fail (uri != NULL, NULL);

    handle = tpa_handle_factory_create_handle (self->priv->handle_factory,
                                               TPA_HANDLE_TYPE_CONTACT, uri);
    contact = tpa_contact_list_add_with_handle (self, handle);

    INFO ("[%s] added", uri);
    VERBOSE ("return %p", contact);
    return contact;
}

/**
 * tpa_contact_list_remove:
 * @self: #TpaContactList instance.
 * @uri: #TpaContact to be removed.
 *
 * Removes contact from contact list.
 */
void
tpa_contact_list_remove (TpaContactList *self,
                         TpaContact *contact)
{
    TpaHandle *handle;
    guint id;
    const gchar *uri;

    VERBOSE ("(%p, %p)", self, contact);
    g_assert (self);
    g_return_if_fail (contact != NULL);

    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (contact));
    tpa_contact_subscribe (contact, FALSE);
    tpa_contact_authorize (contact, FALSE);
    tpa_contact_hide (contact, FALSE);
    tpa_contact_block (contact, FALSE);
    id = tpa_handle_get_id (handle);
    g_hash_table_remove (self->priv->contacts, &id);

    uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (contact));
    INFO ("[%s] removed", uri);
    VERBOSE ("return");
    return;
}

/**
 * tpa_contact_list_get_known:
 * @self: #TpaContactList instance.
 * @returns: #GPtrArray with known contacts.
 *
 * Get contacts all contacts available.
 */
GPtrArray *
tpa_contact_list_get_known (TpaContactList *self)
{
    GPtrArray *contacts = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->fetched) {
        tpa_contact_list_load (self);
        self->priv->fetched = TRUE;
    }
    contacts = g_ptr_array_new ();
    g_hash_table_foreach (self->priv->contacts, _get_all_values, contacts);

    VERBOSE ("return %p", contacts);
    return contacts;
}

/**
 * tpa_contact_list_get_subscribed:
 * @self: #TpaContactList instance.
 * @returns: #GPtrArray with susbscribed #TpaContact contacts.
 *
 * Get contacts that user is susbscribed to see its presences.
 */
GPtrArray *
tpa_contact_list_get_subscribed (TpaContactList *self)
{
    GPtrArray *contacts = NULL;
    guint i;

    VERBOSE ("(%p)", self);
    g_assert (self);

    contacts = tpa_contact_list_get_known (self);

    for (i = 0; i < contacts->len; i++) {
        TpaContact *contact = g_ptr_array_index (contacts, i);

        if (tpa_contact_get_subscription_status (contact) !=
            TPA_SUBSCRIPTION_STATUS_CURRENT)
            g_ptr_array_remove_index (contacts, i--);
    }

    VERBOSE ("return %p", contacts);
    return contacts;
}

/**
 * tpa_contact_list_get_authorized:
 * @self: #TpaContactList instance.
 * @returns: #GPtrArray with authorized #TpaContact contacts.
 *
 * Get contacts that are authorized to see user presence.
 */
GPtrArray *
tpa_contact_list_get_authorized (TpaContactList *self)
{
    GPtrArray *contacts = NULL;
    guint i;

    VERBOSE ("(%p)", self);

    contacts = tpa_contact_list_get_known (self);

    for (i = 0; i < contacts->len; i++) {
        TpaContact *contact = g_ptr_array_index (contacts, i);

        if (tpa_contact_get_authorization_status (contact) !=
            TPA_AUTHORIZATION_STATUS_AUTHORIZED)
            g_ptr_array_remove_index (contacts, i--);
    }

    VERBOSE ("return %p", contacts);
    return contacts;
}

/**
 * tpa_contact_list_get_hidden:
 * @self: #TpaContactList instance.
 * @returns: #GPtrArray with hidden #TpaContact contacts.
 *
 * Get contacts that user is hidden.
 */
GPtrArray *
tpa_contact_list_get_hidden (TpaContactList *self)
{
    GPtrArray *contacts = NULL;
    guint i;

    VERBOSE ("(%p)", self);

    contacts = tpa_contact_list_get_known (self);

    for (i = 0; i < contacts->len; i++) {
        TpaContact *contact = g_ptr_array_index (contacts, i);

        if (!tpa_contact_is_hidden (contact))
            g_ptr_array_remove_index (contacts, i--);
    }

    VERBOSE ("return %p", contacts);
    return contacts;
}

/**
 * tpa_contact_list_get_blocked:
 * @self: #TpaContactList instance.
 * @returns: #GPtrArray with blocked #TpaContact contacts.
 *
 * Get contacts that are blocked.
 */
GPtrArray *
tpa_contact_list_get_blocked (TpaContactList *self)
{
    GPtrArray *contacts = NULL;
    guint i;

    VERBOSE ("(%p)", self);

    contacts = tpa_contact_list_get_known (self);

    for (i = 0; i < contacts->len; i++) {
        TpaContact *contact = g_ptr_array_index (contacts, i);

        if (!tpa_contact_is_blocked (contact))
            g_ptr_array_remove_index (contacts, i--);
    }

    VERBOSE ("return %p", contacts);
    return contacts;
}

/**
 * tpa_contact_list_get_contact:
 * @self: #TpaContactList instance
 * @uri: Given uri.
 * @returns: #TpaContact corresponding to the given uri.
 *
 * Get contact by its string uri.
 */
TpaContact *
tpa_contact_list_get_contact (TpaContactList *self,
                              const gchar *uri)
{
    GPtrArray *contacts;
    TpaContact *contact = NULL;
    const gchar *str;
    guint i;

    VERBOSE ("(%p, %s)", self, uri);
    g_return_val_if_fail (uri, NULL);

    contacts = tpa_contact_list_get_known (self);

    for (i = 0; i < contacts->len; i++) {
        TpaChannelTarget *target = g_ptr_array_index (contacts, i);
        str = tpa_channel_target_get_uri (target);

        if (g_str_equal (str, uri)) {
            contact = TPA_CONTACT (target);
            break;
        }
    }

    g_ptr_array_free (contacts, FALSE);

    VERBOSE ("return %p", contact);
    return contact;
}

/**
 * tpa_contact_list_get_contact_by_id:
 * @self: #TpaContactList instance
 * @id: interger handle id
 * @returns: #TpaContact corresponding to givin id.
 *
 * WARNING: This is a private function.
 * Get contact by its handle id.
 */
const TpaContact *
tpa_contact_list_get_contact_by_id (TpaContactList *self,
                                    guint id)
{
    TpaContact *contact;

    VERBOSE ("(%p, %d)", self, id);
    g_assert (self);

    if (!self->priv->fetched) {
        tpa_contact_list_load (self);
        self->priv->fetched = TRUE;
    }

    contact = g_hash_table_lookup (self->priv->contacts,
                                   GUINT_TO_POINTER (id));

    VERBOSE ("return %p", contact);
    return contact;
}

gboolean
tpa_contact_list_is_local (TpaContactList *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->local ? "true" : "false");
    return self->priv->local;
}

