/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 * @author  Marcio Macedo <marcio.macedo@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-connection-priv.h"
#include "tpa-handle-factory-priv.h"
#include "tpa-utils.h"
#include "tpa-contact-list-priv.h"
#include "tpa-contact-base-priv.h"
#include "tpa-channel-priv.h"
#include "tpa-channel-target-priv.h"
#include "tpa-text-channel-priv.h"
#include "tpa-stream-channel-priv.h"
#include "tpa-user-contact-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-bindings.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

#include <dbus/dbus-glib-bindings.h>

struct _TpaConnectionPrivate {
    gboolean disposed;
    gint status;
    GHashTable *channels;
    DBusGProxy *proxy;
    TpaHandleFactory *handle_factory;
    TpaHandle *handle;
    TpaContactList *contact_list;
    TpaUserContact *user_contact;
};

enum
{
    STATUS_CHANGED,
    CHANNEL_CREATED,
    LAST_SIGNAL
};

static guint tpa_connection_signals[LAST_SIGNAL] = { 0 };

/* Private Functions */
static void             tpa_connection_dispose              (GObject *object);
static guint            tpa_connection_get_this_handle      (TpaConnection *conn);
static DBusGProxy*      tpa_connection_request_channel      (TpaConnection *conn,
                                                             const gchar *interface,
                                                             TpaHandle *handle);
static TpaChannel *     tpa_connection_incoming_channel     (TpaConnection *conn,
                                                             const gchar *type,
                                                             const gchar *path,
                                                             guint handle_type,
                                                             guint handle_number);
static TpaChannel *     tpa_connection_incoming_text_channel(TpaConnection *conn,
                                                             DBusGProxy *channel_proxy,
                                                             guint handle_number);
static TpaChannel *     tpa_connection_incoming_stream_channel
                                                            (TpaConnection *conn,
                                                             DBusGProxy *channel_proxy,
                                                             guint handle_number);
/* Signals*/
static void             proxy_status_changed_signal         (DBusGProxy *proxy,
                                                             guint status,
                                                             guint reasob,
                                                             TpaConnection *conn);
static void             proxy_new_channel_signal            (DBusGProxy *proxy,
                                                             const gchar* obj_path,
                                                             const gchar* channel_type,
                                                             guint handle_type,
                                                             guint handle_id,
                                                             gboolean suppress_handle,
                                                             TpaConnection* conn);
static void             channel_closed_signal               (TpaChannel *channel,
                                                             gpointer user_data);

G_DEFINE_TYPE(TpaConnection, tpa_connection, TPA_TYPE_OBJECT)

static void
tpa_connection_load (TpaConnection *self,
                     gboolean loaded)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->handle) {
        INFO ("creating handle");
        self->priv->handle = tpa_handle_factory_create_handle_by_id (self->priv->handle_factory,
                                                                     TPA_HANDLE_TYPE_CONTACT,
                                                                     tpa_connection_get_this_handle (self));
    }

    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->proxy);

    if (!self->priv->contact_list) {
        INFO ("creating contact list");
        self->priv->contact_list = tpa_contact_list_new (TPA_OBJECT (self), loaded);
    }

    if (!self->priv->user_contact) {
        INFO ("creating user contact");
        self->priv->user_contact = tpa_user_contact_new (self->priv->handle, TPA_OBJECT (self));
    }

    VERBOSE ("return");
}

static GObject*
tpa_connection_constructor (GType type,
                            guint n_construct_params,
                            GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaConnection *self;

    object = G_OBJECT_CLASS (tpa_connection_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_CONNECTION (object);

    self->priv->proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CONNECTION);

    self->priv->handle_factory = tpa_handle_factory_new (TPA_OBJECT (self));

    self->priv->status = tpa_connection_get_status (self);

    if (self->priv->status == TPA_CONNECTION_STATUS_CONNECTED)
        tpa_connection_load (self, TRUE);

    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_CONNECTION,
                               "StatusChanged",
                               G_CALLBACK (proxy_status_changed_signal),
                               self);
    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_CONNECTION,
                               "NewChannel",
                               G_CALLBACK (proxy_new_channel_signal),
                               self);

   return object;
}

static void
tpa_connection_class_init (TpaConnectionClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_connection_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaConnectionPrivate));

    gobject_class->dispose = tpa_connection_dispose;
    gobject_class->constructor = tpa_connection_constructor;

    /* Let's create our signals */
    tpa_connection_signals[STATUS_CHANGED] = g_signal_new ("status-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        tpa_marshal_VOID__UINT_UINT,
        G_TYPE_NONE,
        2,
        G_TYPE_UINT,
        G_TYPE_UINT);

    tpa_connection_signals[CHANNEL_CREATED] = g_signal_new ("channel-created",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__OBJECT,
        G_TYPE_NONE,
        1,
        TPA_TYPE_CHANNEL);
}

static void
tpa_connection_init (TpaConnection *self)
{
    self->priv = TPA_CONNECTION_GET_PRIVATE (self);
    self->priv->channels = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
    self->priv->disposed = FALSE;
    self->priv->handle = NULL;
    self->priv->handle_factory = NULL;
    self->priv->user_contact = NULL;
    self->priv->status = TPA_CONNECTION_STATUS_DISCONNECTED;
}

static void
tpa_connection_dispose (GObject *object)
{
    TpaConnection *self = TPA_CONNECTION (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->contact_list)
        g_object_unref (self->priv->contact_list);

    if (self->priv->user_contact)
        g_object_unref (self->priv->user_contact);

    if (self->priv->proxy)
        g_object_unref (self->priv->proxy);

    if (self->priv->channels)
        g_hash_table_destroy (self->priv->channels);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_connection_parent_class)->dispose (object);
}

/**
 * tpa_connection_new:
 * @service: Service name.
 * @path: Object path.
 * @protocol: Protocol name.
 * @returns: new #TpaConnection instance.
 */
TpaConnection*
tpa_connection_new (const gchar *service,
                    const gchar *path,
                    const gchar *protocol)
{
    TpaConnection *self;

    VERBOSE ("(%s, %s, %s)", service, path, protocol);
    g_return_val_if_fail (service != NULL, NULL);
    g_return_val_if_fail (path != NULL, NULL);
    g_return_val_if_fail (protocol != NULL, NULL);

    self = g_object_new (TPA_TYPE_CONNECTION,
                         "service", service,
                         "path", path,
                         "interface", TPA_INTERFACE_CONNECTION,
                         NULL);

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_connection_connect:
 * @self: #TpaConnection instance.
 *
 * Request the connection establishment.
 * Result will be return asynchronously by signal status-changed.
 */
void
tpa_connection_connect (TpaConnection *self)
{
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Connection_connect (self->priv->proxy, &error) || error)
    {
        if (error)
        {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }
    VERBOSE ("return");
}

/**
 * tpa_connection_disconnect:
 * @self: #TpaConnection instance.
 *
 * Closes the connection.
 */
void
tpa_connection_disconnect (TpaConnection *self)
{
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Connection_disconnect (self->priv->proxy, &error) || error)
    {
        if(error)
        {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }
    VERBOSE ("return");
}

/**
 * tpa_connection_get_protocol:
 * @self: #TpaConnection instance.
 *
 * Return the connection's protocol.
 */
const gchar*
tpa_connection_get_protocol (TpaConnection *self)
{
    gchar *protocol;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Connection_get_protocol (self->priv->proxy, &protocol, &error)
       || error)
    {
        ERROR ("%s", error->message);
        g_error_free (error);
    }
    VERBOSE ("return %s", protocol);
    return protocol;
}

/**
 * tpa_connection_get_status:
 * @self: #TpaConnection instance.
 * @returns: #TpaConnectionStatus status.
 *
 * Return the connection status.
 */
TpaConnectionStatus
tpa_connection_get_status (TpaConnection *self)
{
    TpaConnectionStatus status;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Connection_get_status (self->priv->proxy, &status, &error)
       || error)
    {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    VERBOSE ("return %d", status);
    return status;
}

/**
 * tpa_connection_get_contactlist:
 * @self: #TpaConnection instance.
 *
 * Return connection's contact list.
 */
TpaContactList *
tpa_connection_get_contactlist (TpaConnection *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %p", self->priv->contact_list);
    return g_object_ref (self->priv->contact_list);
}

/**
 * tpa_connection_get_open_channels:
 * @self: #TpaConnection instance.
 * @returns: #GPtrArray array of #TpaChannel channels.
 *
 * Return all currently open channels on the connection.
 */
GPtrArray *
tpa_connection_get_open_channels (TpaConnection *self)
{
    GPtrArray* channels;
    GPtrArray* ret;
    GValueArray* value;
    GError *error = NULL;
    guint i;

    // Channel fields
    gchar *obj_path;
    gchar *type;
    guint handle_type;
    guint handle_number;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Connection_list_channels (self->priv->proxy, &channels, &error)
       || error) {
        VERBOSE ("%s", error->message);
        g_error_free (error);
        return NULL;
    }

    ret = g_ptr_array_new ();

    for (i = 0; i < channels->len; i++) {
        TpaChannel *channel;

        value = g_ptr_array_index (channels, i);

        obj_path = g_strdup (g_value_get_boxed (&(value->values[0])));
        type = g_value_dup_string (&(value->values[1]));
        handle_type = g_value_get_uint (&(value->values[2]));
        handle_number = g_value_get_uint (&(value->values[3]));

        channel = tpa_connection_incoming_channel (self,
                                                   type,
                                                   obj_path,
                                                   handle_type,
                                                   handle_number);
        if (channel)
            g_ptr_array_add (ret, g_object_ref (channel));

        g_value_array_free (value);
    }

    INFO ("number of opened channels: %d", ret->len);

    //    g_hash_table_foreach (self->priv->channels, tpa_connection_get_channels, channels);

    return ret;
}

/**
 * tpa_connection_create_channel:
 * @self: #TpaConnection instance.
 * @type: #TpaChannelType type.
 * @target: #TpaChannelTarget instance.
 *
 * Create a new channel of the given type to the given target.
 */
void
tpa_connection_create_channel (TpaConnection *self,
                               TpaChannelType type,
                               TpaChannelTarget *target)
{
    TpaHandle *handle;

    VERBOSE ("(%p, %d, %p)", self, type, target);
    g_assert (self);
    g_return_if_fail (target != NULL);

    handle = tpa_channel_target_get_handle (target);

    if (type == TPA_CHANNEL_TYPE_TEXT) {
        tpa_connection_request_channel (self, TPA_INTERFACE_TEXT, handle);
    }
    else if (type == TPA_CHANNEL_TYPE_STREAM) {
        tpa_connection_request_channel (self, TPA_INTERFACE_STREAMED_MEDIA, handle);
    }

    VERBOSE ("return");
}

/**
 * tpa_connection_get_user_contact:
 * @self: #TpaConnection instance.
 *
 * Return connection's user contact.
 */
TpaUserContact*
tpa_connection_get_user_contact (TpaConnection *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    return g_object_ref (self->priv->user_contact);
}

static guint
tpa_connection_get_this_handle (TpaConnection *self)
{
    GError *error = NULL;
    guint handle = 0;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!(org_freedesktop_Telepathy_Connection_get_self_handle (self->priv->proxy,
                                                                &handle,
                                                                &error))) {
        ERROR ("%s", error->message);

        g_error_free (error);
    }

    INFO ("connection handle: %d", handle);

    VERBOSE ("return %d", handle);
    return handle;
}

/**
 * Request a channel to a specified type.
 *
 * @conn: #TpaConnection instance.
 * @interface: The #TpaChannelType.
 * @handle destination contact.
 * @returns a #TpaChannel instace or NULL if it fails.
 */
DBusGProxy *
tpa_connection_request_channel (TpaConnection *self,
                                const gchar *interface,
                                TpaHandle *handle)
{
    GError *error = NULL;
    gchar *obj_path;
    TpaHandleType handle_type = 0;
    guint handle_id = 0;

    g_return_val_if_fail (self != NULL, NULL);
    g_return_val_if_fail (interface != NULL, NULL);
    g_return_val_if_fail (self->priv->status == TPA_CONNECTION_STATUS_CONNECTED, NULL);

    if (handle) {
        handle_type = tpa_handle_get_handle_type (handle);
        handle_id = tpa_handle_get_id (handle);
    }

    if (!(org_freedesktop_Telepathy_Connection_request_channel (self->priv->proxy,
                                                                interface,
                                                                handle_type,
                                                                handle_id,
                                                                TRUE,
                                                                &obj_path,
                                                                &error))) {
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }

        return NULL;
    }

    return dbus_g_proxy_new_from_proxy (self->priv->proxy, interface, obj_path);
}

static TpaChannel *
tpa_connection_incoming_text_channel (TpaConnection *self,
                                      DBusGProxy *channel_proxy,
                                      guint handle_number)
{
    TpaChannel *channel;
    TpaChannelTarget *target;

    VERBOSE ("(%p, %p, %d)", self, channel_proxy, handle_number);
    g_assert (self);

    target = TPA_CHANNEL_TARGET (
        tpa_contact_list_get_contact_by_id (self->priv->contact_list, handle_number));

    channel = TPA_CHANNEL (tpa_text_channel_new (channel_proxy,
                                                 self->priv->user_contact,
                                                 target));

    INFO ("new text channel %p", channel);

    VERBOSE ("return %p", channel);
    return channel;
}

static TpaChannel *
tpa_connection_incoming_stream_channel (TpaConnection *self,
                                        DBusGProxy *channel_proxy,
                                        guint handle_number)
{
    TpaChannel *channel = NULL;
    TpaChannelTarget *target = NULL;

    VERBOSE ("(%p, %p, %d)", self, channel_proxy, handle_number);
    g_assert (self);

    if (handle_number != 0) {
        target = TPA_CHANNEL_TARGET (
            tpa_contact_list_get_contact_by_id (self->priv->contact_list, handle_number));
    }
    else { // VERY good idea to have anonymous channels...
        GError *error = NULL;
        DBusGProxy *proxy_group;
        gchar **ifaces;
        guint i;
        GArray *handles;

        if (!org_freedesktop_Telepathy_Channel_get_interfaces (channel_proxy, &ifaces, &error)
            || error) {
            ERROR ("%s", error->message);

            g_error_free (error);
        }
        else { // Look for Group interface and use first member as target
            for (i = 0; ifaces[i]; i++) {
                if (g_str_equal (TPA_INTERFACE_GROUP, ifaces[i])) {
                    proxy_group = dbus_g_proxy_new_from_proxy (channel_proxy, TPA_INTERFACE_GROUP, NULL);

                    if (!org_freedesktop_Telepathy_Channel_Interface_Group_get_remote_pending_members (proxy_group, &handles, &error)
                        || error) {
                        ERROR ("%s", error->message);

                        g_error_free (error);
                    }
                    else if (handles->len) {
                        guint id = g_array_index (handles, guint, 0);
                        target = TPA_CHANNEL_TARGET (
                            tpa_contact_list_get_contact_by_id (self->priv->contact_list, id));
                    }
                    else if (!org_freedesktop_Telepathy_Channel_Interface_Group_get_members (proxy_group, &handles, &error)
                        || error) {
                        ERROR ("%s", error->message);

                        g_error_free (error);
                    }
                    else if (handles->len) {
                        guint id = g_array_index (handles, guint, 0);
                        target = TPA_CHANNEL_TARGET (
                            tpa_contact_list_get_contact_by_id (self->priv->contact_list, id));
                    }
                }
            }
        }
        channel = TPA_CHANNEL (tpa_stream_channel_new (
                               channel_proxy,
                               self->priv->user_contact,
                               target));

        INFO ("new stream channel %p", channel);
    }

    VERBOSE ("return %p", channel);
    return channel;
}

static TpaChannel *
tpa_connection_incoming_channel (TpaConnection *self,
                                 const gchar *type,
                                 const gchar *path,
                                 guint handle_type,
                                 guint handle_number)
{
    TpaChannel *channel = NULL;

    VERBOSE ("(%p, %s, %s, %d)", self, type, path, handle_number);
    g_assert (self);

    if ((handle_type == TPA_HANDLE_TYPE_CONTACT &&
        !g_hash_table_lookup (self->priv->channels, path))
        || handle_number == 0) { // Why wouldnt gabble gives us a valid handle?
        DBusGProxy *channel_proxy = dbus_g_proxy_new_from_proxy (
                                    self->priv->proxy,
                                    TPA_INTERFACE_CHANNEL,
                                    path);

        INFO ("creating channel %s", path);
        if (g_str_equal (type, TPA_INTERFACE_TEXT)) {
            channel = tpa_connection_incoming_text_channel (self,
                                                            channel_proxy,
                                                            handle_number);
        }
        else if (g_str_equal (type, TPA_INTERFACE_STREAMED_MEDIA)) {
             channel = tpa_connection_incoming_stream_channel (self,
                                                               channel_proxy,
                                                               handle_number);
        }

        if (channel) {
            g_hash_table_insert (self->priv->channels, g_strdup (path), channel);

            /* connect to channel closed signal */
            g_signal_connect (channel, "closed",
                              G_CALLBACK (channel_closed_signal),
                              self);
        }
    }

    VERBOSE ("return %p", channel);
    return channel;
}

const gchar *
tpa_connection_get_path (TpaConnection *self)
{
    g_assert (self);

    return dbus_g_proxy_get_path (self->priv->proxy);
}

/* Signal Callbacks */
static void
proxy_status_changed_signal (DBusGProxy *proxy,
                             guint status,
                             guint reason,
                             TpaConnection *conn)
{
    VERBOSE ("(%d, %d, %p)", status, reason, conn);

    conn->priv->status = status;
    if (status == TPA_CONNECTION_STATUS_CONNECTED){
        tpa_connection_load (conn, TRUE);
    }

    INFO ("[status-changed] status: %d, reason: %d", status, reason);
    g_signal_emit(conn, tpa_connection_signals[STATUS_CHANGED], 0, status, reason);
    VERBOSE ("return");
}

static void
proxy_new_channel_signal (DBusGProxy *proxy,
                          const gchar *obj_path,
                          const gchar *channel_type,
                          guint handle_type,
                          guint handle_id,
                          gboolean suppress_handle,
                          TpaConnection *conn)
{
    TpaChannel *channel;
    const gchar *uri;

    VERBOSE ("(%s, %s, %d, %d, %d, %p)",
           obj_path, channel_type, handle_type, handle_id, suppress_handle, conn);

    channel = tpa_connection_incoming_channel (conn, channel_type, obj_path, handle_type, handle_id);

    if (channel) {
        TpaChannelType type = tpa_channel_get_channel_type (channel);
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (tpa_channel_get_target (channel));
            INFO ("[channel-created] target: %s type: %d", uri, type);
        }
        else
            INFO ("[channel-created] type: %d", type);
        g_signal_emit (conn, tpa_connection_signals[CHANNEL_CREATED], 0, channel, NULL);
    }

    VERBOSE ("return");
}

static void
channel_closed_signal (TpaChannel *channel,
                       gpointer user_data)
{
    TpaConnection *self = TPA_CONNECTION (user_data);
    const gchar *path = tpa_channel_get_object_path (channel);

    VERBOSE ("(%p)", self);
    g_assert (self);

    INFO ("[channel-closed] %s", path);
    g_hash_table_remove (self->priv->channels, path);
    VERBOSE ("return");
}
