#!/usr/local/bin/perl
#
# This represents a collection of hosts, either a group (including all hosts),
# a single host, or a list of hosts by FDQN name.
#
#   name          - name of the group the HostList represents
#   hostnames     - a list of string names of the hosts in the list
#   hosthash      - a hash of Host objects keyed by hostname.
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# + display_problems() - shows information only about hosts that have services
#                        that indicate a problem
#
# + host          - used to get at a specific host by name
# + hosts         - used to get back a generic list of hosts that are stored
# + add           - adds a specific host to the list.
#
# History:
# (1) Cleaned up (Ed July 31, 1997);

use Spong::Host;

package Spong::HostList;

my %COLORS = ( 'red', 5, 'blue', 4, 'yellow', 3, 'purple', 2, 'green', 1,
               'clear', 0 );

# This constructor expects one of four different types of options.  Either an
# empty string specifying that no actual hosts will be loaded, the string
# "all" to load all the hosts, the string containing a group name that
# represents a list of hosts, or a reference to an array that contains a list
# of hostnames in which case each host will be loaded individually.

sub new {
   my( $class, $group ) = @_;
   my( @names, %hosts, $name ) = ();
   my $self = {};

   if( $group eq "" ) {
      %hosts = ();
      @names = ();
      $self->{'name'} = "";
   } elsif( ref( $group ) eq "ARRAY" ) {
      foreach $name ( @$group ) {
	 my $object = Spong::Host->new( $name );
	 if( $object ) { $hosts{$name} = $object;  push( @names, $name ); }
      }
      $self->{'name'} = "";
   } else {
      if( $group eq "all" ) {
	 foreach $name ( @main::HOSTS_LIST ) {
	    my $object = Spong::Host->new( $name );
	    if( $object ) { $hosts{$name} = $object; push( @names, $name ); }
	 } 
      } else {
	 if( ref( $main::GROUPS{$group}->{'members'} ) eq "ARRAY" ) {
	    foreach $name ( @{$main::GROUPS{$group}->{'members'}} ) {
	       my $object = Spong::Host->new( $name );
	       if( $object ) { $hosts{$name} = $object; push( @names, $name );}
	    }
	 }
      }

      $self->{'name'} = $group;
   }

   $self->{'hosthash'}      = \%hosts;
   $self->{'hostnames'}     = [ @names ];

   bless $self;
   return $self;
}


# Get/Set methods, hostnames() is a little fancy in that it return a list
# rather then just the list reference (easier for others to deal with).  The
# hosthash function returns a reference however.

sub name { my $var = 'name';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub hosthash { my $var = 'hosthash';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub hostnames { my $var = 'hostnames';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return @{$_[0]->{$var}}; }

# Some specific functions that get at and manipulate the data in the instance
# vars in more convenient ways

sub host { 
   return $_[0]->{'hosthash'}->{$_[1]}; }

sub hosts { 
   my $self = shift;
   my( @tmp );

   foreach( $self->hostnames() ) { push( @tmp, $self->host( $_ ) ); }
   return @tmp;
}

sub add {
   my( $self, $host ) = @_;
   my $name = $host->name();

   $self->{'hosthash'}->{$name} = $host;
   push( @{$self->{'hostnames'}}, $name );
}

# Calculate summary color for a service
sub services_color {
   my( $self, $service ) = @_;
   my( $color ) = "";

   foreach( $self->hosts() ) {
      my $servobj = $_->service($service);
      my $hostcolor = $servobj->color() if $servobj;
      $color = $hostcolor if $COLORS{$hostcolor} > $COLORS{$color}; 
   }
   return $color;
}

# Calculate an overall summary color for the entire host list
sub summary_color {
   my( $self ) = @_;
   my( $color ) = "";
   my( $service, @names );

   foreach $host ( $self->hosts() ) {
      foreach $service (keys %main::SERVICES) {
         my $servobj = $host->service($service);
         my $hostcolor = $servobj->color() if $servobj;
         $color = $hostcolor if $COLORS{$hostcolor} > $COLORS{$color}; 
      }
   }

   return $color;

}

# Display summary.  Does both text and html, does rely on both the Host and
# Service objects for some help.
#
# brief       Hostname and "one" color reflecting overall status of the host
# standard    Table showing hosts vs services and the state of each service
# full        Records for each host showing detailed info about its state

sub display {
   my( $self, $type, $view ) = @_;

   $self->display_text( $view ) if $type eq "text";
   $self->display_html( $view ) if $type eq "html";
   $self->display_wml( $view ) if $type eq "wml";
}

# This displays a summary of all the hosts in this list in a text format 
# suitable for displaying on dumb ascii terminals

sub display_text {
   my( $self, $format ) = @_;

   if( $format eq "standard" ) {
      my( %services, $host, $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).

#      foreach $host ( $self->hosts() ) {
#	 foreach $service ( $host->service_names() ) { $services{$service}++;}}

      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }
     
      # Print the horizontal axis of the table (names of the services)

      print "-"x30, "-"x(($#names+1)*3), "\n";
      foreach $cnt ( 0..4 ) {
	 if( $cnt == 0 ) {  print "Key: . = green,  ? = purple   "; }
	 if( $cnt == 1 ) {  print "     Y = yellow, R = red      "; }
	 if( $cnt == 2 ) {  print "     B = blue,   o = clear    "; }
	 if( $cnt == 3 ) { print " "x30; }
	 if( $cnt == 4 ) { print "Host:", " "x25; }
	 foreach $service ( @names ) { 
	    if( length($service) - 5 + $cnt >= 0 ) {
	       print substr( $service, (length($service) - 5 + $cnt), 1 ),"  ";
	    } else {
	       print "   ";
	    }
	 }
	 print "\n";
      }
      print "-"x30, "-"x(($#names+1)*3), "\n";

      # Now go through each host, and fill in the table.

      foreach $host ( $self->hosts() ) {
	 print substr( $host->name(), 0, 29 );
	 print " "x(30-length(substr($host->name(), 0, 29)));
	 foreach $service ( @names ) {
	    my $servobj = $host->service( $service );
	    if( $servobj ) {
	       $servobj->display_text( "really_brief" );
	    } else {
	       print "   ";
	    }
	 }
	 print "\n";
      }
      print "\n";
   } elsif( $format eq "full" ) {

      # This goes through each host, and has each one print a textual record
      # of the problem it is currently having, this would include a summary
      # of the problem services, date/time the problem occurred, and contact
      # information (as well as acknowledgments if they are available).

      foreach $host ( $self->hosts() ) {
	 $host->display_problem( "text" ); print "\n";
      }
   } elsif( $format eq "brief" ) {

      # This goes through each host, and has each one print a single summary
      # line of the current status of the host. The consists of the overall
      # status color and the host name.

      foreach $host ( $self->hosts() ) {
         printf "%-6s %s\n",$host->color(),$host->name();
      }

   }
}

# This displays a summary of all the hosts in this list in an HTML format
# suitable for displaying on web clients capable of displaying tables.

sub display_html { 
   my( $self, $format ) = @_;

   if( $format eq "standard" ) {
      my( %services, $host, $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).
      
      # If group has a compress attribute, only print services in use
#&main::debug("self->name = " . $self->{'name'} . 
#       "compress = " . $main::GROUPS{$self->{'name'}}->{'compress'} );
      if ( $main::GROUPS{$self->{'name'}}->{'compress'} ) {
         foreach $host ( $self->hosts() ) {
            foreach $service ( $host->service_names() ) {
               $services{$service}++; }}
         @s = (keys %services);
      } else {
         @s = (keys %main::SERVICES);
      }

      if( grep( /^ping$/, (@s) ) ) { push( @names, "ping" ); }
      foreach $service ( sort (@s) ) {
	 push( @names, $service ) unless $service eq "ping"; }
      
      # Print the horizontal axis of the table (names of the services)
      
      print "<table border=1 cellspacing=0 cellpadding=1";
      print " bgcolor=" . $main::WWW_TITLE_COLOR if $main::WWW_TITLE_COLOR;
      print "><tr>";
      print "<td align=center width=80 nowrap><b>Host</b></td>\n";
      foreach $service ( @names ) { 
	 print "<td align=center valign=bottom width=25>\n";
	 print "<font size=-1><a href=\"!!WWWSPONG!!/help/$service\">";
	 print "$service</a></font></td>\n"; 
      }
      
      print "</tr>\n\n";
      
      # Now go through each host, and fill in the table.
      
      foreach $host ( $self->hosts() ) { 
	 my $hostname = $host->name(); 
         # Use display-name attrib if defined
	 my $display = $main::HOSTS{$hostname}->{'display_name'};
         if ( ! $display ) {
            # Otherwise use the hostname if $WWW_FQDN or short name
            $display = $main::WWW_FQDN ? $hostname :
                               (split( /\./, $hostname ))[0];
         }
	 print "<tr><td align=left bgcolor=#ffffff nowrap>\n";
	 print "<a href=\"!!WWWSPONG!!/host/$hostname\">$display</a></td>\n";
	 
	 foreach $service ( @names ) {
	    my $servobj = $host->service( $service );
	    
	    if( $servobj ) {
	       my $col = $servobj->color();
	       
	       if( $main::WWW_USE_IMAGES == 1 ) {
		  print "<td align=center";
                  print " bgcolor=" . $main::WWW_CELL_COLOR 
                     if $main::WWW_CELL_COLOR;
                  print "><a href=\"!!WWWSPONG!!/service/";
		  print "$hostname/$service\">";
                  my $alt = $servobj->summary();
		  print "<img src=\"!!WWWGIFS!!/$col.gif\" alt=\"$alt\" border=0>";
		  print "</a>";
	       } else {
		  print "<td align=center bgcolor=\"";
		  print $main::WWW_COLOR{$col} . "\" width=25>";
		  print "<a href=\"!!WWWSPONG!!/service/$hostname/$service\">";
		  print "<font color=\"" . $main::WWW_COLOR{$col} . "\">";
		  print "___</font></a>";
	       }
	       print "</td>";
	    } else {
	       if( $main::WWW_USE_IMAGES == 1 ) {
		  print "<td align=center width=25";
                  print " bgcolor=" . $main::WWW_CELL_COLOR
                      if $main::WWW_CELL_COLOR;
                  print "> - </td>\n"; 
	       } else {
		  print "<td align=center width=25";
                  print " bgcolor=" . $main::WWW_CELL_COLOR
                      if $main::WWW_CELL_COLOR;
                  print ">&nbsp;</td>\n";
	       }
	    }
	 }
      }	
      print "</tr></table>";

   } elsif( $format eq "full" ) {

      # This goes through each host, and has each one print a textual record
      # of the problem it is currently having, this would include a summary
      # of the problem services, date/time the problem occurred, and contact
      # information (as well as acknowledgments if they are available).
      
      foreach $host ( $self->hosts() ) {
	 $host->display_problem( "html" ); print "\n";
      }
   }
}


# These methods all display summary information about only those machines
# that are currently registering a problem, or that have services that have
# not been updated (showing purple) - it also relays information about 
# services that have been acknowledged.
#
# brief       Hostname and "one" color reflecting overall status of the host
# standard    Table showing hosts vs services and the state of each service
# full        Records for each host showing detailed info about the problems

sub display_problems {
   my( $self, $type, $view ) = @_;
   my( $badhosts ) = Spong::HostList->new( "" );
   my( $host, $problem, $purple );

   foreach $host ( $self->hosts() ) {
      if( $host->has_problem() ) {
         $problem = 1;
	 $badhosts->add( $host );
      } elsif( $host->has_color( "purple" ) ) {
	 $purple = 1; 
      }
   }

   if( $problem ) {
      $badhosts->display( $type, $view );
   } else {      
      if( $view eq "full" ) {
	 if( $purple ) { 
	    print "<p><font color=green><b>"       if( $type eq "html" );
	    print "No known problems.\n";
	    print "</b></font>"                    if( $type eq "html" );
	    print "<p><font size=-1 color=purple>" if( $type eq "html" );
	    print "Although some information is out of date which might ";
	    print "indicate a problem.\n";
	    print "</font>"                        if( $type eq "html" );
	 } else {
	    print "<p><font color=green><b>"       if( $type eq "html" );
	    print "No current problems.\n";
	    print "</b></font>"                    if( $type eq "html" );
	 }
      }
   }
}

# This displays a summary of all the hosts in this list in an WML format
# suitable for displaying on WAP enabled devices

sub display_wml {
   my( $self, $format ) = @_;

   # Standard and brief view are equivalent for WML
   if( $format eq "standard" || $format eq "brief") {

      foreach $host ($self->hosts() ) {
         my $color = substr($host->color(),0,2);
         $color =~ tr/a-z/A-Z/;
         my $name = $host->name();

         print "<b><anchor title=\"$name\">$color";
         print "<go href=\"!!WAPSPONG!!/services/$name\"/>";
         print "</anchor></b> $name<br/>\n";
      }
   }

}


1;
