/* spfmilter - SPF mail filter module
**
** Copyright  2004 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**	notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**	notice, this list of conditions and the following disclaimer in the
**	documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/


#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include <libmilter/mfapi.h>

#include "version.h"
#include "match.h"
#include "iparray.h"


/* Defines. */

#define HEADER_NAME "Received-SPF"

#define SPFMILTER_RESULT_PASS 0
#define SPFMILTER_RESULT_FAIL 1
#define SPFMILTER_RESULT_SOFTFAIL 2
#define SPFMILTER_RESULT_NEUTRAL 3
#define SPFMILTER_RESULT_UNKNOWN 4
#define SPFMILTER_RESULT_UNMECH 5
#define SPFMILTER_RESULT_ERROR 6
#define SPFMILTER_RESULT_NONE 7

#define SPFMILTER_ACTION_UNKNOWN 0
#define SPFMILTER_ACTION_REJECT 1
#define SPFMILTER_ACTION_MARK 2
#define SPFMILTER_ACTION_TEMPFAIL 3


/* Structs. */


/* Fallback list. */
struct lib_fallback_s;
typedef struct lib_fallback_s lib_fallback_t;
typedef struct {
    const char* pattern;
    const char* str;
    lib_fallback_t* lib_fallback;
    } fallback_t;
static fallback_t* fallbacks;
static int max_fallbacks, num_fallbacks;


/* Per-connection data structure. */
#define DEL_HEADER_MAX 10
struct lib_data_s;
typedef struct lib_data_s lib_data_t;
typedef struct {
    int whitelisted;
    int authenticated;
    const char* ip_str;
    const char* helo;
    const char* mail_from;
    const char* use_mail_from;
    lib_data_t* lib_data;
    int result;
    int action;
    int n_my_headers;
    int n_del_headers;
    int del_header_list[DEL_HEADER_MAX];
    char buf1[1000];
    char buf2[2000];
    } connection_data_t;

/* A word about buf1 and buf2, above.  These are scratch buffers for
** temporary use by any routine that needs them.  Normally we'd just
** declare them locally, on the stack.  The problem is that we are
** running in a thread, each thread has a separate stack of fixed size,
** we don't know what that size is, and if we exceed it we dump core.
** So we really want to avoid using up the stack with large local variable
** frames; instead, we use the per-connection data object which is
** allocated from the heap.  It makes the connection objects fairly
** large, but not unreasonably so.
**
** We do have to be a little careful using the buffers, so that two
** routines in the same call stack don't try to make use of them
** simultaneously.  Indeed, there's one case where we need two buffers
** at the same time so that's why we have two here.
*/


/* Forwards. */

static void usage( void );
static void init_fallback( const char* fallback_filename, const char* guess_str );
static void add_fallback( const char* pattern, const char* str );
static void fini_fallback( void );
static int find_fallback( const char* from );
static void init_whitelist( const char* whitelist_filename );
static void fini_whitelist( void );
static void trim( char* str );
static void init_uid( const char* user );
static void init_socket( const char* sockpath );
static void fini_socket( const char* sockpath );
static void unlink_socket( const char* sockpath );
static void init_daemon( void );
static void init_pidfile( const char* pidfile );
static void fini_pidfile( const char* pidfile );

/* The milter calback routines.  Signatures for these are fixed
** by the milter API.
*/
static sfsistat spf_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr );
static sfsistat spf_helo( SMFICTX* ctx, char* helohost );
static sfsistat spf_envfrom( SMFICTX* ctx, char** fromargs );
static sfsistat spf_envrcpt( SMFICTX* ctx, char** rcptargs );
static sfsistat spf_header( SMFICTX* ctx, char* name, char* value );
static sfsistat spf_eoh( SMFICTX* ctx );
static sfsistat spf_eom( SMFICTX* ctx );
static sfsistat spf_abort( SMFICTX* ctx );
static sfsistat spf_close( SMFICTX* ctx );

static char* trim_address( char* address );
static sfsistat handle_result( SMFICTX* ctx );
static void escape_percents( const char* in, char* out, int out_size );
static int receiver_is_me( const char* value );
static void build_header( connection_data_t* cd, char* header, size_t header_size, const char* reason );
static connection_data_t* init_connection_data( void );
static int init_message_data( connection_data_t* cd );
static void fini_message_data( connection_data_t* cd );
static void fini_connection_data( connection_data_t* cd );
static char* result_str( int result );
static void reason_str( connection_data_t* cd, char* str, size_t size );

/* Library interface routines.  These are implemented in parallel for
** both libspf and libspf2.
*/
static int lib_init( void );
static const char* lib_version( void );
static lib_fallback_t* lib_init_fallback( const char* str );
static int lib_set_local_hostname( lib_data_t* ld );
static lib_data_t* lib_init_connection_data( void );
static int lib_init_message_data( lib_data_t* ld );
static int lib_set_ipv4( lib_data_t* ld, struct in_addr ipv4_addr, char* ipv4_str );
static int lib_set_ipv6( lib_data_t* ld, struct in6_addr ipv6_addr, char* ipv6_str );
static int lib_set_helo_hostname( lib_data_t* ld, char* helo_hostname );
static int lib_set_from( lib_data_t* ld, const char* from );
static int lib_do_check( lib_data_t* ld, const char* from );
static int lib_do_check_recipient( lib_data_t* ld, const char* to );
static int lib_do_check_final( lib_data_t* ld );
static int lib_get_result( lib_data_t* ld );
static const char* lib_get_explanation( lib_data_t* ld );
static const char* lib_get_error( lib_data_t* ld );
static void lib_fini_message_data( lib_data_t* ld );
static void lib_fini_connection_data( lib_data_t* ld );
static void lib_fini_fallback( lib_fallback_t* lf );
static void lib_fini( void );


/* Globals. */

static char* argv0;

static char* localpolicy_str;
static int trustedforwarders;
static iparray whitelist;
int recipientmx;
static char* explanation_str;
static int markonly;
static int debug;
static char* local_hostname;
static int local_hostname_len;

static int header_name_len;

static struct smfiDesc smfilter = {
    "SPF",				/* filter name */
    SMFI_VERSION,			/* version code -- do not change */
    SMFIF_CHGHDRS|SMFIF_ADDHDRS,	/* flags */
    spf_connect,			/* connection info filter */
    spf_helo,				/* SMTP HELO command filter */
    spf_envfrom,			/* envelope sender filter */
    spf_envrcpt,			/* envelope recipient filter */
    spf_header,				/* header filter */
    spf_eoh,				/* end of header */
    NULL,				/* body block filter */
    spf_eom,				/* end of message */
    spf_abort,				/* message aborted */
    spf_close				/* connection cleanup */
    };

#ifdef HAVE_GETOPT_LONG
static const struct option	longopts[] = {
    { "localpolicy",		required_argument,	NULL,	'l', },
    { "trustedforwarders",	no_argument,		NULL,	't', },
    { "guess",			required_argument,	NULL,	'g', },
    { "fallback",		required_argument,	NULL,	'f', },
    { "whitelist",		required_argument,	NULL,	'w', },
    { "recipientmx",		no_argument,		NULL,	'r', },
    { "explanation",		required_argument,	NULL,	'e', },
    { "markonly",		no_argument,		NULL,	'm', },
    { "user",			required_argument,	NULL,	'u', },
    { "pidfile",		required_argument,	NULL,	'p', },
    { "nodaemon",		no_argument,		NULL,	'X', },
    { "help",			no_argument,		NULL,	'h', },
    { "debug",			optional_argument,	NULL,	'd', },
    { 0, 0, 0, 0 },
};
#define DOC_LONGOPT(l, v, t, p1) \
    do { \
	(void) fprintf( stderr, "    --%s%c%s%*s" t "\n", l, (v ? '=' : ' '),  (v ? v : ""), p1, "" ); \
    } while( 0 )
#else /* HAVE_GETOPT_LONG */
#define DOC_LONGOPT(l, v, t, p1) do { } while( 0 )
#endif /* HAVE_GETOPT_LONG */
static const char* shortopts = "l:tg:f:w:re:mu:p:Xhd::";

#define DOC_OPT(s, l, v, t, p0, p1) \
    do { \
	(void) fprintf( stderr, "    -%c%c%s%*s" t "\n", s, (v ? ' ' : ' '), (v ? v : ""), p0, "" ); \
	DOC_LONGOPT(l, v, t, p1); \
    } while( 0 )


int
main( int argc, char** argv )
    {
    char* guess_str;
    char* fallback_filename;
    char* whitelist_filename;
    char* user;
    char* pidfile;
    int nodaemon;
    char* sockpath;
#ifdef HAVE_GETOPT_LONG
    int idx;
#endif /* HAVE_GETOPT_LONG */
    char c;

    localpolicy_str = (char*) 0;
    trustedforwarders = 0;
    guess_str = (char*) 0;
    fallback_filename = (char*) 0;
    whitelist_filename = (char*) 0;
    recipientmx = 0;
    explanation_str = (char*) 0;
    markonly = 0;
    user = (char*) 0;
    pidfile = (char*) 0;
    nodaemon = 0;
    debug = 0;

    /* Figure out the program's name. */
    argv0 = strrchr( argv[0], '/' );
    if ( argv0 != (char*) 0 )
	++argv0;
    else
	argv0 = argv[0];

    header_name_len = strlen( HEADER_NAME );

    /* Parse args. */
    while ( ( c =
#ifdef HAVE_GETOPT_LONG
		  getopt_long( argc, argv, shortopts, longopts, &idx )
#else /* HAVE_GETOPT_LONG */
		  getopt( argc, argv, shortopts )
#endif /* HAVE_GETOPT_LONG */
			) != -1 )
	{
	switch (c)
	    {
	    case 'l':
		localpolicy_str = optarg;
		break;
	    case 't':
		trustedforwarders = 1;
		break;
	    case 'g':
		guess_str = optarg;
		break;
	    case 'f':
		fallback_filename = optarg;
		break;
	    case 'w':
		whitelist_filename = optarg;
		break;
	    case 'r':
		recipientmx = 1;
		break;
	    case 'e':
		explanation_str = optarg;
		break;
	    case 'm':
		markonly = 1;
		break;
	    case 'u':
		user = optarg;
		break;
	    case 'p':
		pidfile = optarg;
		break;
	    case 'X':
		nodaemon = 1;
		break;
	    case 'h':
	    case '?':
		usage();
		exit( 1 );
	    case 'd':
		if ( optarg )
		    debug = atoi( optarg );
		else
		    debug = 1;
		break;
	    default:
		(void) fprintf( stderr, "Unrecognised option '%c'\n", c );
		exit( 1 );
	    }
	}
    argv += optind;
    argc -= optind;

    if ( argc != 1 )
	{
	usage();
	exit( 1 );
	}
    sockpath = argv[0];

    local_hostname = (char*) 0;
    if ( ! lib_init() )
	{
	(void) fprintf( stderr, "%s: lib_init() failed\n", argv0 );
	exit( 1 );
	}
    init_fallback( fallback_filename, guess_str );
    init_whitelist( whitelist_filename );
    init_uid( user );
    init_socket( sockpath );
    openlog( argv0, 0, LOG_MAIL );
    if ( ! nodaemon )
	init_daemon();
    init_pidfile( pidfile );

    syslog( LOG_NOTICE, "%s %s with %s starting", SPFMILTER_PROGRAM, SPFMILTER_VERSION, lib_version() );
    if ( smfi_main() == MI_FAILURE )
	{
	syslog( LOG_ERR, "smfi_main() failed" );
	exit( 1 );
	}
    syslog( LOG_NOTICE, "%s %s with %s terminating", SPFMILTER_PROGRAM, SPFMILTER_VERSION, lib_version() );

    fini_pidfile( pidfile );
    fini_socket( sockpath );
    fini_whitelist();
    fini_fallback();
    lib_fini();
    if ( local_hostname != (char*) 0 )
	free( (void*) local_hostname );

    closelog();

    return 0;
    }


static void
usage( void )
    {
    (void) fprintf( stderr, "usage: %s [options] /path/to/socket\n", argv0 );
    DOC_OPT( 'l', "localpolicy", "<spf-mechanism>", "Add local SPF policy mechanisms.", 5, 4 );
    DOC_OPT( 't', "trustedforwarders", (char*) 0, "Use the trusted-forwarder.org whitelist.", 20, 14 );
    DOC_OPT( 'g', "guess", "<spf-mechanisms>", "Add guess SPF policy mechanisms.", 4, 4 );
    DOC_OPT( 'f', "fallback", "<filename>", "A file of fallback SPF mechanisms.", 10, 14 );
    DOC_OPT( 'w', "whitelist", "<filename>", "A file of IP addresses to always accept mail from.", 10, 14 );
    DOC_OPT( 'r', "recipientmx", (char*) 0, "Check the MX records of the message's reqipients.", 20, 14 );
    DOC_OPT( 'e', "explanation", "<exp>", "Change the message returned in mail bounces.", 15, 14 );
    DOC_OPT( 'm', "markonly", (char*) 0, "Mark email instead of rejecting it.", 20, 22 );
    DOC_OPT( 'u', "user", "<user|uid>", "Run as specified user or UID.", 10, 16 );
    DOC_OPT( 'p', "pidfile", "<filename>", "Write the process i.d. to the specified file.", 10, 16 );
    DOC_OPT( 'X', "nodaemon", (char*) 0, "Do not fork into the background.", 20, 26 );
    DOC_OPT( 'h', "help", (char*) 0, "Show this help.", 20, 26 );
    DOC_OPT( 'd', "debug", "[<int>]", "Enable debugging to syslog.", 13, 18 );
    }


static void
init_fallback( const char* fallback_filename, const char* guess_str )
    {
    FILE* fp;
    char line[1000];
    char* cp;

    max_fallbacks = num_fallbacks = 0;
    fallbacks = (fallback_t*) 0;

    if ( fallback_filename != (char*) 0 )
	{
	fp = fopen( fallback_filename, "r" );
	if ( fp == (FILE*) 0 )
	    {
	    perror( fallback_filename );
	    exit( 1 );
	    }
	while ( fgets( line, sizeof(line), fp ) != (char*) 0 )
	    {
	    trim( line );
	    if ( line[0] == '\0' )
		continue;
	    cp = line;
	    cp += strcspn( cp, " \t" );
	    *cp = '\0';
	    ++cp;
	    cp += strspn( cp, " \t" );
	    add_fallback( line, cp );
	    }
	(void) fclose( fp );
	}

    if ( guess_str != (char*) 0 )
	add_fallback( "*", guess_str );
    }


static void
add_fallback( const char* pattern, const char* str )
    {
    if ( num_fallbacks >= max_fallbacks )
	{
	if ( max_fallbacks == 0 )
	    {
	    max_fallbacks = 50;		/* arbitrary */
	    fallbacks = (fallback_t*) malloc( max_fallbacks * sizeof(fallback_t) );
	    }
	else
	    {
	    max_fallbacks *= 2;
	    fallbacks = (fallback_t*) realloc( (void*) fallbacks, max_fallbacks * sizeof(fallback_t) );
	    }
	if ( fallbacks == (fallback_t*) 0 )
	    {
	    (void) fprintf( stderr, "%s: out of memory enlarging fallbacks array\n", argv0 );
	    exit( 1 );
	    }
	}
    fallbacks[num_fallbacks].pattern = strdup( pattern );
    fallbacks[num_fallbacks].str = strdup( str );
    if ( fallbacks[num_fallbacks].pattern == (char*) 0 ||
         fallbacks[num_fallbacks].str == (char*) 0 )
	{
	(void) fprintf( stderr, "%s: out of memory saving a fallback entry\n", argv0 );
	exit( 1 );
	}

    fallbacks[num_fallbacks].lib_fallback = lib_init_fallback( fallbacks[num_fallbacks].str );
    if ( fallbacks[num_fallbacks].lib_fallback == (lib_fallback_t*) 0 )
	{
	(void) fprintf( stderr, "%s: error compiling fallback mechanism '%s'\n", argv0, fallbacks[num_fallbacks].str );
	exit( 1 );
	}

    ++num_fallbacks;
    }


static void
fini_fallback( void )
    {
    if ( fallbacks != (fallback_t*) 0 )
	{
	int i;

	for ( i = 0; i < num_fallbacks; ++i )
	    {
	    free( (void*) fallbacks[i].pattern );
	    free( (void*) fallbacks[i].str );
	    lib_fini_fallback( fallbacks[i].lib_fallback );
	    }
	max_fallbacks = num_fallbacks = 0;
	free( (void*) fallbacks );
	fallbacks = (fallback_t*) 0;
	}
    }


static int
find_fallback( const char* from )
    {
    const char* from_host;
    int i;

    from_host = strchr( from, '@' );
    if ( from_host == (char*) 0 )
	return -1;
    ++from_host;
    if ( fallbacks == (fallback_t*) 0 )
	return -1;
    for ( i = 0; i < num_fallbacks; ++i )
	if ( match( fallbacks[i].pattern, from_host ) )
	    return i;
    return -1;
    }


static void
init_whitelist( const char* whitelist_filename )
    {
    FILE* fp;
    char line[500];
    octets ip;

    if ( whitelist_filename != (char*) 0 )
	{
	whitelist = iparray_new();
	if ( whitelist == (iparray) 0 )
	    {
	    (void) fprintf( stderr, "%s: whitelist create failed\n", argv0 );
	    exit( 1 );
	    }
	fp = fopen( whitelist_filename, "r" );
	if ( fp == (FILE*) 0 )
	    {
	    perror( whitelist_filename );
	    exit( 1 );
	    }
	while ( fgets( line, sizeof(line), fp ) != (char*) 0 )
	    {
	    trim( line );
	    if ( line[0] == '\0' )
		continue;
	    if ( iparray_parse_octets( line, &ip ) )
		(void) iparray_incr( whitelist, ip );
	    else
		(void) fprintf( stderr, "%s: unparsable IP address - \"%s\"\n", argv0, line );
	    }
	(void) fclose( fp );
	}
    else
	whitelist = (iparray) 0;
    }


static void
fini_whitelist( void )
    {
    if ( whitelist != (iparray) 0 )
	{
	iparray_delete( whitelist );
	iparray_fini();
	}
    }


static void
trim( char* str )
    {
    char* cp;
    int len;

    cp = strchr( str, '#' );
    if ( cp != (char*) 0 )
	*cp = '\0';
    len = strlen( str );
    while ( str[len-1] == '\n' || str[len-1] == '\r' || str[len-1] == ' ' || str[len-1] == '\t' )
	{
	--len;
	str[len] = '\0';
	}
    }


static void
init_uid( const char* user )
    {
    struct passwd* pwd;
    char* ep;
    int uid;

    if ( getuid() == 0 )
	{
	/* If we're root, the very first thing we do is become another user. */
	if ( user == (char*) 0 )
	    (void) fprintf( stderr, "%s: warning: started as root but no --user flag specified\n", argv0 );
	else
	    {
	    /* Is it a number? */
	    uid = strtol( user, &ep, 0 );
	    if ( *ep == '\0' )
		pwd = getpwuid( uid );
	    else
		pwd = getpwnam( user );
	    if ( pwd == (struct passwd*) 0 )
		{
		(void) fprintf( stderr, "%s: unknown user: '%s'\n", argv0, user );
		exit( 1 );
		}
	    /* Set aux groups to null. */
	    if ( setgroups( 0, (gid_t*) 0 ) < 0 )
		{
		perror( "setgroups" );
		exit( 1 );
		}
	    /* Set primary group. */
	    if ( setgid( pwd->pw_gid ) < 0 )
		{
		perror( "setgid" );
		exit( 1 );
		}

	    /* Try setting aux groups correctly - not critical if this fails. */
	    if ( initgroups( user, pwd->pw_gid ) < 0 )
		    perror( "initgroups" );
	    /* Set uid. */
	    if ( setuid( pwd->pw_uid ) < 0 )
		{
		perror( "setuid" );
		exit( 1 );
		}
	    }
	}
    else
	{
	/* If we're not root but got a -user flag anyway, that's an error. */
	if ( user != (char*) 0 )
	    {
	    (void) fprintf( stderr, "%s: can't switch users if not started as root\n", argv0 );
	    exit( 1 );
	    }
	}
    }


static void
init_socket( const char* sockpath )
    {
    unlink_socket( sockpath );

    /* Harden our umask so that the new socket gets created securely. */
    umask( 0077 );

    /* Initialize milter stuff. */
    smfi_setconn( (char*) sockpath );
    if ( smfi_register( smfilter ) == MI_FAILURE )
	{
	(void) fprintf( stderr, "%s: smfi_register() failed\n", argv0 );
	exit( 1 );
	}
    }


static void
fini_socket( const char* sockpath )
    {
    unlink_socket( sockpath );
    }


static void
unlink_socket( const char* sockpath )
    {
    /* Remove old local socket.  Actually we should probably stat
    ** the file and check that it is in fact a socket before removing
    ** it.
    */
    if ( strncasecmp( sockpath, "unix:", 5 ) == 0 )
	{
	if ( unlink( &sockpath[5] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockpath[5] );
	}
    else if ( strncasecmp( sockpath, "local:", 6 ) == 0 )
	{
	if ( unlink( &sockpath[6] ) < 0 )
	    if ( errno != ENOENT )
		perror( &sockpath[6] );
	}
    else if ( strchr( sockpath, ':' ) == (char*) 0 )
	{
	if ( unlink( sockpath ) < 0 )
	    if ( errno != ENOENT )
		perror( sockpath );
	}
    }


static void
init_daemon( void )
    {
    /* Daemonize. */
#ifdef HAVE_DAEMON
    if ( daemon( 0, 0 ) < 0)
	{
	perror( "daemon" );
	exit( 1 );
	}
#else /* HAVE_DAEMON */
    switch ( fork() )
	{
	case 0:
	    break;  
	case -1:
	    syslog( LOG_CRIT, "fork: %m" );
	    perror( "fork" );
	    exit( 1 );
	default:
	    exit( 0 );
	}
#ifdef HAVE_SETSID
    setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
    }


static void
init_pidfile( const char* pidfile )
    {
    if ( pidfile != (char*) 0 )
	{
	FILE* fp;

	fp = fopen( pidfile, "w" );
	if ( fp == (FILE*) 0 )
	    syslog( LOG_ERR, "unable to write PID file - %m" );
	else
	    {
	    (void) fprintf( fp, "%ld\n", (long) getpid() );
	    (void) fclose( fp );
	    }
	}
    }


static void
fini_pidfile( const char* pidfile )
    {
    if ( pidfile != (char*) 0 )
	(void) unlink( pidfile );
    }


/* Milter routines. */


/* spf_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
static sfsistat
spf_connect( SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr )
    {
    connection_data_t* cd;
    struct sockaddr_in* sin;
    struct sockaddr_in6* sin6;
    const char* r;

    if ( connaddr == (_SOCK_ADDR*) 0 )
	return SMFIS_ACCEPT;	/* can't deal with it */

    /* Create & initialize the connection_data object. */
    cd = init_connection_data();
    if ( cd == (connection_data_t*) 0 )
	{
	syslog( LOG_ERR, "couldn't allocate connection_data" );
	return SMFIS_TEMPFAIL;
	}
    if ( smfi_setpriv( ctx, (void*) cd ) != MI_SUCCESS )
	{
	syslog( LOG_ERR, "smfi_setpriv() failed" );
	fini_connection_data( cd );
	return SMFIS_TEMPFAIL;
	}

    /* The first time through, we figure out the local hostname. */
    if ( local_hostname == (char*) 0 )
	{
	char* jmac;
	char lh[200];

	jmac = smfi_getsymval( ctx, "{j}" );
	if ( jmac != (char*) 0 )
	    local_hostname = strdup( jmac );
	else
	    {
	    if ( gethostname( lh, sizeof(lh) ) == 0 )
		local_hostname = strdup( lh );
	    else
		{
		syslog( LOG_ERR, "couldn't get local hostname" );
		return SMFIS_TEMPFAIL;
		}
	    }
	if ( local_hostname == (char*) 0 )
	    {
	    syslog( LOG_ERR, "couldn't strdup local hostname" );
	    return SMFIS_TEMPFAIL;
	    }
	local_hostname_len = strlen( local_hostname );
	if ( ! lib_set_local_hostname( cd->lib_data ) )
	    {
	    syslog( LOG_ERR, "lib_set_local_hostname() failed" );
	    return SMFIS_TEMPFAIL;
	    }
	}

    /* Set the IP address. */
    switch ( connaddr->sa_family )
	{
	case AF_INET:
	    sin = (struct sockaddr_in*) connaddr;
	    if ( whitelist != (iparray) 0 )
		{
		char* char_addr = (char*) &sin->sin_addr.s_addr;
		octets ip;

		ip.a = char_addr[0];
		ip.b = char_addr[1];
		ip.c = char_addr[2];
		ip.d = char_addr[3];
		ip.w = 32;
		if ( iparray_get( whitelist, ip ) > 0 )
		    {
		    syslog( LOG_INFO, "whitelist \"%s\" [%d.%d.%d.%d]", connhost, (int) ip.a, (int) ip.b, (int) ip.c, (int) ip.d );
		    cd->whitelisted = 1;
		    }
		}
	    r = inet_ntop( AF_INET, &sin->sin_addr, cd->buf1, sizeof(cd->buf1) );
	    if ( r == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't convert IPv4 address to a string - %m" );
		return SMFIS_TEMPFAIL;
		}
	    if ( ! lib_set_ipv4( cd->lib_data, sin->sin_addr, cd->buf1 ) )
		{
		syslog( LOG_ERR, "lib_set_ipv4() failed" );
		return SMFIS_TEMPFAIL;
		}
	    break;
	case AF_INET6:
	    sin6 = (struct sockaddr_in6*) connaddr;
	    r = inet_ntop( AF_INET6, &sin6->sin6_addr, cd->buf1, sizeof(cd->buf1) );
	    if ( r == (char*) 0 )
		{
		syslog( LOG_ERR, "couldn't convert IPv6 address to a string - %m" );
		return SMFIS_TEMPFAIL;
		}
	    if ( ! lib_set_ipv6( cd->lib_data, sin6->sin6_addr, cd->buf1 ) )
		{
		syslog( LOG_ERR, "lib_set_ipv6() failed" );
		return SMFIS_TEMPFAIL;
		}
	    break;
	default:
	    syslog( LOG_ERR, "unknown address family" );
	    return SMFIS_TEMPFAIL;
	}
    cd->ip_str = strdup( cd->buf1 );
    if ( cd->ip_str == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup ip_str" );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* spf_helo - handle the HELO command
**
** Called at the start of a connection.
**
** helohost: The string passed to the HELO/EHLO command.
*/
static sfsistat
spf_helo( SMFICTX* ctx, char* helohost )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd->helo != (char*) 0 )
	syslog( LOG_NOTICE, "multiple HELOs from %s [%s] - ignoring '%s'", cd->helo, cd->ip_str, helohost );
    else
	{
	cd->helo = strdup( helohost );
	if ( cd->helo == (char*) 0 )
	    {
	    syslog( LOG_ERR, "couldn't strdup helo" );
	    return SMFIS_TEMPFAIL;
	    }
	}

    if ( cd->whitelisted )
	return SMFIS_CONTINUE;

    if ( ! lib_set_helo_hostname( cd->lib_data, helohost ) )
	{
	syslog( LOG_ERR, "lib_set_helo_hostname() failed" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - lib_set_helo_hostname() failed." );
	return SMFIS_TEMPFAIL;
	}

    return SMFIS_CONTINUE;
    }


/* spf_envfrom - handle the MAIL FROM:<> command
**
** Called at the start of each message.  There may be multiple messages
** in a connection.  Any per-message data should be initialized here.
**
** fromargs: Null-terminated SMTP command arguments; fromargs[0] is the
**		   sender address.
*/
static sfsistat
spf_envfrom( SMFICTX* ctx, char** fromargs )
    {
    connection_data_t* cd;
    char* auth_type;
    char* verify;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( ! init_message_data( cd ) )
	{
	syslog( LOG_ERR, "init_message_data() failed" );
	return SMFIS_TEMPFAIL;
	}

    cd->mail_from = strdup( trim_address( fromargs[0] ) );
    if ( cd->mail_from == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup mail_from" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - couldn't strdup mail_from." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}

    if ( cd->whitelisted )
	return SMFIS_CONTINUE;

    /* Authenticated messages get accepted without question. */
    auth_type = smfi_getsymval( ctx, "{auth_type}" );
    verify = smfi_getsymval( ctx, "{verify}" );
    if ( auth_type != (char*) 0 ||
	 ( verify != (char*) 0 && strcmp( verify, "OK" ) == 0 ) )
	{
	cd->authenticated = 1;
	return SMFIS_CONTINUE;
	}

    /* If the mail-from is empty or doesn't have an @, we make a fake one
    ** using the HELO domain (or if that's missing, the IP address).
    ** It's possible the SPF library we're using already does this, but at
    ** least some versions of some of the libraries do not, so we might
    ** as well do it here.
    */
    if ( cd->mail_from[0] == '\0' || strchr( cd->mail_from, '@' ) == (char*) 0 )
	{
	if ( cd->helo != (char*) 0 )
	    (void) snprintf( cd->buf1, sizeof(cd->buf1), "mailer_daemon@%s", cd->helo );
	else
	    (void) snprintf( cd->buf1, sizeof(cd->buf1), "mailer_daemon@[%s]", cd->ip_str );
	cd->use_mail_from = strdup( cd->buf1 );
	}
    else
	cd->use_mail_from = strdup( cd->mail_from );
    if ( cd->use_mail_from == (char*) 0 )
	{
	syslog( LOG_ERR, "couldn't strdup use_mail_from" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - couldn't strdup use_mail_from." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}

    if ( ! lib_set_from( cd->lib_data, cd->use_mail_from ) )
	{
	syslog( LOG_ERR, "lib_set_from() failed" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - lib_set_from() failed." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}

    /* In MX mode we do a check for each recipient, and only later do the
    ** check for the whole message.
    */
    if ( recipientmx )
	return SMFIS_CONTINUE;

    if ( ! lib_do_check( cd->lib_data, cd->use_mail_from ) )
	{
	syslog( LOG_ERR, "lib_do_check() failed" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - lib_do_check() failed." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}
    cd->result = lib_get_result( cd->lib_data );

    return handle_result( ctx );
    }


/* spf_envrcpt - handle a RCPT TO:<> command
**
** Called separately for each recipient of a message.
**
** rcptargs: Null-terminated SMTP command arguments; rcptargs[0] is the
**           recipient address.
*/
static sfsistat
spf_envrcpt( SMFICTX* ctx, char** rcptargs )
    {
    connection_data_t* cd;
    char* rcpt;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd->whitelisted || cd->authenticated )
	return SMFIS_CONTINUE;

    /* If we're not in MX mode then we've already checked the message. */
    if ( ! recipientmx )
	return SMFIS_CONTINUE;

    rcpt = trim_address( rcptargs[0] );

    if ( ! lib_do_check_recipient( cd->lib_data, rcpt ) )
	{
	syslog( LOG_ERR, "lib_do_check_recipient() failed" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - lib_do_check_recipient() failed." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}
    cd->result = lib_get_result( cd->lib_data );

    return handle_result( ctx );
    }


/* spf_header - handle a header line
**
** Called separately for each header line in a message.
**
** name:  Header field name.
** value: Header vield value, including folded whitespace.  The final CRLF
**		is removed.
*/
static sfsistat
spf_header( SMFICTX* ctx, char* name, char* value )
    {
    connection_data_t* cd;
    
    cd = (connection_data_t*) smfi_getpriv( ctx );

    /* We look for any instances of our own header in the incoming
    ** message, so we can remove them later.
    */
    if ( strcasecmp( name, HEADER_NAME ) == 0 )
	{
	++cd->n_my_headers;

	/* Does this SPF header claim to be from us? */
	if ( receiver_is_me( value ) )
	    {
	    syslog( LOG_NOTICE, "found possible spoofed header - '%s'", value );
	    if ( cd->n_del_headers < DEL_HEADER_MAX )
		{
		cd->del_header_list[cd->n_del_headers] = cd->n_my_headers;
		++cd->n_del_headers;
		}
	    }
	}

    return SMFIS_CONTINUE;
    }


/* spf_eoh - handle the end of the headers
**
** Called once per message after all headers have been processed.
*/
static sfsistat
spf_eoh( SMFICTX* ctx )
    {
    connection_data_t* cd;
    
    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd->whitelisted || cd->authenticated )
	return SMFIS_CONTINUE;

    if ( ! recipientmx )
	return SMFIS_CONTINUE;

    if ( ! lib_do_check_final( cd->lib_data ) )
	{
	syslog( LOG_ERR, "lib_do_check_final() failed" );
	smfi_setreply( ctx, "452", "4.3.1", "Please try again later - lib_do_check_final() failed." );
	fini_message_data( cd );
	return SMFIS_TEMPFAIL;
	}
    cd->result = lib_get_result( cd->lib_data );

    cd->action = SPFMILTER_ACTION_MARK;
    return SMFIS_CONTINUE;
    }


/* spf_eom - handle the end of the message
**
** Called once per message after all body blocks have been processed.
** Any per-message data should be freed both here and in spf_abort(),
** and wherever you return any of SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
static sfsistat
spf_eom( SMFICTX* ctx )
    {
    connection_data_t* cd;
    int i;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    /* Header deleting and adding can only happen in the eom handler. */

    /* Remove any previous SPF headers, to prevent spoofing. */
    if ( cd->n_del_headers >= DEL_HEADER_MAX )
	{
	/* There were too many bogus SPF headers to keep track of, so to
	** be safe we just delete all SPF headers.
	*/
	for ( i = cd->n_my_headers; i >= 1; --i )
	    smfi_chgheader( ctx, HEADER_NAME, i, (char*) 0 );
	}
    else
	{
	/* We have a list of only the headers purporting to be from us;
	** those get deleted.  We do the deletions in reverse order in case
	** deleting one header causes subsequent headers to get re-numbered.
	** (I actually checked whether this happens in the current milter
	** implementation, and it does NOT, but perhaps some future version
	** will get it wrong.)
	*/
	for ( i = cd->n_del_headers - 1 ; i >= 0; --i )
	    smfi_chgheader( ctx, HEADER_NAME, cd->del_header_list[i], (char*) 0 );
	}

    if ( cd->whitelisted || cd->authenticated )
	{
	cd->result = SPFMILTER_RESULT_PASS;
	cd->action = SPFMILTER_ACTION_MARK;
	if ( cd->whitelisted )
	    (void) snprintf( cd->buf1, sizeof(cd->buf1), "%s: %s is whitelisted", local_hostname, cd->ip_str );
	else if ( cd->authenticated )
	    (void) snprintf( cd->buf1, sizeof(cd->buf1), "%s: authenticated connection", local_hostname );
	else
	    (void) strncpy( cd->buf1, "???", sizeof(cd->buf1) );
	}
    else
	reason_str( cd, cd->buf1, sizeof(cd->buf1) );

    if ( cd->action == SPFMILTER_ACTION_MARK )
	{
	/* Build and add the SPF header. */
	build_header( cd, cd->buf2, sizeof(cd->buf2), cd->buf1 );
	/* The SPF spec wants us to prepend the new header to the existing
	** ones, but not all versions of the milter API have the necessary
	** function.
	*/
#ifdef HAVE_SMFI_INSHEADER
	smfi_insheader( ctx, 0, HEADER_NAME, cd->buf2 );
#else
	smfi_addheader( ctx, HEADER_NAME, cd->buf2 );
#endif
	}

    fini_message_data( cd );

    return SMFIS_CONTINUE;
    }


/* spf_abort - handle the message being aborted
**
** May be called at any time during processing of a message, indicating
** that something went wrong.  Any per-message data should be freed both
** here and in sample_eom(), and wherever you return any of
** SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
static sfsistat
spf_abort( SMFICTX* ctx )
    {
    connection_data_t* cd;
    
    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd != (connection_data_t*) 0 )
	fini_message_data( cd );
    else
	syslog( LOG_NOTICE, "spf_abort called but cd is null" );

    return SMFIS_CONTINUE;
    }


/* spf_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
static sfsistat
spf_close( SMFICTX* ctx )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv( ctx );

    if ( cd != (connection_data_t*) 0 )
	{
	smfi_setpriv( ctx, (void*) 0 );
	fini_connection_data( cd );
	}
    else
	syslog( LOG_NOTICE, "spf_close called but cd is null" );

    return SMFIS_CONTINUE;
    }


static char*
trim_address( char* address )
    {
    int len;

    /* The milter API sometimes gives us strings with angle brackets. */
    while ( *address == '<' )
	++address;
    len = strlen( address );
    while ( len > 0 && address[len - 1] == '>' )
	address[--len] = '\0';

    /* We also sometimes get CRs on the ends of addresses. */
    while ( len > 0 && address[len - 1] == '\015' )
	address[--len] = '\0';

    return address;
    }


static sfsistat
handle_result( SMFICTX* ctx )
    {
    connection_data_t* cd;
    const char* exp;
    char exp_escaped[1000];

    cd = (connection_data_t*) smfi_getpriv( ctx );

    switch ( cd->result )
	{
	case SPFMILTER_RESULT_PASS:
	case SPFMILTER_RESULT_SOFTFAIL:
	case SPFMILTER_RESULT_NEUTRAL:
	case SPFMILTER_RESULT_UNKNOWN:
	case SPFMILTER_RESULT_UNMECH:
	case SPFMILTER_RESULT_ERROR:	/* typically "DNS lookup failure" */
	case SPFMILTER_RESULT_NONE:
	    cd->action = SPFMILTER_ACTION_MARK;
	    break;
	case SPFMILTER_RESULT_FAIL:
	    if ( markonly )
		cd->action = SPFMILTER_ACTION_MARK;
	    else
		cd->action = SPFMILTER_ACTION_REJECT;
	    break;
	}

    switch ( cd->action )
	{
	case SPFMILTER_ACTION_REJECT:
	    exp = lib_get_explanation( cd->lib_data );
	    syslog( LOG_INFO, "rejecting mail from [%s] - %s", cd->ip_str, result_str( cd->result ) );
	    if ( exp != (char*) 0 )
		escape_percents( exp, exp_escaped, sizeof(exp_escaped) );
	    else
		(void) strncpy( exp_escaped, "rejected by spfmilter", sizeof(exp_escaped) - 1 );
	    smfi_setreply( ctx, "550", "5.7.1", exp_escaped );
	    fini_message_data( cd );
	    return SMFIS_REJECT;

	case SPFMILTER_ACTION_TEMPFAIL:
	    syslog( LOG_INFO, "temporarily failing mail from [%s] - %s", cd->ip_str, lib_get_error( cd->lib_data ) );
	    smfi_setreply( ctx, "451", "4.3.2", "Please try again later." );
	    fini_message_data( cd );
	    return SMFIS_TEMPFAIL;

	default:
	    return SMFIS_CONTINUE;
	}
    }


/* escape_percents - escape '%' characters in a string
**
** Sendmail doesn't like sending percent characters in bounce messages.
** If a milter returns an error message with percents in it, sendmail
** will ignore that and instead return an uninformative default
** message saying "Command rejected".  To avoid this we escape the
** percents witrh a second percent.
*/
static void
escape_percents( const char* in, char* out, int out_size )
    {
    const char* i;
    char* o;
    int out_len;

    i = in;
    o = out;
    out_len = 0;
    while ( *i != '\0' && out_len + 2 < out_size )
	{
	if ( *i == '%' )
	    {
	    /* The character that most often comes back escaped is
	    ** the @ sign (%40). Since it doesn't actually need
	    ** escaping, unescape it.
	    */
	    if ( i[1] == '4' && i[2] == '0' )
		{
		*o++ = '@';
		++out_len;
		i += 3;
		}
	    else
		{
		/* Otherwise, escape the %-sign. */
		*o++ = '%';
		*o++ = '%';
		out_len += 2;
		++i;
		}
	    }
	else
	    {
	    *o++ = *i++;
	    ++out_len;
	    }
	}
    *o = '\0';
    }


static int
receiver_is_me( const char* value )
    {
    const char* cp;

    /* First look for a receiver= setting after the comment part.
    ** The hostname immediately follows the equals and is immediately
    ** followed by a semicolon.
    */
    cp = strrchr( value, ')' );
    if ( cp != (char*) 0 )
	++cp;
    else
	cp = value;
    cp = strstr( cp, " receiver=" );
    if ( cp != (char*) 0 )
	{
	cp += 10;
	if ( strncasecmp( cp, local_hostname, local_hostname_len ) == 0 )
	    {
	    cp += local_hostname_len;
	    if ( *cp == ';' )
		return 1;	/* Yes! */
	    }
	}

    /* If there was no receiver= (it's optional), try checking the
    ** comment part for a hostname.  That's even more optional
    ** (it's not in the spec at all), but many implementations do add it.
    **
    ** To find the comment hostname, skip one word, whitespace, and
    ** an open-paren, then check for the hostname followed by a colon.
    */
    cp = value;
    cp += strcspn( cp, " \t" );
    cp += strspn( cp, " \t" );
    if ( *cp == '(' )
	{
	++cp;
	if ( strncasecmp( cp, local_hostname, local_hostname_len ) == 0 )
	    {
	    cp += local_hostname_len;
	    if ( *cp == ':' )
		return 1;	/* Yes! */
	    }
	}

    /* Nope. */
    return 0;
    }


static void
build_header( connection_data_t* cd, char* header, size_t header_size, const char* reason )
    {
    int len;

    (void) snprintf( header, header_size, "%s", result_str( cd->result ) );
    len = strlen( header );
    if ( reason != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " (%s)", reason );
	len = strlen( header );
	}
    if ( local_hostname != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " receiver=%s;", local_hostname );
	len = strlen( header );
	}
    if ( cd->ip_str!= (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " client-ip=%s;", cd->ip_str);
	len = strlen( header );
	}
    if ( cd->helo != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " helo=%s;", cd->helo );
	len = strlen( header );
	}
    if ( cd->mail_from != (char*) 0 )
	{
	(void) snprintf( &header[len], header_size - len, " envelope-from=%s;", cd->mail_from );
	len = strlen( header );
	}
    /*!!! Do something about the problem= field. */
    (void) snprintf( &header[len], header_size - len, " x-software=%s %s %s with %s;", SPFMILTER_PROGRAM, SPFMILTER_VERSION, SPFMILTER_URL, lib_version() );
    }


static connection_data_t*
init_connection_data( void )
    {
    connection_data_t* cd;

    cd = (connection_data_t*) malloc( sizeof(connection_data_t) );
    if ( cd == (connection_data_t*) 0 )
	return (connection_data_t*) 0;
    cd->whitelisted = 0;
    cd->ip_str = (char*) 0;
    cd->helo = (char*) 0;
    cd->lib_data = lib_init_connection_data();
    if ( cd->lib_data == (lib_data_t*) 0 )
	{
	free( (void*) cd );
	return (connection_data_t*) 0;
	}
    if ( ! init_message_data( cd ) )
	{
	fini_connection_data( cd );
	return (connection_data_t*) 0;
	}
    return cd;
    }


static int
init_message_data( connection_data_t* cd )
    {
    cd->authenticated = 0;
    cd->mail_from = (char*) 0;
    cd->use_mail_from = (char*) 0;
    cd->result = -1;
    cd->action = SPFMILTER_ACTION_UNKNOWN;
    cd->n_my_headers = 0;
    cd->n_del_headers = 0;
    if ( ! lib_init_message_data( cd->lib_data ) )
	return 0;
    return 1;
    }


/* Frees or resets any items in the connection_data that are for
** an individual message.
*/
static void
fini_message_data( connection_data_t* cd )
    {
    if ( cd->mail_from != (char*) 0 )
	{
	free( (void*) cd->mail_from );
	cd->mail_from = (char*) 0;
	}
    if ( cd->use_mail_from != (char*) 0 )
	{
	free( (void*) cd->use_mail_from );
	cd->use_mail_from = (char*) 0;
	}
    lib_fini_message_data( cd->lib_data );
    }


/* Frees all items in the connection_data. */
static void
fini_connection_data( connection_data_t* cd )
    {
    fini_message_data( cd );
    if ( cd->ip_str != (char*) 0 )
	free( (void*) cd->ip_str );
    if ( cd->helo != (char*) 0 )
	free( (void*) cd->helo );
    lib_fini_connection_data( cd->lib_data );
    free( (void*) cd );
    }


static char*
result_str( int result )
    {
    switch ( result )
	{
	case SPFMILTER_RESULT_PASS: return "pass";
	case SPFMILTER_RESULT_FAIL: return "fail";
	case SPFMILTER_RESULT_SOFTFAIL: return "softfail";
	case SPFMILTER_RESULT_NEUTRAL: return "neutral";
	case SPFMILTER_RESULT_UNKNOWN: return "unknown";
	case SPFMILTER_RESULT_UNMECH: return "unknown-mechanisms";
	case SPFMILTER_RESULT_ERROR: return "error";
	case SPFMILTER_RESULT_NONE: return "none";
	default: return "???";
	}
    }


static void
reason_str( connection_data_t* cd, char* str, size_t size )
    {
    switch ( cd->result )
	{
	case SPFMILTER_RESULT_PASS:
	(void) snprintf( str, size, "%s: domain of %s designates %s as permitted sender", local_hostname, cd->use_mail_from, cd->ip_str );
	break;
	case SPFMILTER_RESULT_FAIL:
	(void) snprintf( str, size, "%s: domain of %s does not designate %s as permitted sender", local_hostname, cd->use_mail_from, cd->ip_str );
	break;
	case SPFMILTER_RESULT_SOFTFAIL:
	(void) snprintf( str, size, "%s: domain of transitioning %s does not designate %s as permitted sender", local_hostname, cd->use_mail_from, cd->ip_str );
	break;
	case SPFMILTER_RESULT_NEUTRAL:
	(void) snprintf( str, size, "%s: %s is neither permitted nor denied by domain of %s", local_hostname, cd->ip_str, cd->use_mail_from );
	break;
	case SPFMILTER_RESULT_UNKNOWN:
	(void) snprintf( str, size, "%s: unknown result during lookup of %s", local_hostname, cd->use_mail_from );
	break;
	case SPFMILTER_RESULT_UNMECH:
	(void) snprintf( str, size, "%s: domain of %s uses a mechanism not recognized by this client", local_hostname, cd->use_mail_from );
	break;
	case SPFMILTER_RESULT_ERROR:
	(void) snprintf( str, size, "%s: error in processing during lookup of %s", local_hostname, cd->use_mail_from );
	break;
	case SPFMILTER_RESULT_NONE:
	(void) snprintf( str, size, "%s: %s does not designate permitted sender hosts", local_hostname, cd->use_mail_from );
	break;
	default:
	(void) snprintf( str, size, "%s: ???", local_hostname );
	break;
	}
    }


/* Decide which library we can/should use. */
#ifdef HAVE_LIBSPF_SPF_H
#define USE_LIBSPF
#endif
#ifdef HAVE_SPF2_SPF_H
#define USE_LIBSPF2
#endif

#if defined(USE_LIBSPF) && defined(USE_LIBSPF2)
/* #error "both libspf and libspf2 are present - please pick one" */
#undef USE_LIBSPF2	/* if both are present, use libspf */
#endif




#ifdef USE_LIBSPF

/* Libspf data and routines. */


#include <libspf/spf.h>


struct lib_fallback_s {
    int junk;	/* not used, but mallocing zero bytes can be a problem */
    };


struct lib_data_s {
    peer_info_t* peer_info;
    SPF_RESULT res;
    };


static int
lib_init( void )
    {
    if ( recipientmx )
	{
	(void) fprintf( stderr, "Sorry, --recipientmx mode is not implemented with libspf; if you want" );
	(void) fprintf( stderr, "to use that flag you will need to rebuild spfmilter with libspf2.\n" );
	return 0;
	}

    /* SPF_dbg_level( debug ); */

    return 1;
    }


static const char*
lib_version( void )
    {
    return "libspf-unknown";	/* the library doesn't tell us its version */
    }


static lib_fallback_t*
lib_init_fallback( const char* str )
    {
    lib_fallback_t* lf;

    lf = (lib_fallback_t*) malloc( sizeof(lib_fallback_t) );
    if ( lf == (lib_fallback_t*) 0 )
	return (lib_fallback_t*) 0;
    return lf;
    }


static int
lib_set_local_hostname( lib_data_t* ld )
    {
    return 1;
    }


static lib_data_t*
lib_init_connection_data( void )
    {
    lib_data_t* ld;

    ld = (lib_data_t*) malloc( sizeof(lib_data_t) );
    if ( ld == (lib_data_t*) 0 )
	return (lib_data_t*) 0;
    return ld;
    }


static int
lib_init_message_data( lib_data_t* ld )
    {
    return 1;
    }


static int
lib_set_ipv4( lib_data_t* ld, struct in_addr ipv4_addr, char* ipv4_str )
    {
    ld->peer_info = SPF_init( local_hostname, ipv4_str, explanation_str, (char*) 0, (char*) 0, trustedforwarders, 0 );
    if ( ld->peer_info == (peer_info_t*) 0 )
	return 0;
    ld->res = -1;
    return 1;
    }


static int
lib_set_ipv6( lib_data_t* ld, struct in6_addr ipv6_addr, char* ipv6_str )
    {
    ld->peer_info = SPF_init( local_hostname, ipv6_str, explanation_str, (char*) 0, (char*) 0, trustedforwarders, 0 );
    if ( ld->peer_info == (peer_info_t*) 0 )
	return 0;
    ld->res = -1;
    return 1;
    }


static int
lib_set_helo_hostname( lib_data_t* ld, char* helo_hostname )
    {
    if ( ! SPF_smtp_helo( ld->peer_info, helo_hostname ) )
	return 0;
    return 1;
    }


static int
lib_set_from( lib_data_t* ld, const char* from )
    {
    if ( ! SPF_smtp_from( ld->peer_info, from ) )
	return 0;
    return 1;
    }


static int
lib_do_check( lib_data_t* ld, const char* from )
    {
    ld->res = SPF_policy_main( ld->peer_info );

    /* If there was no SPF record, then try finding a fallback mechanism. */
    if ( ld->res == SPF_NONE )
	{
	int i = find_fallback( from );

	if ( i != -1 )
	    {
	    if ( ! SPF_parse_policy( ld->peer_info, fallbacks[i].str ) )
		return 0;
	    ld->res = ld->peer_info->RES;
	    }
	}

    return 1;
    }


static int
lib_do_check_recipient( lib_data_t* ld, const char* to )
    {
    /* Not implemented for libspf. */
    return 0;
    }


static int
lib_do_check_final( lib_data_t* ld )
    {
    /* Not implemented for libspf. */
    return 0;
    }


static int
lib_get_result( lib_data_t* ld )
    {
    /* Convert libspf result to spfmilter result. */
    switch ( ld->res )
	{
	case SPF_PASS: return SPFMILTER_RESULT_PASS;
	case SPF_NONE: return SPFMILTER_RESULT_NONE;
	case SPF_S_FAIL: return SPFMILTER_RESULT_SOFTFAIL;
	case SPF_H_FAIL: return SPFMILTER_RESULT_FAIL;
	case SPF_ERROR: return SPFMILTER_RESULT_ERROR;
	case SPF_NEUTRAL: return SPFMILTER_RESULT_NEUTRAL;
	case SPF_UNKNOWN: return SPFMILTER_RESULT_UNKNOWN;
	case SPF_UNMECH: return SPFMILTER_RESULT_UNMECH;
	default: return -1;
	}
    }


static const char*
lib_get_explanation( lib_data_t* ld )
    {
    return ld->peer_info->explain;
    }


static const char*
lib_get_error( lib_data_t* ld )
    {
    return ld->peer_info->error;
    }


static void
lib_fini_message_data( lib_data_t* ld )
    {
    }


static void
lib_fini_connection_data( lib_data_t* ld )
    {
    if ( ld != (lib_data_t*) 0 )
	{
	if ( ld->peer_info != (peer_info_t*) 0 )
	    ld->peer_info = SPF_close( ld->peer_info );
	free( (void*) ld );
	}
    }


static void
lib_fini_fallback( lib_fallback_t* lf )
    {
    if ( lf != (lib_fallback_t*) 0 )
	free( (void*) lf );
    }


static void
lib_fini( void )
    {
    }

#endif /* USE_LIBSPF */


#ifdef USE_LIBSPF2

/* Libspf2 data and routines. */

#include <spf2/spf.h>

/* Which version of libspf2 do we have? */
#if ( SPF_LIB_VERSION_MAJOR == 1 && SPF_LIB_VERSION_MINOR < 2 )


/* Old libspf2 API. */

#undef USE_DNS_CACHE


#include <spf2/spf_dns_resolv.h>
#ifdef USE_DNS_CACHE
#include <spf2/spf_dns_cache.h>
#endif /* USE_DNS_CACHE */


static SPF_config_t spfc_global;
static SPF_c_results_t spfr_localpolicy;
static SPF_c_results_t spfr_explanation;


struct lib_fallback_s {
    SPF_c_results_t results;
    };


struct lib_data_s {
    SPF_config_t spfcid;
    const char* from;
    SPF_output_t output;
    int got_output;
    };


/* The libspf2 code uses a resolver pool to work around a memory leak
** in the resolver library.
*/
static void init_resolvers( void );
static void fini_resolvers( void );
static int find_resolver( void );
static SPF_dns_config_t get_resolver( int r );
static void free_resolver( int r );


static int
lib_init( void )
    {
    spfc_global = SPF_create_config();
    if ( spfc_global == (SPF_config_t) 0 )
	{
	(void) fprintf( stderr, "%s: SPF_create_config() failed\n", argv0 );
	return 0;
	}

    SPF_set_debug( spfc_global, debug );

    if ( localpolicy_str != (char*) 0 )
	{
	SPF_init_c_results( &spfr_localpolicy );
	if ( SPF_compile_local_policy( spfc_global, localpolicy_str, trustedforwarders, &spfr_localpolicy ) )
	    {
	    (void) fprintf( stderr, "%s: error compiling local policy - %s\n", argv0, spfr_localpolicy.err_msg );
	    return 0;
	    }
	SPF_set_local_policy( spfc_global, spfr_localpolicy );
	}

    if ( explanation_str != (char*) 0 )
	{
	SPF_init_c_results( &spfr_explanation );
	if ( SPF_compile_exp( spfc_global, explanation_str, &spfr_explanation ) )
	    {
	    (void) fprintf( stderr, "%s: error compiling explanation - %s\n", argv0, spfr_explanation.err_msg );
	    return 0;
	    }
	SPF_set_exp( spfc_global, spfr_explanation );
	}

    init_resolvers();
    return 1;
    }


static const char*
lib_version( void )
    {
    static char version[100];

    (void) snprintf( version, sizeof(version), "libspf2-%d.%d.%d", SPF_LIB_VERSION_MAJOR, SPF_LIB_VERSION_MINOR, SPF_LIB_VERSION_PATCH );
    return version;
    }


static lib_fallback_t*
lib_init_fallback( const char* str )
    {
    lib_fallback_t* lf;

    lf = (lib_fallback_t*) malloc( sizeof(lib_fallback_t) );
    if ( lf == (lib_fallback_t*) 0 )
	return (lib_fallback_t*) 0;
    SPF_init_c_results( &lf->results );
    if ( SPF_compile_local_policy( spfc_global, str, trustedforwarders, &lf->results ) )
	{
	free( (void*) lf );
	return (lib_fallback_t*) 0;
	}
    return lf;
    }


static int
lib_set_local_hostname( lib_data_t* ld )
    {
    if ( SPF_set_rec_dom( spfc_global, local_hostname ) )
	return 0;
    return 1;
    }


static lib_data_t*
lib_init_connection_data( void )
    {
    lib_data_t* ld;

    ld = (lib_data_t*) malloc( sizeof(lib_data_t) );
    if ( ld == (lib_data_t*) 0 )
	return (lib_data_t*) 0;
    ld->from = (char*) 0;
    ld->got_output = 0;

    /* Create the per-connection config object by duping the global one. */
    ld->spfcid = SPF_dup_config( spfc_global );
    if ( ld->spfcid == (SPF_config_t) 0 )
	{
	free( (void*) ld );
	return (lib_data_t*) 0;
	}

    return ld;
    }


static int
lib_init_message_data( lib_data_t* ld )
    {
    ld->from = (char*) 0;
    return 1;
    }


static int
lib_set_ipv4( lib_data_t* ld, struct in_addr ipv4_addr, char* ipv4_str )
    {
    if ( SPF_set_ipv4( ld->spfcid, ipv4_addr ) )
	return 0;
    return 1;
    }


static int
lib_set_ipv6( lib_data_t* ld, struct in6_addr ipv6_addr, char* ipv6_str )
    {
    if ( SPF_set_ipv6( ld->spfcid, ipv6_addr ) )
	return 0;
    return 1;
    }


static int
lib_set_helo_hostname( lib_data_t* ld, char* helo_hostname )
    {
    if ( SPF_set_helo_dom( ld->spfcid, helo_hostname ) )
	return 0;
    return 1;
    }


static int
lib_set_from( lib_data_t* ld, const char* from )
    {
    ld->from = strdup( from );
    if ( ld->from == (char*) 0 )
	return 0;
    if ( SPF_set_env_from( ld->spfcid, ld->from ) )
	return 0;
    return 1;
    }


static int
lib_do_check( lib_data_t* ld, const char* from )
    {
    int r;

    /* Find a resolver to use. */
    r  = find_resolver();
    if ( r < 0 )
	return 0;

    /* Do the SPF lookup. */
    ld->output = SPF_result( ld->spfcid, get_resolver( r ) );
    ld->got_output = 1;

    /* If there was no SPF record, then try finding a fallback mechanism. */
    if ( ld->output.result == SPF_RESULT_NONE )
	{
	int i = find_fallback( from );
	if ( i != -1 )
	    {
	    SPF_free_output( &ld->output );
	    ld->output = SPF_eval_id( ld->spfcid, fallbacks[i].lib_fallback->results.spfid, get_resolver( r ), 1, 0, (int*) 0 );
	    SPF_result_comments( ld->spfcid, get_resolver( r ), fallbacks[i].lib_fallback->results, &ld->output );
	    }
	}

    /* And put the resolver back into the pool. */
    free_resolver( r );

    return 1;
    }


static int
lib_do_check_recipient( lib_data_t* ld, const char* to )
    {
    int r;

    /* Find a resolver to use. */
    r  = find_resolver();
    if ( r < 0 )
	return 0;

    /* If we're already checked a previous recipient, discard that output,
    ** it's no longer needed.
    */
    if ( ld->got_output )
	{
	SPF_free_output( &ld->output );
	ld->got_output = 0;
	}

    /* Do the SPF lookup. */
    ld->output = SPF_result_2mx( ld->spfcid, get_resolver( r ), to );
    ld->got_output = 1;

    /* If there was no SPF record, then try finding a fallback mechanism. */
    if ( ld->output.result == SPF_RESULT_NONE )
	{
	int i = find_fallback( ld->from );
	if ( i != -1 )
	    {
	    SPF_free_output( &ld->output );
	    ld->output = SPF_eval_id( ld->spfcid, fallbacks[i].lib_fallback->results.spfid, get_resolver( r ), 1, 0, (int*) 0 );
	    SPF_result_comments( ld->spfcid, get_resolver( r ), fallbacks[i].lib_fallback->results, &ld->output );
	    }
	}

    /* And put the resolver back into the pool. */
    free_resolver( r );

    return 1;
    }


static int
lib_do_check_final( lib_data_t* ld )
    {
    int r;

    /* Find a resolver to use. */
    r  = find_resolver();
    if ( r < 0 )
	return 0;

    /* Discard output from any recipient checks. */
    if ( ld->got_output )
	{
	SPF_free_output( &ld->output );
	ld->got_output = 0;
	}

    /* Get the accumulated results for all recipients. */
    ld->output = SPF_result_2mx_msg( ld->spfcid, get_resolver( r ) );
    ld->got_output = 1;

    /* And put the resolver back into the pool. */
    free_resolver( r );

    return 1;
    }


static int
lib_get_result( lib_data_t* ld )
    {
    /* Convert libspf2 result to spfmilter result. */
    switch ( ld->output.result )
	{
	case SPF_RESULT_PASS: return SPFMILTER_RESULT_PASS;
	case SPF_RESULT_FAIL: return SPFMILTER_RESULT_FAIL;
	case SPF_RESULT_SOFTFAIL: return SPFMILTER_RESULT_SOFTFAIL;
	case SPF_RESULT_NEUTRAL: return SPFMILTER_RESULT_NEUTRAL;
	case SPF_RESULT_UNKNOWN: return SPFMILTER_RESULT_UNKNOWN;
	case SPF_RESULT_ERROR: return SPFMILTER_RESULT_ERROR;
	case SPF_RESULT_NONE: return SPFMILTER_RESULT_NONE;
	default: return -1;
	}
    }


static const char*
lib_get_explanation( lib_data_t* ld )
    {
    return ld->output.smtp_comment;
    }


static const char*
lib_get_error( lib_data_t* ld )
    {
    return SPF_strerror( ld->output.err );
    }


static void
lib_fini_message_data( lib_data_t* ld )
    {
    if ( ld->from != (char*) 0 )
	{
	free( (void*) ld->from );
	ld->from = (char*) 0;
	}
    if ( ld->got_output )
	{
	SPF_free_output( &ld->output );
	ld->got_output = 0;
	}
    }


static void
lib_fini_connection_data( lib_data_t* ld )
    {
    if ( ld != (lib_data_t*) 0 )
	{
	if ( ld->spfcid != (SPF_config_t) 0 )
	    SPF_destroy_config( ld->spfcid );
	free( (void*) ld );
	}
    }


static void
lib_fini_fallback( lib_fallback_t* lf )
    {
    if ( lf != (lib_fallback_t*) 0 )
	{
	SPF_free_c_results( &lf->results );
	free( (void*) lf );
	}
    }


static void
lib_fini( void )
    {
    fini_resolvers();
    if ( localpolicy_str != (char*) 0 )
	SPF_free_c_results( &spfr_localpolicy );
    if ( explanation_str != (char*) 0 )
	SPF_free_c_results( &spfr_explanation );
    SPF_destroy_config( spfc_global );
    SPF_destroy_default_config();
    }


/* Resolver pool. */


/* Resolver data. */
typedef struct {
    int initialized, inuse;
    int next_free_resolver;
    SPF_dns_config_t spfdcid_r;
#ifdef USE_DNS_CACHE
    SPF_dns_config_t spfdcid_c;
#endif /* USE_DNS_CACHE */
    } resolver_t;
static resolver_t* resolvers;
static int max_resolvers, first_free_resolver, num_resolvers_inuse;
static pthread_mutex_t resolver_mutex;


static void
init_resolvers( void )
    {
    int r;

    max_resolvers = 100;	/* arbitrary */
    resolvers = (resolver_t*) malloc( max_resolvers * sizeof(resolver_t) );
    if ( resolvers == (resolver_t*) 0 )
	{
	(void) fprintf( stderr, "%s: out of memory allocating resolver pool\n", argv0 );
	exit( 1 );
	}
    for ( r = 0; r < max_resolvers; ++r )
	{
	resolvers[r].initialized = 0;
	resolvers[r].inuse = 0;
	resolvers[r].next_free_resolver = r + 1;
	}
    resolvers[max_resolvers - 1].next_free_resolver = -1;
    first_free_resolver = 0;
    num_resolvers_inuse = 0;
    if ( pthread_mutex_init( &resolver_mutex, (pthread_mutexattr_t*) 0 ) != 0 )
	{
	(void) fprintf( stderr, "%s: pthread_mutex_init() failed\n", argv0 );
	exit( 1 );
	}
    }


static void
fini_resolvers( void )
    {
    int r;

    for ( r = 0; r < max_resolvers; ++r )
	if ( resolvers[r].initialized )
	    {
#ifdef USE_DNS_CACHE
	    SPF_dns_destroy_config_cache( resolvers[r].spfdcid_c );
#endif /* USE_DNS_CACHE */
	    SPF_dns_destroy_config_resolv( resolvers[r].spfdcid_r );
	    }
    pthread_mutex_destroy( &resolver_mutex );
    free( (void*) resolvers );
    }


static int
find_resolver( void )
    {
    int r;

    /* Find a free slot. */
    pthread_mutex_lock( &resolver_mutex );
    if ( num_resolvers_inuse >= max_resolvers )
	{
	/* All are in use, so let's expand the table. */
	int new_max_resolvers;
	resolver_t* new_resolvers;

	new_max_resolvers = max_resolvers * 2;
	syslog( LOG_NOTICE, "enlarging the resolver pool from %d to %d", max_resolvers, new_max_resolvers );
	new_resolvers = (resolver_t*) realloc( (void*) resolvers, new_max_resolvers * sizeof(resolver_t) );
	if ( new_resolvers == (resolver_t*) 0 )
	    {
	    pthread_mutex_unlock( &resolver_mutex );
	    syslog( LOG_ERR, "out of memory enlarging the resolver pool" );
	    return -1;
	    }
	for ( r = max_resolvers; r < new_max_resolvers; ++r )
	    {
	    new_resolvers[r].initialized = 0;
	    new_resolvers[r].inuse = 0;
	    new_resolvers[r].next_free_resolver = r + 1;
	    }
	new_resolvers[new_max_resolvers - 1].next_free_resolver = first_free_resolver;
	first_free_resolver = max_resolvers;
	resolvers = new_resolvers;
	max_resolvers = new_max_resolvers;
	}
    if ( first_free_resolver == -1 || resolvers[first_free_resolver].inuse )
	{
	pthread_mutex_unlock( &resolver_mutex );
	syslog( LOG_ERR, "the resolver pool free list is messed up" );
	return -1;
	}
    r = first_free_resolver;
    first_free_resolver = resolvers[r].next_free_resolver;
    resolvers[r].next_free_resolver = -1;
    resolvers[r].inuse = 1;
    ++num_resolvers_inuse;
    pthread_mutex_unlock( &resolver_mutex );

    /* Is it initialized? */
    if ( ! resolvers[r].initialized )
	{
	/* Create the DNS objects. */
	resolvers[r].spfdcid_r = SPF_dns_create_config_resolv( (SPF_dns_config_t) 0, debug );
	if ( resolvers[r].spfdcid_r == (SPF_dns_config_t) 0 )
	    {
	    syslog( LOG_ERR, "SPF_dns_create_config_resolv() failed" );
	    free_resolver( r );
	    return -1;
	    }
#ifdef USE_DNS_CACHE
	resolvers[r].spfdcid_c = SPF_dns_create_config_cache( resolvers[r].spfdcid_r, 8, debug );
	if ( resolvers[r].spfdcid_c == (SPF_dns_config_t) 0 )
	    {
	    syslog( LOG_ERR, "SPF_dns_create_config_cache() failed"  );
	    free_resolver( r );
	    return -1;
	    }
#endif /* USE_DNS_CACHE */
	resolvers[r].initialized = 1;
	}

    return r;
    }


static SPF_dns_config_t
get_resolver( int r )
    {
#ifdef USE_DNS_CACHE
    return resolvers[r].spfdcid_c;
#else /* USE_DNS_CACHE */
    return resolvers[r].spfdcid_r;
#endif /* USE_DNS_CACHE */
    }


static void
free_resolver( int r )
    {
    pthread_mutex_lock( &resolver_mutex );
    resolvers[r].inuse = 0;
    resolvers[r].next_free_resolver = first_free_resolver;
    first_free_resolver = r;
    --num_resolvers_inuse;
    pthread_mutex_unlock( &resolver_mutex );
    }


#else

/* New libspf2 API. */

static SPF_server_t* spf_server;

struct lib_fallback_s {
    /*!!!*/
    int junk;
    };

struct lib_data_s {
    SPF_request_t* spf_request;
    SPF_response_t* spf_response;
    };


static int
lib_init( void )
    {
    spf_server = SPF_server_new( SPF_DNS_CACHE, debug );
    if ( spf_server == (SPF_server_t*) 0 )
	{
	(void) fprintf( stderr, "%s: SPF_server_new() failed\n", argv0 );
	return 0;
	}

    if ( localpolicy_str != (char*) 0 )
	{
	int err, i;
	spf_response_t* lp_response;

	err = SPF_server_set_localpolicy( spf_server, localpolicy_str, trustedforwarders, &lp_response );
	if ( err )
	    {
	    (void) fprintf( stderr, "%s: error compiling local policy - %d (%s)\n", argv0, err, SPF_strerror( err ) );
	    for ( i = 0; i < SPF_response_messages( lp_response ); ++i )
		{
		SPF_error_t* lp_error = SPF_response_message( lp_response, i );
		(void) fprintf( stderr, "  %s: %s\n",
		    SPF_error_errorp( lp_error ) ? "Error" : "Warning",
		    SPF_error_message( lp_error ) );
		}
	    SPF_response_free( lp_response );
	    return 0;
	    }
	SPF_response_free( lp_response );
	}

    if ( explanation_str != (char*) 0 )
	{
	int err, i;
	spf_response_t* ex_response;

	err = SPF_server_set_explanation( spf_server, explanation_str, &ex_response );
	if ( err )
	    {
	    (void) fprintf( stderr, "%s: error compiling explanation - %d (%s)\n", argv0, err, SPF_strerror( err ) );
	    for ( i = 0; i < SPF_response_messages( ex_response ); ++i )
		{
		SPF_error_t* ex_error = SPF_response_message( ex_response, i );
		(void) fprintf( stderr, "  %s: %s\n",
		    SPF_error_errorp( ex_error ) ? "Error" : "Warning",
		    SPF_error_message( ex_error ) );
		}
	    SPF_response_free( ex_response );
	    return 0;
	    }
	SPF_response_free( ex_response );
	}

    return 1;
    }


static const char*
lib_version( void )
    {
    static char version[100];

    (void) snprintf( version, sizeof(version), "libspf2-%d.%d.%d", SPF_LIB_VERSION_MAJOR, SPF_LIB_VERSION_MINOR, SPF_LIB_VERSION_PATCH );
    return version;
    }


static lib_fallback_t*
lib_init_fallback( const char* str )
    {
    lib_fallback_t* lf;

    lf = (lib_fallback_t*) malloc( sizeof(lib_fallback_t) );
    if ( lf == (lib_fallback_t*) 0 )
	return (lib_fallback_t*) 0;
    /*!!!*/
    return lf;
    }


static int
lib_set_local_hostname( lib_data_t* ld )
    {
    if ( SPF_server_set_rec_dom( spf_server, local_hostname ) )
	return 0;
    return 1;
    }


static lib_data_t*
lib_init_connection_data( void )
    {
    lib_data_t* ld;

    ld = (lib_data_t*) malloc( sizeof(lib_data_t) );
    if ( ld == (lib_data_t*) 0 )
	return (lib_data_t*) 0;

    ld->spf_request = SPF_request_new( spf_server );
    if ( ld->spf_request == (spf_request_t*) 0 )
	{
	free( (void*) ld );
	return (lib_data_t*) 0;
	}
    ld->spf_response = (spf_response_t*) 0;

    return ld;
    }


static int
lib_init_message_data( lib_data_t* ld )
    {
    /*!!!*/
    return 0;
    }


static int
lib_set_ipv4( lib_data_t* ld, struct in_addr ipv4_addr, char* ipv4_str )
    {
    if ( SPF_request_set_ipv4_str( ld->spf_request, ipv4_str ) )
	return 0;
    return 1;
    }


static int
lib_set_ipv6( lib_data_t* ld, struct in6_addr ipv6_addr, char* ipv6_str )
    {
    if ( SPF_request_set_ipv6_str( ld->spf_request, ipv4_str ) )
	return 0;
    return 1;
    }


static int
lib_set_helo_hostname( lib_data_t* ld, char* helo_hostname )
    {
    if ( SPF_request_set_helo_dom( ld->spf_request, helo_hostname ) )
	return 0;
    return 1;
    }


static int
lib_set_from( lib_data_t* ld, const char* from )
    {
    if ( SPF_request_set_env_from( ld->spf_request, from ) )
	return 0;
    return 1;
    }


static int
lib_do_check( lib_data_t* ld, const char* from )
    {
    /*!!!*/
    return 0;
    }


static int
lib_do_check_recipient( lib_data_t* ld, const char* to )
    {
    /*!!!*/
    return 0;
    }


static int
lib_do_check_final( lib_data_t* ld )
    {
    /*!!!*/
    return 0;
    }


static int
lib_get_result( lib_data_t* ld )
    {
    /* Convert libspf2 result to spfmilter result. */
    /*!!!*/
    return -1;
    }


static const char*
lib_get_explanation( lib_data_t* ld )
    {
    /*!!!*/
    return (char*) 0;
    }


static const char*
lib_get_error( lib_data_t* ld )
    {
    /*!!!*/
    return (char*) 0;
    }


static void
lib_fini_message_data( lib_data_t* ld )
    {
    /*!!!*/
    }


static void
lib_fini_connection_data( lib_data_t* ld )
    {
    if ( ld != (lib_data_t*) 0 )
	{
	if ( ld->spf_request != (spf_request_t*) 0 )
	    SPF_request_free( ld->spf_request );
	if ( ld->spf_response != (spf_response_t*) 0 )
	    SPF_response_free( ld->spf_response );
	free( (void*) ld );
	}
    }


static void
lib_fini_fallback( lib_fallback_t* lf )
    {
    if ( lf != (lib_fallback_t*) 0 )
	{
	/*!!!*/
	free( (void*) lf );
	}
    }


static void
lib_fini( void )
    {
    SPF_server_free( spf_server );
    }

#endif

#endif /* USE_LIBSPF2 */
