package SliMP3::Command;

# $Id: Command.pm,v 1.18 2003/08/07 17:33:15 blackketter Exp $

# SliMP3 Server Copyright (C) 2001,2002,2003 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

use strict;

use File::Basename;
use File::Spec::Functions qw(:ALL);
use FileHandle;
use IO::Socket qw(:DEFAULT :crlf);
use Time::HiRes;
use SliMP3::Display;
use SliMP3::Misc;
use SliMP3::Scan;
use SliMP3::Strings qw(string);

my %executeCallbacks;

#############################################################################
# execute - does all the hard work.  Use it.
# takes:
#   a client reference
#   a reference to an array of parameters
#   a reference to a callback function
#   a list of callback function args
#
# returns an array containing the given parameters

sub execute {
	my($client, $parrayref, $callbackf, $callbackargs) = @_;
	my $callcallback = 1;
	my $p0 = $$parrayref[0];
	my $p1 = $$parrayref[1];
	my $p2 = $$parrayref[2];
	my $p3 = $$parrayref[3];
	my $p4 = $$parrayref[4];
	my $p5 = $$parrayref[5];
	my $p6 = $$parrayref[6];

	$::d_command && msg(" Executing command " . ($client ? SliMP3::Client::id($client) : "no client") . ": $p0 (" .
			(defined $p1 ? $p1 : "") . ") (" .
			(defined $p2 ? $p2 : "") . ") (" .
			(defined $p3 ? $p3 : "") . ") (" .
			(defined $p4 ? $p4 : "") . ") (" .
			(defined $p5 ? $p5 : "") . ") (" .
			(defined $p6 ? $p6 : "") . ")\n");

	# The first parameter is the client identifier to execute the command.
	# If a parameter is "?" it is replaced by the current value in the array
	# returned by the function
	# Below are a list of the commands:
	#	
	# P0	    	P1		    P2							P3			P4			P5				P6
	# play		
	# pause 		(0|1|)	
	# stop		
	# mode			<play|pause|stop>	
	# mode			?	
	# sleep 		(0..n)																	
	# sleep 		?																		
	# gototime		(0..n sec)|(-n..+n sec)	
	# power 		(0|1|?)	
	# time			?	
	# genre			?	
	# artist		?	
	# album			?	
	# title			?	
	# duration  	?	
			
	# playlist 		play 	    <item>			(item can be a song, playlist or directory. synonym: load)
	# playlist 		insert		<item>			(item can be a song, playlist or directory. synonym: insertlist)
	# playlist 		add 	    <item>			(item can be a song, playlist or directory. synonym: append)

	# playlist  	playalbum   <genre>						<artist>	<album>	 	<songtitle>  (synonym: loadalbum)
	# playlist  	insertalbum <genre> 					<artist>	<album>	 	<songtitle>
	# playlist  	addalbum 	<genre> 					<artist>	<album>	 	<songtitle>
	
	# playlist		deletealbum	<genre>						<artist>	<album>		<songtitle>
	# playlist		deleteitem	<filename/playlist>	
		
	# playlist 		resume 		<playlist>	
	# playlist 		save 		<playlist>	
		
	# playlist 		clear	
	# playlist 		move 		<fromoffset> 				<tooffset>	
	# playlist 		delete 		<songoffset>	
			
	# playlist 		jump 		<index>	
	# playlist 		index		<index>						?	
	# playlist		genre		<index>						?	
	# playlist		artist		<index>						?	
	# playlist		album		<index>						?	
	# playlist		title		<index>						?	
	# playlist		duration	<index>						?	
	# playlist		tracks		?	
	# mixer			volume		(0 .. 100)|(-100 .. +100)	
	# mixer			volume		?	
	# mixer			balance		(-100 .. 100)|(-200 .. +200)			(not implemented!)
	# mixer			bass		(0 .. 100)|(-100 .. +100)									
	# mixer			treble		(0 .. 100)|(-100 .. +100)									
	# display   	<line1> 	<line2>                     <duration>
	# display   	? 			?	
	# button   		<buttoncode>	
	# player		count		?	
	# player		name		<playerindex|playerid>				?	
	# player		address		<playerindex|playerid>				?	
	# player		ip			<playerindex|playerid>				?	
	# pref			<prefname>	<prefvalue>	
	# pref			<prefname>	?		
	# playerpref	<prefname>	<prefvalue>	
	# playerpref	<prefname>	?		
	
	# rescan	
	
	if (!defined($p0)) {
		# ignore empty commands
	# these commands don't require a valid client
	} elsif ($p0 eq "player") {
		if (!defined($p1)) {
		
		} elsif ($p1 eq "count") {
			$p2 = SliMP3::Client::clientCount();
		} elsif ($p1 eq "name" || $p1 eq "address" || $p1 eq "ip") {
		
			my $p2client;
			
			# were we passed an ID?
			if (defined $p2 && SliMP3::Client::getClient($p2)) {
				$p2client = SliMP3::Client::getClient($p2);
			} else {
			
			# otherwise, try for an index
				my @clients = SliMP3::Client::clients();
				if (defined $p2 && defined $clients[$p2]) {
					$p2client = $clients[$p2];
				}
			}
			
			if (defined $p2client) {
				if ($p1 eq "name") {
					$p3 = SliMP3::Client::name($p2client);
				} elsif ($p1 eq "address") {
					$p3 = SliMP3::Client::id($p2client);
				} elsif ($p1 eq "ip") {
					$p3 = SliMP3::Client::ipaddress($p2client);
				}
			}
		} 
	} elsif ($p0 eq "pref") {
		if (defined($p2) && $p2 ne '?' && !$::nosetup) {
			SliMP3::Prefs::set($p1, $p2);
		}
		$p2 = SliMP3::Prefs::get($p1);
	} elsif ($p0 eq "rescan") {
		SliMP3::MusicFolderScan::startScan();
	} elsif ($p0 eq "debug") {
		if ($p1 =~ /^d_/) {
			my $debugsymbol = "::" . $p1;
			no strict 'refs';
			if (!defined($p2)) {
				$$debugsymbol = ! $$debugsymbol;
				$p2 = $$debugsymbol;

			} elsif ($p2 eq "?")  {
				$p2 = $$debugsymbol;
			} else {
				$$debugsymbol = $p2;
			}	
			use strict 'refs';
			if (!$p2) { $p2 = 0; }
		}
	#the following commands require a valid client to be specified
	} elsif ($client) {
		if ($p0 eq "playerpref") {
			if (defined($p2) && $p2 ne '?' && !$::nosetup) {
				SliMP3::Prefs::clientSet($client, $p1, $p2);
			}
			$p2 = SliMP3::Prefs::clientGet($client, $p1);
		} elsif ($p0 eq "play") {
			SliMP3::Playlist::playmode($client, "play");
			SliMP3::Playlist::rate($client,1);
		} elsif ($p0 eq "pause") {
			if (defined($p1)) {
				if ($p1 && $client->playmode eq "play") {
					SliMP3::Playlist::playmode($client, "pause");
				} elsif (!$p1 && $client->playmode eq "pause") {
					SliMP3::Playlist::playmode($client, "resume");
				}
			} else {
				if ($client->playmode eq "pause") {
					SliMP3::Playlist::playmode($client, "resume");
				} elsif ($client->playmode eq "play") {
					SliMP3::Playlist::playmode($client, "pause");
				}
			}
		} elsif ($p0 eq "rate") {
			if ($client->mp3filehandleIsSocket) {
				SliMP3::Playlist::rate($client, 1);
			} elsif (!defined($p1) || $p1 eq "?") {
				$p1 = SliMP3::Playlist::rate($client);
			} else {
				if ($p1 == 0) {
					SliMP3::Playlist::playmode($client, "pausenow");
				} elsif ($p1 == 1) {
					SliMP3::Playlist::playmode($client, "play");
				}
				SliMP3::Playlist::rate($client,$p1);
			}
		} elsif ($p0 eq "stop") {
			SliMP3::Playlist::playmode($client, "stop");
		} elsif ($p0 eq "mode") {
			if (!defined($p1) || $p1 eq "?") {
				$p1 = $client->playmode;
			} else {
				if ($client->playmode eq "pause" && $p1 eq "play") {
					SliMP3::Playlist::playmode($client, "resume");
				} else {
					SliMP3::Playlist::playmode($client, $p1);
				}
			}
		} elsif ($p0 eq "sleep") {
	
			if ($p1 eq "?") {
				$p1 = $client->sleepTime() - Time::HiRes::time();
				if ($p1 < 0) {
					$p1 = 0;
				}
			} else {
				SliMP3::Timers::killTimers($client, \&gotosleep);
				
				if ($p1 != 0) {
					SliMP3::Timers::setTimer($client, Time::HiRes::time() + $p1, \&gotosleep);
					$client->sleepTime(Time::HiRes::time() + $p1);
				} else {
					$client->sleepTime(0);
				}
			}	
		} elsif ($p0 eq "gototime" || $p0 eq "time") {
			if ($p1 eq "?") {
				$p1 = SliMP3::Playlist::songTime($client);
			} else {
				SliMP3::Playlist::gototime($client, $p1, 1);
			}
		} elsif ($p0 eq "duration") {
			$p1 = SliMP3::Info::durationSeconds(SliMP3::Playlist::song($client)) || 0;
		} elsif ($p0 eq "artist") {
			$p1 = SliMP3::Info::artist(SliMP3::Playlist::song($client)) || 0;
		} elsif ($p0 eq "album") {
			$p1 = SliMP3::Info::album(SliMP3::Playlist::song($client)) || 0;
		} elsif ($p0 eq "title") {
			$p1 = SliMP3::Info::title(SliMP3::Playlist::song($client)) || 0;
		} elsif ($p0 eq "genre") {
			$p1 = SliMP3::Info::genre(SliMP3::Playlist::song($client)) || 0;
		} elsif ($p0 eq "path") {
			$p1 = SliMP3::Playlist::song($client) || 0;
		} elsif ($p0 eq "power") {
			if (!defined $p1) {
				SliMP3::Client::power($client, !SliMP3::Client::power($client));
			} elsif ($p1 eq "?") {
				$p1 = SliMP3::Client::power($client);
			} else {
				SliMP3::Client::power($client, $p1)
			}
		} elsif ($p0 eq "playlist") {
			# here are all the commands that add/insert/replace songs/directories/playlists on the current playlist
			if ($p1 =~ /^(play|load|append|add|resume|insert|insertlist)$/) {
			
				my $jumptoindex = undef;
				my $path = $p2;
				my $easypath = catfile(SliMP3::Prefs::get('playlistdir'), basename ($p2) . ".m3u");
				if (-e $easypath) {
					$path = $easypath;
				} else {
					$easypath = catfile(SliMP3::Prefs::get('playlistdir'), basename ($p2) . ".pls");
					if (-e $easypath) {
						$path = $easypath;
					}
				}
				
				if ($p1 =~ /^(play|load|resume)$/) {
					SliMP3::Playlist::playmode($client, "stop");
					SliMP3::Playlist::clear($client);
				}				
				
				$path = SliMP3::Misc::virtualToAbsolute($path);
				
				if ($p1 =~ /^(play|load)$/) { 
					$jumptoindex = 0;
				} elsif ($p1 eq "resume" && SliMP3::Info::isM3U($path)) {
					# do nothing to the index if we can't open the list
					my $playlist = new FileHandle($path, "r");
					if ($playlist) {
						# retrieve comment with track number in it
						$jumptoindex = $playlist->getline;
						if ($jumptoindex =~ /^#CURTRACK (\d+)$/) {
							$jumptoindex = $1;
						} else {
							$jumptoindex = 0;
						}
						close $playlist;
					}
             	}   
             	
				if ($p1 =~ /^(insert|insertlist)$/) {
					my $playListSize = SliMP3::Playlist::count($client);
					my @dirItems;
					SliMP3::Scan::addToList(\@dirItems, $path, 1, undef);
					SliMP3::Scan::addToList(SliMP3::Playlist::playList($client), $path, 1, undef, \&insert_done, $client, $playListSize, scalar(@dirItems),$callbackf, $callbackargs);			
				} else {
					SliMP3::Scan::addToList(SliMP3::Playlist::playList($client), $path, 1, undef, \&load_done, $client, $jumptoindex, $callbackf, $callbackargs);				
				}
				
				$callcallback = 0;
				$p2 = $path;
				
			} elsif ($p1 eq "loadalbum" | $p1 eq "playalbum") {
				SliMP3::Playlist::playmode($client, "stop");
				SliMP3::Playlist::clear($client);
				push(@{SliMP3::Playlist::playList($client)}, SliMP3::Info::songs(singletonRef($p2), singletonRef($p3), singletonRef($p4), singletonRef($p5), $p6));
				SliMP3::Playlist::reshuffle($client);
				SliMP3::Playlist::jumpto($client, 0);
			} elsif ($p1 eq "addalbum") {
				push(@{SliMP3::Playlist::playList($client)}, SliMP3::Info::songs(singletonRef($p2), singletonRef($p3), singletonRef($p4), singletonRef($p5), $p6));
				SliMP3::Playlist::reshuffle($client);
			} elsif ($p1 eq "insertalbum") {
				my @songs = SliMP3::Info::songs(singletonRef($p2), singletonRef($p3), singletonRef($p4), singletonRef($p5), $p6);
				my $playListSize = SliMP3::Playlist::count($client);
				my $size = scalar(@songs);
				push(@{SliMP3::Playlist::playList($client)}, @songs);
				insert_done($client, $playListSize, $size);
				#SliMP3::Playlist::reshuffle($client);
			} elsif ($p1 eq "save") {
				if ( SliMP3::Prefs::get('playlistdir')) {

					# just use the filename to avoid nasties
					my $savename = basename ($p2);
                    # save the current playlist position as a comment at the head of the list
					my @annotatedlist = @{SliMP3::Playlist::playList($client)};
					SliMP3::Parse::writeM3U( \@annotatedlist, catfile(SliMP3::Prefs::get('playlistdir'), $savename . ".m3u"), 1, SliMP3::Playlist::currentSongIndex($client));
				}				
			} elsif ($p1 eq "deletealbum") {
				my @listToRemove=SliMP3::Info::songs(singletonRef($p2),singletonRef($p3),singletonRef($p4),singletonRef($p5),$p6);
				SliMP3::Playlist::removeMultipleTracks($client,\@listToRemove);
			} elsif ($p1 eq "deleteitem") {
				if (defined($p2) && $p2 ne '') {
					$p2 = SliMP3::Misc::virtualToAbsolute($p2);
					my $contents;
					if (!SliMP3::Info::isList($p2)) {
						SliMP3::Playlist::removeMultipleTracks($client,[$p2]);
					} elsif (SliMP3::Info::isDir($p2)) {
						SliMP3::Scan::addToList(\@{$contents}, $p2, 1);
						SliMP3::Playlist::removeMultipleTracks($client,\@{$contents});
					} else {
						$contents = SliMP3::Info::cachedPlaylist($p2);
						if (defined($contents)) {
							$contents = [SliMP3::Info::cachedPlaylist($p2)];
						} else {
							my $playlist_filehandle;
							if (!open($playlist_filehandle, $p2)) {
								$::d_files && msg("Couldn't open playlist file $p2 : $!");
								$playlist_filehandle = undef;				
							} else {
								$contents = [SliMP3::Parse::parseList($p2,$playlist_filehandle,dirname($p2))];
							}
						}
						if (defined($contents)) {
							SliMP3::Playlist::removeMultipleTracks($client,$contents);
						}
					}
				}
			} elsif ($p1 eq "repeat") {
				# change between repeat values 0 (don't repeat), 1 (repeat the current song), 2 (repeat all)
				$client->htmlstatusvalid(0);
				if (!defined($p2)) {
					SliMP3::Playlist::repeat($client, (SliMP3::Playlist::repeat($client) + 1) % 3);
				} elsif ($p2 eq "?") {
					$p2 = SliMP3::Playlist::repeat($client);
				} else {
					SliMP3::Playlist::repeat($client, $p2);
				}
			} elsif ($p1 eq "shuffle") {
				if (!defined($p2)) {
				    my $nextmode=(1,2,0)[SliMP3::Playlist::shuffle($client)];
					SliMP3::Playlist::shuffle($client, $nextmode);
					SliMP3::Playlist::reshuffle($client);
				} elsif ($p2 eq "?") {
					$p2 = SliMP3::Playlist::shuffle($client);
				} else {
					SliMP3::Playlist::shuffle($client, $p2);
					SliMP3::Playlist::reshuffle($client);
				}
			} elsif ($p1 eq "clear") {
				SliMP3::Playlist::clear($client);
				SliMP3::Playlist::playmode($client, "stop");
			} elsif ($p1 eq "move") {
				SliMP3::Playlist::moveSong($client, $p2, $p3);
			} elsif ($p1 eq "delete") {
				if (defined($p2)) {
					SliMP3::Playlist::removeTrack($client,$p2);
				}
			} elsif (($p1 eq "jump") || ($p1 eq "index")) {
				if ($p2 eq "?") {
					$p2 = SliMP3::Playlist::currentSongIndex($client);
				} else {
				#	if (SliMP3::Playlist::playmode($client) eq 'play') {
				#		SliMP3::Control::fade_volume($client, -0.3125, \&jumpto, [$client, $p2]);
				#	} else {
						SliMP3::Playlist::jumpto($client, $p2);
				#	}
				}
			} elsif ($p1 eq "tracks") {
				$p2 = SliMP3::Playlist::count($client);
			} elsif ($p1 eq "duration") {
				$p3 = SliMP3::Info::durationSeconds(SliMP3::Playlist::song($client,$p2)) || 0;
			} elsif ($p1 eq "artist") {
				$p3 = SliMP3::Info::artist(SliMP3::Playlist::song($client,$p2)) || 0;
			} elsif ($p1 eq "album") {
				$p3 = SliMP3::Info::album(SliMP3::Playlist::song($client,$p2)) || 0;
			} elsif ($p1 eq "title") {
				$p3 = SliMP3::Info::title(SliMP3::Playlist::song($client,$p2)) || 0;
			} elsif ($p1 eq "genre") {
				$p3 = SliMP3::Info::genre(SliMP3::Playlist::song($client,$p2)) || 0;
			} elsif ($p1 eq "path") {
				$p3 = SliMP3::Playlist::song($client,$p2) || 0;
			}
		} elsif ($p0 eq "mixer") {
			if ($p1 eq "volume") {
				my $newvol;
				my $oldvol = SliMP3::Prefs::clientGet($client, "volume");
				if ($p2 eq "?") {
					$p2 = $oldvol;
				} else {
					if($oldvol < 0) {
						# volume was previously muted
						$oldvol *= -1;      # un-mute volume
					} 
					
					if ($p2 =~ /^[\+\-]/) {
						$newvol = $oldvol + $p2;
					} else {
						$newvol = $p2;
					}
					
					if ($newvol> 100) { $newvol = $SliMP3::Control::maxVolume; }
					if ($newvol < 0) { $newvol = 0; }
					SliMP3::Prefs::clientSet($client, "volume", $newvol);
					SliMP3::Control::volume($client, $newvol);
					if (SliMP3::Playlist::isSynced($client)) {syncFunction($client, $newvol, "volume",\&SliMP3::Control::volume);};
				}
			} elsif ($p1 eq "muting") {
				my $vol = SliMP3::Prefs::clientGet($client, "volume");
				my $fade;
	
				if($vol < 0) {
					# need to un-mute volume
					SliMP3::Prefs::clientSet($client, "mute",0);
					$fade = 0.3125;
				} else {
					# need to mute volume
					SliMP3::Prefs::clientSet($client, "mute",1);
					$fade = -0.3125;
				}
				SliMP3::Control::fade_volume($client, $fade, \&SliMP3::Control::mute, [$client]);
				if (SliMP3::Playlist::isSynced($client)) {syncFunction($client, $fade, "mute",undef);};
			} elsif ($p1 eq "balance") {
				# unsupported yet
			} elsif ($p1 eq "treble") {
				my $newtreb;
				my $oldtreb = SliMP3::Prefs::clientGet($client, "treble");
				if ($p2 eq "?") {
					$p2 = $oldtreb;
				} else {
	
					if ($p2 =~ /^[\+\-]/) {
						$newtreb = $oldtreb + $p2;
					} else {
						$newtreb = $p2;
					}
					if ($newtreb > $SliMP3::Control::maxTreble) { $newtreb = $SliMP3::Control::maxTreble; }
					if ($newtreb < $SliMP3::Control::minTreble) { $newtreb = $SliMP3::Control::minTreble; }
					SliMP3::Prefs::clientSet($client, "treble", $newtreb);
					SliMP3::Control::treble($client, $newtreb);
					if (SliMP3::Playlist::isSynced($client)) {syncFunction($client, $newtreb, "treble",\&SliMP3::Control::treble);};
				}
			} elsif ($p1 eq "bass") {
				my $newbass;
				my $oldbass = SliMP3::Prefs::clientGet($client, "bass");
				if ($p2 eq "?") {
					$p2 = $oldbass;
				} else {
	
					if ($p2 =~ /^[\+\-]/) {
						$newbass = $oldbass + $p2;
					} else {
						$newbass = $p2;
					}
					if ($newbass > $SliMP3::Control::maxBass) { $newbass = $SliMP3::Control::maxBass; }
					if ($newbass < $SliMP3::Control::minBass) { $newbass = $SliMP3::Control::minBass; }
					SliMP3::Prefs::clientSet($client, "bass", $newbass);
					SliMP3::Control::bass($client, $newbass);
					if (SliMP3::Playlist::isSynced($client)) {syncFunction($client, $newbass, "bass",\&SliMP3::Control::bass);};
				}
			}
		} elsif ($p0 eq "display") {
			if ($p1 eq "?" && $p2 eq "?") {
				my ($line1, $line2) = SliMP3::Display::curLines($client);
				$p1 = $line1;
				$p2 = $line2;
			} else {
				SliMP3::Buttons::ScreenSaver::wakeup($client);
				SliMP3::Animation::showBriefly($client, $p1, $p2, $p3, $p4);
			}
		} elsif ($p0 eq "button") {
			# all buttons now go through execute()
			SliMP3::IR::executeButton($client, $p1, $p2);
		} elsif ($p0 eq "ir") {
			# all ir signals go through execute()
			SliMP3::IR::processIR($client, $p1, $p2);
		}
	# mark html status page invalid so it will update
	$client->htmlstatusvalid(0);
	}	
	
	my @returnArray = ();
	
	if (defined($p0)) { push @returnArray, $p0 };
	if (defined($p1)) { push @returnArray, $p1 };
	if (defined($p2)) { push @returnArray, $p2 };
	if (defined($p3)) { push @returnArray, $p3 };
	if (defined($p4)) { push @returnArray, $p4 };
	if (defined($p5)) { push @returnArray, $p5 };
	if (defined($p6)) { push @returnArray, $p6 };
	
	$callcallback && $callbackf && (&$callbackf(@$callbackargs, \@returnArray));

	executeCallback($client, \@returnArray);

	$::d_command && msg(" Returning array: $p0 (" .
			(defined $p1 ? $p1 : "") . ") (" .
			(defined $p2 ? $p2 : "") . ") (" .
			(defined $p3 ? $p3 : "") . ") (" .
			(defined $p4 ? $p4 : "") . ") (" .
			(defined $p5 ? $p5 : "") . ") (" .
			(defined $p6 ? $p6 : "") . ")\n");
	
	return @returnArray;
}

sub syncFunction {
	my $client = shift;
	my $newval = shift;
	my $setting = shift;
	my $controlRef = shift;
	
	my @buddies = SliMP3::Playlist::syncedWith($client);
	if (scalar(@buddies) > 0) {
		foreach my $eachclient (@buddies) {
			if (SliMP3::Prefs::clientGet($eachclient,'syncVolume')) {
				if ($setting eq "mute") {
					SliMP3::Control::fade_volume($eachclient, $newval, \&SliMP3::Control::mute, [$eachclient]);
				} else {
					SliMP3::Prefs::clientSet($eachclient, $setting, $newval);
					&$controlRef($eachclient, $newval);
				}
				if ($setting eq "volume") {
					SliMP3::Display::volumeDisplay($eachclient);
				}
			}
		}
	}
}

sub setExecuteCallback {
	my $callbackRef = shift;
	$executeCallbacks{$callbackRef} = $callbackRef;
}

sub clearExecuteCallback {
	my $callbackRef = shift;
	delete $executeCallbacks{$callbackRef};
}

sub executeCallback {
	my $client = shift;
	my $paramsRef = shift;

	no strict 'refs';
		
	foreach my $executecallback (keys %executeCallbacks) {
		$executecallback = $executeCallbacks{$executecallback};
		&$executecallback($client, $paramsRef);
	}
}

sub load_done {
	my ($client, $index, $callbackf, $callbackargs)=@_;
	SliMP3::Playlist::reshuffle($client);
	if (defined($index)) {
		SliMP3::Playlist::jumpto($client, $index);
	}
	$callbackf && (&$callbackf(@$callbackargs));
	SliMP3::Command::executeCallback($client, ['playlist','load_done']);
}

sub insert_done {
	my ($client, $listsize, $size,$callbackf, $callbackargs)=@_;
	my $i;
	my $playlistIndex = SliMP3::Playlist::currentSongIndex($client)+1;
	my @reshuffled;

	if (SliMP3::Playlist::shuffle($client)) {
		for ($i = 0; $i < $size; $i++) {
			push @reshuffled,$listsize+$i;
		};
		if (SliMP3::Playlist::isSlave($client)) {
			$client = $client->master;
		}
		splice @{$client->shufflelist},$playlistIndex, 0, @reshuffled;
	} else {
		SliMP3::Playlist::moveSong($client, $listsize, $playlistIndex,$size);
		SliMP3::Playlist::reshuffle($client);
	};
	SliMP3::Playlist::refreshPlaylist($client);
	$callbackf && (&$callbackf(@$callbackargs));
	SliMP3::Command::executeCallback($client, ['playlist','load_done']);
}

sub singletonRef {
    my $arg = shift;

	if (!defined($arg)) {
		return [];
    } elsif ($arg eq '*') {
        return [];
    } elsif ($arg) {
        return [$arg];
    } else {
        return [];
    }
}

sub gotosleep {
	my $client = shift;
	if (SliMP3::Client::isSliMP3($client)) {
		SliMP3::Control::fade_volume($client,-60,\&turnitoff,[$client]);
	}
	$client->sleepTime(0);
	$client->currentSleepTime(0);
}


sub turnitoff {
	my $client = shift;
	
	# Turn off quietly
	execute($client, ['stop', 0]);
	execute($client, ['power', 0]);
}
  
1;
__END__

# Local Variables:
# tab-width:4
# indent-tabs-mode:t
# End:

