/* $Id: arp.c,v 1.5 2003/05/30 00:42:39 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "libc.h"
#include "arp.h"
#include "global.h"
//#include "simether.h"
#include "eth.h"
#include "misc.h"

/* 
 * Routines for resolving IP-addresses to ethernet addresses via the
 * Address Resolution Protocol (ARP)
 */

#define ARP_HARD_TYPE_ETH         1
#define ARP_PROT_TYPE_IP      0x800
#define ARP_HARD_SIZE_ETH         6
#define ARP_PROT_SIZE_IP          4

#define ARP_OP_REQUEST            1
#define ARP_OP_REPLY              2

#define ARP_CACHE_SIZE            4


/* 
 * Data Structures
 */
typedef struct {
	uint8_t hard_type[2];
	uint8_t prot_type[2];
	uint8_t hard_size;
	uint8_t prot_size;
	uint8_t op[2];
	uint8_t send_eth_addr[6];
	uint8_t send_ip_addr[4];
	uint8_t targ_eth_addr[6];
	uint8_t targ_ip_addr[4];
	uint8_t pad[18];
} arp_packet_t;

typedef struct arp_response_node_struct {
	uint32_t ip;
	uint64_t hw_addr;
	struct arp_response_node_struct *next;
} arp_response_node_t;

typedef struct {
	uint64_t hw_addr;
	uint32_t ip;
	int valid;
} arp_cache_entry_t;

typedef struct {
	arp_cache_entry_t entries[ARP_CACHE_SIZE];
	int next_to_replace;
} arp_cache_t;

typedef struct {
	uint32_t ip;
	uint64_t hw_addr;
	unsigned int flag;
} arp_req_handler_data_t;



/*
 * Static data
 */

static arp_cache_t arp_cache;
static arp_response_node_t *response_chain = 0;


/*
 * Static functions
 */


/* 
 *  Handler for when we're doing an arp request and expecting
 *  a response from someone else.  
 */
static int arp_req_handler(mbuf_t *buf, void *ptr, uint16_t ignored) 
{
	arp_req_handler_data_t *data;
	arp_packet_t *arp_req;
	int retval = 0;
	if (buf->size >= sizeof(arp_packet_t)) {
		arp_req = (arp_packet_t *)buf->front_ptr;
		data = (arp_req_handler_data_t *)ptr;
		if ((get_uint32_field(arp_req->send_ip_addr) == data->ip)
		    && (get_uint16_field(arp_req->op) == ARP_OP_REPLY)
		    && (get_uint16_field(arp_req->hard_type) == ARP_HARD_TYPE_ETH)
		    && (get_uint16_field(arp_req->prot_type) == ARP_PROT_TYPE_IP)) {
			data->hw_addr = get_uint48_field(arp_req->send_eth_addr);
			data->flag = 1;
			retval = 1;
		}
	}
	return retval;
}


/*
 * Handler for ethernet addresses we own, for responding to arp requests
 * from elsewhere in the system
 */
static int arp_resp_handler(mbuf_t *buf, void *ptr, uint16_t frame_type)
{
	arp_packet_t *arp_req;
	arp_response_node_t *tmp;
	uint32_t ip;
	int retval = 0;
	if (buf->size > sizeof(arp_packet_t)) {
		arp_req = (arp_packet_t *)buf->front_ptr;
		ip = get_uint32_field(arp_req->send_ip_addr);
		if ((get_uint16_field(arp_req->op) == ARP_OP_REQUEST)
		    && (get_uint16_field(arp_req->hard_type) == ARP_HARD_TYPE_ETH)
		    && (get_uint16_field(arp_req->prot_type) == ARP_PROT_TYPE_IP)) {
			/* See if we've got a handler for this */
			for (tmp = response_chain; tmp; tmp = tmp->next) {
				if (tmp->ip == ip) {
					arp_packet_t *resp;
					mbuf_t *mbuf = lib_malloc(sizeof(mbuf_t));
					if (!mbuf) {
						lib_die("Out of memory");
					}
					mbuf_init(mbuf, 0);
					mbuf_append_space(mbuf, sizeof(arp_packet_t));
					resp = (arp_packet_t *)(mbuf->front_ptr);
					set_uint16_field(resp->hard_type, ARP_HARD_TYPE_ETH);
					set_uint16_field(resp->prot_type, ARP_PROT_TYPE_IP);
					resp->hard_size = ARP_HARD_SIZE_ETH;
					resp->prot_size = ARP_PROT_SIZE_IP;
					set_uint16_field(resp->op, ARP_OP_REPLY);
					set_uint48_field(resp->send_eth_addr, net_cfg.mac_addr);
					set_uint48_field(resp->send_ip_addr, ip);
					lib_memcpy(resp->targ_eth_addr, arp_req->send_eth_addr, 6);
					lib_memcpy(resp->targ_ip_addr, arp_req->send_ip_addr, 4);
					eth_send_packet(mbuf, get_uint48_field(arp_req->send_eth_addr), net_cfg.mac_addr, ETH_FRAME_TYPE_ARP);
					lib_free(mbuf);
				}
			}
			retval = 1;
		}
	}
	return retval;
}



/* 
 * Exported interface.  These are described in the header file
 */

int arp_resolve(uint32_t ip, uint64_t *resolved_mac_addr)
{
	int tries = 4;
	arp_packet_t *req;
	int retval = -1;
	int i;
	arp_req_handler_data_t handler_data;
	mbuf_t *mbuf;
	unsigned int timeout = 500000;

	for (i = 0; i < ARP_CACHE_SIZE; i++) {
		if (arp_cache.entries[i].valid 
		    && arp_cache.entries[i].ip == ip) {
			*resolved_mac_addr = arp_cache.entries[i].hw_addr;
#if 0
			lib_printf("(Cached) resolved ip ");
			lib_print_ip(ip);
			lib_printf(" to mac address ");
			lib_print_mac_addr(arp_cache.entries[i].hw_addr);
			lib_printf("\n");
#endif
			retval = 0;
			goto out;
		}
	}
	
	mbuf = lib_malloc(sizeof(mbuf_t));
	if (!mbuf) {
		lib_die("Out of memory");
	}

	eth_add_handler(arp_req_handler, &handler_data, ETH_FRAME_TYPE_ARP);
	while(tries--) {
		mbuf_init(mbuf, 0);
		mbuf_append_space(mbuf, sizeof(arp_packet_t));
		req = (arp_packet_t *)(mbuf->front_ptr);
		set_uint16_field(req->hard_type, ARP_HARD_TYPE_ETH);
		set_uint16_field(req->prot_type, ARP_PROT_TYPE_IP);
		req->hard_size = ARP_HARD_SIZE_ETH;
		req->prot_size = ARP_PROT_SIZE_IP;
		set_uint16_field(req->op, ARP_OP_REQUEST);
		set_uint48_field(req->send_eth_addr, net_cfg.mac_addr);
		set_uint32_field(req->send_ip_addr, net_cfg.ip_addr);
		set_uint48_field(req->targ_eth_addr, 0);
		set_uint32_field(req->targ_ip_addr, ip);
		lib_bzero(&(req->pad), sizeof(req->pad));
		eth_send_packet(mbuf, ETH_ADDR_BROADCAST, net_cfg.mac_addr, ETH_FRAME_TYPE_ARP);

		handler_data.ip = ip;
		handler_data.flag = 0;
		eth_recv_loop(timeout, &handler_data.flag);
		if (handler_data.flag) {
		       	/* This is our resolution.  Rejoice.  Yay.  */
			(*resolved_mac_addr) = handler_data.hw_addr;
#if 0
			lib_printf("Resolved ip ");
			lib_print_ip(handler_data.ip);
			lib_printf(" to mac address ");
			lib_print_mac_addr(handler_data.hw_addr);
			lib_printf("\n");
#endif
			arp_cache.entries[arp_cache.next_to_replace].valid = 1;
			arp_cache.entries[arp_cache.next_to_replace].ip = ip;
			arp_cache.entries[arp_cache.next_to_replace].hw_addr = handler_data.hw_addr;
			arp_cache.next_to_replace = (arp_cache.next_to_replace + 1) % ARP_CACHE_SIZE;
			tries = 0;
			retval = 0;
			
		}
		timeout <<= 1;
	}
	eth_del_handler(arp_req_handler, &handler_data, ETH_FRAME_TYPE_ARP);
	lib_free(mbuf);
 out:
	return retval;
}

/* 
 * Set up a handler to respond to arp requests to the passed IP with
 * our mac address.  
 */
void arp_add_response(uint32_t ip, uint64_t hw_addr)
{
	arp_response_node_t *new_node = lib_malloc(sizeof(arp_response_node_t));
	if (!new_node) {
		lib_die("Out of memory");
	}
	new_node->next = response_chain;
	new_node->ip = ip;
	new_node->hw_addr = hw_addr;
	if (!response_chain) {
		eth_add_handler(arp_resp_handler, 0, ETH_FRAME_TYPE_ARP);
	}
	response_chain = new_node;
}

void arp_del_response(uint32_t ip, uint64_t hw_addr)
{
	arp_response_node_t *node, *prev = 0;
	for (node = response_chain; node; node = node->next) {
		if ((node->ip == ip)
		    && (node->hw_addr == hw_addr)) {
			if (prev) {
				prev->next = node->next;
			} else {
				response_chain = node->next;
			}
			lib_free(node);
			break;
		}
	}
	if (!response_chain) {
		eth_del_handler(arp_resp_handler, 0, ETH_FRAME_TYPE_ARP);
	}
}
