// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "text.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "rotstr.h"

#include <Xutil.h>

Text::Text(const Text& a) : Marker(a) {}

Text::Text(FrameBase* p, const Vector& ctr,
	   double ang,
	   const char* clr, int wth, const char* fnt, const char* txt, 
	   unsigned short prop, const char* cmt,
	   const List<Tag>& tg, const List<CallBack>& cb) 
  : Marker(p, ctr, ang, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type,"text");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Text::updateBBox()
{
  Vector cc = center * parent->refToCanvas;
  bbox = BBox(cc);

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    // generate handles
    Vector s(width, metrics.linespace);
    Matrix mm = Rotate(calcAngle()) * Translate(cc);

    handle[0] = Vector(-s[0],-s[1])/2 * mm;
    handle[1] = Vector( s[0],-s[1])/2 * mm;
    handle[2] = Vector( s[0], s[1])/2 * mm;
    handle[3] = Vector(-s[0], s[1])/2 * mm;

    // bbox
    for (int i=0; i<4; i++)
      bbox.bound(handle[i]);
  }
  else
    bbox.expand(5);

  // make room for handles
  bbox.expand(3);
  allBBox = bbox;
}

void Text::draw(Drawable drawable)
{
  renderHandles(drawable);
  render(drawable, parent->refToWidget, SRC);
}

void Text::drawXOR()
{
  render(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);
}

void Text::drawMagnifier(const Matrix& mx)
{
  render(parent->magnifierPixmap, mx, SRC);
}

void Text::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }
  setLineNoDash();

  if (font)
    XSetFont(display, gc, Tk_FontId(font));

  double ang = calcAngle();
  if (text && font)
    if (ang>-.01 && ang<+.01) {
      Tk_FontMetrics metrics;
      Tk_GetFontMetrics(font, &metrics);

      Vector t = (center * mx) * 
	Translate(-Tk_TextWidth(font, text, strlen(text))/2., 
		  (metrics.ascent-metrics.descent)/2.);

      XDrawString(display, drawable, gc, (int)t[0], (int)t[1], 
		  text, strlen(text));
    }
    else {
      XDrawRotString(display, drawable, gc, 
		     parent->baseXImage->byte_order, 
		     parent->getBlackColor(), parent->getWhiteColor(),
		     text, font, ang, center*mx);
    }
}

void Text::ps(int mode)
{
  if (text && font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    double ang = calcAngle();
    Vector cc = center * parent->refToCanvas;
    Matrix m = Translate(-cc) * Rotate(-ang) * 
      Translate(-width/2.,metrics.descent) *
      Rotate(ang) * Translate(cc);

    ostringstream str;
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
      
    Tcl_DStringFree(&psFont);

    psLineNoDash();
    psColor(mode, colorName);

    str << "gsave "
        << (cc*m).TkCanvasPs(parent->canvas) << " moveto" << endl
      	<< radToDeg(ang) << " rotate " 
	<< '(' << psQuote(text) << ')' 
	<< " show grestore" << endl << ends;
    
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

// list

void Text::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v = ptr->mapFromRef(center,sys,sky);
	      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] 
		  << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      ptr->mapFromRef(center,sys,sky,format,buf,64);
	      char ra[16];
	      char dec[16];

	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra >> dec;
	      str << type << '(' << ra << ',' << dec << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v = ptr->mapFromRef(center,sys);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';
	}
      }
    }

    listPost(str, sys, sky, conj);
  }
}

void Text::listPost(ostream& str, CoordSystem sys, SkyFrame sky, int conj)
{
  if (conj)
    str << " ||";

  if (angle != 0)
    str << " textangle=" << radToDeg(parent->mapAngleFromRef(angle,sys,sky));
  listProperties(str, 0);
}

void Text::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  if (properties&INCLUDE)
    str << '+';
  else
    str << '-';

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] 
	  << ", \"" << text << "\")";
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<", \""
	      << text << "\")";
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec
	      << ", \"" << text << "\")";
	}
	break;
      }
    }
  }

  listSAOtngPost(str, strip);
}




