// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <limits.h>

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "colorbarrgbtrue.h"
#include "util.h"
#include "ps.h"

ColorbarRGBTrueColor::ColorbarRGBTrueColor(Tcl_Interp* i, Tk_Canvas c,
					   Tk_Item* item) 
  : ColorbarRGB(i, c, item)
{
  colorCount = 0;
  xmap = NULL;
}

ColorbarRGBTrueColor::~ColorbarRGBTrueColor()
{
  if (xmap)
    XDestroyImage(xmap);
}

void ColorbarRGBTrueColor::invalidPixmap()
{
  Widget::invalidPixmap();

  if (xmap)
    XDestroyImage(xmap);
  xmap = NULL;
}

int ColorbarRGBTrueColor::updatePixmap(const BBox& bb)
{
  // create a valid pixmap if needed
  // bb is in canvas coords

  if (!pixmap || !xmap) {
    int& width = options->width;
    int& height = options->height;
    
    // pixmap

    if (!pixmap)
      if (!(pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
				  width, height, depth))) {
	cerr << "ColorbarRGBTrueColor Internal Error: Unable to Create Pixmap" 
	     << endl;
	exit(1);
      }

    // xmap

    if (!xmap) {
      if (!(xmap = XGETIMAGE(display, pixmap, 0, 0, width, height, 
			     AllPlanes, ZPixmap))){
	cerr << "ColorbarRGBTrueColor Internal Error: Unable to Create XImage" 
	     << endl;
	exit(1);
      }

      updateColors();
    }
  }

  return TCL_OK;
}

int ColorbarRGBTrueColor::initColormapForReal()
{
  colorCells = new unsigned char[colorCount*3];

 // needed to initialize colorCells
  reset();

  return TCL_OK;
}

void ColorbarRGBTrueColor::updateColorCells()
{
  // fill rgb table
  // note: its filled bgr to match XImage

  for(int i=0; i<colorCount; i++) {
    int idr = calcContrastBias(i,bias[0],contrast[0]);
    int idg = calcContrastBias(i,bias[1],contrast[1]);
    int idb = calcContrastBias(i,bias[2],contrast[2]);

    colorCells[i*3]   = (int)(256.*idr/colorCount);
    colorCells[i*3+1] = (int)(256.*idg/colorCount);
    colorCells[i*3+2] = (int)(256.*idb/colorCount);
  }
}

void ColorbarRGBTrueColor::psLevel1(PSColorSpace mode, int width, int height, 
				    float scale)
{
  if (!colorCells)
    return;

  psLevel1Head(mode, width, height);
  NoCompressAsciiHex filter;

  // red
  for (int j=0; j<(int)(height/3.); j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned char green = 0;
      unsigned char blue = 0;

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  // green
  for (int j=(int)(height/3.); j<(int)(height*2/3.); j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = 0;
      unsigned char green = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned char blue = 0;

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  // blue
  for (int j=(int)(height*2/3.); j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = 0;
      unsigned char green = 0;
      unsigned char blue = colorCells[(int)(double(i)/width*colorCount)*3];

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void ColorbarRGBTrueColor::psLevel2(PSColorSpace mode, int width, int height, 
				    float scale)
{
  if (!colorCells)
    return;

  psLevel2Head(mode, width, height);
  RLEAscii85 filter;

  // red
  for (int j=0; j<(int)(height/3.); j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = colorCells[(int)(double(i)/width*colorCount)*3+2];
      unsigned char green = 0;
      unsigned char blue = 0;

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  // green
  for (int j=(int)(height/3.); j<(int)(height*2/3.); j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = 0;
      unsigned char green = colorCells[(int)(double(i)/width*colorCount)*3+1];
      unsigned char blue = 0;

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  // blue
  for (int j=(int)(height*2/3.); j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      unsigned char red = 0;
      unsigned char green = 0;
      unsigned char blue = colorCells[(int)(double(i)/width*colorCount)*3];

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }

    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

