# Samizdat resource representation
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

class Resource

  # _id_ should be translatable to an Integer above zero
  #
  # returns nil if _id_ is invalid
  #
  def Resource.validate_id(id)
    return 0 if 'upload' == id   # hack for PublishMessage#new
    return nil if
      id.to_i.to_s != id.to_s or
      not id.to_i > 0
    id.to_i
  end

  # validate _id_, generate _uriref_, gets references for _session_ and
  # _template_
  #
  def initialize(session, id)
    @session = session
    @template = session.template
    @id = Resource.validate_id(id)
    raise ResourceNotFoundError if @id.nil?
    @uriref = session.base + @id.to_s
  end

  attr_reader :session, :template, :id, :uriref

  # in :list mode (default) return rendered desc
  #
  # in :title mode return resource title (HTML-escaped)
  #
  # in :short mode return hash with :type, :head, and :desc keys
  #
  # in :full mode, add :messages and :focuses to the hash
  # (only one full resource per page is possible)
  #
  def render(mode=:list)
    cache_key = %{#{@session.accept_language.join(':')}/#{id}/#{mode}/#{@session['skip'].to_i}/#{@session.showhidden?}}
    if :full != mode and r = cache[cache_key]
      # :full rendering can include session-specific data
      return r
    end
    r = {}

    external, literal, label =
      db.select_one("SELECT uriref, literal, label FROM Resource WHERE literal = 'false' AND id = ?", @id)
    if external
      r[:type] = 'Uriref'
      head = Samizdat::SquishQuery.ns_shrink(label)
      head.gsub!(/\Afocus::/, '') and head = _(head)   # focus is special
      r[:head] = head
      r[:desc] = '<div>' + sprintf(_('refers to <a href="%s">external uriref</a>'), label) + '</div>'
      # todo: select all statements with this subject
    elsif literal
      r[:type] = 'Literal'
      r[:head] = label
      r[:desc] = %{<a href="#{@uriref}">#{Samizdat::SquishQuery.ns_shrink(@uriref)}</a> = #{label}}
    else   # internal resource
      r =
        case label
        when nil then raise ResourceNotFoundError, id.to_s
        when 'Member' then render_member mode
        when 'Message' then render_message mode
        when 'Statement' then render_statement mode
        when 'Vote' then render_vote mode
        when 'Item' then render_item mode
        else raise RuntimeError,
          sprintf(_("Unknown resource type '%s'"), CGI.escapeHTML(label))
        end
      r[:type] = label
    end

    r[:head] = CGI.escapeHTML(Template.limit_string(r[:head]))
    :title == mode and return cache[cache_key] = r[:head]
    :list == mode and return cache[cache_key] = template.resource(id, r[:head], r[:desc])

    r[:focuses] = render_focuses if :full == mode
    class << r
      def to_s
        self.values_at(:desc, :focuses, :messages, :buttons).join
      end
    end
    # don't cache this: dRuby doesn't work with singleton classes
    r
  end

  # render the most recent related resource in :short mode and a list of less
  # recent related resources in :title mode
  #
  def render_related
    resources = rdf.select_all( %{
SELECT ?resource
WHERE (rdf::subject ?stmt ?resource)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{id})
      (s::inReplyTo ?resource ?parent)
      (dct::isVersionOf ?resource ?current)
      (s::rating ?stmt ?rating)
      (dc::date ?resource ?date)
LITERAL ?rating > 0 AND ?parent IS NULL AND ?current IS NULL
ORDER BY ?date DESC}, limit_page)
    related = Resource.new(@session, resources.shift[0]
      ).render(:short)[:desc] if resources.size > 0
    related += _('Also related to this focus: ') +
      resources.collect {|resource,|
        template.resource_href(resource,
          Resource.new(@session, resource).render(:title))
      }.join(', ') if resources.size > 0
    related = template.box(_('Recent updates in focus: ') +
      template.resource_href(id, render(:title)), related)
    related
  end

private

  def render_focuses
    focuses = {}
    rdf.select_all( %{
SELECT ?focus
WHERE (rdf::subject ?stmt #{id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt ?focus)
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0
ORDER BY ?rating DESC}, limit_page).each {|focus,|
      f = Focus.new(@session, focus, self)
      focuses[f.id] = f
    }
    if @session.access('vote')   # always show selected focus and translation
      if not rdf.get_property(@id, 's::inReplyTo').nil?
        # publish translations as replies (todo: dct:isPartOf)
        translation = Focus.new(@session, 'focus::Translation', self)
        focuses.update({translation.id => translation})
      end
    end
    template.focus_box(self, focuses.values)
  end

  def render_member(mode)
    r = {}
    login, r[:head] = rdf.select_one %{
SELECT ?login, ?full_name
WHERE (s::login #{id} ?login)
      (s::fullName #{id} ?full_name)}
    r[:desc] = _('Login') + ": #{login}"
    if :full == mode
      skip, = @session.params %w[skip]
      skip = skip.to_i
      messages = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::creator ?msg #{id})
      (dct::isVersionOf ?msg ?current)
LITERAL ?current IS NULL
ORDER BY ?date DESC}, limit_page, limit_page * skip
      ).collect {|msg,| Resource.new(@session, msg).render }
      r[:desc] = '<p>' + r[:desc] + '</p>'
      # check if account is blocked
      passwd, prefs = db.select_one(
        'SELECT passwd, prefs FROM Member WHERE id = ?', id)
      if passwd.nil?   # account is blocked
        blocked_by = yaml_hash(prefs)['blocked_by'].to_i
        if blocked_by > 0
          b_name, = db.select_one(
            'SELECT full_name FROM Member WHERE id = ?', blocked_by)
          r[:desc] += '<p>' + sprintf(
            _('Account blocked by moderator: %s.'),
            template.resource_href(blocked_by,
              CGI.escapeHTML(b_name))) + '</p>'
        end
      end
      if @session.moderator and not config['access']['moderators'].include? login
        # show block/unblock button
        r[:desc] += template.form('member.rb', 
          passwd.nil? ?
            [:submit_moderator, 'unblock', _('UNBLOCK')] :
            [:submit_moderator, 'block', _('BLOCK')],
          [:hidden, 'id', id])
      end
      r[:desc] = template.box(nil, r[:desc])
      r[:messages] = template.box( _('Latest Messages') +
        (skip > 0? sprintf(_(', page %s'), skip + 1) : ''),
        template.list( messages, template.nav(messages.size,
          skip + 1, %{resource.rb?id=#{id}&amp;}) )
      ) if messages.size > 0
    end
    r
  end

  def render_message(mode)
    message = Message.cached(@id, @session.showhidden?)

    # use translation to preferred language if available
    translation = (:full == mode)? message :
      message.select_translation(@session.accept_language)
    head = translation.title

    return { :head => head } if :title == mode

    head = Focus.focus_title(head) if :list == mode and message.nrelated > 0

    return { :head => head, :desc => template.message_info(message, mode) } if
      :list == mode

    # rely on Template#message to take care of translation in :short mode
    r = { :head => head, :desc => template.message(message, mode) }

    return r if :short == mode or @session.has_key?('skip_related')

    if message.nreplies + message.translations.size > 0   # add replies
      skip, = @session.params %w[skip]
      skip = skip.to_i
      hidden = " AND ?hidden = 'false'" unless @session.showhidden?
      replies = rdf.select_all( %{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (s::inReplyTo ?msg #{id})
      (s::hidden ?msg ?hidden)
LITERAL ?msg IS NOT NULL #{hidden}
ORDER BY ?date}, limit_page, limit_page * skip
      ).collect {|reply,| Resource.new(@session, reply).render(:short)[:desc] }
      r[:messages] = template.box(
        _('Replies') + (skip > 0? sprintf(_(', page %s'), skip + 1) : ''),
        template.list( replies, template.nav(replies.size, skip + 1,
          %{resource.rb?id=#{id}&amp;}) ),
        'replies'
      ) if replies.size > 0
    end

    r[:buttons] = template.message_buttons(message) if :full == mode

    # navigation links
    template.link['made'] = message.creator
    template.link['up'] = message.parent if message.parent
    r
  end

  def render_statement(mode)
    r = { :head => _('Statement') + ' ' + id.to_s }
    return r if :title == mode
    stmt = rdf.select_one %{
SELECT ?p, ?s, ?o
WHERE (rdf::predicate #{id} ?p)
      (rdf::subject #{id} ?s)
      (rdf::object #{id} ?o)}
    n = [_('Predicate'), _('Subject'), _('Object')]
    if :list == mode
      r[:desc] = '(' + stmt.collect {|id|
        %{<a href="#{id}">#{n.shift} #{id}</a>}
      }.join(', ') + ')'
    else
      r[:desc] = stmt.collect {|id|
        template.box(n.shift, Resource.new(@session, id).render)
      }.join
    end
    # todo: display votes
    r
  end

  def render_vote(mode)
    r = { :head => _('Vote') + ' ' + id.to_s }
    return r if :title == mode
    date, stmt, member, name, rating = rdf.select_one %{
SELECT ?date, ?stmt, ?member, ?name, ?rating
WHERE (dc::date #{id} ?date)
      (s::voteProposition #{id} ?stmt)
      (s::voteMember #{id} ?member)
      (s::fullName ?member ?name)
      (s::voteRating #{id} ?rating)}
    r[:desc] = sprintf(_('<a href="%s">%s</a> gave rating %4.2f to the <a href="%s">Statement %s</a> on %s.'),
      member, name, rating, stmt, stmt, template.format_date(date).to_s)
    if :full == mode
      r[:desc] = template.box(nil, r[:desc]) +
        template.box(_('Vote Proposition'),
          Resource.new(@session, stmt).render(:short)[:desc])
      template.link['made'] = member
    end
    r
  end

  def render_item(mode)
    # todo: short mode
    msg, date, creator, name, head, format, content, parent,
      contributor, c_name, possessor, p_name = rdf.select_one %{
SELECT ?msg, ?date, ?creator, ?name, ?title, ?content, ?parent,
       ?contributor, ?c_name, ?possessor, ?p_name
WHERE (s::description #{id} ?msg)
      (dc::date ?msg ?date)
      (dc::creator ?msg ?creator)
      (s::fullName ?creator ?name)
      (dc::title ?msg ?title)
      (dc::format ?msg ?format)
      (s::content ?msg ?content)
      (s::inReplyTo ?msg ?parent)
      (s::contributor #{id} ?contributor)
      (s::fullName ?contributor ?c_name)
      (s::possessor #{id} ?possessor)
      (s::fullName ?possessor ?p_name)}
    desc =
%'<p><b>Contributor:</b> <a href="#{contributor}">#{c_name}</a></p>
<p><b>Possessor:</b> <a href="#{possessor}">#{p_name}</a></p>
<p><b>Description:</b></p>' +
      template.message(msg, date, creator, name, nil, \
        head, format, content, parent)
    head += ' / ' + possessor
    if :full == mode
      query = %{
SELECT ?item, ?name
WHERE (s::description #{id} ?msg)
      (s::description ?item ?msg)
      (s::possessor ?item ?possessor)
      (s::fullName ?possessor ?name)}
      desc += template.box(_('Similar Items'),
        _('<p><b>Possessed by:</b></p>') +
        db.execute(*rdf.select(query)) {|sth|
          sth.fetch_many(limit_page).collect {|item,name|
            %'<a href="#{item}">#{name}</a><br>'
          }.join })
    end
    return head, desc
  end
end
