/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>

#include <bluetooth/bluetooth.h>
#include <openobex/obex.h>

#include "btobexmain.h"
#include "btobexclient.h"


typedef struct {
	
	int opdone;
	int timeout;
	int lastrsp;
	
	// connect_done etc.
} obexconfig_t;


/*
 * Reads specified number of bytes from a file.
 * The returned buffer must be freed.
 */
static uint8_t *btobexclient_readfile(FILE *file, uint32_t len)
{
	int actual;
	uint8_t *filebuf = NULL;
	
	if ( !(filebuf = malloc(len)) ) {
		DEBUG("[btobexclient_readfile] Error creating file buffer!\n");
		return NULL;
	}	
	
        actual = fread(filebuf, 1, len, file);	
        if (actual != len) {
		DEBUG("[btobexclient_readfile] Error reading file!\n");
		return NULL;
        }
        return filebuf;
}

/* 
 * Make a request to the server and wait for a response.
 */
static int btobexclient_request(obex_t *handle, obex_object_t *object) 
{
	obexconfig_t *config;
	int result;
	
	/* send the request */
	result = OBEX_Request(handle, object);
	if (result < 0) {
		DEBUG("[btobexclient_request] Error sending request!\n");
		return result;
	}		
	
	/* synchronously wait for a server response */
	
	config = OBEX_GetUserData(handle);
	config->lastrsp = OBEX_RSP_SUCCESS;
	config->opdone = FALSE;
	
	while (!config->opdone) {
		result = OBEX_HandleInput(handle, config->timeout);
		if (result < 0) {
			DEBUG("[btobexclient_request] OBEX_HandleInput() error!\n");
			return result;
		}
		if (result == 0) {
			DEBUG("[btobexclient_request] Operation timed out!\n");
			OBEX_CancelRequest(handle, FALSE);
			return -1;		/* should return some other timeout error code? */
		}
	}
	return (config->lastrsp == OBEX_RSP_SUCCESS)? 0 : -1;
}

/* 
 * Called when a request is complete.
 */
static void btobexclient_done(obex_t *handle, obex_object_t *object, int cmd, int rsp)
{
	obexconfig_t *config;
	
	config = OBEX_GetUserData(handle);
	if (rsp == OBEX_RSP_SUCCESS) {	
		DEBUG("[btobexclient_done] Request 0x%02x successful\n", cmd);
	} else {
		DEBUG("[btobexclient_done] Request 0x%02x failed! (response = '%s')\n", cmd, OBEX_ResponseToString(rsp));	
	}
	
	config->lastrsp = rsp;
	config->opdone = TRUE;
}

/*
 * Called when an obex event occurs.
 */
void btobexclient_event(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
    switch (event)  {
		case OBEX_EV_PROGRESS:
			break;
		case OBEX_EV_REQDONE:
			btobexclient_done(handle, object, obex_cmd, obex_rsp);
			break;
		case OBEX_EV_LINKERR:
			DEBUG("[btobexclient_event] Transport link broken, terminating connection!\n");
			OBEX_TransportDisconnect(handle);
			break;
		default:
			DEBUG("[btobexclient_event] Unknown event (%d)\n", event);
			break;
    }
}

/* 
 * Creates and sets up an obex client.
 */
obex_t *btobexclient_init(unsigned int flags)
{
	obex_t *handle;
	obexconfig_t *config;
	
	handle = OBEX_Init(OBEX_TRANS_BLUETOOTH, btobexclient_event, flags);
	if (!handle) return NULL;
	
	
	// set server config data
	config = malloc(sizeof(obexconfig_t));
	if (!config) {
		DEBUG("[btobexclient_init] Error allocating internal client config data\n");
		return NULL;
	}	
	
	config->timeout = 30;		/* default time out */
	OBEX_SetUserData(handle, config);
	return handle;
}

void btobexclient_cleanup(obex_t *handle)
{
	obexconfig_t *config;
	
	DEBUG("[btobexserver_cleanup] entry.\n");
	
	config = OBEX_GetUserData(handle);
	if (config) free(config);
	OBEX_SetUserData(handle, NULL);	
}

/* 
 * Connects a client to the server on the given address and channel.
 */
int btobexclient_connect(obex_t *handle, bdaddr_t *bdaddr, uint8_t channel)
{
	obex_object_t *object;
	int result;
	
	DEBUG("[btobexclient_connect] Connecting transport...\n");	
	
	result = BtOBEX_TransportConnect(handle, BDADDR_ANY, bdaddr, channel);
	if (result < 0) {
		DEBUG("[btobexclient_connect] Error opening Bluetooth transport connection!\n");
	return result;
	}	
	
	DEBUG("[btobexclient_connect] Connecting OBEX session...\n");
	
	if( !(object = OBEX_ObjectNew(handle, OBEX_CMD_CONNECT)) ) {
		DEBUG("[btobexclient_connect] Error creating connection request!\n");
		return -1;
	}
	
	//return btobexclient_request(handle, object);
	result = btobexclient_request(handle, object);
	return result;
}

/* 
 * Disconnects the obex session.
 */
int btobexclient_disconnect(obex_t *handle) 
{
	obex_object_t *object;
	
	DEBUG("[btobexclient_disconnect] Disconnecting...\n");
	
	if( !(object = OBEX_ObjectNew(handle, OBEX_CMD_DISCONNECT)) ) {
		DEBUG("[btobexclient_disconnect] Error creating disconnection request!\n");
		return -1;
	}
	return btobexclient_request(handle, object);
}

/* 
 * Performs a PUT to send a file to the server.
 */
int btobexclient_put(obex_t *handle, const char *name, FILE *file, uint32_t filesize)
{
	int result;
		
	obex_headerdata_t header;
	obex_object_t *object = NULL;
	
	int uname_size = -1;
	char *uname = NULL;
	uint8_t *body = NULL;

	DEBUG("[btobexclient_put] Sending file '%s' (%d bytes)...\n", name, filesize);	
	
	/* encode name to unicode */
	uname_size = (strlen(name)+1)*2;
	if ( !(uname = malloc(uname_size)) ) {
		DEBUG("[btobexclient_put] Error creating name buffer!\n");
		goto out_err; 
	}	
	result = OBEX_CharToUnicode((uint8_t*)uname, (uint8_t*)name, uname_size);	
	if (result < 0) {
		DEBUG("[btobexclient_put] Error encoding name to unicode!\n");
		goto out_err;
	}
	
	/* read the file */
	body = btobexclient_readfile(file, filesize); 
	if (!body) {
		DEBUG("[btobexclient_put] Error reading file!\n");
		goto out_err;
	}
	
	/* build request object */
	if( !(object = OBEX_ObjectNew(handle, OBEX_CMD_PUT)) ) {
		DEBUG("[btobexclient_put] Error creating PUT request!\n");
		goto out_err;
	}	
	
	/* add unicode name header */	
	header.bs = (uint8_t*)uname;
	result = OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME, header, uname_size, 0);
	if (result < 0) {
		DEBUG("[btobexclient_put] Error adding name header!\n");
		goto out_err;
	}	
	
	/* add length header */
	header.bq4 = filesize;
	result = OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, header, 4, 0);
	if (result < 0) {
		DEBUG("[btobexclient_put] Error adding file length header!\n");
		goto out_err;
	}	
	
	/* add body header */
	header.bs = body;
	result = OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, header, filesize, 0);
	if (result < 0) {
		DEBUG("[btobexclient_put] Error adding file body header!\n");
		goto out_err;
	}		
	
	return btobexclient_request(handle, object);
		
out_err:
	DEBUG("[btobexclient_put] Clean up after error\n");
	if (body) free(body);
	if (uname) free(uname);
	if (object) OBEX_ObjectDelete(handle, object);
	
	return -1;
}
