/***************************************************************************
                          gui_listing.cpp  -  description
                             -------------------
    begin                : Thu May 3 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "gui_listing.h"

#include "gui.h"
#include "datadispatcher.h"
#include "configuration.h"
#include "jobmanager.h"
#include "job_deletetrackphysically.h"
#include "lvi_track.h"
#include "job_modifyplaylist_tracks.h"
#include "job_getextradata.h"
#include "pixmapcache.h"
#ifdef HAVE_MIXXX
#include "mixxxclient.h"
#endif /* HAVE_MIXXX */

#include <qlayout.h>
#include <qhgroupbox.h>
#include <qvgroupbox.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qheader.h>
#include <qpopupmenu.h>
#include <qtooltip.h>
#include <qmessagebox.h>
#include <qstringlist.h>

#include <stdlib.h>

// ##############################################
// # initialize listing gui
// ##############################################
GUI_Listing::GUI_Listing(QWidget *parent, const char *name )
  : QFrame(parent, name)
{
  QVBoxLayout *blay = new QVBoxLayout(this);
  
  tabbar = new QTabWidget(this);
  
  selectionframe = new QFrame(this);
  QVBoxLayout *selblay = new QVBoxLayout(selectionframe);
  
  // radiobuttons for selecting sources
  QFrame *hframe = new QFrame(selectionframe);
  QHBoxLayout *hlay = new QHBoxLayout(hframe);
	
  selbox = new QButtonGroup(1, Qt::Horizontal, _("Sources"), hframe);
  QFrame *selframe = new QFrame(selbox);
  QGridLayout *selgrid = new QGridLayout(selframe, 3, 1);
  QRadioButton *hddradio = new QRadioButton(_("Only Available Tracks"), selframe);
  selgrid->addWidget(hddradio, 0, 0);
  QRadioButton *cdromradio = new QRadioButton(_("Only NOT Available Tracks"), selframe);
  selgrid->addWidget(cdromradio, 1, 0);
  QRadioButton *bothradio = new QRadioButton(_("All Tracks"), selframe);
  selgrid->addWidget(bothradio, 2, 0);
  selbox->insert(hddradio, 0);
  selbox->insert(cdromradio, 1);
  selbox->insert(bothradio, 2);
  connect(selbox, SIGNAL(clicked(int)), SLOT(slot_changeRadio(int)));
  
  hlay->addWidget(selbox);
  
  // checkboxes for additional informations
  QGroupBox *infobox = new QGroupBox(1, Qt::Horizontal, _("Additional Information"), hframe);
  QFrame *infoframe = new QFrame(infobox);
  QGridLayout *infogrid = new QGridLayout(infoframe, 4, 3);
  
  info_lengthcheck = new QCheckBox(_("Length"), infoframe);
  infogrid->addWidget(info_lengthcheck, 0, 0);
  info_bitratecheck = new QCheckBox(_("Bitrate"), infoframe);
  infogrid->addWidget(info_bitratecheck, 0, 1);
  info_yearcheck = new QCheckBox(_("Year"), infoframe);
  infogrid->addWidget(info_yearcheck, 0, 2);
  info_genrecheck = new QCheckBox(_("Genre"), infoframe);
  infogrid->addWidget(info_genrecheck, 1, 0);
  info_albumcheck = new QCheckBox(_("Album"), infoframe);
  infogrid->addWidget(info_albumcheck, 1, 1);	
  info_commentcheck = new QCheckBox(_("Comment"), infoframe);
  infogrid->addWidget(info_commentcheck, 1, 2);
  info_pathcheck = new QCheckBox(_("Path"), infoframe);
  infogrid->addWidget(info_pathcheck, 2, 0);	
  info_filenamecheck = new QCheckBox(_("Filename"), infoframe);
  infogrid->addWidget(info_filenamecheck, 2, 1);
  info_mediacheck = new QCheckBox(_("Medium"), infoframe);
  infogrid->addWidget(info_mediacheck, 2, 2);
  info_notescheck = new QCheckBox(_("Notes"), infoframe);
  infogrid->addWidget(info_notescheck, 3, 0);
  info_ratingcheck = new QCheckBox(_("Rating"), infoframe);
  infogrid->addWidget(info_ratingcheck, 3, 1);
  
  connect(info_lengthcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Length(int)));
  connect(info_bitratecheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Bitrate(int)));
  connect(info_yearcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Year(int)));
  connect(info_genrecheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Genre(int)));
  connect(info_albumcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Album(int)));
  connect(info_commentcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Comment(int)));
  connect(info_pathcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Path(int)));
  connect(info_filenamecheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Filename(int)));
  connect(info_mediacheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Media(int)));
  connect(info_notescheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Notes(int)));
  connect(info_ratingcheck, SIGNAL(stateChanged(int)), this, SLOT(slot_Rating(int)));

  hlay->addWidget(infobox, 1);
  
  // the selection buttons
  QFrame *butframe = new QFrame(selectionframe);
  QGridLayout *butlay = new QGridLayout(butframe, 1, 7);
  
  button_All = new QPushButton(_("Select All"), butframe);
  butlay->addWidget(button_All, 0, 1);
  button_None = new QPushButton(_("Select None"), butframe);
  butlay->addWidget(button_None, 0, 3);
  button_Invert = new QPushButton(_("Invert Selection"), butframe);
  butlay->addWidget(button_Invert, 0, 5);
  
  connect(button_All, SIGNAL(clicked()), this, SLOT(slot_selectionAll()));
  connect(button_None, SIGNAL(clicked()), this, SLOT(slot_selectionNone()));
  connect(button_Invert, SIGNAL(clicked()), this, SLOT(slot_selectionInvert()));
  
  selblay->addWidget(hframe);
  selblay->addWidget(butframe, 1);
  
  // external tabs
  searcher   = new GUI_Searcher();
  sql        = new GUI_SQL();
#ifdef HAVE_MIXXX
  mixxxQueue = new GUI_MixxxQueue();
#endif /* HAVE_MIXXX */
  
  trackframe = new QFrame();
  trackframe->setFixedHeight(1);
	
  tabbar->addTab(trackframe, pixmapcache->getFadeIconSet("music_sixteenthnote.png"), _("Tracks"));

  connect(tabbar, SIGNAL(currentChanged(QWidget*)), this, SLOT(slot_tabChanged(QWidget*)));
  
  // the listing widget
  listing = new ExtendedListView(this);
  QString temps = _( "Multiselect and right click to play titles,\nadd them to a playlist, or edit tags.\nYou can also drag and drop tracks to xmms." ); 
  QToolTip::add( listing, temps);
  QToolTip::add( listing->viewport(), temps);
  listing->addColumn("", -1);
  columnMedium   = listing->addColumn(_("Medium"), 100);
  columnArtist   = listing->addColumn(_("Artist"), 120);
  columnTitle    = listing->addColumn(_("Title"), 160);
  columnRating   = listing->addColumn(_("Rating"), 70 + fontMetrics().maxWidth());
  columnLength   = listing->addColumn(_("Length"), -1);
  listing->setColumnAlignment(columnLength, Qt::AlignRight);
  columnBitrate  = listing->addColumn(_("Bitrate"), -1);
  listing->setColumnAlignment(columnBitrate, Qt::AlignRight);
  columnAlbum    = listing->addColumn(_("Album"), 160);
  columnComment  = listing->addColumn(_("Comment"), 160);
  columnYear     = listing->addColumn(_("Year"), -1);
  listing->setColumnAlignment(columnYear, Qt::AlignCenter);
  columnGenre    = listing->addColumn(_("Genre"), 60);
  columnPath     = listing->addColumn(_("Path"), 200);
  columnFilename = listing->addColumn(_("Filename"), 200);
  columnNotes    = listing->addColumn(_("Notes"), 200);
    
  listing->setItemMargin(0);
  
  listing->setAllColumnsShowFocus(true);
  listing->setSelectionMode(QListView::Extended);
  listing->setShowSortIndicator(true);
  
  connect(listing, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
	  this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));

  connect(listing, SIGNAL(selectionChanged()), this, SLOT(slot_selectionChanged()));
  connect(listing, SIGNAL(currentChanged(QListViewItem*)), this, SLOT(slot_currentChanged(QListViewItem*)));
  connect(listing, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slot_doubleClicked(QListViewItem*)));
  
  connect(listing->header(), SIGNAL(clicked(int)), this, SLOT(slot_selectionChanged2(int)));
  
  blay->addWidget(tabbar);
  blay->addWidget(listing, 1);
  
  widthes.resize(listing->columns());
  for(int i = 0; i < listing->columns(); i++) widthes[i] = listing->columnWidth(i);
  
  // set default states
  info_lengthcheck->setChecked( config->readInfo_Length() );
  info_mediacheck->setChecked( config->readInfo_Media() );
  info_bitratecheck->setChecked( config->readInfo_Bitrate() );
  info_albumcheck->setChecked( config->readInfo_Album() );
  info_commentcheck->setChecked( config->readInfo_Comment() );
  info_yearcheck->setChecked( config->readInfo_Year() );
  info_genrecheck->setChecked( config->readInfo_Genre() );
  info_pathcheck->setChecked( config->readInfo_Path() );
  info_filenamecheck->setChecked( config->readInfo_Filename() );
  info_notescheck->setChecked( config->readInfo_Notes() );
  info_ratingcheck->setChecked( config->readInfo_Rating() );
  
  slot_Length( config->readInfo_Length() );
  slot_Media( config->readInfo_Media() );
  slot_Bitrate( config->readInfo_Bitrate() );
  slot_Album( config->readInfo_Album() );
  slot_Comment( config->readInfo_Comment() );
  slot_Year( config->readInfo_Year() );
  slot_Genre( config->readInfo_Genre() );
  slot_Path( config->readInfo_Path() );
  slot_Filename( config->readInfo_Filename() );
  slot_Notes( config->readInfo_Notes() );
  slot_Rating( config->readInfo_Rating() );
  
  radioState = config->readinfo_radiostate();
  slot_changeRadio( radioState );
  switch( radioState ) {
  case 0 : hddradio->setChecked(true);
    break;
  case 1 : cdromradio->setChecked(true);
    break;
  default : bothradio->setChecked(true);
  }

  if ( !config->getMaxWidthMode() ) {
    QStringList qs = config->getinfo_columnwidth();
    int loop = 0;
    for ( QStringList::Iterator it = qs.begin(); it != qs.end(); ++it ) {
      listing->header()->resizeSection( loop, (*it).toInt());
      ++loop;
    }
  }
  if ( config->getAllowGuiRestore() ) {
    QStringList qs = config->getinfo_columnmap();
    int loop = 0;
    for ( QStringList::Iterator it = qs.begin(); it != qs.end(); ++it ) {
      listing->header()->moveSection(loop,(*it).toInt());
      ++loop;
    }  
  }

  listing->triggerUpdate();

  editmode = false;
#ifdef HAVE_MIXXX
  mixxxMode = false;
#endif /* HAVE_MIXXX */
}

void GUI_Listing::polish()
{
  QFrame::polish();
  tabbarMinSize = tabbar->minimumSizeHint().height();

  /* add tabs 'Selection' and 'Search' */
  /*-----------------------------------*/
  tabbar->addTab(selectionframe, pixmapcache->getFadeIconSet("filter.png"), _("Selection"));
  tabbar->addTab(searcher, pixmapcache->getFadeIconSet("action_updateonly.png"), _("Search"));
  tabbar->addTab(sql, pixmapcache->getFadeIconSet("action_updateonly.png"), _("SQL"));

  /* determine maximum size of tabbar */
  /*----------------------------------*/
  tabbarMaxSize = tabbar->minimumSizeHint().height();

#ifdef HAVE_MIXXX
  tabbar->addTab(mixxxQueue, pixmapcache->getFadeIconSet("mixxx.png"), _("Mixxx Queue"));
#endif /* HAVE_MIXXX */

  /* set tabbar size for tab 'Tracks' */
  /*----------------------------------*/
  tabbar->setFixedHeight(tabbarMinSize);
}

void GUI_Listing::slot_tabChanged(QWidget *widget) {

#ifdef HAVE_MIXXX
    if(widget == mixxxQueue) {
        mixxxMode = true;
        gui->enterMixxxMode();
        tabbar->setMinimumHeight(tabbarMaxSize);
        tabbar->setMaximumHeight(listing->maximumHeight());
    } else {
        if(mixxxMode) {
            mixxxMode = false;
            gui->leaveMixxxMode();
        }
#endif /* HAVE_MIXXX */

        if(widget != trackframe) {
            // set tabbar to polished size
            tabbar->setFixedHeight(tabbarMaxSize);
        } else {
            // hide tabbar
            tabbar->setFixedHeight(tabbarMinSize);
        }
#ifdef HAVE_MIXXX
    }
#endif /* HAVE_MIXXX */
}

// ##############################################
// # callbacks for the checkboxes
// ##############################################
// always show columnArtist and columnTitle
void GUI_Listing::slot_Media(int state)    { toggleColumn(columnMedium, state); }
void GUI_Listing::slot_Length(int state)   { toggleColumn(columnLength, state); }
void GUI_Listing::slot_Bitrate(int state)  { toggleColumn(columnBitrate, state); }
void GUI_Listing::slot_Album(int state)    { toggleColumn(columnAlbum, state); }
void GUI_Listing::slot_Comment(int state)  { toggleColumn(columnComment, state); }
void GUI_Listing::slot_Year(int state)     { toggleColumn(columnYear, state); }
void GUI_Listing::slot_Genre(int state)    { toggleColumn(columnGenre, state); }
void GUI_Listing::slot_Path(int state)     { toggleColumn(columnPath, state);}
void GUI_Listing::slot_Filename(int state) { toggleColumn(columnFilename, state);}
void GUI_Listing::slot_Notes(int state)    { toggleColumn(columnNotes, state);}
void GUI_Listing::slot_Rating(int state)   { toggleColumn(columnRating, state);}

// ##############################################
// # callbacks for the buttons
// ##############################################
void GUI_Listing::slot_selectionAll()    { listing->selectAll(true);  }
void GUI_Listing::slot_selectionNone()   { listing->selectAll(false); }
void GUI_Listing::slot_selectionInvert() { listing->invertSelection();}

// ##############################################
// # callback for selection
// ##############################################
void GUI_Listing::slot_selectionChanged()     {
  gui->getTagListing()->notifySelectionChanged();
}

void GUI_Listing::slot_currentChanged(QListViewItem *lvi) {
    gui->getTagListing()->notifyCurrentChanged(lvi);
}

void GUI_Listing::slot_doubleClicked(QListViewItem *lvi) {
  slot_playWithXMMS();
}

void GUI_Listing::slot_selectionChanged2(int c) {
  if(!editmode) gui->getTagListing()->notifySelectionChanged();
  else gui->getTagListing()->notifySortingInEditModeChanged();
}

// ##############################################
// # callback for context menu
// ##############################################
void GUI_Listing::slot_contextMenu(QListViewItem *lvi, const QPoint &p, int)
{
  if(!lvi || !lvi->isSelectable()) return;
  listing->setCurrentItem(lvi);
  listing->setSelected(lvi, true);

  if(editmode) return;

  QPopupMenu *menu = new QPopupMenu(this);

  LVI_Track *item = static_cast<LVI_Track*>(lvi);
  
  int i=0;
  if(item->getIsAvailable()) {
#ifdef HAVE_MIXXX
    if(config->RegularPlayerEnabled()) {
#endif /* HAVE_MIXXX */
      menu->insertItem(pixmapcache->get("stock_media-play.png"), _("Play track(s) with regular player"), this, SLOT(slot_playWithXMMS()), 0, i++);
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("Enqueue track(s) at regular player"), this, SLOT(slot_enqueueInXMMS()), 0, i++);
      menu->insertSeparator();
#ifdef HAVE_MIXXX
    }
    if(config->MixxxPlayerEnabled() && isMixxxConnected) {
      menu->insertItem(pixmapcache->get("play-off-Ch1.png"), _("mixxx: load at channel 1"), this, SLOT(slot_mixxxLoadAtChannel1()), 0, i++);
      menu->insertItem(pixmapcache->get("play-off-Ch2.png"), _("mixxx: load at channel 2"), this, SLOT(slot_mixxxLoadAtChannel2()), 0, i++);
      menu->insertItem(pixmapcache->get("play-on-Ch1.png"), _("mixxx: load at channel 1 and play"), this, SLOT(slot_mixxxLoadAndPlayAtChannel1()), 0, i++);
      menu->insertItem(pixmapcache->get("play-on-Ch2.png"), _("mixxx: load at channel 2 and play"), this, SLOT(slot_mixxxLoadAndPlayAtChannel2()), 0, i++);
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue at channel 1"), this, SLOT(slot_mixxxEnqueueAtChannel1()), 0, i++);
      menu->insertItem(pixmapcache->get("stock_music-library.png"), _("mixxx: enqueue at channel 2"), this, SLOT(slot_mixxxEnqueueAtChannel2()), 0, i++);
      menu->insertSeparator();
    }
#endif /* HAVE_MIXXX */
    
  }

  if(gui->getPlaylisting()->getDisplayedPlaylistID() > 0) {
    menu->insertItem(pixmapcache->get("lvi_playlist.png"), _("append track(s) to selected playlist"), this, SLOT(slot_addToPlaylist()), 0, i++);
    menu->insertSeparator();
  }

  menu->insertItem(pixmapcache->get("lvi_changed.png"), _("Edit Selected Track(s)"), gui->getTagListing(), SLOT(slot_editbutton()), 0, i++);
  menu->insertSeparator();

  if(item->getIsAvailable() && item->getType() != MEDIUM_CDROM) {
    menu->insertItem(pixmapcache->get("action_remove.png"), _("Delete Track Physically"), this, SLOT(slot_deleteTrack()), 0, i);
    if(busyMap.contains(item->getMedium()) && busyMap[item->getMedium()]) menu->setItemEnabled(i, false);
  }

  menu->exec(p, 0);
}

void GUI_Listing::slot_deleteTrack()
{
  LVI_Track *lvi;
  for (QListViewItem *item = listing->firstChild(); item != 0; item = item->nextSibling())
    if ( item->isSelected() && ( lvi = dynamic_cast<LVI_Track*>(item))) 
    {
      TRACK *track = new TRACK();
      track->id     = lvi->getOriginalTrack()->id;
      track->artist = lvi->getOriginalTrack()->artist;
      track->path   = lvi->getOriginalTrack()->path;
      track->filename = lvi->getOriginalTrack()->filename;
      track->medium = lvi->getOriginalTrack()->medium;

      jobman->addJob(new Job_DeleteTrackPhysically(track));
    }
}


// ##############################################
// # callback for context menu items
// ##############################################

void GUI_Listing::slot_playWithXMMS(bool selected) {
  callRegularPlayer(selected, false);
}

void GUI_Listing::slot_enqueueInXMMS(bool selected) {
  callRegularPlayer(selected, true);
}

void GUI_Listing::callRegularPlayer(bool selected, bool enqueue)
{
  QStringList fileList;
  
  for (QListViewItem *item = listing->firstChild(); item != 0; item = item->nextSibling())
  {
     if ( ( !selected || item->isSelected()) && dynamic_cast<LVI_Track*>(item)->getIsAvailable() )
         fileList.append(QString(item->text(columnPath) + "/" + item->text(columnFilename)));
  }
  
  commandLineCallToRegularPlayer(fileList, enqueue);
}

void GUI_Listing::slot_mixxxLoadAtChannel1() {
#ifdef HAVE_MIXXX
  mixxxLoadAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Listing::slot_mixxxLoadAtChannel2() {
#ifdef HAVE_MIXXX
  mixxxLoadAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Listing::mixxxLoadAtChannel(int channel)
{
  QListViewItem *item = listing->currentItem ();
  if (dynamic_cast<LVI_Track*>(item)->getIsAvailable() ) {
    mixxxClient->loadAtChannel(QString(item->text(columnPath) + "/" + item->text(columnFilename)), channel);
  } else {
    QMessageBox::information(this, "Message", _("File(s) not available for playing"), QMessageBox::Ok);
  }
}
#endif /* HAVE_MIXXX */

void GUI_Listing::slot_mixxxLoadAndPlayAtChannel1() {
#ifdef HAVE_MIXXX
  mixxxLoadAndPlayAtChannel(1);
#endif /* HAVE_MIXXX */
}

void GUI_Listing::slot_mixxxLoadAndPlayAtChannel2() {
#ifdef HAVE_MIXXX
  mixxxLoadAndPlayAtChannel(2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Listing::mixxxLoadAndPlayAtChannel(int channel)
{
  QListViewItem *item = listing->currentItem ();
  if (dynamic_cast<LVI_Track*>(item)->getIsAvailable() ) {
    mixxxClient->loadAndPlayAtChannel(QString(item->text(columnPath) + "/" + item->text(columnFilename)), channel);
  } else {
    QMessageBox::information(this, "Message", _("File(s) not available for playing"), QMessageBox::Ok);
  }
}
#endif /* HAVE_MIXXX */

void GUI_Listing::slot_mixxxEnqueueAtChannel1() {
#ifdef HAVE_MIXXX
  mixxxEnqueueAtChannel(true, 1);
#endif /* HAVE_MIXXX */
}

void GUI_Listing::slot_mixxxEnqueueAtChannel2() {
#ifdef HAVE_MIXXX
  mixxxEnqueueAtChannel(true, 2);
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI_Listing::mixxxEnqueueAtChannel(bool selected, int channel)
{
  QList<LVI_Track> list;
  
  for (QListViewItem *item = listing->firstChild(); item != 0; item = item->nextSibling())
  {
     if ( ( !selected || item->isSelected()) && dynamic_cast<LVI_Track*>(item)->getIsAvailable() ) {
         list.append(dynamic_cast<LVI_Track*>(item));
     }
  }
  
  mixxxQueue->enqueueAtChannel(list, channel);
}
#endif /* HAVE_MIXXX */

void GUI_Listing::slot_addToCurrent() {
  QList<TRACK> *templist;
  templist = new QList<TRACK>;
  for(QListViewItem *item = listing->firstChild(); item != 0; item = item->nextSibling()) {
    if(item->isSelected() && dynamic_cast<LVI_Track*>(item)->getIsAvailable())
      templist->append(dynamic_cast<LVI_Track*>(item)->getOriginalTrack());
  }

  jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, 0));
}

void GUI_Listing::slot_addToPlaylist() {
  int ID;
  if((ID = gui->getPlaylisting()->getDisplayedPlaylistID()) < 0)
    return;

  QList<TRACK> *templist;
  templist = new QList<TRACK>;
  for(QListViewItem *item = listing->firstChild(); item != 0; item = item->nextSibling()) {
    if(item->isSelected() )
      templist->append(dynamic_cast<LVI_Track*>(item)->getOriginalTrack());
  }

  jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, ID));
}

// ##############################################
// # callback for the radiobuttons
// ##############################################
void GUI_Listing::slot_changeRadio(int id)
{
  if(id == radioState) return;
  
  for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next()) if(item->getIsShowed()) { listing->takeItem(item); item->setIsShowed(false); }

  for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next()) {
    if((id == 0 && item->getIsAvailable())
       || (id == 1 && !item->getIsAvailable())
       || (id == 2))
      {
	listing->insertItem(item);
	item->setIsShowed(true);
      }
  }
  
  radioState = id;
  
  // restore sorting;
  sort();
  
  if(listing->childCount() != 0) listing->setSelected(listing->firstChild(), true);
  slot_selectionChanged();
}
  
// ##############################################
// # show or hide columns
// ##############################################
void GUI_Listing::toggleColumn(int column, int state)
{
  // hide column
  if(state == 0) {
    widthes[column] = listing->columnWidth(column);
    listing->setColumnText(column, "");

    listing->header()->resizeSection(column, 0);
    listing->triggerUpdate();
    // ^^^ you need this instead of listing->setColumnWidth(column, 0) ==> BUG?! in Qt

    listing->setColumnWidthMode(column, QListView::Manual);
    listing->header()->setClickEnabled(false, column);
    listing->header()->setResizeEnabled(false, column);
  }
  // show column
  if(state == 2) {

    listing->header()->resizeSection(column, widthes[column]);
    listing->triggerUpdate();
    // ^^^ you need this instead of listing->setColumnWidth(column, widthes[column]) ==> BUG?! in Qt
    
    if(column == columnMedium)        listing->setColumnText(columnMedium,   _("Medium"));
    else if(column == columnLength)   listing->setColumnText(columnLength,   _("Length"));
    else if(column == columnBitrate)  listing->setColumnText(columnBitrate,  _("Bitrate"));
    else if(column == columnAlbum)    listing->setColumnText(columnAlbum,    _("Album"));
    else if(column == columnComment)  listing->setColumnText(columnComment,  _("Comment"));
    else if(column == columnYear)     listing->setColumnText(columnYear,     _("Year"));
    else if(column == columnGenre)    listing->setColumnText(columnGenre,    _("Genre"));
    else if(column == columnPath)     listing->setColumnText(columnPath,     _("Path"));
    else if(column == columnFilename) listing->setColumnText(columnFilename, _("Filename"));
    else if(column == columnNotes)    listing->setColumnText(columnNotes,    _("Notes"));
    else if(column == columnRating)   listing->setColumnText(columnRating,   _("Rating"));
    
    if ( config->getMaxWidthMode() ) 
	       listing->setColumnWidthMode(column, QListView::Maximum);
    else
      listing->setColumnWidthMode(column, QListView::Manual);
    listing->header()->setClickEnabled(true, column);
    listing->header()->setResizeEnabled(true, column);
  }
  
  // adjust width for scrollingbar
  int width = 0;
  for(int i=0; i<= 11; i++) width += listing->header()->sectionSize(i);
  listing->resizeContents(width, listing->contentsHeight());
}

void GUI_Listing::notifyMediumBusy(int id, bool status) {
  busyMap[id] = status;
}

void GUI_Listing::notifyMediumAvailabilityChanged(int id, QString path, bool state)
{
  mapAvail[id] = state;
  if(state == true) mapPath[id] = path;

  for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next()) if(item->getMedium() == id) {
    if(state == true && item->getType() != MEDIUM_HARDDISK) { item->applyPath(path); }
    item->setAvailability(state);
    if(!editmode) {
      int c=0;
      if((radioState == 0 || radioState == 2) && item->getIsAvailable() && !item->getIsShowed()) { listing->insertItem(item); item->setIsShowed(true); c++; }
      if((radioState == 1) && item->getIsAvailable() && item->getIsShowed()) { listing->takeItem(item); item->setIsShowed(false); c++; }
      if((radioState == 1 || radioState == 2) && !item->getIsAvailable() && !item->getIsShowed()) { listing->insertItem(item); item->setIsShowed(true); c++; }
      if((radioState == 0) && !item->getIsAvailable() && item->getIsShowed()) { listing->takeItem(item); item->setIsShowed(false); c++; }
      if(c > 0) {
	// restore sorting;
	sort();
	if(listing->childCount() != 0) listing->setSelected(listing->firstChild(), true);
	slot_selectionChanged();
      }
    }
  }
 if(editmode) gui->getTagListing()->notifySelectionChanged();
}

void GUI_Listing::notifyNewMediumBasis(QList<MEDIUM> *list)
{
  for(MEDIUM *medium = list->first(); medium != 0; medium = list->next()) notifyNewMedium(medium);
}

void GUI_Listing::notifyNewTrackListing(QList<TRACK> *list)
{
  if(editmode) return;
  
  // clear old data
  listing->clear();    
  listAll.clear();

  if(list != 0) for(TRACK *entry = list->first(); entry != 0; entry = list->next()) {
    LVI_Track *item = new LVI_Track(listing, entry);
    item->applyMedium(mapType[entry->medium], mapLabel[entry->medium]);
    if(mapAvail[entry->medium] && item->getType() != MEDIUM_HARDDISK) item->applyPath(mapPath[entry->medium]);
    item->setAvailability(mapAvail[entry->medium]);
     
    listAll.append(item);
    if(radioState == 0 && !item->getIsAvailable()) { listing->takeItem(item); item->setIsShowed(false); }
    if(radioState == 1 && item->getIsAvailable()) { listing->takeItem(item); item->setIsShowed(false); }
  }
  
  // restore sorting;
  sort();
  
  if(listing->childCount() != 0) listing->setSelected(listing->firstChild(), true);
    if(listing->childCount() != 0) listing->setCurrentItem(listing->firstChild());
  slot_selectionChanged();
}

void GUI_Listing::notifyNewMedium(MEDIUM *medium)
{
  mapLabel[medium->id] = medium->label;
  mapType[medium->id] = medium->type;
  mapAvail[medium->id] = (medium->type == MEDIUM_HARDDISK ? true : false);
}

void GUI_Listing::notifyMediumRemoved(int id)
{
  mapLabel.remove(id);
  mapType.remove(id);
  mapAvail.remove(id);
}

void GUI_Listing::notifyMediumRenamed(MEDIUM *medium)
{
  mapLabel[medium->id] = medium->label;
}

void GUI_Listing::notifyFilenameHasChanged(int id, QString filename) {
  for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next())
    if(item->getOriginalTrack()->id == id) item->setNewFilename(filename);
}

void GUI_Listing::setEditMode(bool state)
{
  if ( verbose == 4 ) qWarning( "GUI_Listing::setEditMode(state=%s) starts", state ? "true" : "false");
  editmode = state;

  if (editmode) { // save previous tooltip
    oldtip = QToolTip::textFor( listing, QPoint(0,0));
    QToolTip::remove( listing );
    QToolTip::remove( listing->viewport() );
    QString temps = _( "Select one track to edit.\nAll modifications will be flagged,\nand need to be saved" );
    QToolTip::add( listing, temps );
    QToolTip::add( listing->viewport(), temps );
  } else { // restore previous tooltip
    QToolTip::remove( listing );
    QToolTip::remove( listing->viewport() );
    QToolTip::add( listing, oldtip);
    QToolTip::add( listing->viewport(), oldtip);
  }

#ifdef HAVE_MIXXX
  tabbar->setTabEnabled(mixxxQueue, !editmode);
#endif /* HAVE_MIXXX */
  button_All->setEnabled(!editmode);
  button_None->setEnabled(!editmode);
  button_Invert->setEnabled(!editmode);
  selbox->setEnabled(!editmode);
  searcher->slot_setConnectionState(!editmode);
  sql->slot_setConnectionState(!editmode);

  if(editmode) datadispatcher->eventAddReadWriteLock();
  else datadispatcher->eventRemoveReadWriteLock();

  if(editmode) {
    for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next())
      if(item->getIsShowed() && !listing->isSelected(item)) { listing->takeItem(item); item->setIsShowed(false); }
    listing->selectAll(false);

    // listing->setSelectionMode(QListView::Extended)  multi selection work in progress... 
    listing->setSelectionMode(QListView::Single);
    
    if(listing->childCount() != 0) listing->setSelected(listing->firstChild(), true);
    QToolTip::add(this, _("Select to modify tag or file name"));
  } else {
    listing->setSelectionMode(QListView::Extended);
    listing->selectAll(true);
    for(LVI_Track *item = listAll.first(); item != 0; item = listAll.next()) {
      if((radioState == 0 || radioState == 2) && item->getIsAvailable() && !item->getIsShowed()) { listing->insertItem(item); item->setIsShowed(true); }
      if((radioState == 1) && item->getIsAvailable() && item->getIsShowed()) { listing->takeItem(item); item->setIsShowed(false); }
      if((radioState == 1 || radioState == 2) && !item->getIsAvailable() && !item->getIsShowed()) { listing->insertItem(item); item->setIsShowed(true); }
      if((radioState == 0) && !item->getIsAvailable() && item->getIsShowed()) { listing->takeItem(item); item->setIsShowed(false); }
    }
    // restore sorting;
    sort();
    QToolTip::add(this, _("Select and right click to play titles\nor to add them to a playlist"));
 }
  if ( verbose == 4 ) qWarning( "GUI_Listing::setEditMode() ends");
}


int GUI_Listing::getHeaderIndex( QString s) {
  // return colum index whose header name is s
  // Column position is fully user configurable
  // Index column will not change however.
  int i;
  QHeader* qh( listing->header() );
  for (i = 0 ; i < listing->header()->count(); i++) 
    if ( qh->label(qh->mapToSection(i)) == s ) return i;
  return 999;   // not found, do not sort on this index
}

void GUI_Listing::sort()
{
  int i;
  LVI_Track *item = static_cast<LVI_Track*>(listing->firstChild());
  if (config->getComboSortingAuto() && item && item->getTracknumber() ) 
    i = getHeaderIndex( _("Album") );
  else 
    i = getHeaderIndex(  config->getComboSortingMode() );

  // if index not found then use filename as a default.
  if ( i== 999 )  i = getHeaderIndex( _("Filename") );

  listing->setSorting( listing->header()->mapToSection(i), true); 

  //  qWarning(" sorting on idx:%d sec:%d title:%s", i, listing->header()->mapToSection(i),
  //listing->header()->label(listing->header()->mapToSection(i)).local8Bit().data());
}

void GUI_Listing::slot_setConnectionState(bool state)
{
  if(state == false) clear();
  if(state == false) editmode = false;
  isConnected = state;
}

#ifdef HAVE_MIXXX
void GUI_Listing::setMixxxConnectionState(bool state)
{
  mixxxQueue->setMixxxConnectionState(state);
  isMixxxConnected = state;
}
#endif /* HAVE_MIXXX */

QStringList GUI_Listing::readColumnWidth()
{
  QStringList qs;
  for ( int i=0; i<listing->columns(); i++ ) {
    qs += QString::number(listing->columnWidth(i));
   }
  return qs;
}

QStringList GUI_Listing::readColumnMap()
{
  QStringList qs;
  for ( int i=0; i<listing->columns(); i++ ) {
    qs += QString::number(listing->header()->mapToIndex(i));
  }
  return qs;
}

void GUI_Listing::clear()
{
  notifyNewTrackListing(0);
}

GUI_Listing::~GUI_Listing() {}
