/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <errno.h>

#include "config_pm.h"
#include <dirent.h>
#include "config.h"
#include <list>

const string GeneralConfig::config_files[] = { "common", "thermal", "cpufreq", "battery", "sleep", "events" };

GeneralConfig::GeneralConfig() 
 : current_scheme(NULL), AC_scheme(NULL), Battery_scheme(NULL), disable_CPU_freq(false), scheme_configs(NULL) 
{
	/*****************************************
        Initialise all events */
	eventMap.insert(pair < string, Event > ("acadapter.offline", Event("acadapter.offline")));
	eventMap.insert(pair < string, Event > ("acadapter.online", Event("acadapter.online")));

	eventMap.insert(pair < string, Event > ("button.sleep", Event("button.sleep")));
	eventMap.insert(pair < string, Event > ("button.power", Event("button.power")));
	eventMap.insert(pair < string, Event > ("button.lid.open", Event("button.lid.open")));
	eventMap.insert(pair < string, Event > ("button.lid.closed", Event("button.lid.closed")));

	eventMap.insert(pair < string, Event > ("temperature.ok", Event("temperature.ok")));
	eventMap.insert(pair < string, Event > ("temperature.active", Event("temperature.active")));
	eventMap.insert(pair < string, Event > ("temperature.passive", Event("temperature.passive")));
	eventMap.insert(pair < string, Event > ("temperature.hot", Event("temperature.hot")));
	eventMap.insert(pair < string, Event > ("temperature.critical", Event("temperature.critical")));

	eventMap.insert(pair < string, Event > ("battery.normal", Event("battery.normal")));
	eventMap.insert(pair < string, Event > ("battery.warning", Event("battery.warning")));
	eventMap.insert(pair < string, Event > ("battery.low", Event("battery.low")));
	eventMap.insert(pair < string, Event > ("battery.critical", Event("battery.critical")));

	eventMap.insert(pair < string, Event > ("battery.info", Event("battery.info")));

	eventMap.insert(pair < string, Event > ("global.suspend2disk", Event("global.suspend2disk")));
	eventMap.insert(pair < string, Event > ("global.suspend2ram", Event("global.suspend2ram")));
	eventMap.insert(pair < string, Event > ("global.standby", Event("global.standby")));
	eventMap.insert(pair < string, Event > ("global.resume.suspend2disk", Event("global.resume.suspend2disk")));
	eventMap.insert(pair < string, Event > ("global.resume.suspend2ram", Event("global.resume.suspend2ram")));
	eventMap.insert(pair < string, Event > ("global.resume.standby", Event("global.resume.standby")));
	eventMap.insert(pair < string, Event > ("global.suspend2disk.other", Event("global.suspend2disk.other")));

	eventMap.insert(pair < string, Event > ("processor.performance", Event("processor.performance")));
	eventMap.insert(pair < string, Event > ("processor.powersave", Event("processor.powersave")));
	eventMap.insert(pair < string, Event > ("processor.dynamic", Event("processor.dynamic")));
	eventMap.insert(pair < string, Event > ("processor.busy", Event("processor.busy")));
	eventMap.insert(pair < string, Event > ("processor.idle", Event("processor.idle")));
	eventMap.insert(pair < string, Event > ("processor.notify", Event("processor.notify")));

	eventMap.insert(pair < string, Event > ("daemon.start", Event("daemon.start")));
	eventMap.insert(pair < string, Event > ("daemon.terminate", Event("daemon.terminate")));

	eventMap.insert(pair < string, Event > ("daemon.scheme.change", Event("daemon.scheme.change")));

	eventMap.insert(pair < string, Event > ("other", Event("other")));
	/*****************************************/

	config_dir = CONFIG_DIR;
	config_dir.append("/");
	script_dir = SCRIPT_DIR;
	script_dir.append("/");
	pub_script_dir = PUB_SCRIPT_DIR;
	pub_script_dir.append("/");
	scheme_dir = CONFIG_DIR;
	scheme_dir.append("/");
};

GeneralConfig::~GeneralConfig()
{

	SchemeConfig *temp = NULL;

	/* delete scheme list */
	if (scheme_configs) {
		SchemeConfig *temp1 = scheme_configs;
		for (temp = scheme_configs; temp != NULL;) {
			temp1 = temp;
			temp = temp->next;
			delete temp1;
			temp1 = NULL;
		}
		current_scheme = NULL;
		Battery_scheme = NULL;
		AC_scheme = NULL;
		scheme_configs = NULL;
	}
};

int GeneralConfig::readConfFiles()
{

	for (int x = 0; x < (int)(sizeof(config_files) / sizeof(string)); x++)
		if (readConfigFile(config_dir + config_files[x]) < 0)
			pDebug(DBG_ERR, "ERROR Could not open config file %s: %s",
			       (config_dir + config_files[x]).c_str(), strerror(errno));
	assignConfigEntries();
	initSchemes();
	if (!scheme_configs) {
		scheme_configs = new SchemeConfig("", *this);
		current_scheme = (SchemeConfig *) scheme_configs;
		Battery_scheme = (SchemeConfig *) scheme_configs;
		AC_scheme = scheme_configs;
		pDebug(DBG_WARN, "no schemes found, applying default scheme");
	}
	if (!current_scheme) {
		current_scheme = scheme_configs;
	}
	if (!AC_scheme) {
		AC_scheme = scheme_configs;
		pDebug(DBG_WARN, "No AC scheme has been declared/found.");
	}
	parseSchemes();
	return 1;
}

int GeneralConfig::initSchemes()
{

	int no = 0;
	DIR *dir;
	struct dirent *dirent;
	string name;
	SchemeConfig *temp;
	int isACScheme;
	int isBatteryScheme;

	/* first delete old scheme configurations */

	if (scheme_configs) {
		SchemeConfig *temp1 = scheme_configs;
		for (temp = scheme_configs; temp != NULL;) {
			temp1 = temp;
			temp = temp->next;
			delete temp1;
			temp1 = NULL;
		}
		current_scheme = NULL;
		Battery_scheme = NULL;
		AC_scheme = NULL;
		scheme_configs = NULL;
	}

	dir = opendir(scheme_dir.c_str());
	if (!dir) {
		pDebug(DBG_ERR, "Cannot open scheme dir(%s): %s", scheme_dir.c_str(), strerror(errno));
		return -1;
	}

	/* scan all the battery directories (e.g. BAT0, ...) */
	/* expect that dirent will always give back the same order 
	   of files(directories) */
	while ((dirent = readdir(dir)) != NULL) {
		isACScheme = isBatteryScheme = 0;
		name = string(dirent->d_name);
		if (name.find("scheme_") != 0)
			continue;	/* skip files not beginning with scheme_ */
		/* skip files ending on ~ */
		if (name.at(name.length() - 1) == '~')
			continue;
		/* skip files ending on .rpmnew or .rpmsave */
		if (name.substr(name.length() - 7) == ".rpmnew" || 
		  	name.substr(name.length() - 8) == ".rpmsave")
			continue;

		if (name.substr(7) == BATTERY_SCHEME) {
			pDebug(DBG_INFO, "Default Battery scheme is %s", dirent->d_name);
			isBatteryScheme = 1;
		}
		if (name.substr(7) == AC_SCHEME) {
			pDebug(DBG_INFO, "Default AC scheme is %s", dirent->d_name);
			isACScheme = 1;
		}

		if (no == 0) {
			scheme_configs = new SchemeConfig(dirent->d_name, *this);
			scheme_configs->next = NULL;
			if (isACScheme)
				AC_scheme = scheme_configs;
			if (isBatteryScheme)
				Battery_scheme = scheme_configs;
		} else {
			if (scheme_configs) {
				for (temp = scheme_configs; temp->next != NULL; temp = temp->next) ;
				temp->next = new SchemeConfig(dirent->d_name, *this);
				temp->next->next = NULL;
				if (isACScheme)
					AC_scheme = temp->next;
				if (isBatteryScheme)
					Battery_scheme = temp->next;
			}
		}

		no++;
		pDebug(DBG_INFO, "Scheme %s found.", dirent->d_name);
	}
	closedir(dir);
	if (!current_scheme)
		current_scheme = AC_scheme;
	return no;
}

bool GeneralConfig::parseSchemes()
{
	SchemeConfig *temp;
	bool ret = true;
	typedef std::list<std::string> StringList;
	StringList mySchemeNameList;

	if (!scheme_configs)
		return false;
	for (temp = scheme_configs; temp != NULL; temp = temp->next) {
		if (temp->readConfigFile() < 0) {
			// should never happen
			pDebug(DBG_WARN, "Could not read scheme config file");
			ret = false;
		} else {
			temp->assignConfigEntries();

			/* check if a equally named scheme was already parsed */
			for (StringList::iterator myiter=mySchemeNameList.begin();
				myiter != mySchemeNameList.end() ; myiter++) {
				if (*myiter == temp->SCHEME_NAME) {
					pDebug(DBG_WARN, "config file %s overrides previous %s scheme configuration", 
						(temp->file_name).c_str(), (temp->SCHEME_NAME).c_str());
					break;
				}
			}
			mySchemeNameList.push_back(temp->SCHEME_NAME);
		}
#ifdef CREATE_DEBUG_OUTPUT
		cout << *temp;
#endif
	}
#ifdef CREATE_DEBUG_OUTPUT
	cout << *this;
#endif
	return ret;
}

/* Be careful, the name of the file must be unique! */
SchemeConfig *GeneralConfig::getScheme(const string &scheme_name)
{

	SchemeConfig *temp;
	if (scheme_configs) {
		for (temp = scheme_configs; temp != NULL; temp = temp->next) {
			if (temp->SCHEME_NAME == scheme_name)
				return temp;
		}
	}
	return NULL;
}

/* use for iteration to get all schemes...
   returns NULL if scheme at position no does not exist */
SchemeConfig *GeneralConfig::getSchemeByNumber(int no)
{

	SchemeConfig *temp;
	int x;
	if (scheme_configs) {
		for (x = 0, temp = scheme_configs; temp != NULL && x < no; temp = temp->next, x++) ;
		if (x == no)
			return temp;
		else
			return NULL;
	} else
		return NULL;
}

int GeneralConfig::setActiveScheme(const string &scheme_name)
{

	SchemeConfig *temp;
	int isSet = 0;
	for (temp = scheme_configs; temp != NULL; temp = temp->next) {
		if (!temp->SCHEME_NAME.compare(scheme_name)) {
			if (temp == current_scheme) {
				return 0;
			}
			current_scheme = temp;
			isSet = 1;
			break;
		}
	}
	if (!isSet) {
		return -1;
	}
	return 1;
}

string GeneralConfig::getBatteryScheme()
{

	if (Battery_scheme == NULL) {
		pDebug(DBG_ERR, "No battery scheme defined");
		return "";
	} else
		return Battery_scheme->SCHEME_NAME;
}

string GeneralConfig::getACScheme()
{

	if (AC_scheme == NULL) {
		pDebug(DBG_ERR, "No AC scheme defined");
		return "";
	} else
		return AC_scheme->SCHEME_NAME;
}

#ifdef CREATE_DEBUG_OUTPUT

ostream & operator<<(ostream & os, const GeneralConfig & gc)
{
	string ret;
	int x;

	os << endl << endl << "GENERAL CONFIGURATIONS:";
	os << endl << "Force CPU frequency scaling to be disabled: " << (gc.disable_CPU_freq == 1 ? "yes" : "no");
	os << endl << "gc.AC_SCHEME: " << ((gc.AC_scheme == NULL) ? "Not available" : gc.AC_scheme->SCHEME_NAME);
	os << endl << "gc.BATTERY_SCHEME: " << ((gc.Battery_scheme == NULL) ? "Not available" : gc.Battery_scheme->
						SCHEME_NAME);
	os << endl << "gc.CURRENT_SCHEME: " << ((gc.current_scheme == NULL) ? "Not available" : gc.current_scheme->
						SCHEME_NAME);
	os << endl << endl << "Config files:" << endl;

	for (x = 0; x < (int)(sizeof(gc.config_files) / sizeof(string)); x++)
		os << gc.config_dir << gc.config_files[x] << endl;

	os << endl << endl;

	return os;
}

#endif
