/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef CPUFREQ_MANAGEMENT_H
#define CPUFREQ_MANAGEMENT_H

#include <list>
#include "cpufreq_interface.h"

/** @brief class caring about all cpus */
class CpufreqManagement {
public:
	/** @brief constructor */
	CpufreqManagement();

	/** @brief destructor */
	~CpufreqManagement();

	/** @brief creates the cpufreq objects for each base cpu */
	void init();

	/** @brief This can be called outside after calling createCPUFreqObject()
	 *  
	 *  @return true if supported
	 */
	bool isSupported();

	/** @brief adjusts the speeds of all CPUs to their load (if in
	 *         dynamic mode)
	 */
	void adjustSpeeds();

	/** @brief set cpufreq policy for all CPUs
	 *
	 * @param mode the new mode to set
	 * @param eM object to throw apporpriate events
	 */
	int setModes(CPUFREQ_MODE mode, EventManagement *eM);

	/** @brief get the current cpu policy
	 *
	 * @return the current cpu policy
	 */
	CPUFREQ_MODE getMode();

	/** @brief This function can be used if the userspace class thinks it
	 *  run out of sync, what means it has another frequency stored
	 *  that actually is active.
	 *  IIRC this has been used after suspend, but should be obsolute.
	 *  The same for the according reinitSpeed function
	 *
	 *  @todo: Delete this one if running stable
	*/
	void reinitSpeeds();

	/** @brief Reinitialises speeds. This function should be called if the
	 *  frequencies could change unnoticably (e.g. after supsend, AC events, ...)
	 *  This funcion should only be called if the userspace class is used
	 */
	void rereadFrequencies();

	/** @brief set the governor on all CPUs */
	int setGovernors(const string &new_governor);

	/** @brief set configs of all cpus */
	void setConfigs();

	/** @brief get the current control mode used
	 *
	 * @return @ref CPUFREQ_CONTROL_MODE
	 */
	CPUFREQ_CONTROL_MODE controlMode();

	/** @brief enables a CPU
	 *
	 * @param cpu CPU to disable starting from 0
	 * @return @ref CPUFreq_Interface::enable()
	 */
	int enableCPU(int cpu);

	/** @brief disables a CPU
	 *
	 * @param cpu CPU to disable starting from 0
	 * @return @ref CPUFreq_Interface::enable()
	 */	
	int disableCPU(int cpu);
	
	/** @brief check if all cpufreq objects contain a control mode */
	bool hasControlMode(CPUFREQ_CONTROL_MODE control_mode);

#ifdef CPUFREQ_MEASURE
	/** @brief append measure output to /tmp/cpufreq.log statically for now */
	void startMeasures();
	/** @brief append measure output to /tmp/cpufreq.log statically for now */
	void stopMeasures();
#endif

private:
	/** @brief inits one single cpufreq interface
	 *
	 * this function tries to initialize a cpufreq interface, either
	 * kernel or userspace. If kernel does not work, it falls back to
	 * a userspace object
	 *
	 * @param cores list of cores this interface has to care for
	 * @param control_mode @ref CPUFREQ_CONTROL_MODE
	 *
	 * @return false if neither interface could be set up
	 */
	bool initSingleInterface(std::list< int > &cores, CPUFREQ_CONTROL_MODE control_mode);

	/** @brief get cpu cores which belong together via the affectec_cpus file
	 *
	 * @param cpu_list list containing list of cpus
	 * @param num_cpus number of cores in system
	 *
	 * the given list is something like
	 *
	 *   sockel 0 -------> core 0
	 *              |
	 *              |----> core 1
	 *              |
	 *             ...
	 *   sockel 1 -------> core 0
	 *              |
	 *              |----> core 1
	 *              |
	 *             ...
	 *
	 * @return true if dependencies could be fugured out, false otherwise
	 */
	bool getCoresAffectedCpus(std::list< std::list< int> > &cpu_list, int num_cpus);
	
	/** @brief get cpucores which belong together via parsing /proc/cpuinfi
	 *
	 * this function is kept to get cpufreq and multicores working with older
	 * kernels. It should not be needed with a kernel > 2.6.13
	 *
     	 * @param cpu_list stl list containing list of cpus
	 * @param num_cpus number of cores in system
	 *
	 * @return true if dependencies could be fugured out, false otherwise
	 */
	bool getCoresProcfile(list< list< int > > & cpu_list, int num_cpus);

	/** @brief split up the given cpu in its siblings
	 *
	 * @param core_list stl list of cores which gets filled
	 * @param cpu cpu number starting with 0
	 *
	 * @return false on error, true on succes
	 */
	bool readCpu(std::list< int > & core_list, int cpu);

	/** @brief check whether the given list contains the given core
	 *
	 * @param core_list list holding alrady added cores
	 * @param core core to check whether it is in core_list
	 *
	 * @return true if already inside core_list, false otherwise
	 */
	bool siblingExists(list< int > core_list, int core);

	/** @brief checks CPU hotplugging depending on the scheme */
	void checkCPUHotplug();

	/** @brief stl list holding all cpufreq objects */
	std::list< CPUFreq_Interface* > _cpufreq_objects;

	/** @brief stl list holding all cpus */
	std::list< std::list< int > > _cpus;

	/** @brief bitmask containin control modes used */
	int _control_modes;

	/** @brief default control mode
	 *
	 * needed to check whether we automatically switched to userspace
	 * if kernel does not work. On scheme switch, we need this as
	 * reference whether the settings for the control mode changed.
	 */
	CPUFREQ_CONTROL_MODE _initial_control_mode;

	/** @brief number of CPUs on startup */
	int _initial_cpu_count;
};


#endif // CPUFREQ_MANAGEMENT_H
