//
// FTPStreamFactory.h
//
// $Id: //poco/1.2/Net/include/Poco/Net/FTPStreamFactory.h#1 $
//
// Library: Net
// Package: FTP
// Module:  FTPStreamFactory
//
// Definition of the FTPStreamFactory class.
//
// Copyright (c) 2005-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Net_FTPStreamFactory_INCLUDED
#define Net_FTPStreamFactory_INCLUDED


#include "Poco/Net/Net.h"
#include "Poco/Net/HTTPSession.h"
#include "Poco/URIStreamFactory.h"


namespace Poco {
namespace Net {


class Net_API FTPPasswordProvider
	/// The base class for all password providers.
	/// An instance of a subclass of this class can be
	/// registered with the FTPStreamFactory to 
	/// provide a password
{
public:
	virtual std::string password(const std::string& username, const std::string& host) = 0;
		/// Provide the password for the given user on the given host.

protected:
	FTPPasswordProvider();
	virtual ~FTPPasswordProvider();
};


class Net_API FTPStreamFactory: public Poco::URIStreamFactory
	/// An implementation of the URIStreamFactory interface
	/// that handles File Transfer Protocol (ftp) URIs.
	///
	/// The URI's path may end with an optional type specification
	/// in the form (;type=<typecode>), where <typecode> is
	/// one of a, i or d. If type=a, the file identified by the path
	/// is transferred in ASCII (text) mode. If type=i, the file
	/// is transferred in Image (binary) mode. If type=d, a directory
	/// listing (in NLST format) is returned. This corresponds with
	/// the FTP URL format specified in RFC 1738.
	///
	/// If the URI does not contain a username and password, the
	/// username "anonymous" and the password "
{
public:
	FTPStreamFactory();
		/// Creates the FTPStreamFactory.

	~FTPStreamFactory();
		/// Destroys the FTPStreamFactory.
		
	std::istream* open(const Poco::URI& uri);
		/// Creates and opens a HTTP stream for the given URI.
		/// The URI must be a ftp://... URI.
		///
		/// Throws a NetException if anything goes wrong.
		
	static void setAnonymousPassword(const std::string& password);
		/// Sets the password used for anonymous FTP.
		///
		/// WARNING: Setting the anonymous password is not
		/// thread-safe, so it's best to call this method
		/// during application initialization, before the
		/// FTPStreamFactory is used for the first time.
		
	static const std::string& getAnonymousPassword();
		/// Returns the password used for anonymous FTP.
		
	static void setPasswordProvider(FTPPasswordProvider* pProvider);
		/// Sets the FTPPasswordProvider. If NULL is given,
		/// no password provider is used.
		///
		/// WARNING: Setting the password provider is not
		/// thread-safe, so it's best to call this method
		/// during application initialization, before the
		/// FTPStreamFactory is used for the first time.
		
	static FTPPasswordProvider* getPasswordProvider();
		/// Returns the FTPPasswordProvider currently in use,
		/// or NULL if no one has been set.

	static void registerFactory();
		/// Registers the FTPStreamFactory with the
		/// default URIStreamOpener instance.

protected:
	static void splitUserInfo(const std::string& userInfo, std::string& username, std::string& password);
	static void getUserInfo(const Poco::URI& uri, std::string& username, std::string& password);
	static void getPathAndType(const Poco::URI& uri, std::string& path, char& type);
	
private:
	static std::string          _anonymousPassword;
	static FTPPasswordProvider* _pPasswordProvider;
};


} } // namespace Poco::Net


#endif // Net_FTPStreamFactory_INCLUDED
