/**
 *
 * @file     player.c
 * @brief    Player interface
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Wed Nov 23, 2005 00:37
 *
 * Copyright (C) 2005, 2006 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>

#include "player.h"

#include "interface.h"
#include "plugins.h"
#include "preferences.h"

#include <dirent.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>

/* Mazimum path size */
enum { MAX_PATH = 4096 };

/* Maximum number of plugins */
enum { MAX_PLUGINS = 25 };

/* Internal plugin structure */
typedef struct
{
  gchar file_name[MAX_PATH];    /* Plugin file name  */
  void *dl_handle;              /* Dynamic library handle */
  gchar *description;           /* Plugin short description (also in plugin) */
  pg_plugin_t *plugin;          /* External plugin structure */
} pg_in_plugin_t;

struct pg_player_t
{
  gint count;                   /* Number of available plugins */
  pg_in_plugin_t *current;      /* Current loaded plugin */
  pg_in_plugin_t *plugins[MAX_PLUGINS]; /* List of available player plugins */
};

/* Static functions definition */

static void plugins_init (pg_player_t *player);
static gint plugin_search (pg_player_t *player, gchar const *file_name);
static pg_plugin_t* current_plugin (pg_player_t *player);
static pg_in_plugin_t* current_in_plugin (pg_player_t *player);

/* Static variables definition */

static gchar *plugin_public_sym = "playground_plugin_info";

/* Module functions definition */

pg_player_t*
player_create (void)
{
  pg_player_t *player = g_new0 (pg_player_t, 1);
  if (player)
    {
      plugins_init (player);
    }
  return player;
}

void
player_destroy (pg_player_t *player)
{
  if (player)
    {
      gint i = 0;

      player_plugin_unload (player);
      for (i = 0; i < player->count; i++)
        {
          g_free (player->plugins[i]);
        }
      g_free (player);
    }
}

gint
player_plugin_count (pg_player_t *player)
{
  return player == NULL ? 0 : player->count;
}

gchar*
player_plugin_description (pg_player_t *player, gint pos)
{
  if ((player == NULL) || (pos >= player->count))
    {
      return NULL;
    }

  return player->plugins[pos]->description;
}

gchar*
player_plugin_file (pg_player_t *player, gint pos)
{
  if ((player == NULL) || (pos >= player->count))
    {
      return NULL;
    }

  return player->plugins[pos]->file_name;
}

gboolean
player_plugin_load (pg_player_t *player, gchar const *plugin_file)
{
  gint pos = -1;
  void *hnd = NULL;
  gchar file_name[MAX_PATH] = "";
  pg_in_plugin_t *plugin = NULL;
  pg_plugin_t* (*plugin_info)(void) = NULL;

  if ((player == NULL) || (pos >= player->count))
    {
      return FALSE;
    }

  pos = plugin_search (player, plugin_file);
  plugin = player->plugins[pos];

  if (plugin->dl_handle != NULL)
    {
      /* Plugin already loaded */
      return TRUE;
    }
  else
    {
      /* Unload current plugin */
      player_plugin_unload (player);
    }

  sprintf (file_name, "%s/%s", PG_PLUGINSDIR, plugin_file);

  hnd = plugin->dl_handle = dlopen (file_name, RTLD_NOW);
  if (hnd == NULL)
    {
      return FALSE;
    }

  /* Search for public known symbol in dynamic library */
  plugin_info = dlsym (hnd, plugin_public_sym);
  if (plugin_info == NULL)
    {
      return FALSE;
    }

  plugin->plugin = plugin_info ();
  if (plugin->plugin == NULL)
    {
      return FALSE;
    }
  plugin->plugin->plugin_init ();

  /* Set current plugin */
  player->current = plugin;

  /* Obey preferences */
  preferences_startup ();

  return TRUE;
}

void
player_plugin_unload (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  pg_in_plugin_t *in_plugin = current_in_plugin (player);

  if (plugin)
    {
      plugin->plugin_close ();
      plugin->player_quit ();
      /* Let player time to close */
      sleep (1);
    }

  if (in_plugin && in_plugin->dl_handle)
    {
      dlclose (in_plugin->dl_handle);
      in_plugin->dl_handle = NULL;
    }

  player->current = NULL;
}

void
player_plugin_reload_list (pg_player_t *player)
{
  if (player == NULL)
    {
      return;
    }
  plugins_init (player);
}

void
player_start (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  if (!plugin->player_start ())
    {
      enum { MAX_BUFFER = 512 };
      gchar str[MAX_BUFFER] = "";

      sprintf (str, _("Unable to launch %s player."), plugin->short_desc);
      gui_show_error (str);
    }
}

void
player_quit  (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
     }
  plugin->player_quit ();
}

void
player_prev (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->prev_song ();
}

void
player_play_pause (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }

  if (!plugin->is_running ())
    {
      plugin->player_start ();
    }

  plugin->play_pause ();
}

void
player_stop (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->stop ();
}

void
player_next (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->next_song ();
}

void
player_eject (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->eject ();
}

void
player_seek (pg_player_t *player, gdouble t)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->seek (t);
}

void
player_repeat (pg_player_t *player, gboolean active)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->repeat (active);
}

void
player_shuffle (pg_player_t *player, gboolean active)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->shuffle (active);
}

void
player_main_win (pg_player_t *player, gboolean toggle)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->main_win (toggle);
}

void
player_playlist_win (pg_player_t *player, gboolean toggle)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->playlist_win (toggle);
}

void
player_show_hide (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->show_hide ();
}

void
player_show_all (pg_player_t *player, gboolean show)
{
  player_main_win (player, show);
  player_playlist_win (player, show);
}

void
player_preferences (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->preferences ();
}

void
player_lower_volume (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->lower_volume ();
}

void
player_raise_volume (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return;
    }
  plugin->raise_volume ();
}

gboolean
player_is_running (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_running ();
}

gboolean
player_is_playing (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_playing ();
}

gboolean
player_is_paused (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_paused ();
}

gboolean
player_is_repeat (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_repeat ();
}

gboolean
player_is_shuffle (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_shuffle ();
}

gboolean
player_is_main_win (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_main_win ();
}

gboolean
player_is_playlist_win (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return FALSE;
    }
  return plugin->is_playlist_win ();
}

gchar*
player_track_name (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return NULL;
    }
  return plugin->track_name ();
}

gchar*
player_track_album (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return NULL;
    }
  return plugin->track_album ();
}

gchar*
player_track_artist (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return NULL;
    }
  return plugin->track_artist ();
}

gint
player_track_current_time (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return -1;
    }
  return plugin->track_current_time ();
}

gint
player_track_total_time (pg_player_t *player)
{
  pg_plugin_t *plugin = current_plugin (player);
  if (plugin == NULL)
    {
      return -1;
    }
  return plugin->track_total_time ();
}

/* Static functions definition */

static void
plugins_init (pg_player_t *player)
{
  static char const *so_ext = ".so";
  DIR *dir = NULL;
  struct dirent *next = NULL;

  if (player == NULL)
    {
      return;
    }

  dir = opendir (PG_PLUGINSDIR);
  if (dir == NULL)
    {
      gchar str_error[MAX_PATH] = "";
      sprintf (str_error, "Unable to open plugins directory (%s).",
               PG_PLUGINSDIR);
      gui_show_error (str_error);
      return;
    }

  next = readdir (dir);
  while (next)
    {
      gchar file_name[MAX_PATH] = "";

      sprintf (file_name, "%s/%s", PG_PLUGINSDIR, next->d_name);
      if (strstr (file_name, so_ext))
        {
          /* Make sure we only read non-versioned libraries */
          if (!strcmp (strstr (file_name, so_ext), so_ext))
            {
              void *hnd = dlopen (file_name, RTLD_NOW);
              if (hnd)
                {
                  pg_plugin_t* (*plugin_info)(void) = NULL;
                  plugin_info = dlsym (hnd, plugin_public_sym);
                  if (plugin_info)
                    {
                      /* Append plugin if not present */
                      gint pos = plugin_search (player, next->d_name);
                      if (pos == -1)
                        {
                          /* Get plugin so we can obtain short description */
                          pg_plugin_t *plugin = plugin_info ();

                          pos = player->count;
                          player->plugins[pos] = g_new0 (pg_in_plugin_t, 1);
                          strcpy (player->plugins[pos]->file_name,
                                  next->d_name);
                          player->plugins[pos]->description =
                            g_strdup (plugin->short_desc);
                          player->count++;
                        }
                    }
                  dlclose (hnd);
                }
            }
        }
      next = readdir(dir);
    }
}

static gint
plugin_search (pg_player_t *player, gchar const *file_name)
{
  gint i = 0;
  gint ret = -1;
  for (i = 0; (ret == -1) && (i < player->count); i++)
    {
      if (!strcmp (player->plugins[i]->file_name, file_name))
        {
          ret = i;
        }
    }

  return ret;
}

static pg_plugin_t*
current_plugin (pg_player_t *player)
{
  if ((player == NULL) || (player->current == NULL))
    {
      return NULL;
    }

  return player->current->plugin;
}

static pg_in_plugin_t*
current_in_plugin (pg_player_t *player)
{
  if (player == NULL)
    {
      return NULL;
    }

  return player->current;
}
