/**
 *
 * @file     xmms.c
 * @brief    XMMS control functions
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Thu Aug 07, 2003 21:16
 *
 * Copyright (C) 2005, 2006 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>

#include <playground/plugins.h>

#include <xmms/xmmsctrl.h>

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>


/* Static functions declaration */

static void xmms_init (void);
static void xmms_close (void);
static gboolean xmms_start (void);
static void xmms_quit (void);
static void xmms_prev (void);
static void xmms_play_pause (void);
static void xmms_stop (void);
static void xmms_next (void);
static void xmms_eject (void);
static void xmms_seek (gdouble t);
static void xmms_repeat (gboolean enable);
static void xmms_shuffle (gboolean enable);
static void xmms_main_win (gboolean enable);
static void xmms_pl_win (gboolean enable);
static void xmms_show_hide (void);
static void xmms_preferences (void);
static void xmms_lower_volume (void);
static void xmms_raise_volume (void);
static gint xmms_is_running (void);
static gint xmms_is_playing (void);
static gint xmms_is_paused (void);
static gint xmms_is_repeat (void);
static gint xmms_is_shuffle (void);
static gint xmms_is_main_win (void);
static gint xmms_is_pl_win (void);
static gchar* xmms_track_name (void);
static gchar* xmms_track_album (void);
static gchar* xmms_track_artist (void);
static gdouble xmms_track_rating (void);
static gint xmms_track_current_time (void);
static gint xmms_track_total_time (void);

static pg_plugin_t xmms_info =
  {
    VERSION,
    "0.3",
    "Aleix Conchillo Flaque <aleix@member.fsf.org>",
    "X Multimedia System (XMMS)",
    "",
    xmms_init,
    xmms_close,
    xmms_start,
    xmms_quit,
    xmms_prev,
    xmms_play_pause,
    xmms_stop,
    xmms_next,
    xmms_eject,
    xmms_seek,
    xmms_repeat,
    xmms_shuffle,
    xmms_main_win,
    xmms_pl_win,
    xmms_show_hide,
    xmms_preferences,
    xmms_lower_volume,
    xmms_raise_volume,
    xmms_is_running,
    xmms_is_playing,
    xmms_is_paused,
    xmms_is_repeat,
    xmms_is_shuffle,
    xmms_is_main_win,
    xmms_is_pl_win,
    xmms_track_name,
    xmms_track_album,
    xmms_track_artist,
    xmms_track_rating,
    xmms_track_current_time,
    xmms_track_total_time
  };

/* XMMS session */
static gint session = 0;

/* Module functions definition */

extern pg_plugin_t*
playground_plugin_info (void)
{
  return &xmms_info;
}

/* Static functions definition */

void
xmms_init (void)
{
}

void
xmms_close (void)
{
}

/* This function has been copied from the gkrellmms plugin. */
gboolean
xmms_start (void)
{
  static gchar *exec_cmd = "xmms &";
  gint timer;
  time_t lt;

  if (xmms_is_running ())
    {
      return TRUE;
    }

  system (exec_cmd);

  timer = time (&lt);
  while (!xmms_is_running () && ((time (&lt) - timer) < 4))
    {
      usleep (0);
    }

  return xmms_is_running();
}

void
xmms_quit (void)
{
  if (xmms_is_running ())
    {
      xmms_remote_quit (session);
    }
}

void
xmms_prev (void)
{
  if (xmms_is_running ())
    {
      xmms_remote_playlist_prev (session);
    }
}

void
xmms_play_pause (void)
{
  if (xmms_is_running ())
    {
      if (xmms_is_playing ()
          && !xmms_is_paused ())
        {
          xmms_remote_pause (session);
        }
      else
        {
          xmms_remote_play (session);
        }
    }
}

void
xmms_stop (void)
{
  if (xmms_is_running ())
    {
      xmms_remote_stop (session);
    }
}

void
xmms_next (void)
{
  if (xmms_is_running())
    {
      xmms_remote_playlist_next(session);
    }
}

void
xmms_eject (void)
{
  if (xmms_is_running ())
    {
      xmms_remote_eject (session);
    }
}

void
xmms_seek (gdouble t)
{
  if (xmms_is_running ())
    {
      gint total_time = xmms_track_total_time ();
      gint new_time = ceil (total_time * t) - 1;
      xmms_remote_jump_to_time (session, new_time);
    }
}

void
xmms_repeat (gboolean enable)
{
  if (xmms_is_running () && (xmms_is_repeat () != enable))
    {
      xmms_remote_toggle_repeat (session);
    }
}

void
xmms_shuffle (gboolean enable)
{
  if (xmms_is_running () && (xmms_is_shuffle () != enable))
    {
      xmms_remote_toggle_shuffle (session);
    }
}

void
xmms_main_win (gboolean enable)
{
  if (xmms_is_running ())
    {
      xmms_remote_main_win_toggle (session, enable);
    }
}

void
xmms_pl_win (gboolean enable)
{
  if (xmms_is_running ())
    {
      xmms_remote_pl_win_toggle (session, enable);
    }
}

void
xmms_show_hide (void)
{
  if (xmms_is_running ())
    {
      gint show_hide = !(xmms_is_main_win()
                        | xmms_is_pl_win());

      xmms_main_win (show_hide);
      xmms_pl_win (show_hide);
    }
}

void
xmms_preferences (void)
{
  if (xmms_is_running ())
    {
      xmms_remote_show_prefs_box (session);
    }
}

void
xmms_lower_volume (void)
{
  if (xmms_is_running())
    {
      gint new_volume = xmms_remote_get_main_volume (session) - 3;
      if(new_volume < 0)
        {
          new_volume = 0;
        }
      xmms_remote_set_main_volume (session, new_volume);
    }
}

void
xmms_raise_volume (void)
{
  if (xmms_is_running ())
    {
      gint new_volume = xmms_remote_get_main_volume (session) + 3;
      if(new_volume < 0)
        {
          new_volume = 0;
        }
      xmms_remote_set_main_volume (session, new_volume);
    }
}

gint
xmms_is_running (void)
{
  return xmms_remote_is_running (session);
}

gint
xmms_is_playing (void)
{
  return xmms_remote_is_playing (session);
}

gint
xmms_is_paused (void)
{
  return xmms_remote_is_paused (session);
}

gint
xmms_is_repeat (void)
{
  return xmms_remote_is_repeat (session);
}

gint
xmms_is_shuffle (void)
{
  return xmms_remote_is_shuffle (session);
}

gint
xmms_is_main_win (void)
{
  return xmms_remote_is_main_win (session);
}

gint
xmms_is_pl_win (void)
{
  return xmms_remote_is_pl_win (session);
}

gchar*
xmms_track_name (void)
{
  gint pos = xmms_remote_get_playlist_pos (session);
  return xmms_remote_get_playlist_title (session, pos);
}

gchar*
xmms_track_album (void)
{
  return NULL;
}

gchar*
xmms_track_artist (void)
{
  return NULL;
}

gdouble
xmms_track_rating (void)
{
  return 0.0;
}

gint
xmms_track_current_time (void)
{
  return xmms_remote_get_output_time (session);
}

gint
xmms_track_total_time (void)
{
  gint pos = xmms_remote_get_playlist_pos (session);
  return xmms_remote_get_playlist_time (session, pos);
}
