# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/

"""
Store internally as a string.
Provide validation routines.
"""

#---import modules

#standard library
import glob, os, textwrap, types

if '_' not in dir():
    _ = str

#gui independent (core.lib)
from odict import odict as Fields

NO_FIELDS               = Fields()
_t                      = unicode
#---image
ALIGN_HORIZONTAL        = [_t('left'),_t('center'),_t('right')]
ALIGN_VERTICAL          = [_t('top'),_t('middle'),_t('bottom')]

FONT_EXTENSIONS         = ['ttf','otf','ttc']

IMAGE_EXTENSIONS        = ['bmp','dib','gif','jpe','jpeg','jpg','im','msp',
                        'pcx','png','pbm','pgm','ppm','tif','tiff','xbm']
IMAGE_READ_EXTENSIONS   = IMAGE_EXTENSIONS + ['cur','dcx','fli','flc','fpx',
                        'gbr','gd','ico','imt','mic','mcidas','pcd',
                        'psd','bw','rgb','cmyk','sun','tga','xpm']
IMAGE_READ_EXTENSIONS.sort()
IMAGE_READ_MIMETYPES    = ['image/'+ext for ext in IMAGE_READ_EXTENSIONS]
IMAGE_WRITE_EXTENSIONS  = IMAGE_EXTENSIONS + ['eps','ps','pdf']
IMAGE_WRITE_EXTENSIONS.sort()
IMAGE_MODES             = [_t('Monochrome (1-bit pixels, black and white)'),
                        _t('Grayscale (8-bit pixels, black and white)'),
                        _t('RGB (3x8-bit pixels, true colour)'),
                        _t('RGBA (4x8-bit pixels, RGB with transparency mask)'),
                        _t('CMYK (4x8-bit pixels, colour separation)'),
                        _t('P (8-bit pixels, mapped using a colour palette)'),
                        _t('YCbCr (3x8-bit pixels, colour video format)'),
                        _t('I (32-bit integer pixels)'),
                        _t('F (32-bit floating point pixels)')]
IMAGE_EFFECTS           = [_t('blur'), _t('contour'), _t('detail'), 
                            _t('edge enhance'), _t('edge enhance more'), 
                            _t('emboss'), _t('find edges'), _t('smooth'), 
                            _t('smooth more'), _t('sharpen')]
IMAGE_FILTERS           = [_t('nearest'),_t('bilinear'),_t('bicubic')]
IMAGE_RESAMPLE_FILTERS  = IMAGE_FILTERS + [_t('antialias')]
IMAGE_TRANSPOSE         = [_t('Rotate 90'), _t('Rotate 180'), _t('Rotate 270'),
                            _t('Flip Left Right'),_t('Flip Top Bottom')]

TEXT_ORIENTATION        = [_t('Normal')] + IMAGE_TRANSPOSE

IMAGE_MODELS_WRITE_EXTENSIONS = ['<type>']+IMAGE_WRITE_EXTENSIONS

IMAGE_READ_EXTENSIONS.sort()
IMAGE_WRITE_EXTENSIONS.sort()

RANK_SIZES              = [3,5]

#---os
def ensure_path(path):
    return _ensure_path(path.rstrip('/').rstrip('\\'))

def _ensure_path(path):
    """Ensure a path exists, create all not existing paths."""
    if not os.path.exists(path):
        parent  = os.path.dirname(path)
        if parent:
            _ensure_path(parent)
            os.mkdir(path)
        else:
            raise OSError, "The path '%s' is not valid."%path

def is_www_file(value):
    return value.startswith('http://') or value.startswith('ftp://')

def is_file(value):
    return os.path.isfile(value) or is_www_file(value)

def files_dictionary(paths,extensions):
    files       = []
    for path in paths:
        for extension in extensions:
            files   += glob.glob(os.path.join(path,'*'+extension))
    d           = {}
    for f in files:
        d[os.path.splitext(os.path.basename(f))[0].replace('_',' ').replace('-',' ').title()]= f
    return d
        
#---form
class Form(object):
    #todo: move this as instance attributes
    dpi     = 'dpi'
    label   = 'label'
    icon    = 'ART_TIP'
    tags    = []
    __doc__ = ''
    
    FILENAME        = '<%s>'%_t('filename')
    FOLDER          = '<%s>'%_t('folder')
    FOLDER_PHATCH   = FOLDER+'_phatch'
    DPI             = '<%s>'%_t('dpi')
    DATE            = '<%s>-<%s>-<%s>'%(_t('year'),_t('month'),_t('day'))
    DATETIME        = DATE+'-<%s>-<%s>-<%s>'%(_t('hour'),_t('minute'),_t('second'))
    ROOT            = '<%s>'%_t('root')
    BYSIZE          = '%s/phatch/<%s>x<%s>'%(ROOT,_t('width'),_t('height'))
    SUBFOLDER       = '<%s>'%_t('subfolder')
    DEFAULT_FOLDER  = '%s/%s'%(FOLDER_PHATCH,SUBFOLDER)

    # choices
    DPIS    = [DPI,DPI+'/2','72','144','300']
    PIXELS  = ['10','25','50','100','200']
    SMALL_PIXELS    = ['1','2','5','10']
    FILENAMES       = [
        FILENAME,
        '%s<###>'%_t('Image'),
        DATETIME,
    ]
    FOLDERS  = [
        '%s/phatch/%s'%(ROOT,SUBFOLDER),
        ROOT+'/phatch/'+DATE.replace('-','/'),
        BYSIZE,
        BYSIZE+'/'+SUBFOLDER,
        DEFAULT_FOLDER,
        FOLDER_PHATCH,
        FOLDER,
    ]
    STAMPS  = [
        'Phatch',
        'Phatch (c)<%s> www.stani.be'%_t('year'),
        DATE,
        DATETIME,
        FILENAME,
        '<%s>'%_t('path'),
    ]
    
    def __init__(self,**options):
        """For the possible options see the source code."""
        fields  = Fields()
        fields['__enabled__']   = BooleanField(True,visible=False)
        self.interface(fields)
        self._fields            = fields
        self._fields.update(options)
        
    def interface(self,fields):
        pass
        
    def __cmp__(self, other):
        label       = _(self.label)
        other_label = _(other_label)
        if label < other_label: return -1
        elif label == other_label: return 0
        else: return 1
        
    def _get_fields(self):
        return self._fields
    
    def get_field_labels(self):
        return self._get_fields().keys()
    
    def _get_field(self,label):
        return self._fields[label]
    
    def get_field(self,label,info={}):
        return self._get_field(label).get(info,label)
    
    def get_fields(self,info,convert=False,pixel_fields={}):
        result  = {}
        for label in self.get_field_labels():
            if label[:2] != '__':
                param = None
                #skip hidden fields such as __enabled__
                if label in pixel_fields:
                    #pixel size -> base, dpi needed
                    param       = pixel_fields[label]
                    if type(param) != types.TupleType:
                        param   = (param,info[self.dpi])
                elif self._get_field(label).__class__ == PixelField:
                    param       = (1,1)
                if param:
                    value       = self.get_field_size(label,info,*param)
                else:
                    #retrieve normal value
                    value       = self.get_field(label,info)
                #convert field labels to function parameters
                if convert:
                    label       = label.lower().replace(' ','_')
                result[label]   = value
        return result
    
    def get_field_size(self,label,info,base,dpi):
        return self._get_field(label).get_size(info,base,dpi,label)
    
    def get_field_filesize(self,label,info,base):
        return self._get_field(label).get_size(info,base,label)
    
    def get_field_string(self,label):
        return self._get_field(label).get_as_string()
    
    def is_enabled(self):
        return self.get_field('__enabled__',None)
    
    def _set_field(self,label,field):
        self._fields[label] = field
        
    def set_field(self,label,value):
        self._get_field(label).set(value)
        return self
        
    def set_fields(self,**options):
        for label, value in options.items():
            self.set_field(label, value)
    
    def set_field_as_string(self,label,value_as_string):
        self._get_field(label).set_as_string(value_as_string)
        return self
        
    def load(self,fields):
        """Load dumped, raw strings."""
        invalid_labels  = []
        for label, value in fields.items():
            if self._fields.has_key(label):
                self.set_field_as_string(label,value)
            else:
                invalid_labels.append(label)
        return invalid_labels
            
    def dump(self):
        """Dump as raw strings"""
        fields_as_strings  = {}
        for label in self.get_field_labels():
            fields_as_strings[label]  = self.get_field_string(label)
        return {'label':self.label,'fields':fields_as_strings}
    
    #tools
    def ensure_path(self,path):
        return ensure_path(path)
    
    def is_www_file(self,path):
        return is_www_file(path)
    
    def is_file(self,path):
        return is_file(path)

    
#---errors
class ValidationError(Exception):
    
    def __init__(self, expected, message, details=None):
        """ValidationError for invalid input.
        
        expected - description of the expected value
        message  - message why validation failed
        details  - eg. which variables are allowed"""
        self.expected       = expected
        self.message        = message
        self.details        = details
        
    def __str__(self):
        return self.message
    
#---field mixins
class PilConstantMixin:
    def to_python(self,x,label):
        return x.upper().replace(' ','_')
    
class TestFieldMixin:
    """ Mixin class, the to_python method should
    
    def to_python(self,x,label,test=False):
        "test parameter to signal test-validate"
        return x
    
    See set_form_field_value in treeEdit.py
    """
        
    def get(self,info=None,label='?',value_as_string=None,test=False):
        """Use this method to test-validate the user input, for example:
            field.get(IMAGE_TEST_INFO, value_as_string, label, test=True)"""
        if value_as_string is None:
            value_as_string = self.value_as_string
        return self.to_python(self.interpolate(value_as_string,info,label),
                label,test)
        
#---fields
class Field(object):
    """Base class for fields. This will be subclassed but, 
    never used directly.
    
    Required to overwrite:
    description - describes the expected value
    
    Optional to overwrite
    to_python   - raise here exceptions in case of validation errors (defaults 
                  to string). 
    to_string   - (defaults to string)
    
    Never overwrite:
    validate    - will work right out of the box as exceptions are raised by
                  the to_python method
    get         - gets the current value as a string
    set         - sets the current value as a string
    
    You can access the value by self.value_as_string
    
    This field interpolates <variables> within a info.
    << or >> will be interpolated as < or >
    """
    
    description             = '<?>'
    
    def __init__(self,value,visible=True):
        self.visible    = visible
        if isinstance(value, (str, unicode)):
            self.set_as_string(value)
        else:
            self.set(value)
        
    def interpolate(self,x,info,label):
        if info == None:
            return self.value_as_string
        else:
            try:
                return x.replace('%','%%')\
                    .replace('<','%(').replace('>',')s')\
                    .replace('%(%(','<').replace(')s)s','>')%info
            except KeyError, details:
                try:
                    variable  = unicode(details)
                except:
                    reason  = '?'
                raise ValidationError(self.description,
                "%s: %s '%s' %s."%(_(label),_("the variable"),
                    variable,_("doesn't exist")),
                    _('Use the Image Inspector to list all the variables.'))

    def to_python(self,x,label):
        return x
    
    def to_string(self,x):
        return unicode(x)
    
    def get_as_string(self):
        """For GUI: Translation, but no interpolation here"""
        return self.value_as_string
    
    def set_as_string(self,x):
        """For GUI: Translation, but no interpolation here"""
        self.value_as_string    = x
        
    def get(self,info=None,label='?',value_as_string=None,test=False):
        """For code: Interpolated, but not translated
        - value_as_string can be optionally provided to test the expression
        
        Ignore test parameter (only for compatiblity with TestField)"""
        if value_as_string is None:
            value_as_string = self.value_as_string
        return self.to_python(self.interpolate(value_as_string,info,label),
                label)
   
    def set(self,x):
        """For code: Interpolated, but not translated"""
        self.value_as_string  = self.to_string(x)
        
    def eval(self,x,label):
        try:
            return eval(x)
        except SyntaxError:
            pass
        except NameError:
            pass
        raise ValidationError(self.description, 
            '%s: %s "%s" %s.'%(_(label),_('invalid syntax'),x,
                _('for integer')))
            
class _CharField(Field):
    description             = _('string')
    
class CharField(Field):
    description             = _('string')
    def __init__(self,value=None,visible=True,choices=None):
        if value is None and choices:
            value           = choices[0]
        if choices is None: choices = []
        super(CharField,self).__init__(value,visible)
        self.choices        = choices
    
class IntegerField(CharField):
    """"""
    description             = _('integer')
    
    def to_python(self,x,label):
        error   = ValidationError(self.description, 
            '%s: %s "%s" %s.'%(_(label),_('invalid literal'),x,
                _('for integer')))
        try:
            return int(round(self.eval(x,label)))
        except ValueError:
            raise error
        except TypeError:
            raise error

class PositiveIntegerField(IntegerField):
    """"""
    description = _('positive integer')

    def to_python(self,x,label):
        value = super(PositiveIntegerField, self).to_python(x,label)
        if value < 0:
            raise ValidationError(self.description,
            '%s: %s "%s" %s.'%(_(label),('the integer value'),x,
                _('is negative, but should be positive')))
        return value
    
class PositiveNonZeroIntegerField(PositiveIntegerField):
    """"""
    
    description = _('positive, non-zero integer')

    def to_python(self,x,label):
        value = super(PositiveNonZeroIntegerField, self).to_python(x,label)
        if value == 0:
            raise ValidationError(self.description,
                '%s: %s "%s" %s.'%(_(label),_('the integer value'),x,
                    _('is zero, but should be non-zero.')))
        return value
                
class DpiField(PositiveNonZeroIntegerField):
    """PIL defines the resolution in two dimensions as a tuple (x,y).
    Phatch ignores this possibility and simplifies by using only one resolution
    """
    
    description = _('resolution')
    
class FloatField(Field):
    description             = _('float')
    
    def to_python(self,x,label):
        try:
            return float(self.eval(x,label))
        except ValueError, message:
            raise ValidationError(self.description, 
            '%s: %s "%s" %s.'%(_(label),_('invalid literal'),x,_('for float')))

class PositiveFloatField(FloatField):
    """"""
    description = _('positive integer')

    def to_python(self,x,label):
        value = super(PositiveFloatField, self).to_python(x,label)
        if value < 0:
            raise ValidationError(self.description,
            '%s: %s "%s" %s.'%(_(label),('the float value'),x,
                _('is negative, but should be positive')))
        return value
    
class PositiveNonZeroFloatField(PositiveIntegerField):
    """"""
    
    description = _('positive, non-zero integer')

    def to_python(self,x,label):
        value = super(PositiveNonZeroIntegerField, self).to_python(x,label)
        if value == 0:
            raise ValidationError(self.description,
                '%s: %s "%s" %s.'%(_(label),_('the float value'),x,
                    _('is zero, but should be non-zero.')))
        return value
                
class BooleanField(Field):
    description             = _('boolean')
    
    def to_string(self,x):
        return super(BooleanField,self).to_string(x).lower()
    
    def to_python(self,x,label):
        if x.lower() in ['1','true','yes']: return True
        if x.lower() in ['0','false','no']: return False
        raise ValidationError(self.description, 
            '%s: %s "%s" %s (%s,%s).'%(_(label),_('invalid literal'), x,
                _('for boolean'),_('true'),_('false')))
    
class ChoiceField(CharField):
    description             = _('choice')
    def __init__(self,value,choices,**keyw):
        super(ChoiceField,self).__init__(value,**keyw)
        self.choices    = choices
        
class ComboField(CharField):
    description             = 'dropdown'
    def __init__(self,value,choices,style=['DROPDOWN'],**keyw):
        "style can consist of 'DROPDOWN','SORT','READONLY','SIMPLE'"
        super(ComboField,self).__init__(value,**keyw)
        self.choices    = choices
        self.style      = style
        
class FileField(CharField):
    extensions  = []
    allow_empty = False
    
    def to_python(self,x,label):
        value   = super(FileField, self).to_python(x,label).strip()
        if not value and self.allow_empty:
            return ''
        ext     = os.path.splitext(value)[-1][1:]
        if self.extensions and not (ext.lower() in self.extensions):
            if ext:
                raise ValidationError(self.description,
                '%s: %s "%s" %s\n\n%s:\n%s.'%(_(label),
                    _('the file extension'),ext,
                    _('is invalid.'),
                    _('You can only use files with the following extensions'),
                    ', '.join(self.extensions)))
            else:
                raise ValidationError(self.description,
                '%s: %s\n%s:\n%s.'%(
                    _(label),
                    _('a filename with a valid extension was expected.'),
                    _('You can only use files with the following extensions'),
                    textwrap.fill(', '.join(self.extensions),70)))
        return value
    
class ReadFileField(TestFieldMixin,FileField):
    """This is a test field to ensure that the file exists.
    It could also have been called the MustExistFileField."""
    
    def to_python(self,x,label,test=False):
        value = super(ReadFileField, self).to_python(x,label)
        if not value.strip() and self.allow_empty:
            return ''
        if (x==value or not test) and (not is_file(value)):
            raise ValidationError(self.description,
            '%s: %s "%s" %s.'%(_(label),_('the filename'),value,
                _('does not exist.')))
        return value
    
class DictionaryReadFileField(ReadFileField):
    dictionary      = None
    
    def init_dictionary(self):
        self.dictionary    = {}

    def to_python(self,x,label,test=False):
        if self.dictionary is None:
            self.init_dictionary()
        try:
            x   = self.dictionary[x]
        except KeyError:
            pass
        return super(DictionaryReadFileField,self).to_python(x,label,test)
            
class FontFileField(DictionaryReadFileField):
    extensions      = FONT_EXTENSIONS
    allow_empty     = True
    
    def init_dictionary(self):
        from fonts import font_dictionary
        self.dictionary    = font_dictionary()
        
class ImageReadFileField(DictionaryReadFileField):
    extensions  = IMAGE_READ_EXTENSIONS
    
class FileNameField(CharField):
    """Without extension"""
    pass
    
class FilePathField(CharField):
    pass
    
class ImageTypeField(ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageTypeField,self).__init__(value,IMAGE_EXTENSIONS,**keyw)
        
    def set_as_string(self,x):
        #ignore translation
        if x and x[0]=='.':
            x = x[1:]
        super(ImageTypeField,self).set_as_string(x)
        
class ImageReadTypeField(ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageReadTypeField,self).__init__(\
            value,IMAGE_READ_EXTENSIONS,**keyw)
        
class ImageWriteTypeField(ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageWriteTypeField,self).__init__(\
            value,IMAGE_MODELS_WRITE_EXTENSIONS,**keyw)
        
class ImageModeField(ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageModeField,self).__init__(value,IMAGE_MODES,**keyw)
        
    def to_python(self,x,label):
        return x.split(' ')[0].replace('Grayscale','L').replace('Monochrome','1')
        
class ImageEffectField(PilConstantMixin,ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageEffectField,self).__init__(\
            value,IMAGE_EFFECTS,**keyw)
            
class ImageFilterField(PilConstantMixin,ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageFilterField,self).__init__(\
            value,IMAGE_FILTERS,**keyw)

class ImageResampleField(PilConstantMixin,ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageResampleField,self).__init__(\
            value,IMAGE_RESAMPLE_FILTERS,**keyw)
            
class ImageTransposeField(PilConstantMixin,ChoiceField):
    def __init__(self,value,**keyw):
        super(ImageTransposeField,self).__init__(\
            value,IMAGE_TRANSPOSE,**keyw)
            
class TextOrientationField(PilConstantMixin,ChoiceField):
    def __init__(self,value,**keyw):
        super(TextOrientationField,self).__init__(\
            value,TEXT_ORIENTATION,**keyw)
            
    def to_python(self,x,label):
        if x == _t('Normal'):
            return None
        return super(TextOrientationField,self).to_python(x,label)
    
class AlignHorizontalField(ChoiceField):
    def __init__(self,value,**keyw):
        super(AlignHorizontalField,self).__init__(\
            value,ALIGN_HORIZONTAL,**keyw)
            
class AlignVerticalField(ChoiceField):
    def __init__(self,value,**keyw):
        super(AlignVerticalField,self).__init__(\
            value,ALIGN_VERTICAL,**keyw)
            
class RankSizeField(IntegerField,ChoiceField):
    def __init__(self,value,**keyw):
        super(RankSizeField,self).__init__(\
            value,RANK_SIZES,**keyw)
            
class PixelField(IntegerField):
    """Can be pixels, cm, inch, %."""
    def get_size(self,info,base,dpi,label,value_as_string=None):
        if value_as_string is None:
            value_as_string = self.value_as_string
        for unit, value in self._units(base,dpi).items():
            value_as_string = value_as_string.replace(unit,value)
        return super(PixelField,self).get(info,label,value_as_string)
        
    def _units(self,base,dpi):
        return {
            'cm'    : '*%f'%(dpi/2.54),
            'mm'    : '*%f'%(dpi/25.4),
            'inch'  : '*%f'%dpi,
            '%'     : '*%f'%(base/100.0),
            'px'    : '',
        }

class FileSizeField(IntegerField):
    """Can be pixels, cm, inch, %."""
    def get_size(self,info,base,label,value_as_string=None):
        if value_as_string is None:
            value_as_string = self.value_as_string
        for unit, value in self._units(base).items():
            value_as_string = value_as_string.replace(unit,value)
        return super(FileSizeField,self).get(info,label,value_as_string)
        
    def _units(self,base):
        return {
            'kb'    : '*1024',
            '%'     : '*%f'%(base/100.0),
            'gb'    : '*1073741824',
            'mb'    : '*1048576',
            'bt'    : '',
        }

class SliderField(IntegerField):
    """A value with boundaries set by a slider."""
    def __init__(self,value,minValue,maxValue,**keyw):
        super(SliderField,self).__init__(value,**keyw)
        self.min    = minValue
        self.max    = maxValue
            
class ColourField(Field):
    pass
    
#todo
##class CommaSeparatedIntegerField(CharField):
##    """Not implemented yet."""
##    pass
##    
##class DateField(Field):
##    """Not implemented yet."""
##    pass
##    
##class DateTimeField(DateField):
##    """Not implemented yet."""
##    pass
##    
##class EmailField(CharField):
##    """Not implemented yet."""
##    pass
##    
##class UrlField(CharField):
##    """Not implemented yet."""
##    pass

#Give Form all the tools
FIELDS = [(name,cls) for name, cls in locals().items() 
            if name[0] != '_' and \
            ((type(cls) == types.TypeType and issubclass(cls,Field)) or\
            type(cls) in [types.StringType,types.UnicodeType,types.ListType,
            types.TupleType])
        ]
            
for name,Field in FIELDS:
    setattr(Form,name,Field)
