
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: logger.c 1416 2006-10-29 08:59:15Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>

#include "heap.h"
#include "i18n.h"
#include "logger.h"

#define DEFAULT_LOGFILE "/tmp/oxine.log"
static char *logfile = NULL;

void
logger_init (const char *filename)
{
    /* We try to open the provided filename. */
    if (filename) {
        logfile = ho_strdup (filename);
        FILE *f = fopen (logfile, "a");
        if (!f) {
            const time_t now = time (NULL);
            struct tm *tm_now = localtime (&now);
            char str_time[64];
            strftime (str_time, 64, "%Y-%m-%d %H:%M:%S", tm_now);
            printf ("%s ERROR: [%15s:%4d] ", str_time, __FILE__, __LINE__);
            printf (_("Could not open '%s': %s!"), logfile, strerror (errno));
            printf ("\n");
            ho_free (logfile);
            logfile = NULL;
        }
    }

    /* If no filename was provided or we could not open the provided filename,
     * we fall back to the default logfile. */
    if (!logfile) {
        if (!filename || (strcmp (filename, DEFAULT_LOGFILE) != 0))
            logger_init (DEFAULT_LOGFILE);
        return;
    }

    /* If there is an old logfile in the same place, we make a backup. */
    if (access (logfile, R_OK) == 0) {
        char *cmd = ho_strdup_printf ("/bin/cp %s %s.0", logfile, logfile);
        system (cmd);
        ho_free (cmd);
    }
}


void
logger_free (void)
{
    ho_free (logfile);
    logfile = NULL;
}


void
_log (int level, const char *file, int line, const char *format, ...)
{
    const char *level_str[] = { "FATAL", "ERROR", "WARN", "INFO", "DEBUG" };

    const time_t now = time (NULL);
    struct tm *tm_now = localtime (&now);
    char str_time[64];
    strftime (str_time, 64, "%Y-%m-%d %H:%M:%S", tm_now);

    char log_format[1024];
    sprintf (log_format, "%s %5s: [%15s:%4d] %s\n",
             str_time, level_str[level], file, line, format);

    va_list args;
    va_start (args, format);
    if (level < LEVEL_INFO) {
        vfprintf (stderr, log_format, args);
    } else {
        vfprintf (stdout, log_format, args);
    }
    va_end (args);

    if (logfile) {
        FILE *f = fopen (logfile, "a");

        if (f) {
            va_start (args, format);
            vfprintf (f, log_format, args);
            va_end (args);
            fclose (f);
        }

        /* We tried to catch any problems with the logfile at the top. If we
         * get problems here we fail quietly and give up. */
        else {
            ho_free (logfile);
            logfile = NULL;
        }
    }
}
