/* OmniVision OV511 Camera-to-USB Bridge Driver
 *
 * Copyright (c) 1999-2006 Mark McClelland <mark@ovcam.org>
 * Original decompression code Copyright 1998-2000 OmniVision Technologies
 * Many improvements by Bret Wallach <bwallac1@san.rr.com>
 * Color fixes by by Orion Sky Lawlor <olawlor@acm.org> (2/26/2000)
 * Snapshot code by Kevin Moore
 * OV7620 fixes by Charl P. Botha <cpbotha@ieee.org>
 * Changes by Claudio Matsuoka <claudio@conectiva.com>
 * Kernel I2C interface improvements by Kysti Mlkki
 * URB error messages and device_hint from pwc driver by Nemosoft
 * Small V4L pieces from bttv by Ralph & Marcus Metzler, Gerd Knorr, and
 *  Justin Schoeman
 * Memory management (rvmalloc) code from bttv driver, by Gerd Knorr and others
 *
 * Based on the Linux CPiA driver written by Peter Pregler,
 * Scott J. Bertin and Johannes Erdfelt.
 * 
 * Please see the file: Documentation/usb/ov511.txt 
 * and the website at:  http://ovcam.org/ov511
 * for more info.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version. NO WARRANTY OF ANY KIND is expressed or implied.
 */

#include <linux/config.h>
#include <linux/version.h>

/* 2.6 Doesn't support /proc/video, but this is still defined somewhere */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
#  undef CONFIG_VIDEO_PROC_FS
#endif

#if defined(CONFIG_VIDEO_PROC_FS)
#  include <asm/io.h>
#endif
#include <asm/semaphore.h>
#include <asm/processor.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#if defined(CONFIG_VIDEO_PROC_FS)
#  include <linux/fs.h>
#  include <linux/proc_fs.h>
#endif
#include <linux/ctype.h>
#include <linux/pagemap.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 68)
#  include <linux/wrapper.h>
#endif
#include <linux/video_decoder.h>
#include <linux/mm.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
#  include <linux/device.h>
#endif

#include "ov511.h"
#include "id.h"
#include "driver_version.h"

#if defined(OUTSIDE_KERNEL)
#  include "tuner.h"
#else
#  if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
#    include "../media/video/tuner.h"
#  else
#    include <media/tuner.h>
#  endif
#endif

/* Driver will compile with older kernels, but will oops on open() */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 19)
#  error "Kernel 2.4.19 is the minimum for this driver"
#endif

/*
 * Version Information
 */
#if 0	/* Version is in driver_version.h */
#define DRIVER_VERSION "vX.XX"
#define DRIVER_VERSION_CODE KERNEL_VERSION(X,XX,0)
#endif
#define EMAIL "mark@ovcam.org"
#define DRIVER_AUTHOR "Mark McClelland <mark@ovcam.org> & Bret Wallach \
	& Orion Sky Lawlor <olawlor@acm.org> & Kevin Moore & Charl P. Botha \
	<cpbotha@ieee.org> & Claudio Matsuoka <claudio@conectiva.com>"
#define DRIVER_DESC "ov511 USB Camera Driver"

#define OV511_I2C_RETRIES    6
#define OV511_MAX_UNIT_VIDEO 16
#define OV511_MAX_DEV_HINTS  16

/* Pixel count * bytes per YUV420 pixel (1.5) */
#define MAX_FRAME_SIZE(w, h) ((w) * (h) * 3 / 2)

#define MAX_DATA_SIZE(w, h) (MAX_FRAME_SIZE(w, h) + sizeof(struct timeval))

/* Max size * bytes per YUV420 pixel (1.5) + one extra isoc frame for safety */
#define MAX_RAW_DATA_SIZE(w, h) ((w) * (h) * 3 / 2 + 1024)

#define FATAL_ERROR(rc) ((rc) < 0 && (rc) != -EPERM)

#if defined (HAVE_V4L2)
/* No private controls defined yet */
#define OV51X_CID_LASTP1     (V4L2_CID_PRIVATE_BASE + 0)

/* CID offsets for various components */
#define OV51X_CID_OFFSET    0
#define SENSOR_CID_OFFSET   (OV51X_CID_OFFSET + \
                             (OVCAMCHIP_V4L2_CID_LASTP1 - \
                              V4L2_CID_PRIVATE_BASE))
#endif

/**********************************************************************
 * Module Parameters
 * (See ov511.txt for detailed descriptions of these)
 **********************************************************************/

/* These variables (and all static globals) default to zero */
static int autobright		= 1;
static int autoexp		= 1;
#ifdef OV511_DEBUG
static int debug;
int ov511_debug		= 0; /* Same as debug; for external compilation units */
#endif
static int snapshot;
static int cams			= 1;
static int compress;
static int dumppix;
static int led 			= 1;
static int dump_bridge;
static int printph;
static int phy			= 0x1f;
static int phuv			= 0x05;
static int pvy			= 0x06;
static int pvuv			= 0x06;
static int qhy			= 0x14;
static int qhuv			= 0x03;
static int qvy			= 0x04;
static int qvuv			= 0x04;
static int lightfreq;
static int bandingfilter;
static int clockdiv		= -1;
static int packetsize		= -1;
static int framedrop		= -1;
static int fastset;
static int force_palette;
static int tuner		= -1;
static int backlight;
static int unit_video[OV511_MAX_UNIT_VIDEO];
static int remove_zeros;
static int mirror;
int ov518_color 		= 1;
static int i2c_clockdiv		= 1;
static char *dev_hint[OV511_MAX_DEV_HINTS] = { };
#if defined (HAVE_V4L2)
static int v4l2;
#endif

module_param(autobright, int, 0);
MODULE_PARM_DESC(autobright, "Sensor automatically changes brightness");
module_param(autoexp, int, 0);
MODULE_PARM_DESC(autoexp, "Sensor automatically changes exposure");
#ifdef OV511_DEBUG
module_param(debug, int, 0);
MODULE_PARM_DESC(debug,
  "Debug level: 0=none, 1=inits, 2=warning, 3=config, 4=functions, 5=max");
#endif
module_param(snapshot, int, 0);
MODULE_PARM_DESC(snapshot, "Enable snapshot mode");
module_param(cams, int, 0);
MODULE_PARM_DESC(cams, "Number of simultaneous cameras");
module_param(compress, int, 0);
MODULE_PARM_DESC(compress, "Turn on compression");
module_param(dumppix, int, 0);
MODULE_PARM_DESC(dumppix, "Dump raw pixel data");
module_param(led, int, 0);
MODULE_PARM_DESC(led,
  "LED policy (OV511+ or later). 0=off, 1=on (default), 2=auto (on when open)");
module_param(dump_bridge, int, 0);
MODULE_PARM_DESC(dump_bridge, "Dump the bridge registers");
module_param(printph, int, 0);
MODULE_PARM_DESC(printph, "Print frame start/end headers");
module_param(phy, int, 0);
MODULE_PARM_DESC(phy, "Prediction range (horiz. Y)");
module_param(phuv, int, 0);
MODULE_PARM_DESC(phuv, "Prediction range (horiz. UV)");
module_param(pvy, int, 0);
MODULE_PARM_DESC(pvy, "Prediction range (vert. Y)");
module_param(pvuv, int, 0);
MODULE_PARM_DESC(pvuv, "Prediction range (vert. UV)");
module_param(qhy, int, 0);
MODULE_PARM_DESC(qhy, "Quantization threshold (horiz. Y)");
module_param(qhuv, int, 0);
MODULE_PARM_DESC(qhuv, "Quantization threshold (horiz. UV)");
module_param(qvy, int, 0);
MODULE_PARM_DESC(qvy, "Quantization threshold (vert. Y)");
module_param(qvuv, int, 0);
MODULE_PARM_DESC(qvuv, "Quantization threshold (vert. UV)");
module_param(lightfreq, int, 0);
MODULE_PARM_DESC(lightfreq,
  "Light frequency. Set to 50 or 60 Hz, or zero for default settings");
module_param(bandingfilter, int, 0);
MODULE_PARM_DESC(bandingfilter,
  "Enable banding filter (to reduce effects of fluorescent lighting)");
module_param(clockdiv, int, 0);
MODULE_PARM_DESC(clockdiv, "Force pixel clock divisor to a specific value");
module_param(packetsize, int, 0);
MODULE_PARM_DESC(packetsize, "Force a specific isoc packet size");
module_param(framedrop, int, 0);
MODULE_PARM_DESC(framedrop, "Force a specific frame drop register setting");
module_param(fastset, int, 0);
MODULE_PARM_DESC(fastset, "Allows picture settings to take effect immediately");
module_param(force_palette, int, 0);
MODULE_PARM_DESC(force_palette, "Force the palette to a specific value");
module_param(tuner, int, 0);
MODULE_PARM_DESC(tuner, "Set tuner type, if not autodetected");
module_param(backlight, int, 0);
MODULE_PARM_DESC(backlight, "For objects that are lit from behind");
#if defined(module_param_array)
static int num_uv;
module_param_array(unit_video, int, &num_uv, 0);
#else
MODULE_PARM(unit_video, "1-" __MODULE_STRING(OV511_MAX_UNIT_VIDEO) "i");
#endif
MODULE_PARM_DESC(unit_video,
  "Force use of specific minor number(s). 0 is not allowed.");
module_param(remove_zeros, int, 0);
MODULE_PARM_DESC(remove_zeros,
  "Remove zero-padding from uncompressed incoming data");
module_param(mirror, int, 0);
MODULE_PARM_DESC(mirror, "Reverse image horizontally");
module_param(ov518_color, int, 0);
MODULE_PARM_DESC(ov518_color, "Enable OV518 color");
module_param(i2c_clockdiv, int, 0);
MODULE_PARM_DESC(i2c_clockdiv, "Set I2C clock divider");
#if defined(module_param_array)
static int num_dev_hint;
module_param_array(dev_hint, charp, &num_dev_hint, 0);
#else
MODULE_PARM(dev_hint, "0-" __MODULE_STRING(OV511_MAX_DEV_HINTS) "s");
#endif
MODULE_PARM_DESC(dev_hint, "Device node hints");
#if defined HAVE_V4L2
module_param(v4l2, int, 0);
MODULE_PARM_DESC(v4l2, "Enable Video4Linux 2 support");
#endif

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
#if defined(MODULE_LICENSE)	/* Introduced in ~2.4.10 */
MODULE_LICENSE("GPL");
#endif

/**********************************************************************
 * Miscellaneous Globals
 **********************************************************************/

static struct usb_driver ov511_driver;

/* Prevents double-free of ov struct */
static struct semaphore ov_free_lock;

static struct usb_device_id device_table [] = {
	{ USB_DEVICE(VEND_OMNIVISION, PROD_OV511) },
	{ USB_DEVICE(VEND_OMNIVISION, PROD_OV511PLUS) },
	{ USB_DEVICE(VEND_OMNIVISION, PROD_OV518) },
	{ USB_DEVICE(VEND_OMNIVISION, PROD_OV518PLUS) },
	{ USB_DEVICE(VEND_MATTEL, PROD_ME2CAM) },
	{ }  /* Terminating entry */
};

MODULE_DEVICE_TABLE (usb, device_table);

static unsigned char yQuanTable511[] = OV511_YQUANTABLE;
static unsigned char uvQuanTable511[] = OV511_UVQUANTABLE;
static unsigned char yQuanTable518[] = OV518_YQUANTABLE;
static unsigned char uvQuanTable518[] = OV518_UVQUANTABLE;

struct device_hint {
	int cid;
	int minor;
} device_hint[OV511_MAX_UNIT_VIDEO];

/**********************************************************************
 * Symbolic Names
 **********************************************************************/

/* Known OV511-based cameras */
static struct symbolic_list camlist[] = {
	{   0, "Generic Camera" },
	{   1, "Mustek WCam 3X" },
	{   3, "D-Link DSB-C300" },
	{   4, "Generic OV511/OV7610" },
	{   5, "Puretek PT-6007" },
	{   6, "Lifeview USB Life TV (NTSC)" },
	{  10, "Aiptek HyperVcam Home" },
	{  21, "Creative Labs WebCam 3" },
	{  22, "Lifeview USB Life TV (PAL D/K+B/G)" },
	{  36, "Koala-Cam" },
	{  38, "Lifeview USB Life TV (PAL)" },
	{  41, "Samsung Anycam MPC-M10" },
	{  43, "Mtekvision Zeca MV402" },
	{  44, "LG Electronics LPC-UM10" },
	{  46, "Suma eON" },
	{  70, "Lifeview USB Life TV (PAL/SECAM)" },
	{ 100, "Lifeview RoboCam" },
	{ 102, "AverMedia InterCam Elite" },
	{ 108, "Alpha Vision Tech. AlphaCam SE" },
	{ 112, "MediaForte MV300" },	/* or OV7110 evaluation kit */
	{ 134, "Ezonics EZCam II" },
	{ 150, "Lifeview USB Life TV/FM (PAL D/K+B/G)" },
	{ 192, "Webeye 2000B" },
	{ 193, "Interbiometrics Securecam" },
	{ 194, "Interbiometrics Securecam" },
	{ 195, "Interbiometrics Securecam" },
	{ 196, "Interbiometrics Securecam" },
	{ 253, "Alpha Vision Tech. AlphaCam SE" },
	{  -1, NULL }
};

/* Video4Linux1 Palettes */
static struct symbolic_list v4l1_plist[] = {
	{ VIDEO_PALETTE_GREY,	"GREY" },
	{ VIDEO_PALETTE_HI240,	"HI240" },
	{ VIDEO_PALETTE_RGB565,	"RGB565" },
	{ VIDEO_PALETTE_RGB24,	"RGB24" },
	{ VIDEO_PALETTE_RGB32,	"RGB32" },
	{ VIDEO_PALETTE_RGB555,	"RGB555" },
	{ VIDEO_PALETTE_YUV422,	"YUV422" },
	{ VIDEO_PALETTE_YUYV,	"YUYV" },
	{ VIDEO_PALETTE_UYVY,	"UYVY" },
	{ VIDEO_PALETTE_YUV420,	"YUV420" },
	{ VIDEO_PALETTE_YUV411,	"YUV411" },
	{ VIDEO_PALETTE_RAW,	"RAW" },
	{ VIDEO_PALETTE_YUV422P,"YUV422P" },
	{ VIDEO_PALETTE_YUV411P,"YUV411P" },
	{ VIDEO_PALETTE_YUV420P,"YUV420P" },
	{ VIDEO_PALETTE_YUV410P,"YUV410P" },
	{ -1, NULL }
};

static struct symbolic_list brglist[] = {
	{ BRG_OV511,		"OV511" },
	{ BRG_OV511PLUS,	"OV511+" },
	{ BRG_OV518,		"OV518" },
	{ BRG_OV518PLUS,	"OV518+" },
	{ -1, NULL }
};

#if defined(CONFIG_VIDEO_PROC_FS) || defined(HAVE_V4L2)
static struct symbolic_list senlist[] = {
	{ CC_OV76BE,	"OV76BE" },
	{ CC_OV7610,	"OV7610" },
	{ CC_OV7620,	"OV7620" },
	{ CC_OV7620AE,	"OV7620AE" },
	{ CC_OV6620,	"OV6620" },
	{ CC_OV6630,	"OV6630" },
	{ CC_OV6630AE,	"OV6630AE" },
	{ CC_OV6630AF,	"OV6630AF" },
	{ SEN_SAA7111A,	"SAA7111A" },
	{ -1, NULL }
};
#endif

/* URB error codes: */
static struct symbolic_list urb_errlist[] = {
	{ -ENOSR,	"Buffer error (overrun)" },
	{ -EPIPE,	"Stalled (device not responding)" },
	{ -EOVERFLOW,	"Babble (bad cable?)" },
	{ -EPROTO,	"Bit-stuff error (bad cable?)" },
	{ -EILSEQ,	"CRC/Timeout" },
	{ -ETIMEDOUT,	"NAK (device does not respond)" },
	{ -1, NULL }
};

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16)
static const char *v4l1_ioctls[] = {
	"?", "CGAP", "GCHAN", "SCHAN", "GTUNER", "STUNER", "GPICT", "SPICT",
	"CCAPTURE", "GWIN", "SWIN", "GFBUF", "SFBUF", "KEY", "GFREQ",
	"SFREQ", "GAUDIO", "SAUDIO", "SYNC", "MCAPTURE", "GMBUF", "GUNIT",
	"GCAPTURE", "SCAPTURE", "SPLAYMODE", "SWRITEMODE", "GPLAYINFO",
	"SMICROCODE", "GVBIFMT", "SVBIFMT" };
#define NUM_V4L1_IOCTLS (sizeof(v4l1_ioctls)/sizeof(char*))
#endif

/**********************************************************************
 * Prototypes
 **********************************************************************/

static int saa7111a_configure(struct usb_ov511 *);
static int ov7xx0_configure(struct usb_ov511 *);
static int ov6xx0_configure(struct usb_ov511 *);

/**********************************************************************
 * Memory management
 **********************************************************************/

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 10)
/* Here we want the physical address of the memory.
 * This is used when initializing the contents of the area.
 */
static inline unsigned long
kvirt_to_pa(unsigned long adr)
{
	unsigned long kva, ret;

	kva = (unsigned long) page_address(vmalloc_to_page((void *)adr));
	kva |= adr & (PAGE_SIZE-1); /* restore the offset */
	ret = __pa(kva);
	return ret;
}
#endif

static void *
rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr;

	size = PAGE_ALIGN(size);
	mem = vmalloc_32(size);
	if (!mem)
		return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 68)
		SetPageReserved(vmalloc_to_page((void *)adr));
#else
		mem_map_reserve(vmalloc_to_page((void *)adr));
#endif
		adr += PAGE_SIZE;
		size -= PAGE_SIZE;
	}

	return mem;
}

static void
rvfree(void *mem, unsigned long size)
{
	unsigned long adr;

	if (!mem)
		return;

	adr = (unsigned long) mem;
	while ((long) size > 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 68)
		ClearPageReserved(vmalloc_to_page((void *)adr));
#else
		mem_map_unreserve(vmalloc_to_page((void *)adr));
#endif
		adr += PAGE_SIZE;
		size -= PAGE_SIZE;
	}
	vfree(mem);
}

/**********************************************************************
 * /proc interface
 * Based on the CPiA driver version 0.7.4 -claudio
 **********************************************************************/

#if defined(CONFIG_VIDEO_PROC_FS)

static struct proc_dir_entry *ov511_proc_entry = NULL;
extern struct proc_dir_entry *video_proc_entry;

/* Prototypes */
static int sensor_get_picture(struct usb_ov511 *, struct video_picture *);
static int sensor_get_control(struct usb_ov511 *, int, int *);
static int ov51x_check_snapshot(struct usb_ov511 *);
static void ov51x_clear_snapshot(struct usb_ov511 *);
static int ov51x_control_ioctl(struct inode *, struct file *, unsigned int,
			       unsigned long);

static struct file_operations ov511_control_fops = {
	.ioctl =	ov51x_control_ioctl,
};

#define YES_NO(x) ((x) ? "yes" : "no")

/* /proc/video/ov511/<minor#>/info */
static int
ov511_read_proc_info(char *page, char **start, off_t off, int count, int *eof,
		     void *data)
{
	char *out = page;
	int i, len;
	struct usb_ov511 *ov = data;
	struct video_picture p;
	int exp = 0;

	if (!ov || !ov->dev)
		return -ENODEV;

	sensor_get_picture(ov, &p);
	sensor_get_control(ov, OVCAMCHIP_CID_EXP, &exp);

	/* IMPORTANT: This output MUST be kept under PAGE_SIZE
	 *            or we need to get more sophisticated. */

	out += sprintf(out, "driver_version  : %s\n", DRIVER_VERSION);
	out += sprintf(out, "custom_id       : %d\n", ov->customid);
	out += sprintf(out, "model           : %s\n", ov->desc);
	out += sprintf(out, "streaming       : %s\n", YES_NO(ov->streaming));
	out += sprintf(out, "grabbing        : %s\n", YES_NO(ov->grabbing));
	out += sprintf(out, "can_decompress  : %s\n", YES_NO(ov->decomp_ops));
	out += sprintf(out, "compress        : %s\n", YES_NO(ov->compress));
	out += sprintf(out, "subcapture      : %s\n", YES_NO(ov->sub_flag));
	out += sprintf(out, "sub_size        : %d %d %d %d\n",
		       ov->subx, ov->suby, ov->subw, ov->subh);
	out += sprintf(out, "brightness      : %d\n", p.brightness >> 8);
	out += sprintf(out, "colour          : %d\n", p.colour >> 8);
	out += sprintf(out, "contrast        : %d\n", p.contrast >> 8);
	out += sprintf(out, "hue             : %d\n", p.hue >> 8);
	out += sprintf(out, "exposure        : %d\n", exp);
	out += sprintf(out, "num_frames      : %d\n", OV511_NUMFRAMES);
	for (i = 0; i < OV511_NUMFRAMES; i++) {
		out += sprintf(out, "frame           : %d\n", i);
		out += sprintf(out, "  depth         : %d\n",
			       ov->frame[i].depth);
		out += sprintf(out, "  size          : %d %d\n",
			       ov->frame[i].width, ov->frame[i].height);
		out += sprintf(out, "  format        : %s\n",
			       symbolic(v4l1_plist, ov->frame[i].format));
		out += sprintf(out, "  data_buffer   : 0x%p\n",
			       ov->frame[i].data);
	}
	out += sprintf(out, "snap_enabled    : %s\n", YES_NO(ov->snap_enabled));
	out += sprintf(out, "bridge          : %s\n",
		       symbolic(brglist, ov->bridge));
	out += sprintf(out, "sensor          : %s\n",
		       symbolic(senlist, ov->sensor));
	out += sprintf(out, "packet_size     : %d\n", ov->packet_size);
	out += sprintf(out, "framebuffer     : 0x%p\n", ov->fbuf);
	out += sprintf(out, "packet_numbering: %d\n", ov->packet_numbering);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 20)
	out += sprintf(out, "topology        : %s\n", ov->usb_path);
#else
	out += sprintf(out, "usb_bus         : %d\n", ov->dev->bus->busnum);
	out += sprintf(out, "usb_device      : %d\n", ov->dev->devnum);
#endif
	out += sprintf(out, "i2c_adap_id     : %d\n",
	               i2c_adapter_id(&ov->i2c_adap));
	out += sprintf(out, "input           : %d\n", ov->input);

	len = out - page;
	len -= off;
	if (len < count) {
		*eof = 1;
		if (len <= 0)
			return 0;
	} else
		len = count;

	*start = page + off;

	return len;
}

/* /proc/video/ov511/<minor#>/button
 *
 * When the camera's button is pressed, the output of this will change from a
 * 0 to a 1 (ASCII). It will retain this value until it is read, after which
 * it will reset to zero.
 *
 * SECURITY NOTE: Since reading this file can change the state of the snapshot
 * status, it is important for applications that open it to keep it locked
 * against access by other processes, using flock() or a similar mechanism. No
 * locking is provided by this driver.
 */
static int
ov511_read_proc_button(char *page, char **start, off_t off, int count, int *eof,
		       void *data)
{
	char *out = page;
	int len, status;
	struct usb_ov511 *ov = data;

	if (!ov || !ov->dev)
		return -ENODEV;

	status = ov51x_check_snapshot(ov);
	out += sprintf(out, "%d", status);

	if (status)
		ov51x_clear_snapshot(ov);

	len = out - page;
	len -= off;
	if (len < count) {
		*eof = 1;
		if (len <= 0)
			return 0;
	} else {
		len = count;
	}

	*start = page + off;

	return len;
}

static void
create_proc_ov511_cam(struct usb_ov511 *ov)
{
	char dirname[10];

	if (!ov511_proc_entry || !ov)
		return;

	/* Create per-device directory */
	snprintf(dirname, 10, "%d", ov->vdev->minor);
	ov->proc_devdir = create_proc_entry(dirname, S_IFDIR, ov511_proc_entry);
	if (!ov->proc_devdir)
		return;
	ov->proc_devdir->owner = THIS_MODULE;

	/* Create "info" entry (human readable device information) */
	ov->proc_info = create_proc_read_entry("info", S_IFREG|S_IRUGO|S_IWUSR,
		ov->proc_devdir, ov511_read_proc_info, ov);
	if (!ov->proc_info)
		return;
	ov->proc_info->owner = THIS_MODULE;

	/* Don't create it if old snapshot mode on (would cause race cond.) */
	if (!snapshot) {
		/* Create "button" entry (snapshot button status) */
		ov->proc_button = create_proc_read_entry("button",
			S_IFREG|S_IRUGO|S_IWUSR, ov->proc_devdir,
			ov511_read_proc_button, ov);
		if (!ov->proc_button)
			return;
		ov->proc_button->owner = THIS_MODULE;
	}

	/* Create "control" entry (ioctl() interface) */
	lock_kernel();
	ov->proc_control = create_proc_entry("control",	S_IFREG|S_IRUGO|S_IWUSR,
		ov->proc_devdir);
	if (!ov->proc_control) {
		unlock_kernel();
		return;
	}
	ov->proc_control->owner = THIS_MODULE;
	ov->proc_control->data = ov;
	ov->proc_control->proc_fops = &ov511_control_fops;
	unlock_kernel();
}

static void
destroy_proc_ov511_cam(struct usb_ov511 *ov)
{
	char dirname[10];

	if (!ov || !ov->proc_devdir)
		return;

	snprintf(dirname, 10, "%d", ov->vdev->minor);

	/* Destroy "control" entry */
	if (ov->proc_control) {
		remove_proc_entry("control", ov->proc_devdir);
		ov->proc_control = NULL;
	}

	/* Destroy "button" entry */
	if (ov->proc_button) {
		remove_proc_entry("button", ov->proc_devdir);
		ov->proc_button = NULL;
	}

	/* Destroy "info" entry */
	if (ov->proc_info) {
		remove_proc_entry("info", ov->proc_devdir);
		ov->proc_info = NULL;
	}

	/* Destroy per-device directory */
	remove_proc_entry(dirname, ov511_proc_entry);
	ov->proc_devdir = NULL;
}

static void
proc_ov511_create(void)
{
	if (video_proc_entry == NULL) {
		err("Error: /proc/video/ does not exist");
		return;
	}

	ov511_proc_entry = create_proc_entry("ov511", S_IFDIR,
					     video_proc_entry);

	if (ov511_proc_entry)
		ov511_proc_entry->owner = THIS_MODULE;
	else
		err("Unable to create /proc/video/ov511");
}

static void
proc_ov511_destroy(void)
{
	if (ov511_proc_entry == NULL)
		return;

	remove_proc_entry("ov511", video_proc_entry);
}
#else
static inline void create_proc_ov511_cam(struct usb_ov511 *ov) { }
static inline void destroy_proc_ov511_cam(struct usb_ov511 *ov) { }
static inline void proc_ov511_create(void) { }
static inline void proc_ov511_destroy(void) { }
#endif /* #ifdef CONFIG_VIDEO_PROC_FS */

/**********************************************************************
 *
 * Register I/O
 *
 **********************************************************************/

/* Write an OV51x register */
static int
reg_w(struct usb_ov511 *ov, unsigned char reg, unsigned char value)
{
	int rc;

	PDEBUG(5, "0x%02X:0x%02X", reg, value);

	down(&ov->cbuf_lock);

	if (!ov->cbuf) {
		up(&ov->cbuf_lock);
		return -ENODEV;
	}

	ov->cbuf[0] = value;
	rc = usb_control_msg(ov->dev,
			     usb_sndctrlpipe(ov->dev, 0),
			     (ov->bclass == BCL_OV518)?1:2 /* REG_IO */,
			     USB_TYPE_VENDOR | USB_RECIP_DEVICE,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 12)
			     0, (__u16)reg, &ov->cbuf[0], 1, 1000);
#else
			     0, (__u16)reg, &ov->cbuf[0], 1, HZ);
#endif
	up(&ov->cbuf_lock);

	if (rc < 0)
		err("reg write: error %d: %s", rc, symbolic(urb_errlist, rc));

	return rc;
}

/* Read from an OV51x register */
/* returns: negative is error, pos or zero is data */
static int
reg_r(struct usb_ov511 *ov, unsigned char reg)
{
	int rc;

	down(&ov->cbuf_lock);

	if (!ov->cbuf) {
		up(&ov->cbuf_lock);
		return -ENODEV;
	}

	rc = usb_control_msg(ov->dev,
			     usb_rcvctrlpipe(ov->dev, 0),
			     (ov->bclass == BCL_OV518)?1:3 /* REG_IO */,
			     USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 12)
			     0, (__u16)reg, &ov->cbuf[0], 1, 1000);
#else
			     0, (__u16)reg, &ov->cbuf[0], 1, HZ);
#endif
	if (rc < 0) {
		err("reg read: error %d: %s", rc, symbolic(urb_errlist, rc));
	} else {
		rc = ov->cbuf[0];
		PDEBUG(5, "0x%02X:0x%02X", reg, ov->cbuf[0]);
	}

	up(&ov->cbuf_lock);

	return rc;
}

/*
 * Writes bits at positions specified by mask to an OV51x reg. Bits that are in
 * the same position as 1's in "mask" are cleared and set to "value". Bits
 * that are in the same position as 0's in "mask" are preserved, regardless
 * of their respective state in "value".
 */
static int
reg_w_mask(struct usb_ov511 *ov,
	   unsigned char reg,
	   unsigned char value,
	   unsigned char mask)
{
	int ret;
	unsigned char oldval, newval;

	if (mask == 0xff) {
		newval = value;
	} else {
		ret = reg_r(ov, reg);
		if (ret < 0)
			return ret;

		oldval = (unsigned char) ret;
		oldval &= (~mask);		/* Clear the masked bits */
		value &= mask;			/* Enforce mask on value */
		newval = oldval | value;	/* Set the desired bits */
	}

	return (reg_w(ov, reg, newval));
}

/* 
 * Writes multiple (n) byte value to a single register. Only valid with certain
 * registers (0x30 and 0xc4 - 0xce).
 */
static int
ov518_reg_w32(struct usb_ov511 *ov, unsigned char reg, u32 val, int n)
{
	int rc;

	PDEBUG(5, "0x%02X:%7d, n=%d", reg, val, n);

	down(&ov->cbuf_lock);

	if (!ov->cbuf) {
		up(&ov->cbuf_lock);
		return -ENODEV;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9)
	*((__le32 *)ov->cbuf) = __cpu_to_le32(val);
#else
	*((u32 *)ov->cbuf) = __cpu_to_le32(val);
#endif

	rc = usb_control_msg(ov->dev,
			     usb_sndctrlpipe(ov->dev, 0),
			     1 /* REG_IO */,
			     USB_TYPE_VENDOR | USB_RECIP_DEVICE,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 12)
			     0, (__u16)reg, ov->cbuf, n, 1000);
#else
			     0, (__u16)reg, ov->cbuf, n, HZ);
#endif
	up(&ov->cbuf_lock);

	if (rc < 0)
		err("reg write multiple: error %d: %s", rc,
		    symbolic(urb_errlist, rc));

	return rc;
}

static int
ov511_upload_quan_tables(struct usb_ov511 *ov)
{
	unsigned char *pYTable = yQuanTable511;
	unsigned char *pUVTable = uvQuanTable511;
	unsigned char val0, val1;
	int i, rc, reg = R511_COMP_LUT_BEGIN;

	PDEBUG(4, "Uploading quantization tables");

	for (i = 0; i < OV511_QUANTABLESIZE / 2; i++) {
		/* Y */
		val0 = *pYTable++;
		val1 = *pYTable++;
		val0 &= 0x0f;
		val1 &= 0x0f;
		val0 |= val1 << 4;
		rc = reg_w(ov, reg, val0);
		if (rc < 0)
			return rc;

		/* UV */
		val0 = *pUVTable++;
		val1 = *pUVTable++;
		val0 &= 0x0f;
		val1 &= 0x0f;
		val0 |= val1 << 4;
		rc = reg_w(ov, reg + OV511_QUANTABLESIZE/2, val0);
		if (rc < 0)
			return rc;

		reg++;
	}

	return 0;
}

/* OV518 quantization tables are 8x4 (instead of 8x8) */
static int
ov518_upload_quan_tables(struct usb_ov511 *ov)
{
	unsigned char *pYTable = yQuanTable518;
	unsigned char *pUVTable = uvQuanTable518;
	unsigned char val0, val1;
	int i, rc, reg = R511_COMP_LUT_BEGIN;

	PDEBUG(4, "Uploading quantization tables");

	for (i = 0; i < OV518_QUANTABLESIZE / 2; i++) {
		/* Y */
		val0 = *pYTable++;
		val1 = *pYTable++;
		val0 &= 0x0f;
		val1 &= 0x0f;
		val0 |= val1 << 4;
		rc = reg_w(ov, reg, val0);
		if (rc < 0)
			return rc;

		/* UV */
		val0 = *pUVTable++;
		val1 = *pUVTable++;
		val0 &= 0x0f;
		val1 &= 0x0f;
		val0 |= val1 << 4;
		rc = reg_w(ov, reg + OV518_QUANTABLESIZE/2, val0);
		if (rc < 0)
			return rc;

		reg++;
	}

	return 0;
}

static int
ov51x_reset(struct usb_ov511 *ov, unsigned char reset_type)
{
	int rc;

	/* Setting bit 0 not allowed on 518/518Plus */
	if (ov->bclass == BCL_OV518)
		reset_type &= 0xfe;

	PDEBUG(4, "Reset: type=0x%02X", reset_type);

	rc = reg_w(ov, R51x_SYS_RESET, reset_type);
	rc = reg_w(ov, R51x_SYS_RESET, 0);

	if (rc < 0)
		err("reset: command failed");

	return rc;
}

/* returns: negative is error, pos or zero is data */
static int
i2c_r(struct usb_ov511 *ov, unsigned char reg)
{
	return i2c_smbus_read_byte_data(&ov->internal_client, reg);
}

static int
i2c_w(struct usb_ov511 *ov, unsigned char reg, unsigned char value)
{
	return i2c_smbus_write_byte_data(&ov->internal_client, reg, value);
}

/**********************************************************************
 *
 * Low-level I2C I/O functions
 *
 **********************************************************************/

/* NOTE: Do not call this function directly!
 * Sets I2C read and write slave IDs, if they have changed.
 * Returns <0 for error
 */
static int
ov51x_i2c_adap_set_slave(struct usb_ov511 *ov, unsigned char slave)
{
	int rc;

	if (ov->last_slave == slave)
		return 0;

	/* invalidate slave */
	ov->last_slave = 0;

	rc = reg_w(ov, R51x_I2C_W_SID, (slave<<1));
	if (rc < 0)
		return rc;

	rc = reg_w(ov, R51x_I2C_R_SID, (slave<<1) + 1);
	if (rc < 0)
		return rc;

	// FIXME: Is this actually necessary?
	rc = ov51x_reset(ov, OV511_RESET_NOREGS);
	if (rc < 0)
		return rc;

	/* validate slave */
	ov->last_slave = slave;
	return 0;
}

static int
ov511_i2c_adap_read_byte(struct usb_ov511 *ov,
			 unsigned char addr,
			 unsigned char *val)
{
	int rc, retries;

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Two byte read cycle */
	for (retries = OV511_I2C_RETRIES; ; ) {
		/* Initiate 2-byte read cycle */
		rc = reg_w(ov, R511_I2C_CTL, 0x05);
		if (rc < 0)
			goto out;

		do
			rc = reg_r(ov, R511_I2C_CTL);
		while (rc > 0 && ((rc&1) == 0)); /* Retry until idle */
		if (rc < 0)
			goto out;

		if ((rc&2) == 0) /* Ack? */
			break;

		/* I2C abort */
		rc = reg_w(ov, R511_I2C_CTL, 0x10);
		if (rc < 0)
			goto out;

		if (--retries < 0) {
			PDEBUG(5, "i2c read retries exhausted");
			rc = -1;
			goto out;
		}
	}

	rc = reg_r(ov, R51x_I2C_DATA);
	if (rc < 0)
		goto out;
	else
		*val = rc;

	PDEBUG(5, "(0x%02X) 0x%02X", addr<<1, *val);

	/* This is needed to make writes work again */
	rc = reg_w(ov, R511_I2C_CTL, 0x05);
out:
	if (rc < 0)
		PDEBUG(5, "I2C read error (%d)", rc);

	return rc;
}

static int
ov511_i2c_adap_write_byte(struct usb_ov511 *ov,
			  unsigned char addr,
			  unsigned char subaddr)
{
	int rc, retries;

	PDEBUG(5, "(0x%02X) 0x%02X", addr<<1, subaddr);

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Two byte write cycle */
	for (retries = OV511_I2C_RETRIES; ; ) {
		/* Select camera register */
		rc = reg_w(ov, R51x_I2C_SADDR_2, subaddr);
		if (rc < 0)
			goto out;

		/* Initiate 2-byte write cycle */
		rc = reg_w(ov, R511_I2C_CTL, 0x03);
		if (rc < 0)
			goto out;

		do
			rc = reg_r(ov, R511_I2C_CTL);
		while (rc > 0 && ((rc&1) == 0)); /* Retry until idle */
		if (rc < 0)
			goto out;

		if ((rc&2) == 0) /* Ack? */
			break;

		/* I2C abort */
		rc = reg_w(ov, R511_I2C_CTL, 0x10);
		if (rc < 0)
			goto out;

		if (--retries < 0) {
			PDEBUG(5, "i2c write retries exhausted");
			rc = -1;
			goto out;
		}
	}
out:
	if (rc < 0)
		PDEBUG(5, "I2C write error (%d)", rc);

	return rc;
}

static int
ov511_i2c_adap_read_byte_data(struct usb_ov511 *ov,
			      unsigned char addr,
			      unsigned char subaddr,
			      unsigned char *val)
{
	int rc;

	rc = ov511_i2c_adap_write_byte(ov, addr, subaddr);
	if (rc < 0)
		return rc;

	rc = ov511_i2c_adap_read_byte(ov, addr, val);

	return rc;
}

static int
ov511_i2c_adap_write_byte_data(struct usb_ov511 *ov,
			       unsigned char addr,
			       unsigned char subaddr,
			       unsigned char val)
{
	int rc, retries;

	PDEBUG(5, "(0x%02X) 0x%02X:0x%02X", addr<<1, subaddr, val);

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Three byte write cycle */
	for (retries = OV511_I2C_RETRIES; ; ) {
		/* Select camera register */
		rc = reg_w(ov, R51x_I2C_SADDR_3, subaddr);
		if (rc < 0)
			goto out;

		/* Write "value" to I2C data port of OV511 */
		rc = reg_w(ov, R51x_I2C_DATA, val);
		if (rc < 0)
			goto out;

		/* Initiate 3-byte write cycle */
		rc = reg_w(ov, R511_I2C_CTL, 0x01);
		if (rc < 0)
			goto out;

		do
			rc = reg_r(ov, R511_I2C_CTL);
		while (rc > 0 && ((rc&1) == 0)); /* Retry until idle */
		if (rc < 0)
			goto out;

		if ((rc&2) == 0) /* Ack? */
			break;
#if 0
		/* I2C abort */
		reg_w(ov, R511_I2C_CTL, 0x10);
#endif
		if (--retries < 0) {
			PDEBUG(5, "i2c write retries exhausted");
			rc = -1;
			goto out;
		}
	}
out:
	if (rc < 0)
		PDEBUG(5, "I2C write error (%d)", rc);

	return rc;
}

/* NOTE: this function always succeeds regardless of whether the sensor is
 * present and working.
 */
static int
ov518_i2c_adap_read_byte(struct usb_ov511 *ov,
			 unsigned char addr,
			 unsigned char *val)
{
	int rc;

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Initiate 2-byte read cycle */
	rc = reg_w(ov, R518_I2C_CTL, 0x05);
	if (rc < 0)
		goto out;

	rc = reg_r(ov, R51x_I2C_DATA);
	if (rc < 0)
		goto out;
	else
		*val = rc;

	PDEBUG(5, "(0x%02X) 0x%02X", addr<<1, *val);
out:
	if (rc < 0)
		PDEBUG(5, "I2C read error (%d)", rc);

	return rc;
}

/* NOTE: this function always succeeds regardless of whether the sensor is
 * present and working.
 */
static int
ov518_i2c_adap_write_byte(struct usb_ov511 *ov,
			  unsigned char addr,
			  unsigned char subaddr)
{
	int rc;

	PDEBUG(5, "(0x%02X) 0x%02X", addr<<1, subaddr);

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Select camera register */
	rc = reg_w(ov, R51x_I2C_SADDR_2, subaddr);
	if (rc < 0)
		goto out;

	/* Initiate 2-byte write cycle */
	rc = reg_w(ov, R518_I2C_CTL, 0x03);
	if (rc < 0)
		goto out;
out:
	if (rc < 0)
		PDEBUG(5, "I2C write error (%d)", rc);

	return rc;
}

static int
ov518_i2c_adap_read_byte_data(struct usb_ov511 *ov,
			      unsigned char addr,
			      unsigned char subaddr,
			      unsigned char *val)
{
	int rc;

	rc = ov518_i2c_adap_write_byte(ov, addr, subaddr);
	if (rc < 0)
		return rc;

	rc = ov518_i2c_adap_read_byte(ov, addr, val);

	return rc;
}

/* NOTE: this function always succeeds regardless of whether the sensor is
 * present and working.
 */
static int
ov518_i2c_adap_write_byte_data(struct usb_ov511 *ov,
			       unsigned char addr,
			       unsigned char subaddr,
			       unsigned char val)
{
	int rc;

	PDEBUG(5, "(0x%02X) 0x%02X:0x%02X", addr<<1, subaddr, val);

	/* Set slave addresses */	
	rc = ov51x_i2c_adap_set_slave(ov, addr);
	if (rc < 0)
		return rc;

	/* Select camera register */
	rc = reg_w(ov, R51x_I2C_SADDR_3, subaddr);
	if (rc < 0)
		goto out;

	/* Write "value" to I2C data port of OV511 */
	rc = reg_w(ov, R51x_I2C_DATA, val);
	if (rc < 0)
		goto out;

	/* Initiate 3-byte write cycle */
	rc = reg_w(ov, R518_I2C_CTL, 0x01);
	if (rc < 0)
		goto out;

out:
	if (rc < 0)
		PDEBUG(5, "I2C write error (%d)", rc);

	return rc;
}

static int
write_regvals(struct usb_ov511 *ov, struct regval *regvals)
{
	int rc;

	while (regvals->mask != 0) {
		rc = reg_w_mask(ov, regvals->reg, regvals->val, regvals->mask);
		if (rc < 0)
			return rc;

		regvals++;
	}

	return 0;
}

#ifdef OV511_DEBUG
static void
dump_reg_range(struct usb_ov511 *ov, int reg1, int regn)
{
	int i, rc;

	for (i = reg1; i <= regn; i++) {
		rc = reg_r(ov, i);
		info("OV511[0x%02X] = 0x%02X", i, rc);
	}
}

static void
ov511_dump_regs(struct usb_ov511 *ov)
{
	info("CAMERA INTERFACE REGS");
	dump_reg_range(ov, 0x10, 0x1f);
	info("DRAM INTERFACE REGS");
	dump_reg_range(ov, 0x20, 0x23);
	info("ISO FIFO REGS");
	dump_reg_range(ov, 0x30, 0x31);
	info("PIO REGS");
	dump_reg_range(ov, 0x38, 0x39);
	dump_reg_range(ov, 0x3e, 0x3e);
	info("I2C REGS");
	dump_reg_range(ov, 0x40, 0x49);
	info("SYSTEM CONTROL REGS");
	dump_reg_range(ov, 0x50, 0x55);
	dump_reg_range(ov, 0x5e, 0x5f);
	info("OmniCE REGS");
	dump_reg_range(ov, 0x70, 0x79);
	/* NOTE: Quantization tables are not readable. You will get the value
	 * in reg. 0x79 for every table register */
	dump_reg_range(ov, 0x80, 0x9f);
	dump_reg_range(ov, 0xa0, 0xbf);

}

static void
ov518_dump_regs(struct usb_ov511 *ov)
{
	info("VIDEO MODE REGS");
	dump_reg_range(ov, 0x20, 0x2f);
	info("DATA PUMP AND SNAPSHOT REGS");
	dump_reg_range(ov, 0x30, 0x3f);
	info("I2C REGS");
	dump_reg_range(ov, 0x40, 0x4f);
	info("SYSTEM CONTROL AND VENDOR REGS");
	dump_reg_range(ov, 0x50, 0x5f);
	info("60 - 6F");
	dump_reg_range(ov, 0x60, 0x6f);
	info("70 - 7F");
	dump_reg_range(ov, 0x70, 0x7f);
	info("Y QUANTIZATION TABLE");
	dump_reg_range(ov, 0x80, 0x8f);
	info("UV QUANTIZATION TABLE");
	dump_reg_range(ov, 0x90, 0x9f);
	info("A0 - BF");
	dump_reg_range(ov, 0xa0, 0xbf);
	info("CBR");
	dump_reg_range(ov, 0xc0, 0xcf);
}
#endif

/**********************************************************************
 *
 * Kernel I2C Interface
 *
 **********************************************************************/

static int
ov511_i2c_validate_addr(struct usb_ov511 *ov, u16 addr)
{
	switch (addr) {
	case 0x00c2>>1:	/* Tuner */
	case 0x0088>>1:	/* Audiochip */
		if (ov->tuner_type >= 0)
			return 0;
		else
			goto reject;
	case SAA7111A_SID:
		return 0;
	case OV6xx0_SID:
	case OV7xx0_SID:
		/* Got to be careful here... tuner gives a false ACK at
		 * address 0x00c0>>1 */
		if (ov->tuner_type >= 0)
			goto reject;
		else
			return 0;
	}
reject:
	PDEBUG(3, "Rejected slave ID 0x%04X", addr);
	return -ENODEV;
}

static int
ov518_i2c_validate_addr(struct usb_ov511 *ov, u16 addr)
{
	switch (addr) {
	case OV6xx0_SID:
	case OV7xx0_SID:
		return 0;
	default:
		PDEBUG(3, "Rejected slave ID 0x%04X", addr);
		return -ENODEV;
	}
}

static inline int
sensor_cmd(struct usb_ov511 *ov, unsigned int cmd, void *arg)
{
	struct i2c_client *c = ov->sensor_client;

	if (ov->sensor == SEN_SAA7111A)
		return -EPERM;	/* Non-fatal error */

	if (c && c->driver->command)
		return c->driver->command(ov->sensor_client, cmd, arg);
	else
		return -ENODEV;
}

static void
call_i2c_clients(struct usb_ov511 *ov, unsigned int cmd, void *arg)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
	struct i2c_client *client;
	int i;

	for (i = 0; i < I2C_CLIENT_MAX; i++) {
		client = ov->i2c_adap.clients[i];

		/* Sensor_client is not called from here for now, since it
		   has different enumeration for cmd  */
		if (client == ov->sensor_client)
			continue;

		if (client && client->driver->command)
			client->driver->command(client, cmd, arg);
	}
#else
	i2c_clients_command(&ov->i2c_adap, cmd, arg);
#endif
}

static int
ov511_smbus_xfer(struct i2c_adapter *adapter, u16 addr, unsigned short flags,
		 char read_write, u8 command, int size,
		 union i2c_smbus_data *data)
{
	struct usb_ov511 *ov = i2c_get_adapdata(adapter);
	int rc;

	rc = ov511_i2c_validate_addr(ov, addr);
	if (rc < 0)
		return rc;

	if (size == I2C_SMBUS_QUICK) {
		PDEBUG(4, "Got probed at addr 0x%04X", addr);
	} else if (size == I2C_SMBUS_BYTE) {
		if (read_write == I2C_SMBUS_WRITE) {
 			rc = ov511_i2c_adap_write_byte(ov, addr, command);
		} else if (read_write == I2C_SMBUS_READ) {
			rc = ov511_i2c_adap_read_byte(ov, addr, &data->byte);
		}
	} else if (size == I2C_SMBUS_BYTE_DATA) {
		if (read_write == I2C_SMBUS_WRITE) {
 			rc = ov511_i2c_adap_write_byte_data(ov, addr, command,
							    data->byte);
		} else if (read_write == I2C_SMBUS_READ) {
			rc = ov511_i2c_adap_read_byte_data(ov, addr, command,
							   &data->byte);
		}
	} else {
		warn("Unsupported I2C transfer mode (%d)", size);
		return -EINVAL;
	}

	/* This works around a bug in the I2C core */
	if (rc > 0)
		rc = 0;

	return rc;
}

static u32
ov511_i2c_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_SMBUS_QUICK
	     | I2C_FUNC_SMBUS_BYTE
	     | I2C_FUNC_SMBUS_BYTE_DATA;
}

static int
ov518_smbus_xfer(struct i2c_adapter *adapter, u16 addr, unsigned short flags,
		 char read_write, u8 command, int size,
		 union i2c_smbus_data *data)
{
	struct usb_ov511 *ov = i2c_get_adapdata(adapter);
	int rc;

	/* With OV518 we must be more aggressive about rejecting
	 * unknown addresses, since it can't report NAKs. */
	rc = ov518_i2c_validate_addr(ov, addr);
	if (rc < 0)
		return rc;

	if (size == I2C_SMBUS_QUICK) {
		PDEBUG(4, "Got probed at addr 0x%04X", addr);
	} else if (size == I2C_SMBUS_BYTE) {
		if (read_write == I2C_SMBUS_WRITE) {
 			rc = ov518_i2c_adap_write_byte(ov, addr, command);
		} else if (read_write == I2C_SMBUS_READ) {
			rc = ov518_i2c_adap_read_byte(ov, addr, &data->byte);
		}
	} else if (size == I2C_SMBUS_BYTE_DATA) {
		if (read_write == I2C_SMBUS_WRITE) {
 			rc = ov518_i2c_adap_write_byte_data(ov, addr, command,
							    data->byte);
		} else if (read_write == I2C_SMBUS_READ) {
			rc = ov518_i2c_adap_read_byte_data(ov, addr, command,
							   &data->byte);
		}
	} else {
		warn("Unsupported I2C transfer mode (%d)", size);
		return -EINVAL;
	}

	/* This works around a bug in the I2C core */
	if (rc > 0)
		rc = 0;

	return rc;
}

static u32
ov518_i2c_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_SMBUS_QUICK
	     | I2C_FUNC_SMBUS_BYTE
	     | I2C_FUNC_SMBUS_BYTE_DATA;
}

static int
i2c_attach_inform(struct i2c_client *client)
{
	struct usb_ov511 *ov = i2c_get_adapdata(client->adapter);
	int id = client->driver->id;

	if (id == I2C_DRIVERID_OVCAMCHIP) {
		int rc;

		ov->sensor_client = client;

		rc = sensor_cmd(ov, OVCAMCHIP_CMD_INITIALIZE, &ov->sensor_mono);
		if (rc < 0) {
			err("ERROR: Sensor init failed (rc=%d)", rc);
			ov->sensor_client = NULL;
			return rc;
		}

		down(&ov->lock);
		if (sensor_cmd(ov, OVCAMCHIP_CMD_Q_SUBTYPE, &ov->sensor) < 0)
			rc = -EIO;
		else if (client->addr == OV7xx0_SID)
			rc = ov7xx0_configure(ov);
		else if (client->addr == OV6xx0_SID)
			rc = ov6xx0_configure(ov);
		else
			rc = -EINVAL;
		up(&ov->lock);

		if (rc) {
			ov->sensor_client = NULL;
			return rc;
		}
	} else if (id == I2C_DRIVERID_TUNER) {
		if (ov->bclass == BCL_OV518 || ov->tuner_type < 0)
			return -1;

		ov->has_tuner = 1;
		call_i2c_clients(ov, TUNER_SET_TYPE, &ov->tuner_type);
	} else if (id == I2C_DRIVERID_TDA7313) {
		if (ov->bclass == BCL_OV518 || ov->tuner_type < 0)
			return -1;

		ov->has_audiochip = 1;
	} else if (id == I2C_DRIVERID_SAA7111A) {
		int arg;

		if (ov->bclass == BCL_OV518)
			return -1;

		down(&ov->lock);
		saa7111a_configure(ov);
		up(&ov->lock);

		arg = 1; call_i2c_clients(ov, DECODER_ENABLE_OUTPUT, &arg);
	} else	{
		PDEBUG(1, "Rejected client [%s] with [%s]",
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16)
		       client->name, client->driver->name);
#else
		       client->name, client->driver->driver.name);
#endif
		return -1;
	}

	PDEBUG(1, "i2c attach client [%s] with [%s]",
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16)
	       client->name, client->driver->name);
#else
	       client->name, client->driver->driver.name);
#endif

	return 0;
}

static int
i2c_detach_inform(struct i2c_client *client)
{
	struct usb_ov511 *ov = i2c_get_adapdata(client->adapter);

	if (ov->sensor_client == client)
		ov->sensor_client = NULL;

	PDEBUG(1, "i2c detach [%s]", client->name);

	return 0;
}

static int 
ov51x_i2c_control(struct i2c_adapter *adapter, unsigned int cmd,
		  unsigned long arg)
{
	return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
static void
ov51x_i2c_inc_use(struct i2c_adapter *adap)
{
	MOD_INC_USE_COUNT;
}

static void
ov51x_i2c_dec_use(struct i2c_adapter *adap)
{
	MOD_DEC_USE_COUNT;
}
#endif

static struct i2c_algorithm ov518_i2c_algo = {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
	.name =			"OV518 algorithm",
	.id =			I2C_ALGO_SMBUS,
#endif
	.smbus_xfer =		ov518_smbus_xfer,
	.algo_control =		ov51x_i2c_control,
	.functionality =	ov518_i2c_func,
};

static struct i2c_algorithm ov511_i2c_algo = {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
	.name =			"OV511 algorithm",
	.id =			I2C_ALGO_SMBUS,
#endif
	.smbus_xfer =		ov511_smbus_xfer,
	.algo_control =		ov51x_i2c_control,
	.functionality =	ov511_i2c_func,
};

static struct i2c_adapter i2c_adap_template = {
	.name = 		"(unset)",
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 6)
	.class =		I2C_CLASS_CAM_DIGITAL,
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 70)
	.class =		I2C_ADAP_CLASS_CAM_DIGITAL,
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
	.inc_use =		ov51x_i2c_inc_use,
	.dec_use =		ov51x_i2c_dec_use,
#else
	.owner =		THIS_MODULE,
#endif
	.client_register =	i2c_attach_inform,
	.client_unregister =	i2c_detach_inform,
};

static int
ov51x_init_i2c(struct usb_ov511 *ov)
{
	memcpy(&ov->i2c_adap, &i2c_adap_template, sizeof(struct i2c_adapter));

	/* Temporary name. We'll set the final one when we know the minor # */
	sprintf(ov->i2c_adap.name, "OV51x");
	i2c_set_adapdata(&ov->i2c_adap, ov);

	// FIXME: Need separate IDs for OV511+/OV518+
	if (ov->bclass == BCL_OV518) {
		ov->i2c_adap.algo = &ov518_i2c_algo;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
		ov->i2c_adap.id = I2C_ALGO_SMBUS | I2C_HW_SMBUS_OV518;
#else
		ov->i2c_adap.id = I2C_HW_SMBUS_OV518;
#endif
	} else {
		ov->i2c_adap.algo = &ov511_i2c_algo;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
		ov->i2c_adap.id = I2C_ALGO_SMBUS | I2C_HW_SMBUS_OV511;
#else
		ov->i2c_adap.id = I2C_HW_SMBUS_OV511;
#endif
	}

	ov->internal_client.adapter = &ov->i2c_adap;

	PDEBUG(4, "Registering I2C bus with kernel");

	return i2c_add_adapter(&ov->i2c_adap);
}

/*****************************************************************************/

/* Temporarily stops OV511 from functioning. Must do this before changing
 * registers while the camera is streaming */
static inline int
ov51x_stop(struct usb_ov511 *ov)
{
	PDEBUG(4, "stopping");
	ov->stopped = 1;
	if (ov->bclass == BCL_OV518)
		return (reg_w_mask(ov, R51x_SYS_RESET, 0x3a, 0x3a));
	else
		return (reg_w(ov, R51x_SYS_RESET, 0x3d));
}

/* Restarts OV511 after ov511_stop() is called. Has no effect if it is not
 * actually stopped (for performance). */
static inline int
ov51x_restart(struct usb_ov511 *ov)
{
	if (ov->stopped) {
		PDEBUG(4, "restarting");
		ov->stopped = 0;

		/* Reinitialize the stream */
		if (ov->bclass == BCL_OV518)
			reg_w(ov, 0x2f, 0x80);

		return (reg_w(ov, R51x_SYS_RESET, 0x00));
	}

	return 0;
}

/* Sleeps until no frames are active. Returns !0 if got signal or unplugged */
static int
ov51x_wait_frames_inactive(struct usb_ov511 *ov)
{
	int rc;

	rc = wait_event_interruptible(ov->wq, ov->curframe < 0 || !ov->present);
	if (rc)
		return rc;

	if (ov->present)
		return 0;
	else
		return -ENODEV;
}

/* Resets the hardware snapshot button */
static void
ov51x_clear_snapshot(struct usb_ov511 *ov)
{
	if (ov->bclass == BCL_OV511) {
		reg_w(ov, R51x_SYS_SNAP, 0x00);
		reg_w(ov, R51x_SYS_SNAP, 0x02);
		reg_w(ov, R51x_SYS_SNAP, 0x00);
	} else if (ov->bclass == BCL_OV518) {
		warn("snapshot reset not supported yet on OV518(+)");
	} else {
		err("clear snap: invalid bridge type");
	}
}

#if defined(CONFIG_VIDEO_PROC_FS)
/* Checks the status of the snapshot button. Returns 1 if it was pressed since
 * it was last cleared, and zero in all other cases (including errors) */
static int
ov51x_check_snapshot(struct usb_ov511 *ov)
{
	int ret, status = 0;

	if (ov->bclass == BCL_OV511) {
		ret = reg_r(ov, R51x_SYS_SNAP);
		if (ret < 0) {
			err("Error checking snspshot status (%d)", ret);
		} else if (ret & 0x08) {
			status = 1;
		}
	} else if (ov->bclass == BCL_OV518) {
		warn("snapshot check not supported yet on OV518(+)");
	} else {
		err("check snap: invalid bridge type");
	}

	return status;
}
#endif

/* Returns 1 if image streaming needs to be stopped while setting the specified
 * control, and returns 0 if not */
static int
sensor_needs_stop(struct usb_ov511 *ov, int cid)
{
	if (!ov->stop_during_set)
		return 0;

	/* SAA7111A always maintains a stable
	 * image, even when setting controls */
	if (ov->sensor == SEN_SAA7111A)
		return 0;

	switch (cid) {
	case OVCAMCHIP_CID_CONT:
	case OVCAMCHIP_CID_BRIGHT:
	case OVCAMCHIP_CID_SAT:
	case OVCAMCHIP_CID_HUE:
	case OVCAMCHIP_CID_EXP:
		 return 1;
	}

	return 0;
}

static int
sensor_set_control(struct usb_ov511 *ov, int cid, int val)
{
	struct ovcamchip_control ctl;
	int rc;

	if (sensor_needs_stop(ov, cid))
		if (ov51x_stop(ov) < 0)
			return -EIO;

	ctl.id = cid;
	ctl.value = val;

	rc = sensor_cmd(ov, OVCAMCHIP_CMD_S_CTRL, &ctl);

	if (ov51x_restart(ov) < 0)
		return -EIO;

	return rc;
}

static int
sensor_get_control(struct usb_ov511 *ov, int cid, int *val)
{
	struct ovcamchip_control ctl;
	int rc;

	ctl.id = cid;

	rc = sensor_cmd(ov, OVCAMCHIP_CMD_G_CTRL, &ctl);
	if (rc >= 0)
		*val = ctl.value;

	return rc;
}

static int
ov511_set_packet_size(struct usb_ov511 *ov, int size)
{
	int alt, mult;

	if (ov51x_stop(ov) < 0)
		return -EIO;

	mult = size >> 5;

	if (ov->bridge == BRG_OV511) {
		if (size == 0) alt = OV511_ALT_SIZE_0;
		else if (size == 257) alt = OV511_ALT_SIZE_257;
		else if (size == 513) alt = OV511_ALT_SIZE_513;
		else if (size == 769) alt = OV511_ALT_SIZE_769;
		else if (size == 993) alt = OV511_ALT_SIZE_993;
		else {
			err("Set packet size: invalid size (%d)", size);
			return -EINVAL;
		}
	} else if (ov->bridge == BRG_OV511PLUS) {
		if (size == 0) alt = OV511PLUS_ALT_SIZE_0;
		else if (size == 33) alt = OV511PLUS_ALT_SIZE_33;
		else if (size == 129) alt = OV511PLUS_ALT_SIZE_129;
		else if (size == 257) alt = OV511PLUS_ALT_SIZE_257;
		else if (size == 385) alt = OV511PLUS_ALT_SIZE_385;
		else if (size == 513) alt = OV511PLUS_ALT_SIZE_513;
		else if (size == 769) alt = OV511PLUS_ALT_SIZE_769;
		else if (size == 961) alt = OV511PLUS_ALT_SIZE_961;
		else {
			err("Set packet size: invalid size (%d)", size);
			return -EINVAL;
		}
	} else {
		err("Set packet size: Invalid bridge type");
		return -EINVAL;
	}

	PDEBUG(3, "%d, mult=%d, alt=%d", size, mult, alt);

	if (reg_w(ov, R51x_FIFO_PSIZE, mult) < 0)
		return -EIO;

	if (usb_set_interface(ov->dev, ov->iface, alt) < 0) {
		err("Set packet size: set interface error");
		return -EBUSY;
	}

	if (ov51x_reset(ov, OV511_RESET_NOREGS) < 0)
		return -EIO;

	ov->packet_size = size;

	if (ov51x_restart(ov) < 0)
		return -EIO;

	return 0;
}

/* Note: Unlike the OV511/OV511+, the size argument does NOT include the
 * optional packet number byte. The actual size *is* stored in ov->packet_size,
 * though. */
static int
ov518_set_packet_size(struct usb_ov511 *ov, int size)
{
	int alt;

	if (ov51x_stop(ov) < 0)
		return -EIO;

	if (ov->bclass == BCL_OV518) {
		if (size == 0) alt = OV518_ALT_SIZE_0;
		else if (size == 128) alt = OV518_ALT_SIZE_128;
		else if (size == 256) alt = OV518_ALT_SIZE_256;
		else if (size == 384) alt = OV518_ALT_SIZE_384;
		else if (size == 512) alt = OV518_ALT_SIZE_512;
		else if (size == 640) alt = OV518_ALT_SIZE_640;
		else if (size == 768) alt = OV518_ALT_SIZE_768;
		else if (size == 896) alt = OV518_ALT_SIZE_896;
		else {
			err("Set packet size: invalid size (%d)", size);
			return -EINVAL;
		}
	} else {
		err("Set packet size: Invalid bridge type");
		return -EINVAL;
	}

	PDEBUG(3, "%d, alt=%d", size, alt);

	ov->packet_size = size;
	if (size > 0) {
		/* Program ISO FIFO size reg (packet number isn't included) */
		ov518_reg_w32(ov, 0x30, size, 2);

		if (ov->packet_numbering)
			++ov->packet_size;
	}

	if (usb_set_interface(ov->dev, ov->iface, alt) < 0) {
		err("Set packet size: set interface error");
		return -EBUSY;
	}

	/* Initialize the stream */
	if (reg_w(ov, 0x2f, 0x80) < 0)
		return -EIO;

	if (ov51x_restart(ov) < 0)
		return -EIO;

	if (ov51x_reset(ov, OV511_RESET_NOREGS) < 0)
		return -EIO;

	return 0;
}

/* Upload compression params and quantization tables. Returns 0 for success. */
static int
ov511_init_compression(struct usb_ov511 *ov)
{
	int rc = 0;

	if (!ov->compress_inited) {
		reg_w(ov, 0x70, phy);
		reg_w(ov, 0x71, phuv);
		reg_w(ov, 0x72, pvy);
		reg_w(ov, 0x73, pvuv);
		reg_w(ov, 0x74, qhy);
		reg_w(ov, 0x75, qhuv);
		reg_w(ov, 0x76, qvy);
		reg_w(ov, 0x77, qvuv);

		if (ov511_upload_quan_tables(ov) < 0) {
			err("Error uploading quantization tables");
			rc = -EIO;
			goto out;
		}
	}

	ov->compress_inited = 1;
out:
	return rc;
}

/* Upload compression params and quantization tables. Returns 0 for success. */
static int
ov518_init_compression(struct usb_ov511 *ov)
{
	int rc = 0;

	if (!ov->compress_inited) {
		if (ov518_upload_quan_tables(ov) < 0) {
			err("Error uploading quantization tables");
			rc = -EIO;
			goto out;
		}
	}

	ov->compress_inited = 1;
out:
	return rc;
}

static int
sensor_set_picture(struct usb_ov511 *ov, struct video_picture *p)
{
	int rc;

	PDEBUG(4, "sensor_set_picture");

	if (ov->has_decoder) {
		call_i2c_clients(ov, DECODER_SET_PICTURE, p);
		return 0;
	}

	/* Don't return error if a setting is unsupported, or rest of settings
         * will not be performed */

	rc = sensor_set_control(ov, OVCAMCHIP_CID_CONT, p->contrast);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_BRIGHT, p->brightness);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_SAT, p->colour);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_HUE, p->hue);
	if (FATAL_ERROR(rc))
		return rc;

	return 0;
}

static int
sensor_get_picture(struct usb_ov511 *ov, struct video_picture *p)
{
	int rc, v;

	PDEBUG(4, "sensor_get_picture");

	if (ov->has_decoder) {
		call_i2c_clients(ov, VIDIOCGPICT, p);
		return 0;
	}

	/* Don't return error if a setting is unsupported, or rest of settings
         * will not be performed */

	rc = sensor_get_control(ov, OVCAMCHIP_CID_CONT, &v);
	if (FATAL_ERROR(rc))
		return rc;
	p->contrast = v;

	rc = sensor_get_control(ov, OVCAMCHIP_CID_BRIGHT, &v);
	if (FATAL_ERROR(rc))
		return rc;
	p->brightness = v;

	rc = sensor_get_control(ov, OVCAMCHIP_CID_SAT, &v);
	if (FATAL_ERROR(rc))
		return rc;
	p->colour = v;

	rc = sensor_get_control(ov, OVCAMCHIP_CID_HUE, &v);
	if (FATAL_ERROR(rc))
		return rc;
	p->hue = v;

	p->whiteness = 105 << 8;

	return 0;
}

/* Turns on or off the LED. Only has an effect with OV511+/OV518(+) */
static void
ov51x_led_control(struct usb_ov511 *ov, int enable)
{
	PDEBUG(4, " (%s)", enable ? "turn on" : "turn off");

	if (ov->bridge == BRG_OV511PLUS)
		reg_w(ov, R511_SYS_LED_CTL, enable ? 1 : 0);
	else if (ov->bclass == BCL_OV518)
		reg_w_mask(ov, R518_GPIO_OUT, enable ? 0x02 : 0x00, 0x02);

	return;
}

/* Matches the sensor's internal frame rate to the lighting frequency.
 * Valid frequencies are:
 *	50 - 50Hz, for European and Asian lighting
 *	60 - 60Hz, for American lighting
 *
 * Returns: 0 for success
 */
static int
sensor_set_light_freq(struct usb_ov511 *ov, int freq)
{
	if (freq != 50 && freq != 60) {
		err("Invalid light freq (%d Hz)", freq);
		return -EINVAL;
	}

	return sensor_set_control(ov, OVCAMCHIP_CID_FREQ, freq);
}

/* Returns number of bits per pixel (regardless of where they are located;
 * planar or not), or zero for unsupported format.
 */
static inline int
get_depth(int palette)
{
	switch (palette) {
	case VIDEO_PALETTE_GREY:    return 8;
	case VIDEO_PALETTE_YUV420:  return 12;
	case VIDEO_PALETTE_YUV420P: return 12; /* Planar */
	default:		    return 0;  /* Invalid format */
	}
}

/* Bytes per frame. Used by read(). Return of 0 indicates error */
static inline long int
get_frame_length(struct ov511_frame *frame)
{
	if (!frame)
		return 0;
	else
		return ((frame->width * frame->height
			 * get_depth(frame->format)) >> 3);
}

static int
set_ov_sensor_window(struct usb_ov511 *ov, int width, int height, int mode,
		     int sub_flag)
{
	struct ovcamchip_window win;
	int half_w = ov->maxwidth / 2;
	int half_h = ov->maxheight / 2;
	int rc;

	win.format = mode;

	/* Unless subcapture is enabled, center the image window and downsample
	 * if possible to increase the field of view */
	if (sub_flag) {
		win.x = ov->subx;
		win.y = ov->suby;
		win.width = ov->subw;
		win.height = ov->subh;
		win.quarter = 0;
	} else {
		/* NOTE: OV518(+) does downsampling on its own */
		if ((width > half_w && height > half_h)
		    || ov->bclass == BCL_OV518) {
			win.width = ov->maxwidth;
			win.height = ov->maxheight;
			win.quarter = 0;
		} else if (width > half_w || height > half_h) {
			err("Illegal dimensions");
			return -EINVAL;
		} else {
			win.width = half_w;
			win.height = half_h;
			win.quarter = 1;
		}

		/* Center it */
		win.x = (win.width - width) / 2;
		win.y = (win.height - height) / 2;
	}

	if (ov->clockdiv >= 0) {
		/* Manual override */
		win.clockdiv = ov->clockdiv;
	} else if (ov->bridge == BRG_OV518) {
		/* OV518 controls the clock externally */
		win.clockdiv = 0;
	} else if (ov->bridge == BRG_OV518PLUS) {
		/* OV518+ controls the clock externally */
		win.clockdiv = 1;
	} else if (ov->compress) {
		/* Use the highest possible rate, to maximize FPS */
		switch (ov->sensor) {
		case CC_OV6620:
			/* ...except with this sensor, which doesn't like
			 * higher rates yet */
			win.clockdiv = 3;
			break;
		case CC_OV6630:
			win.clockdiv = 0;
			break;
		case CC_OV76BE:
		case CC_OV7610:
		case CC_OV7620:
			win.clockdiv = 1;
			break;
		default:
			err("Invalid sensor");
			return -EINVAL;
		}
	} else {
		switch (ov->sensor) {
		case CC_OV6620:
		case CC_OV6630:
			win.clockdiv = 3;
			break;
		case CC_OV76BE:
		case CC_OV7610:
		case CC_OV7620:
			/* Use slowest possible clock without sacrificing
			 * frame rate */
			win.clockdiv = ((sub_flag ? ov->subw * ov->subh
			      : width * height)
			     * (win.format == VIDEO_PALETTE_GREY ? 2 : 3) / 2)
			    / 66000;
			break;
		default:
			err("Invalid sensor");
			return -EINVAL;
		}
	}

	PDEBUG(4, "Setting clock divider to %d", win.clockdiv);

	rc = sensor_cmd(ov, OVCAMCHIP_CMD_S_MODE, &win);
	if (rc < 0)
		return rc;

	/* A setting of 0x06 reduces frame corruption with certain sensors, in
	 * scenes with moving objects */
	if (framedrop >= 0) {
		i2c_w(ov, 0x16, framedrop);
	} else if (ov->sensor == CC_OV6620 || ov->sensor == CC_OV76BE ||
	         ov->sensor == CC_OV7610) {
		i2c_w(ov, 0x16, 0x06);
	}

	return 0;
}

/* Set up the OV511/OV511+ with the given image parameters.
 *
 * Do not put any sensor-specific code in here (including I2C I/O functions)
 */
static int
ov511_mode_init_regs(struct usb_ov511 *ov,
		     int width, int height, int mode, int sub_flag)
{
	int hsegs, vsegs;

	if (sub_flag) {
		width = ov->subw;
		height = ov->subh;
	}

	PDEBUG(3, "width:%d, height:%d, mode:%d, sub:%d",
	       width, height, mode, sub_flag);

	// FIXME: This should be moved to a 7111a-specific function once
	// subcapture is dealt with properly
	if (ov->sensor == SEN_SAA7111A) {
		if (width == 320 && height == 240) {
			/* No need to do anything special */
		} else if (width == 640 && height == 480) {
			/* Set the OV511 up as 320x480, but keep the
			 * V4L resolution as 640x480 */
			width = 320;
		} else {
			err("SAA7111A only allows 320x240 or 640x480");
			return -EINVAL;
		}
	}

	/* Make sure width and height are a multiple of 8 */
	if (width % 8 || height % 8) {
		err("Invalid size (%d, %d) (mode = %d)", width, height, mode);
		return -EINVAL;
	}

	if (width < ov->minwidth || height < ov->minheight) {
		err("Requested dimensions are too small");
		return -EINVAL;
	}

	if (ov51x_stop(ov) < 0)
		return -EIO;

	if (mode == VIDEO_PALETTE_GREY) {
		reg_w(ov, R511_CAM_UV_EN, 0x00);
		reg_w(ov, R511_SNAP_UV_EN, 0x00);
		reg_w(ov, R511_SNAP_OPTS, 0x01);
	} else {
		reg_w(ov, R511_CAM_UV_EN, 0x01);
		reg_w(ov, R511_SNAP_UV_EN, 0x01);
		reg_w(ov, R511_SNAP_OPTS, 0x03);
	}

	/* Here I'm assuming that snapshot size == image size.
	 * I hope that's always true. --claudio
	 */
	hsegs = (width >> 3) - 1;
	vsegs = (height >> 3) - 1;

	reg_w(ov, R511_CAM_PXCNT, hsegs);
	reg_w(ov, R511_CAM_LNCNT, vsegs);
	reg_w(ov, R511_CAM_PXDIV, 0x00);
	reg_w(ov, R511_CAM_LNDIV, 0x00);

	/* YUV420, low pass filter on */
	reg_w(ov, R511_CAM_OPTS, 0x03);

	/* Snapshot additions */
	reg_w(ov, R511_SNAP_PXCNT, hsegs);
	reg_w(ov, R511_SNAP_LNCNT, vsegs);
	reg_w(ov, R511_SNAP_PXDIV, 0x00);
	reg_w(ov, R511_SNAP_LNDIV, 0x00);

	if (ov->compress) {
		/* Enable Y and UV quantization and compression */
		reg_w(ov, R511_COMP_EN, 0x07);
		reg_w(ov, R511_COMP_LUT_EN, 0x03);
		ov51x_reset(ov, OV511_RESET_OMNICE);
	}

	if (ov51x_restart(ov) < 0)
		return -EIO;

	return 0;
}

/* Sets up the OV518/OV518+ with the given image parameters
 *
 * OV518 needs a completely different approach, until we can figure out what
 * the individual registers do. Also, only 15 FPS is supported now.
 *
 * Do not put any sensor-specific code in here (including I2C I/O functions)
 */
static int
ov518_mode_init_regs(struct usb_ov511 *ov,
		     int width, int height, int mode, int sub_flag)
{
	int hsegs, vsegs, hi_res;

	if (sub_flag) {
		width = ov->subw;
		height = ov->subh;
	}

	PDEBUG(3, "width:%d, height:%d, mode:%d, sub:%d",
	       width, height, mode, sub_flag);

	if (width % 16 || height % 8) {
		err("Invalid size (%d, %d)", width, height);
		return -EINVAL;
	}

	if (width < ov->minwidth || height < ov->minheight) {
		err("Requested dimensions are too small");
		return -EINVAL;
	}

	if (width >= 320 && height >= 240) {
		hi_res = 1;
	} else if (width >= 320 || height >= 240) {
		err("Invalid width/height combination (%d, %d)", width, height);
		return -EINVAL;
	} else {
		hi_res = 0;
	}

	if (ov51x_stop(ov) < 0)
		return -EIO;

	/******** Set the mode ********/

	reg_w(ov, 0x2b, 0);
	reg_w(ov, 0x2c, 0);
	reg_w(ov, 0x2d, 0);
	reg_w(ov, 0x2e, 0);
	reg_w(ov, 0x3b, 0);
	reg_w(ov, 0x3c, 0);
	reg_w(ov, 0x3d, 0);
	reg_w(ov, 0x3e, 0);

	if (ov->bridge == BRG_OV518 && ov518_color) {
	 	if (mode == VIDEO_PALETTE_GREY) {
			/* Set 16-bit input format (UV data are ignored) */
			reg_w_mask(ov, 0x20, 0x00, 0x08);

			/* Set 8-bit (4:0:0) output format */
			reg_w_mask(ov, 0x28, 0x00, 0xf0);
			reg_w_mask(ov, 0x38, 0x00, 0xf0);
		} else {
			/* Set 8-bit (YVYU) input format */
			reg_w_mask(ov, 0x20, 0x08, 0x08);

			/* Set 12-bit (4:2:0) output format */
			reg_w_mask(ov, 0x28, 0x80, 0xf0);
			reg_w_mask(ov, 0x38, 0x80, 0xf0);
		}
	} else {
		reg_w(ov, 0x28, (mode == VIDEO_PALETTE_GREY) ? 0x00:0x80);
		reg_w(ov, 0x38, (mode == VIDEO_PALETTE_GREY) ? 0x00:0x80);
	}

	hsegs = width / 16;
	vsegs = height / 4;

	reg_w(ov, 0x29, hsegs);
	reg_w(ov, 0x2a, vsegs);

	reg_w(ov, 0x39, hsegs);
	reg_w(ov, 0x3a, vsegs);

	/* Windows driver does this here; who knows why */
	reg_w(ov, 0x2f, 0x80);

	/******** Set the framerate (to maximum) ********/

	/* Mode independent, but framerate dependent, regs */

	/* Clock divider; lower==faster */
	if (ov->bridge == BRG_OV518PLUS)
		reg_w(ov, 0x51, 0x02);
	else
		reg_w(ov, 0x51, 0x04);

	reg_w(ov, 0x22, 0x18);
	reg_w(ov, 0x23, 0xff);

	if (ov->bridge == BRG_OV518PLUS)
		reg_w(ov, 0x21, 0x1f);
	else
		reg_w(ov, 0x71, 0x17);	/* Compression-related? */

	// FIXME: Sensor-specific
	/* Bit 5 is what matters here. Of course, it is "reserved" */
	i2c_w(ov, 0x54, 0x23);

	reg_w(ov, 0x2f, 0x80);

	if (ov->bridge == BRG_OV518PLUS) {
		reg_w(ov, 0x24, 0x9f);
		reg_w(ov, 0x25, 0x90);
		ov518_reg_w32(ov, 0xc4,    700, 2);	/* 2bch   */
		ov518_reg_w32(ov, 0xc6,    498, 2);	/* 1f2h   */
		ov518_reg_w32(ov, 0xc7,    498, 2);	/* 1f2h   */
		ov518_reg_w32(ov, 0xc8,    100, 2);	/* 64h    */
		ov518_reg_w32(ov, 0xca, 183331, 3);	/* 2cc23h */
		ov518_reg_w32(ov, 0xcb,    895, 2);	/* 37fh   */
		ov518_reg_w32(ov, 0xcc,   3000, 2);	/* bb8h   */
		ov518_reg_w32(ov, 0xcd,     45, 2);	/* 2dh    */
		ov518_reg_w32(ov, 0xce,    851, 2);	/* 353h   */
	} else {
		reg_w(ov, 0x24, 0x9f);
		reg_w(ov, 0x25, 0x90);
		ov518_reg_w32(ov, 0xc4,    400, 2);	/* 190h   */
		ov518_reg_w32(ov, 0xc6,    381, 2);	/* 17dh   */
		ov518_reg_w32(ov, 0xc7,    381, 2);	/* 17dh   */
		ov518_reg_w32(ov, 0xc8,    128, 2);	/* 80h    */
		ov518_reg_w32(ov, 0xca, 183331, 3);	/* 2cc23h */
		ov518_reg_w32(ov, 0xcb,    746, 2);	/* 2eah   */
		ov518_reg_w32(ov, 0xcc,   1750, 2);	/* 6d6h   */
		ov518_reg_w32(ov, 0xcd,     45, 2);	/* 2dh    */
		ov518_reg_w32(ov, 0xce,    851, 2);	/* 353h   */
	}

	reg_w(ov, 0x2f, 0x80);

	if (ov51x_restart(ov) < 0)
		return -EIO;

	/* Reset it just for good measure */
	if (ov51x_reset(ov, OV511_RESET_NOREGS) < 0)
		return -EIO;

	return 0;
}

/* This is a wrapper around the OV511, OV518, and sensor specific functions */
static int
mode_init_regs(struct usb_ov511 *ov,
	       int width, int height, int mode, int sub_flag)
{
	int rc = 0;

	if (ov->bclass == BCL_OV518) {
		rc = ov518_mode_init_regs(ov, width, height, mode, sub_flag);
	} else {
		rc = ov511_mode_init_regs(ov, width, height, mode, sub_flag);
	}

	if (FATAL_ERROR(rc))
		return rc;

	switch (ov->sensor) {
	case CC_OV7610:
	case CC_OV7620:
	case CC_OV76BE:
	case CC_OV6620:
	case CC_OV6630:
	{
		rc = set_ov_sensor_window(ov, width, height, mode, sub_flag);
		break;
	}
	case SEN_SAA7111A:
	{
//		rc = mode_init_saa_sensor_regs(ov, width, height, mode,
//					       sub_flag);

		PDEBUG(1, "SAA status = 0x%02X", i2c_r(ov, 0x1f));
		break;
	}
	default:
		err("Unknown sensor");
		rc = -EINVAL;
	}

	if (FATAL_ERROR(rc))
		return rc;

	return 0;
}

/* Set up the camera chip with the options provided by the module params */
static int
camchip_init_settings(struct usb_ov511 *ov)
{
	int rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_AUTOBRIGHT, autobright);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_AUTOEXP, autoexp);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_BANDFILT, bandingfilter);
	if (FATAL_ERROR(rc))
		return rc;

	if (lightfreq) {
		rc = sensor_set_light_freq(ov, lightfreq);
		if (FATAL_ERROR(rc))
			return rc;
	}

	rc = sensor_set_control(ov, OVCAMCHIP_CID_BACKLIGHT, backlight);
	if (FATAL_ERROR(rc))
		return rc;

	rc = sensor_set_control(ov, OVCAMCHIP_CID_MIRROR, mirror);
	if (FATAL_ERROR(rc))
		return rc;

	return 0;
}

/* This sets the default image parameters. This is useful for apps that use
 * read() and do not set these.
 */
static int
ov51x_set_default_params(struct usb_ov511 *ov)
{
	int i;

	/* Set default sizes in case IOCTL (VIDIOCMCAPTURE) is not used
	 * (using read() instead). */
	for (i = 0; i < OV511_NUMFRAMES; i++) {
		ov->frame[i].width = ov->maxwidth;
		ov->frame[i].height = ov->maxheight;
		ov->frame[i].bytes_read = 0;
		if (force_palette)
			ov->frame[i].format = force_palette;
		else
			ov->frame[i].format = VIDEO_PALETTE_YUV420;
		ov->frame[i].depth = get_depth(ov->frame[i].format);
	}

	PDEBUG(3, "%dx%d, %s", ov->maxwidth, ov->maxheight,
	       symbolic(v4l1_plist, ov->frame[0].format));

	/* Initialize to max width/height, default palette */
	if (mode_init_regs(ov, ov->maxwidth, ov->maxheight,
			   ov->frame[0].format, 0) < 0)
		return -EINVAL;

	return 0;
}

#if defined(HAVE_V4L2)
static int
ov51x_control_op(struct usb_ov511 *ov, unsigned int cmd, void *arg)
{
	// No controls defined yet
	return -EINVAL;
}
#endif

/**********************************************************************
 *
 * Video decoder stuff
 *
 **********************************************************************/

/* Get ASCII name of video input */
static int
decoder_get_input_name(struct usb_ov511 *ov, int input, char *name)
{
	switch (ov->sensor) {
	case SEN_SAA7111A:
	{
		if (input < 0 || input > 7)
			return -EINVAL;
		else if (input < 4)
			sprintf(name, "CVBS-%d", input);
		else // if (input < 8)
			sprintf(name, "S-Video-%d", input - 4);

		break;
	}
	default:
		sprintf(name, "%s", "Camera");
	}

	return 0;
}

static int
ov51x_set_input(struct usb_ov511 *ov, int i)
{
	if (ov->has_decoder) {
		if (i < 0 || i >= ov->num_inputs)
			return -EINVAL;

		call_i2c_clients(ov, DECODER_SET_INPUT, &i);
		ov->input = i;
	} else {
		if (i != 0)
			return -EINVAL;
	}

	return 0;
}

/**********************************************************************
 *
 * Raw data parsing
 *
 **********************************************************************/

/* Copies a 64-byte segment at pIn to an 8x8 block at pOut. The width of the
 * image at pOut is specified by w.
 */
static inline void
make_8x8(unsigned char *pIn, unsigned char *pOut, int w)
{
	unsigned char *pOut1 = pOut;
	int x, y;

	for (y = 0; y < 8; y++) {
		pOut1 = pOut;
		for (x = 0; x < 8; x++) {
			*pOut1++ = *pIn++;
		}
		pOut += w;
	}
}

/*
 * For RAW BW (YUV 4:0:0) images, data show up in 256 byte segments.
 * The segments represent 4 squares of 8x8 pixels as follows:
 *
 *      0  1 ...  7    64  65 ...  71   ...  192 193 ... 199
 *      8  9 ... 15    72  73 ...  79        200 201 ... 207
 *           ...              ...                    ...
 *     56 57 ... 63   120 121 ... 127        248 249 ... 255
 *
 */ 
static void
yuv400raw_to_yuv400p(struct ov511_frame *frame,
		     unsigned char *pIn0, unsigned char *pOut0)
{
	int x, y;
	unsigned char *pIn, *pOut, *pOutLine;

	/* Copy Y */
	pIn = pIn0;
	pOutLine = pOut0;
	for (y = 0; y < frame->rawheight - 1; y += 8) {
		pOut = pOutLine;
		for (x = 0; x < frame->rawwidth - 1; x += 8) {
			make_8x8(pIn, pOut, frame->rawwidth);
			pIn += 64;
			pOut += 8;
		}
		pOutLine += 8 * frame->rawwidth;
	}
}

/*
 * For YUV 4:2:0 images, the data show up in 384 byte segments.
 * The first 64 bytes of each segment are U, the next 64 are V.  The U and
 * V are arranged as follows:
 *
 *      0  1 ...  7
 *      8  9 ... 15
 *           ...   
 *     56 57 ... 63
 *
 * U and V are shipped at half resolution (1 U,V sample -> one 2x2 block).
 *
 * The next 256 bytes are full resolution Y data and represent 4 squares
 * of 8x8 pixels as follows:
 *
 *      0  1 ...  7    64  65 ...  71   ...  192 193 ... 199
 *      8  9 ... 15    72  73 ...  79        200 201 ... 207
 *           ...              ...                    ...
 *     56 57 ... 63   120 121 ... 127   ...  248 249 ... 255
 *
 * Note that the U and V data in one segment represent a 16 x 16 pixel
 * area, but the Y data represent a 32 x 8 pixel area. If the width is not an
 * even multiple of 32, the extra 8x8 blocks within a 32x8 block belong to the
 * next horizontal stripe.
 *
 * If dumppix module param is set, _parse_data just dumps the incoming segments,
 * verbatim, in order, into the frame. When used with vidcat -f ppm -s 640x480
 * this puts the data on the standard output and can be analyzed with the
 * parseppm.c utility I wrote.  That's a much faster way for figuring out how
 * these data are scrambled.
 */

/* Converts from raw, uncompressed segments at pIn0 to a YUV420P frame at pOut0.
 *
 * FIXME: Currently only handles width and height that are multiples of 16
 */
static void
yuv420raw_to_yuv420p(struct ov511_frame *frame,
		     unsigned char *pIn0, unsigned char *pOut0)
{
	int k, x, y;
	unsigned char *pIn, *pOut, *pOutLine;
	const unsigned int a = frame->rawwidth * frame->rawheight;
	const unsigned int w = frame->rawwidth / 2;

	/* Copy U and V */
	pIn = pIn0;
	pOutLine = pOut0 + a;
	for (y = 0; y < frame->rawheight - 1; y += 16) {
		pOut = pOutLine;
		for (x = 0; x < frame->rawwidth - 1; x += 16) {
			make_8x8(pIn, pOut, w);
			make_8x8(pIn + 64, pOut + a/4, w);
			pIn += 384;
			pOut += 8;
		}
		pOutLine += 8 * w;
	}

	/* Copy Y */
	pIn = pIn0 + 128;
	pOutLine = pOut0;
	k = 0;
	for (y = 0; y < frame->rawheight - 1; y += 8) {
		pOut = pOutLine;
		for (x = 0; x < frame->rawwidth - 1; x += 8) {
			make_8x8(pIn, pOut, frame->rawwidth);
			pIn += 64;
			pOut += 8;
			if ((++k) > 3) {
				k = 0;
				pIn += 128;
			}
		}
		pOutLine += 8 * frame->rawwidth;
	}
}

/**********************************************************************
 *
 * Decompression
 *
 **********************************************************************/

/* Returns error if decompressor is not available */
static int
check_decompressor(struct usb_ov511 *ov)
{
	if (!ov)
		return -ENODEV;

	if (!ov->decomp_ops) {
		err("No decompressor available");
		return -ENOSYS;
	}

	return 0;
}

static void
decompress(struct usb_ov511 *ov, struct ov511_frame *frame,
	   unsigned char *pIn0, unsigned char *pOut0)
{
	if (check_decompressor(ov) < 0)
		return;

	PDEBUG(4, "Decompressing %d bytes", frame->bytes_recvd);

	if (frame->format == VIDEO_PALETTE_GREY
	    && ov->decomp_ops->decomp_400) {
		int ret = ov->decomp_ops->decomp_400(
			pIn0,
			pOut0,
			frame->compbuf,
			frame->rawwidth,
			frame->rawheight,
			frame->bytes_recvd);
		PDEBUG(4, "DEBUG: decomp_400 returned %d", ret);
	} else if (frame->format != VIDEO_PALETTE_GREY
		   && ov->decomp_ops->decomp_420) {
		int ret = ov->decomp_ops->decomp_420(
			pIn0,
			pOut0,
			frame->compbuf,
			frame->rawwidth,
			frame->rawheight,
			frame->bytes_recvd);
		PDEBUG(4, "DEBUG: decomp_420 returned %d", ret);
	} else {
		err("Decompressor does not support this format");
	}
}

/**********************************************************************
 *
 * Image processing
 *
 **********************************************************************/

/* Fuses even and odd fields together, and doubles width.
 * INPUT: an odd field followed by an even field at pIn0, in YUV planar format
 * OUTPUT: a normal YUV planar image, with correct aspect ratio
 */
static void
deinterlace(struct ov511_frame *frame, int rawformat,
            unsigned char *pIn0, unsigned char *pOut0)
{
	const int fieldheight = frame->rawheight / 2;
	const int fieldpix = fieldheight * frame->rawwidth;
	const int w = frame->width;
	int x, y;
	unsigned char *pInEven, *pInOdd, *pOut;

	PDEBUG(5, "fieldheight=%d", fieldheight);

	if (frame->rawheight != frame->height) {
		err("invalid height");
		return;
	}

	if ((frame->rawwidth * 2) != frame->width) {
		err("invalid width");
		return;
	}

	/* Y */
	pInOdd = pIn0;
	pInEven = pInOdd + fieldpix;
	pOut = pOut0;
	for (y = 0; y < fieldheight; y++) {
		for (x = 0; x < frame->rawwidth; x++) {
			*pOut = *pInEven;
			*(pOut+1) = *pInEven++;
			*(pOut+w) = *pInOdd;
			*(pOut+w+1) = *pInOdd++;
			pOut += 2;
		}
		pOut += w;
	}

	if (rawformat == RAWFMT_YUV420) {
	/* U */
		pInOdd = pIn0 + fieldpix * 2;
		pInEven = pInOdd + fieldpix / 4;
		for (y = 0; y < fieldheight / 2; y++) {
			for (x = 0; x < frame->rawwidth / 2; x++) {
				*pOut = *pInEven;
				*(pOut+1) = *pInEven++;
				*(pOut+w/2) = *pInOdd;
				*(pOut+w/2+1) = *pInOdd++;
				pOut += 2;
			}
			pOut += w/2;
		}
	/* V */
		pInOdd = pIn0 + fieldpix * 2 + fieldpix / 2;
		pInEven = pInOdd + fieldpix / 4;
		for (y = 0; y < fieldheight / 2; y++) {
			for (x = 0; x < frame->rawwidth / 2; x++) {
				*pOut = *pInEven;
				*(pOut+1) = *pInEven++;
				*(pOut+w/2) = *pInOdd;
				*(pOut+w/2+1) = *pInOdd++;
				pOut += 2;
			}
			pOut += w/2;
		}
	}
}

static void
ov51x_postprocess_grey(struct usb_ov511 *ov, struct ov511_frame *frame)
{
		/* Deinterlace frame, if necessary */
		if (ov->sensor == SEN_SAA7111A && frame->rawheight >= 480) {
			if (frame->compressed)
				decompress(ov, frame, frame->rawdata,
						 frame->tempdata);
			else
				yuv400raw_to_yuv400p(frame, frame->rawdata,
						     frame->tempdata);

			deinterlace(frame, RAWFMT_YUV400, frame->tempdata,
			            frame->data);
		} else {
			if (frame->compressed)
				decompress(ov, frame, frame->rawdata,
						 frame->data);
			else
				yuv400raw_to_yuv400p(frame, frame->rawdata,
						     frame->data);
		}
}

/* Process raw YUV420 data into standard YUV420P */
static void
ov51x_postprocess_yuv420(struct usb_ov511 *ov, struct ov511_frame *frame)
{
	/* Deinterlace frame, if necessary */
	if (ov->sensor == SEN_SAA7111A && frame->rawheight >= 480) {
		if (frame->compressed)
			decompress(ov, frame, frame->rawdata, frame->tempdata);
		else
			yuv420raw_to_yuv420p(frame, frame->rawdata,
					     frame->tempdata);

		deinterlace(frame, RAWFMT_YUV420, frame->tempdata,
		            frame->data);
	} else {
		if (frame->compressed)
			decompress(ov, frame, frame->rawdata, frame->data);
		else
			yuv420raw_to_yuv420p(frame, frame->rawdata,
					     frame->data);
	}
}

/* Post-processes the specified frame. This consists of:
 * 	1. Decompress frame, if necessary
 *	2. Deinterlace frame and scale to proper size, if necessary
 * 	3. Convert from YUV planar to destination format, if necessary
 * 	4. Fix the RGB offset, if necessary
 */
static void
ov51x_postprocess(struct usb_ov511 *ov, struct ov511_frame *frame)
{
	if (dumppix) {
		memset(frame->data, 0,
			MAX_DATA_SIZE(ov->maxwidth, ov->maxheight));
		PDEBUG(4, "Dumping %d bytes", frame->bytes_recvd);
		memcpy(frame->data, frame->rawdata, frame->bytes_recvd);
	} else {
		switch (frame->format) {
		case VIDEO_PALETTE_GREY:
			ov51x_postprocess_grey(ov, frame);
			break;
		case VIDEO_PALETTE_YUV420:
		case VIDEO_PALETTE_YUV420P:
			ov51x_postprocess_yuv420(ov, frame);
			break;
		default:
			err("Cannot convert data to %s",
			    symbolic(v4l1_plist, frame->format));
		}
	}
}

/**********************************************************************
 *
 * OV51x data transfer, IRQ handler
 *
 **********************************************************************/

static inline void
ov511_move_data(struct usb_ov511 *ov, unsigned char *in, int n)
{
	int num, offset;
	int pnum = in[ov->packet_size - 1];		/* Get packet number */
	int max_raw = MAX_RAW_DATA_SIZE(ov->maxwidth, ov->maxheight);
	struct ov511_frame *frame = &ov->frame[ov->curframe];
	struct timeval *ts;

	/* SOF/EOF packets have 1st to 8th bytes zeroed and the 9th
	 * byte non-zero. The EOF packet has image width/height in the
	 * 10th and 11th bytes. The 9th byte is given as follows:
	 *
	 * bit 7: EOF
	 *     6: compression enabled
	 *     5: 422/420/400 modes
	 *     4: 422/420/400 modes
	 *     3: 1
	 *     2: snapshot button on
	 *     1: snapshot frame
	 *     0: even/odd field
	 */

	if (printph) {
		info("ph(%3d): %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x",
		     pnum, in[0], in[1], in[2], in[3], in[4], in[5], in[6],
		     in[7], in[8], in[9], in[10], in[11]);
	}

	/* Check for SOF/EOF packet */
	if ((in[0] | in[1] | in[2] | in[3] | in[4] | in[5] | in[6] | in[7]) ||
	    (~in[8] & 0x08))
		goto check_middle;

	/* Frame end */
	if (in[8] & 0x80) {
		ts = (struct timeval *)(frame->data
		      + MAX_FRAME_SIZE(ov->maxwidth, ov->maxheight));
		do_gettimeofday(ts);

		/* Get the actual frame size from the EOF header */
		frame->rawwidth = ((int)(in[9]) + 1) * 8;
		frame->rawheight = ((int)(in[10]) + 1) * 8;

 		PDEBUG(4, "Frame end, frame=%d, pnum=%d, w=%d, h=%d, recvd=%d",
			ov->curframe, pnum, frame->rawwidth, frame->rawheight,
			frame->bytes_recvd);

		/* Validate the header data */
		RESTRICT_TO_RANGE(frame->rawwidth, ov->minwidth, ov->maxwidth);
		RESTRICT_TO_RANGE(frame->rawheight, ov->minheight,
				  ov->maxheight);

		/* Don't allow byte count to exceed buffer size */
		RESTRICT_TO_RANGE(frame->bytes_recvd, 8, max_raw);

		if (frame->scanstate == STATE_LINES) {
	    		int nextf;

			frame->grabstate = FRAME_DONE;
			wake_up_interruptible(&frame->wq);

			/* If next frame is ready or grabbing,
			 * point to it */
			nextf = (ov->curframe + 1) % OV511_NUMFRAMES;
			if (ov->frame[nextf].grabstate == FRAME_READY
			    || ov->frame[nextf].grabstate == FRAME_GRABBING) {
				ov->curframe = nextf;
				ov->frame[nextf].scanstate = STATE_SCANNING;
			} else {
				if (ov->frame[nextf].grabstate == FRAME_DONE) {
					PDEBUG(4, "No empty frames left");
				} else {
					PDEBUG(4, "Frame not ready? state = %d",
						ov->frame[nextf].grabstate);
				}

				ov->curframe = -1;
			}
		} else {
			PDEBUG(5, "Frame done, but not scanning");
		}
		/* Image corruption caused by misplaced frame->segment = 0
		 * fixed by carlosf@conectiva.com.br
		 */
	} else {
		/* Frame start */
		PDEBUG(4, "Frame start, framenum = %d", ov->curframe);

		/* Check to see if it's a snapshot frame */
		/* FIXME?? Should the snapshot reset go here? Performance? */
		if (in[8] & 0x02) {
			frame->snapshot = 1;
			PDEBUG(3, "snapshot detected");
		}

		frame->scanstate = STATE_LINES;
		frame->bytes_recvd = 0;
		frame->compressed = in[8] & 0x40;
	}

check_middle:
	/* Are we in a frame? */
	if (frame->scanstate != STATE_LINES) {
		PDEBUG(5, "Not in a frame; packet skipped");
		return;
	}

	/* If frame start, skip header */
	if (frame->bytes_recvd == 0)
		offset = 9;
	else
		offset = 0;

	num = n - offset - 1;

	/* Dump all data exactly as received */
	if (dumppix == 2) {
		frame->bytes_recvd += n - 1;
		if (frame->bytes_recvd <= max_raw)
			memcpy(frame->rawdata + frame->bytes_recvd - (n - 1),
				in, n - 1);
		else
			PDEBUG(3, "Raw data buffer overrun!! (%d)",
				frame->bytes_recvd - max_raw);
	} else if (!frame->compressed && !remove_zeros) {
		frame->bytes_recvd += num;
		if (frame->bytes_recvd <= max_raw)
			memcpy(frame->rawdata + frame->bytes_recvd - num,
				in + offset, num);
		else
			PDEBUG(3, "Raw data buffer overrun!! (%d)",
				frame->bytes_recvd - max_raw);
	} else { /* Remove all-zero FIFO lines (aligned 32-byte blocks) */
		int b, read = 0, allzero, copied = 0;
		if (offset) {
			frame->bytes_recvd += 32 - offset;	// Bytes out
			memcpy(frame->rawdata,	in + offset, 32 - offset);
			read += 32;
		}

		while (read < n - 1) {
			allzero = 1;
			for (b = 0; b < 32; b++) {
				if (in[read + b]) {
					allzero = 0;
					break;
				}
			}

			if (allzero) {
				/* Don't copy it */
			} else {
				if (frame->bytes_recvd + copied + 32 <= max_raw)
				{
					memcpy(frame->rawdata
						+ frame->bytes_recvd + copied,
						in + read, 32);
					copied += 32;
				} else {
					PDEBUG(3, "Raw data buffer overrun!!");
				}
			}
			read += 32;
		}

		frame->bytes_recvd += copied;
	}
}

static inline void
ov518_move_data(struct usb_ov511 *ov, unsigned char *in, int n)
{
	int max_raw = MAX_RAW_DATA_SIZE(ov->maxwidth, ov->maxheight);
	struct ov511_frame *frame = &ov->frame[ov->curframe];
	struct timeval *ts;

	/* Don't copy the packet number byte */
	if (ov->packet_numbering)
		--n;

	/* A false positive here is likely, until OVT gives me
	 * the definitive SOF/EOF format */
	if ((!(in[0] | in[1] | in[2] | in[3] | in[5])) && in[6]) {
		if (printph) {
			info("ph: %2x %2x %2x %2x %2x %2x %2x %2x", in[0],
			     in[1], in[2], in[3], in[4], in[5], in[6], in[7]);
		}

		if (frame->scanstate == STATE_LINES) {
			PDEBUG(4, "Detected frame end/start");
			goto eof;
		} else { //scanstate == STATE_SCANNING
			/* Frame start */
			PDEBUG(4, "Frame start, framenum = %d", ov->curframe);
			goto sof;
		}
	} else {
		goto check_middle;
	}

eof:
	ts = (struct timeval *)(frame->data
	      + MAX_FRAME_SIZE(ov->maxwidth, ov->maxheight));
	do_gettimeofday(ts);

	PDEBUG(4, "Frame end, curframe = %d, hw=%d, vw=%d, recvd=%d",
		ov->curframe,
		(int)(in[9]), (int)(in[10]), frame->bytes_recvd);

	// FIXME: Since we don't know the header formats yet,
	// there is no way to know what the actual image size is
	frame->rawwidth = frame->width;
	frame->rawheight = frame->height;

	/* Validate the header data */
	RESTRICT_TO_RANGE(frame->rawwidth, ov->minwidth, ov->maxwidth);
	RESTRICT_TO_RANGE(frame->rawheight, ov->minheight, ov->maxheight);

	/* Don't allow byte count to exceed buffer size */
	RESTRICT_TO_RANGE(frame->bytes_recvd, 8, max_raw);

	if (frame->scanstate == STATE_LINES) {
    		int nextf;

		frame->grabstate = FRAME_DONE;
		wake_up_interruptible(&frame->wq);

		/* If next frame is ready or grabbing,
		 * point to it */
		nextf = (ov->curframe + 1) % OV511_NUMFRAMES;
		if (ov->frame[nextf].grabstate == FRAME_READY
		    || ov->frame[nextf].grabstate == FRAME_GRABBING) {
			ov->curframe = nextf;
			ov->frame[nextf].scanstate = STATE_SCANNING;
			frame = &ov->frame[nextf];
		} else {
			if (ov->frame[nextf].grabstate == FRAME_DONE) {
				PDEBUG(4, "No empty frames left");
			} else {
				PDEBUG(4, "Frame not ready? state = %d",
				       ov->frame[nextf].grabstate);
			}

			ov->curframe = -1;
			PDEBUG(4, "SOF dropped (no active frame)");
			return;  /* Nowhere to store this frame */
		}
	}
sof:
	PDEBUG(4, "Starting capture on frame %d", frame->framenum);

// Snapshot not reverse-engineered yet.
#if 0
	/* Check to see if it's a snapshot frame */
	/* FIXME?? Should the snapshot reset go here? Performance? */
	if (in[8] & 0x02) {
		frame->snapshot = 1;
		PDEBUG(3, "snapshot detected");
	}
#endif
	frame->scanstate = STATE_LINES;
	frame->bytes_recvd = 0;
	frame->compressed = 1;

check_middle:
	/* Are we in a frame? */
	if (frame->scanstate != STATE_LINES) {
		PDEBUG(4, "scanstate: no SOF yet");
		return;
	}

	/* Dump all data exactly as received */
	if (dumppix == 2) {
		frame->bytes_recvd += n;
		if (frame->bytes_recvd <= max_raw)
			memcpy(frame->rawdata + frame->bytes_recvd - n, in, n);
		else
			PDEBUG(3, "Raw data buffer overrun!! (%d)",
				frame->bytes_recvd - max_raw);
	} else {
		/* All incoming data are divided into 8-byte segments. If the
		 * segment contains all zero bytes, it must be skipped. These
		 * zero-segments allow the OV518 to mainain a constant data rate
		 * regardless of the effectiveness of the compression. Segments
		 * are aligned relative to the beginning of each isochronous
		 * packet. The first segment in each image is a header (the
		 * decompressor skips it later).
		 */

		int b, read = 0, allzero, copied = 0;

		while (read < n) {
			allzero = 1;
			for (b = 0; b < 8; b++) {
				if (in[read + b]) {
					allzero = 0;
					break;
				}
			}

			if (allzero) {
			/* Don't copy it */
			} else {
				if (frame->bytes_recvd + copied + 8 <= max_raw)
				{
					memcpy(frame->rawdata
						+ frame->bytes_recvd + copied,
						in + read, 8);
					copied += 8;
				} else {
					PDEBUG(3, "Raw data buffer overrun!!");
				}
			}
			read += 8;
		}
		frame->bytes_recvd += copied;
	}
}

static void
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 51)
ov51x_isoc_irq(struct urb *urb, struct pt_regs *regs)
#else
ov51x_isoc_irq(struct urb *urb)
#endif
{
	int i;
	struct usb_ov511 *ov;
	struct ov511_sbuf *sbuf;

	if (!urb->context) {
		PDEBUG(4, "no context");
		return;
	}

	sbuf = urb->context;
	ov = sbuf->ov;

	if (!ov || !ov->dev || !ov->user) {
		PDEBUG(4, "no device, or not open");
		return;
	}

	if (!ov->streaming) {
		PDEBUG(4, "hmmm... not streaming, but got interrupt");
		return;
	}

        if (urb->status == -ENOENT || urb->status == -ECONNRESET) {
                PDEBUG(4, "URB unlinked");
                return;
        }

	if (urb->status != -EINPROGRESS && urb->status != 0) {
		err("ERROR: urb->status=%d: %s", urb->status,
		    symbolic(urb_errlist, urb->status));
	}

	/* Copy the data received into our frame buffer */
	PDEBUG(5, "sbuf[%d]: Moving %d packets", sbuf->n,
	       urb->number_of_packets);
	for (i = 0; i < urb->number_of_packets; i++) {
		/* Warning: Don't call *_move_data() if no frame active! */
		if (ov->curframe >= 0) {
			int n = urb->iso_frame_desc[i].actual_length;
			int st = urb->iso_frame_desc[i].status;
			unsigned char *cdata;

			urb->iso_frame_desc[i].actual_length = 0;
			urb->iso_frame_desc[i].status = 0;

			cdata = urb->transfer_buffer
				+ urb->iso_frame_desc[i].offset;

			if (!n) {
				PDEBUG(4, "Zero-length packet");
				continue;
			}

			if (st)
				PDEBUG(2, "data error: [%d] len=%d, status=%d",
				       i, n, st);

			if (ov->bclass == BCL_OV511)
				ov511_move_data(ov, cdata, n);
			else if (ov->bclass == BCL_OV518)
				ov518_move_data(ov, cdata, n);
			else
				err("Unknown bridge device (%d)", ov->bridge);

		} else if (waitqueue_active(&ov->wq)) {
			wake_up_interruptible(&ov->wq);
		}
	}

	/* Resubmit this URB */
	urb->dev = ov->dev;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 4)
	if ((i = usb_submit_urb(urb, GFP_ATOMIC)) != 0)
#else
	if ((i = usb_submit_urb(urb)) != 0)
#endif
		err("usb_submit_urb() ret %d", i);

	return;
}

/****************************************************************************
 *
 * Stream initialization and termination
 *
 ***************************************************************************/

static int
ov51x_init_isoc(struct usb_ov511 *ov)
{
	struct urb *urb;
	int fx, err, n, size;

	PDEBUG(3, "*** Initializing capture ***");

	ov->curframe = -1;

	if (ov->bridge == BRG_OV511) {
		if (cams == 1)				size = 993;
		else if (cams == 2)			size = 513;
		else if (cams == 3 || cams == 4)	size = 257;
		else {
			err("\"cams\" parameter too high!");
			return -1;
		}
	} else if (ov->bridge == BRG_OV511PLUS) {
		if (cams == 1)				size = 961;
		else if (cams == 2)			size = 513;
		else if (cams == 3 || cams == 4)	size = 257;
		else if (cams >= 5 && cams <= 8)	size = 129;
		else if (cams >= 9 && cams <= 31)	size = 33;
		else {
			err("\"cams\" parameter too high!");
			return -1;
		}
	} else if (ov->bclass == BCL_OV518) {
		if (cams == 1)				size = 896;
		else if (cams == 2)			size = 512;
		else if (cams == 3 || cams == 4)	size = 256;
		else if (cams >= 5 && cams <= 8)	size = 128;
		else {
			err("\"cams\" parameter too high!");
			return -1;
		}
	} else {
		err("invalid bridge type");
		return -1;
	}

	if (ov->bclass == BCL_OV518) {
		if (packetsize == -1) {
			ov518_set_packet_size(ov, 896);
		} else {
			info("Forcing packet size to %d", packetsize);
			ov518_set_packet_size(ov, packetsize);
		}
	} else {
		if (packetsize == -1) {
			ov511_set_packet_size(ov, size);
		} else {
			info("Forcing packet size to %d", packetsize);
			ov511_set_packet_size(ov, packetsize);
		}
	}

	for (n = 0; n < OV511_NUMSBUF; n++) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 5)
		urb = usb_alloc_urb(FRAMES_PER_DESC, GFP_KERNEL);
#else
		urb = usb_alloc_urb(FRAMES_PER_DESC);
#endif
		if (!urb) {
			err("init isoc: usb_alloc_urb ret. NULL");
			return -ENOMEM;
		}
		ov->sbuf[n].urb = urb;
		urb->dev = ov->dev;
		urb->context = &ov->sbuf[n];
		urb->pipe = usb_rcvisocpipe(ov->dev, OV511_ENDPOINT_ADDRESS);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 45)
		urb->transfer_flags = URB_ISO_ASAP;
#else
		urb->transfer_flags = USB_ISO_ASAP;
#endif
		urb->transfer_buffer = ov->sbuf[n].data;
		urb->complete = ov51x_isoc_irq;
		urb->number_of_packets = FRAMES_PER_DESC;
		urb->transfer_buffer_length = ov->packet_size * FRAMES_PER_DESC;
		urb->interval = 1;
		for (fx = 0; fx < FRAMES_PER_DESC; fx++) {
			urb->iso_frame_desc[fx].offset = ov->packet_size * fx;
			urb->iso_frame_desc[fx].length = ov->packet_size;
		}
	}

	ov->streaming = 1;

	for (n = 0; n < OV511_NUMSBUF; n++) {
		ov->sbuf[n].urb->dev = ov->dev;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 4)
		err = usb_submit_urb(ov->sbuf[n].urb, GFP_KERNEL);
#else
		err = usb_submit_urb(ov->sbuf[n].urb);
#endif
		if (err) {
			err("init isoc: usb_submit_urb(%d) ret %d", n, err);
			return err;
		}
	}

	return 0;
}

static void
ov51x_unlink_isoc(struct usb_ov511 *ov)
{
	int n;

	/* Unschedule all of the iso td's */
	for (n = OV511_NUMSBUF - 1; n >= 0; n--) {
		if (ov->sbuf[n].urb) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 8)
			usb_kill_urb(ov->sbuf[n].urb);
#else
			usb_unlink_urb(ov->sbuf[n].urb);
#endif
			usb_free_urb(ov->sbuf[n].urb);
			ov->sbuf[n].urb = NULL;
		}
	}
}

static void
ov51x_stop_isoc(struct usb_ov511 *ov)
{
	if (!ov->streaming)
		return;

	PDEBUG(3, "*** Stopping capture ***");

	if (ov->bclass == BCL_OV518)
		ov518_set_packet_size(ov, 0);
	else
		ov511_set_packet_size(ov, 0);

	ov->streaming = 0;

	ov51x_unlink_isoc(ov);
}

static int
ov51x_new_frame(struct usb_ov511 *ov, int framenum)
{
	struct ov511_frame *frame;
	int newnum;

	PDEBUG(4, "ov->curframe = %d, framenum = %d", ov->curframe, framenum);

	/* If we're not grabbing a frame right now and the other frame is */
	/* ready to be grabbed into, then use it instead */
	if (ov->curframe == -1) {
		newnum = (framenum - 1 + OV511_NUMFRAMES) % OV511_NUMFRAMES;
		if (ov->frame[newnum].grabstate == FRAME_READY)
			framenum = newnum;
	} else
		return 0;

	frame = &ov->frame[framenum];

	PDEBUG(4, "framenum = %d, width = %d, height = %d", framenum,
	       frame->width, frame->height);

	frame->grabstate = FRAME_GRABBING;
	frame->scanstate = STATE_SCANNING;
	frame->snapshot = 0;

	ov->curframe = framenum;

	/* Make sure it's not too big */
	if (frame->width > ov->maxwidth)
		frame->width = ov->maxwidth;

	frame->width &= ~7L;		/* Multiple of 8 */

	if (frame->height > ov->maxheight)
		frame->height = ov->maxheight;

	frame->height &= ~3L;		/* Multiple of 4 */

	return 0;
}

/****************************************************************************
 *
 * Buffer management
 *
 ***************************************************************************/

/*
 * - You must acquire buf_lock before entering this function.
 * - Because this code will free any non-null pointer, you must be sure to null
 *   them if you explicitly free them somewhere else!
 */
static void
ov51x_do_dealloc(struct usb_ov511 *ov)
{
	int i;
	PDEBUG(4, "entered");

	rvfree(ov->fbuf, OV511_NUMFRAMES
	       * MAX_DATA_SIZE(ov->maxwidth, ov->maxheight));
	ov->fbuf = NULL;

	vfree(ov->rawfbuf);
	ov->rawfbuf = NULL;

	vfree(ov->tempfbuf);
	ov->tempfbuf = NULL;

	for (i = 0; i < OV511_NUMSBUF; i++) {
		kfree(ov->sbuf[i].data);
		ov->sbuf[i].data = NULL;
	}

	for (i = 0; i < OV511_NUMFRAMES; i++) {
		ov->frame[i].data = NULL;
		ov->frame[i].rawdata = NULL;
		ov->frame[i].tempdata = NULL;
		if (ov->frame[i].compbuf) {
			free_page((unsigned long) ov->frame[i].compbuf);
			ov->frame[i].compbuf = NULL;
		}
	}

	PDEBUG(4, "buffer memory deallocated");
	ov->buf_state = BUF_NOT_ALLOCATED;
	PDEBUG(4, "leaving");
}

static int
ov51x_alloc(struct usb_ov511 *ov)
{
	int i;
	const int w = ov->maxwidth;
	const int h = ov->maxheight;
	const int data_bufsize = OV511_NUMFRAMES * MAX_DATA_SIZE(w, h);
	const int raw_bufsize = OV511_NUMFRAMES * MAX_RAW_DATA_SIZE(w, h);

	PDEBUG(4, "entered");
	down(&ov->buf_lock);

	if (ov->buf_state == BUF_ALLOCATED)
		goto out;

	ov->fbuf = rvmalloc(data_bufsize);
	if (!ov->fbuf)
		goto error;

	ov->rawfbuf = vmalloc(raw_bufsize);
	if (!ov->rawfbuf)
		goto error;

	memset(ov->rawfbuf, 0, raw_bufsize);

	ov->tempfbuf = vmalloc(raw_bufsize);
	if (!ov->tempfbuf)
		goto error;

	memset(ov->tempfbuf, 0, raw_bufsize);

	for (i = 0; i < OV511_NUMSBUF; i++) {
		ov->sbuf[i].data = kmalloc(FRAMES_PER_DESC *
			MAX_FRAME_SIZE_PER_DESC, GFP_KERNEL);
		if (!ov->sbuf[i].data)
			goto error;

		PDEBUG(4, "sbuf[%d] @ %p", i, ov->sbuf[i].data);
	}

	for (i = 0; i < OV511_NUMFRAMES; i++) {
		ov->frame[i].data = ov->fbuf + i * MAX_DATA_SIZE(w, h);
		ov->frame[i].rawdata = ov->rawfbuf
		 + i * MAX_RAW_DATA_SIZE(w, h);
		ov->frame[i].tempdata = ov->tempfbuf
		 + i * MAX_RAW_DATA_SIZE(w, h);

		ov->frame[i].compbuf =
		 (unsigned char *) __get_free_page(GFP_KERNEL);
		if (!ov->frame[i].compbuf)
			goto error;

		PDEBUG(4, "frame[%d] @ %p", i, ov->frame[i].data);
	}

	ov->buf_state = BUF_ALLOCATED;
out:
	up(&ov->buf_lock);
	PDEBUG(4, "leaving");
	return 0;
error:
	ov51x_do_dealloc(ov);
	up(&ov->buf_lock);
	PDEBUG(4, "errored");
	return -ENOMEM;
}

static void
ov51x_dealloc(struct usb_ov511 *ov)
{
	PDEBUG(4, "entered");
	down(&ov->buf_lock);
	ov51x_do_dealloc(ov);
	up(&ov->buf_lock);
	PDEBUG(4, "leaving");
}

/****************************************************************************
 *
 * V4L API
 *
 ***************************************************************************/

#if defined(HAVE_V4L2)
static const struct v4l2_fmtdesc ov51x_fmt_grey = {
	.description = "8 bpp, gray",
	.pixelformat = V4L2_PIX_FMT_GREY,
};

static const struct v4l2_fmtdesc ov51x_fmt_yuv420 = {
	.description = "4:2:0, planar, Y-Cb-Cr",
	.pixelformat = V4L2_PIX_FMT_YUV420,
};
#endif

static int
ov51x_open(struct inode *inode, struct file *file)
{
	struct video_device *vdev = video_devdata(file);
	struct usb_ov511 *ov = video_get_drvdata(vdev);
	int err, i;

	PDEBUG(4, "opening");

	down(&ov->lock);

	err = -ENODEV;
	if (!ov->dev)
		goto out;

	if (ov->sensor == CC_UNKNOWN) {
		err("No sensor is detected yet");
		goto out;
	}

	err = -EBUSY;
	if (ov->user)
		goto out;

	ov->sub_flag = 0;

	/* In case app doesn't set them... */
	err = ov51x_set_default_params(ov);
	if (err < 0)
		goto out;

	/* Make sure frames are reset */
	for (i = 0; i < OV511_NUMFRAMES; i++) {
		ov->frame[i].grabstate = FRAME_UNUSED;
		ov->frame[i].bytes_read = 0;
	}

	/* If compression is on, make sure that decompressor is compiled in */
	if (ov->compress && !dumppix) {
		err = check_decompressor(ov);
		if (err < 0)
			goto out;
	}

	err = ov51x_alloc(ov);
	if (err < 0)
		goto out;

	err = ov51x_init_isoc(ov);
	if (err) {
		ov51x_dealloc(ov);
		goto out;
	}

	ov->user++;
	file->private_data = vdev;

	if (ov->led_policy == LED_AUTO)
		ov51x_led_control(ov, 1);

out:
	up(&ov->lock);

	return err;
}

static int
ov51x_release(struct inode *inode, struct file *file)
{
	struct video_device *vdev = file->private_data;
	struct usb_ov511 *ov = video_get_drvdata(vdev);

	PDEBUG(4, "ov511_close");

	down(&ov->lock);

	ov->user--;
	ov51x_stop_isoc(ov);

	if (ov->led_policy == LED_AUTO)
		ov51x_led_control(ov, 0);

	ov51x_dealloc(ov);

	up(&ov->lock);

	/* Device unplugged while open. Only a minimum of unregistration is done
	 * here; the disconnect callback already did the rest. */
	down(&ov_free_lock);
	if (!ov->dev) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
		if (ov->vdev)
			video_device_release(ov->vdev);
#endif
		kfree(ov);
		ov = NULL;
	}
	up(&ov_free_lock);

	file->private_data = NULL;
	return 0;
}

/* Do not call this function directly! */
static int
ov51x_do_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
               void *arg)
{
	struct video_device *vdev = file->private_data;
	struct usb_ov511 *ov = video_get_drvdata(vdev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16)
	switch (_IOC_TYPE(cmd)) {
	case 'v':
		PDEBUG(4, "ioctl 0x%x (v4l1, VIDIOC%s)", cmd,
		       (_IOC_NR(cmd) < NUM_V4L1_IOCTLS) ?
		       v4l1_ioctls[_IOC_NR(cmd)] : "???");
		break;
#  if defined(HAVE_V4L2)
	case 'V':
		PDEBUG(4, "ioctl 0x%x (v4l2, %s)", cmd,
		       v4l2_ioctl_names[_IOC_NR(cmd)]);
		break;
#  endif
	default:
		PDEBUG(4, "ioctl 0x%x (?)", cmd);
	}
#else
	if (debug >= 4)
		v4l_print_ioctl("ov511", cmd);
#endif

	if (!ov->dev)
		return -EIO;

	switch (cmd) {
	case VIDIOCGCAP:
	{
		struct video_capability *b = arg;

		memset(b, 0, sizeof(struct video_capability));
		sprintf(b->name, "%s USB Camera",
			symbolic(brglist, ov->bridge));
		b->type = VID_TYPE_CAPTURE | VID_TYPE_SUBCAPTURE;
		if (ov->has_tuner)
			b->type |= VID_TYPE_TUNER;
		b->channels = ov->num_inputs;
		b->audios = ov->has_audiochip ? 1:0;
		b->maxwidth = ov->maxwidth;
		b->maxheight = ov->maxheight;
		b->minwidth = ov->minwidth;
		b->minheight = ov->minheight;

		return 0;
	}
	case VIDIOCGCHAN:
	{
		struct video_channel *v = arg;

		if ((unsigned)(v->channel) >= ov->num_inputs)
			return -EINVAL;

		v->norm = ov->norm;
		v->type = (ov->has_tuner) ? VIDEO_TYPE_TV : VIDEO_TYPE_CAMERA;
		v->flags = (ov->has_tuner) ? VIDEO_VC_TUNER : 0;
		v->flags |= (ov->has_audiochip) ? VIDEO_VC_AUDIO : 0;
		v->tuners = (ov->has_tuner) ? 1:0;
		decoder_get_input_name(ov, v->channel, v->name);

		return 0;
	}
	case VIDIOCSCHAN:
	{
		struct video_channel *v = arg;
		int norm, rc;

		if (v->norm != VIDEO_MODE_PAL &&
		    v->norm != VIDEO_MODE_NTSC &&
		    v->norm != VIDEO_MODE_SECAM &&
		    v->norm != VIDEO_MODE_AUTO) {
			err("Invalid norm (%d)", v->norm);
			return -EINVAL;
		}

		rc = ov51x_set_input(ov, v->channel);
		if (rc < 0)
			return rc;

		/* Don't pass a __u16* */
		norm = v->norm;
		call_i2c_clients(ov, DECODER_SET_NORM, &norm);

		return 0;
	}
	case VIDIOCGPICT:
	{
		struct video_picture *p = arg;

		memset(p, 0, sizeof(struct video_picture));
		if (sensor_get_picture(ov, p))
			return -EIO;

		/* Can we get these from frame[0]? -claudio? */
		p->depth = ov->frame[0].depth;
		p->palette = ov->frame[0].format;

		return 0;
	}
	case VIDIOCSPICT:
	{
		struct video_picture *p = arg;
		int i, rc;

		if (!get_depth(p->palette))
			return -EINVAL;

		if (sensor_set_picture(ov, p))
			return -EIO;

		if (force_palette && p->palette != force_palette) {
			info("Palette rejected (%s)",
			     symbolic(v4l1_plist, p->palette));
			return -EINVAL;
		}

		// FIXME: Format should be independent of frames
		if (p->palette != ov->frame[0].format) {
			PDEBUG(4, "Detected format change");

			rc = ov51x_wait_frames_inactive(ov);
			if (rc)
				return rc;

			mode_init_regs(ov, ov->frame[0].width,
				ov->frame[0].height, p->palette, ov->sub_flag);
		}

		PDEBUG(4, "Setting depth=%d, palette=%s",
		       p->depth, symbolic(v4l1_plist, p->palette));

		for (i = 0; i < OV511_NUMFRAMES; i++) {
			ov->frame[i].depth = p->depth;
			ov->frame[i].format = p->palette;
		}

		return 0;
	}
	case VIDIOCGCAPTURE:
	{
		int *vf = arg;

		ov->sub_flag = *vf;
		return 0;
	}
	case VIDIOCSCAPTURE:
	{
		struct video_capture *vc = arg;

		if (vc->flags || vc->decimation)
			return -EINVAL;

		vc->x &= ~3L;
		vc->y &= ~1L;
		vc->y &= ~31L;

		if (vc->width == 0)
			vc->width = 32;

		vc->height /= 16;
		vc->height *= 16;
		if (vc->height == 0)
			vc->height = 16;

		ov->subx = vc->x;
		ov->suby = vc->y;
		ov->subw = vc->width;
		ov->subh = vc->height;

		return 0;
	}
	case VIDIOCSWIN:
	{
		struct video_window *vw = arg;
		int i, rc;

		PDEBUG(4, "Set window: %dx%d", vw->width, vw->height);

/* This code breaks a few apps (Gnomemeeting, possibly
  others). Disable it until a solution can be found */
#if 0
		if (vw->flags || vw->clipcount)
			return -EINVAL;
#endif
		rc = ov51x_wait_frames_inactive(ov);
		if (rc)
			return rc;

		rc = mode_init_regs(ov, vw->width, vw->height,
			ov->frame[0].format, ov->sub_flag);
		if (rc < 0)
			return rc;

		for (i = 0; i < OV511_NUMFRAMES; i++) {
			ov->frame[i].width = vw->width;
			ov->frame[i].height = vw->height;
		}

		return 0;
	}
	case VIDIOCGWIN:
	{
		struct video_window *vw = arg;

		memset(vw, 0, sizeof(struct video_window));
		vw->x = 0;		/* FIXME */
		vw->y = 0;
		vw->width = ov->frame[0].width;
		vw->height = ov->frame[0].height;

		PDEBUG(4, "Get window: %dx%d", vw->width, vw->height);

		return 0;
	}
	case VIDIOCGMBUF:
	{
		struct video_mbuf *vm = arg;
		int i;

		memset(vm, 0, sizeof(struct video_mbuf));
		vm->size = OV511_NUMFRAMES
			   * MAX_DATA_SIZE(ov->maxwidth, ov->maxheight);
		vm->frames = OV511_NUMFRAMES;

		vm->offsets[0] = 0;
		for (i = 1; i < OV511_NUMFRAMES; i++) {
			vm->offsets[i] = vm->offsets[i-1]
			   + MAX_DATA_SIZE(ov->maxwidth, ov->maxheight);
		}

		return 0;
	}
	case VIDIOCMCAPTURE:
	{
		struct video_mmap *vm = arg;
		int rc, depth;
		unsigned int f = vm->frame;

		PDEBUG(4, "MCAPTURE: frame: %d, %dx%d, %s", f, vm->width,
			vm->height, symbolic(v4l1_plist, vm->format));

		depth = get_depth(vm->format);
		if (!depth) {
			PDEBUG(2, "MCAPTURE: invalid format (%s)",
			       symbolic(v4l1_plist, vm->format));
			return -EINVAL;
		}

		if (f >= OV511_NUMFRAMES) {
			err("MCAPTURE: invalid frame (%d)", f);
			return -EINVAL;
		}

		if (vm->width > ov->maxwidth
		    || vm->height > ov->maxheight) {
			err("MCAPTURE: requested dimensions too big");
			return -EINVAL;
		}

		if (ov->frame[f].grabstate == FRAME_GRABBING) {
			PDEBUG(4, "MCAPTURE: already grabbing");
			return -EBUSY;
		}

		if (force_palette && (vm->format != force_palette)) {
			PDEBUG(2, "MCAPTURE: palette rejected (%s)",
			       symbolic(v4l1_plist, vm->format));
			return -EINVAL;
		}

		if ((ov->frame[f].width != vm->width) ||
		    (ov->frame[f].height != vm->height) ||
		    (ov->frame[f].format != vm->format) ||
		    (ov->frame[f].sub_flag != ov->sub_flag) ||
		    (ov->frame[f].depth != depth)) {
			PDEBUG(4, "MCAPTURE: change in image parameters");

			rc = ov51x_wait_frames_inactive(ov);
			if (rc)
				return rc;

			rc = mode_init_regs(ov, vm->width, vm->height,
				vm->format, ov->sub_flag);
#if 0
			if (rc < 0) {
				PDEBUG(1, "Got error while initializing regs ");
				return rc;
			}
#endif
			ov->frame[f].width = vm->width;
			ov->frame[f].height = vm->height;
			ov->frame[f].format = vm->format;
			ov->frame[f].sub_flag = ov->sub_flag;
			ov->frame[f].depth = depth;
		}

		/* Mark it as ready */
		ov->frame[f].grabstate = FRAME_READY;

		return ov51x_new_frame(ov, f);
	}
	case VIDIOCSYNC:
	{
		unsigned int fnum = *((unsigned int *) arg);
		struct ov511_frame *frame;
		int rc;

		if (fnum >= OV511_NUMFRAMES) {
			err("SYNC: invalid frame (%d)", fnum);
			return -EINVAL;
		}

		frame = &ov->frame[fnum];

		PDEBUG(4, "syncing to frame %d, grabstate = %d", fnum,
		       frame->grabstate);

		switch (frame->grabstate) {
		case FRAME_UNUSED:
			return -EINVAL;
		case FRAME_READY:
		case FRAME_GRABBING:
		case FRAME_ERROR:
redo:
			rc = wait_event_interruptible(frame->wq,
			    (frame->grabstate == FRAME_DONE)
			    || (frame->grabstate == FRAME_ERROR)
			    || !ov->present);

			if (rc)
				return rc;

			if (!ov->present)
				return -ENODEV;

			if (frame->grabstate == FRAME_ERROR) {
				if ((rc = ov51x_new_frame(ov, fnum)) < 0)
					return rc;
				goto redo;
			}
			/* Fall through */
		case FRAME_DONE:
			if (ov->snap_enabled && !frame->snapshot) {
				if ((rc = ov51x_new_frame(ov, fnum)) < 0)
					return rc;
				goto redo;
			}

			frame->grabstate = FRAME_UNUSED;

			/* Reset the hardware snapshot button */
			/* FIXME - Is this the best place for this? */
			if ((ov->snap_enabled) && (frame->snapshot)) {
				frame->snapshot = 0;
				ov51x_clear_snapshot(ov);
			}

			/* Decompression, format conversion, etc... */
			ov51x_postprocess(ov, frame);

			break;
		} /* end switch */

		return 0;
	}
	case VIDIOCGFBUF:
	{
		struct video_buffer *vb = arg;

		memset(vb, 0, sizeof(struct video_buffer));

		return 0;
	}
	case VIDIOCGUNIT:
	{
		struct video_unit *vu = arg;

		memset(vu, 0, sizeof(struct video_unit));

		vu->video = ov->vdev->minor;
		vu->vbi = VIDEO_NO_UNIT;
		vu->radio = VIDEO_NO_UNIT;
		vu->audio = VIDEO_NO_UNIT;
		vu->teletext = VIDEO_NO_UNIT;

		return 0;
	}
	case VIDIOCGTUNER:
	{
		struct video_tuner *v = arg;

		if (!ov->has_tuner || v->tuner)	// Only tuner 0
			return -EINVAL;

		strcpy(v->name, "Television");

		// FIXME: Need a way to get the real values
		v->rangelow = 0;
		v->rangehigh = ~0;

		v->flags = VIDEO_TUNER_PAL | VIDEO_TUNER_NTSC
			   | VIDEO_TUNER_SECAM;
		v->mode = 0; 		/* FIXME:  Not sure what this is yet */
		v->signal = 0xFFFF;	/* unknown */

		call_i2c_clients(ov, cmd, v);

		return 0;
	}
	case VIDIOCSTUNER:
	{
		struct video_tuner *v = arg;

		/* Only no or one tuner for now */
		if (!ov->has_tuner || v->tuner)
			return -EINVAL;

		/* and it only has certain valid modes */
		if (v->mode != VIDEO_MODE_PAL &&
		    v->mode != VIDEO_MODE_NTSC &&
		    v->mode != VIDEO_MODE_SECAM)
			return -EOPNOTSUPP;

		call_i2c_clients(ov, cmd, v);

		return 0;
	}
	case VIDIOCGFREQ:
	{
		unsigned long v = *((unsigned long *) arg);

		if (!ov->has_tuner)
			return -EINVAL;

		v = ov->freq;
#if 0
		/* FIXME: this is necessary for testing */
		v = 46*16;
#endif
		return 0;
	}
	case VIDIOCSFREQ:
	{
		unsigned long v = *((unsigned long *) arg);

		if (!ov->has_tuner)
			return -EINVAL;

		ov->freq = v;
		call_i2c_clients(ov, cmd, &v);

		return 0;
	}
	case VIDIOCGAUDIO:
	case VIDIOCSAUDIO:
	{
		if (!ov->has_audiochip)
			return -EINVAL;

		call_i2c_clients(ov, cmd, arg);

		return 0;
	}
#if defined(HAVE_V4L2)
	case VIDIOC_QUERYCAP:
	{
		struct v4l2_capability *c = arg;

		if (!v4l2)
			return -EINVAL;

		strcpy(c->driver, "ov511");
		snprintf(c->card, sizeof(c->card), "%s/%s (cid=%d)",
			symbolic(brglist, ov->bridge),
			symbolic(senlist, ov->sensor),
			ov->customid);
		strncpy(c->bus_info, ov->usb_path, sizeof(c->bus_info));
		c->version = DRIVER_VERSION_CODE;
		c->capabilities =
			V4L2_CAP_VIDEO_CAPTURE |
			V4L2_CAP_READWRITE;

// FIXME: Implement this later
//		c->capabilities |= V4L2_CAP_STREAMING;

#if 0	// FIXME: Implement these ioctls
		if (ov->has_tuner)
			c->capabilities |= V4L2_CAP_TUNER;
		if (ov->has_audiochip)
			c->capabilities |= V4L2_CAP_AUDIO;
#endif
		memset(&c->reserved, 0, sizeof(c->reserved));

		return 0;
	}
	case VIDIOC_CROPCAP:
	{
		struct v4l2_cropcap *c = arg;

		if (c->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		c->bounds.left = 0;
		c->bounds.top = 0;
		c->bounds.width = ov->maxwidth;
		c->bounds.height = ov->maxheight;

		c->defrect.left = 0;
		c->defrect.top = 0;
		c->defrect.width = ov->maxwidth;
		c->defrect.height = ov->maxheight;

		return 0;
	}
	case VIDIOC_ENUM_FMT:
	{
		struct v4l2_fmtdesc *f = arg;
		const struct v4l2_fmtdesc *ptr = NULL;
		int index = f->index;

		if (f->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		if (force_palette) {
			if (index)
				return -EINVAL;

			switch (force_palette) {
			case 1:  ptr = &ov51x_fmt_grey;   break;
			case 10:
			case 15: ptr = &ov51x_fmt_yuv420; break;
			}
		} else if (ov->bclass == BCL_OV511) {
			switch (index) {
			case 0:  ptr = &ov51x_fmt_grey;   break;
			case 1:  ptr = &ov51x_fmt_yuv420; break;
			}
		} else if (ov->bclass == BCL_OV518) {
			switch (index) {
			case 0:  ptr = &ov51x_fmt_yuv420; break;
			}
		}

		if (ptr)
			memcpy(f, ptr, sizeof(*f));
		else
			return -EINVAL;

		f->index = index;
		f->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

		return 0;
	}
	case VIDIOC_G_FMT:
	{
		struct v4l2_format *f = arg;

		if (f->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		memset(&f->fmt.pix, 0, sizeof(struct v4l2_pix_format));
		f->fmt.pix.width = ov->frame[0].width;
		f->fmt.pix.height = ov->frame[0].height;

		switch (ov->frame[0].format) {
		case VIDEO_PALETTE_GREY:
			f->fmt.pix.pixelformat = V4L2_PIX_FMT_GREY;
			f->fmt.pix.bytesperline = f->fmt.pix.width;
			f->fmt.pix.sizeimage = f->fmt.pix.bytesperline *
			    f->fmt.pix.height;
			break;
		case VIDEO_PALETTE_YUV420:
		case VIDEO_PALETTE_YUV420P:
			f->fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
			f->fmt.pix.bytesperline = f->fmt.pix.width;
			f->fmt.pix.sizeimage = (f->fmt.pix.bytesperline *
			    f->fmt.pix.height * 3) / 2 ;
			break;
		default:
			err("Bad pixel format");
			return -ENODATA;
		}

		f->fmt.pix.field = V4L2_FIELD_NONE;

		// FIXME: This is probably only true for some sensors
		f->fmt.pix.colorspace = V4L2_COLORSPACE_JPEG;

		return 0;
	}
	case VIDIOC_S_FMT:
	{
		struct v4l2_format *f = arg;
		int format, i;

		if (f->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		switch (f->fmt.pix.pixelformat) {
		case V4L2_PIX_FMT_GREY:
			format = VIDEO_PALETTE_GREY;
			f->fmt.pix.bytesperline = f->fmt.pix.width;
			f->fmt.pix.sizeimage = f->fmt.pix.bytesperline *
			    f->fmt.pix.height;
			break;
		case V4L2_PIX_FMT_YUV420:
			format = VIDEO_PALETTE_YUV420P;
			f->fmt.pix.bytesperline = f->fmt.pix.width;
			f->fmt.pix.sizeimage = (f->fmt.pix.bytesperline *
			    f->fmt.pix.height * 3) / 2 ;
			break;
		default:
			warn("App requested invalid pix format: 0x%x",
			     f->fmt.pix.pixelformat);
			/* According to V4L2 spec, we're allowed to return
			 * -EINVAL iff v4l2_format is ambiguous */ 
			return -EINVAL;
		}

		if (force_palette && (format != force_palette)) {
			PDEBUG(2, "palette rejected (%s)",
			       symbolic(v4l1_plist, format));
			return -EINVAL;
		}

		f->fmt.pix.field = V4L2_FIELD_NONE;

		// FIXME: This is probably only true for some sensors
		f->fmt.pix.colorspace = V4L2_COLORSPACE_JPEG;

		// FIXME: This sucks. mode_init_regs() should handle this
		for (i = 0; i < OV511_NUMFRAMES; i++) {
			ov->frame[i].width = f->fmt.pix.width;
			ov->frame[i].height = f->fmt.pix.height;
			ov->frame[i].sub_flag = 0;
			ov->frame[i].depth = get_depth(format);
			ov->frame[i].format = format;
		}

		// FIXME: This function gives up rather than forcing the
		// closest supported size.
		return mode_init_regs(ov, f->fmt.pix.width, f->fmt.pix.height,
		    format, 0);
	}
	case VIDIOC_ENUMINPUT:
	{
		struct v4l2_input *i = arg;
		__u32 n;

		n = i->index;
		if (n >= ov->num_inputs)
			return -EINVAL;

		memset(i, 0, sizeof(*i));
		i->index = n;
		decoder_get_input_name(ov, n, i->name);
		if (ov->has_tuner) {		// FIXME: this isn't quite right
			i->type = V4L2_INPUT_TYPE_TUNER;
			i->audioset = 0;	// FIXME
			i->tuner = 0;		// FIXME
			i->std = V4L2_STD_ALL;	// FIXME
		} else if (ov->has_decoder) {
			i->type = V4L2_INPUT_TYPE_CAMERA;
			i->std = V4L2_STD_ALL;	// FIXME
		} else {
			i->type = V4L2_INPUT_TYPE_CAMERA;
			i->std = V4L2_STD_UNKNOWN;
		}

		return 0;
	}
	case VIDIOC_G_INPUT:
	{
		*((int *)arg) = ov->input;
		return 0;		
	}
	case VIDIOC_S_INPUT:
	{
		return ov51x_set_input(ov, *((int *)arg));
	}
	case VIDIOC_QUERYCTRL:
	case VIDIOC_S_CTRL:
	case VIDIOC_G_CTRL:
	{
		struct v4l2_queryctrl *c = arg;
		int rc;

		/* Since controls are implemented at multiple driver layers,
		 * we need translate the IDs in some cases and route them
		 * accordingly. Apps should find private controls by name
		 * rather than by ID */
		if (c->id >= V4L2_CID_BASE || c->id < V4L2_CID_LASTP1) {
			rc = ov51x_control_op(ov, cmd, c);
			if (rc == -EINVAL)
				if (ov->sensor != SEN_SAA7111A)
					rc = sensor_cmd(ov, cmd, c);
		} else if ((c->id >= V4L2_CID_PRIVATE_BASE) &&
		           (c->id < OV51X_CID_LASTP1)) {
			rc = ov51x_control_op(ov, cmd, c);
		} else if (c->id < (SENSOR_CID_OFFSET +
		                    OVCAMCHIP_V4L2_CID_LASTP1)) {
			c->id += SENSOR_CID_OFFSET;
			rc = sensor_cmd(ov, cmd, c);
			c->id -= SENSOR_CID_OFFSET;
		} else {
			rc = -EINVAL;
		}


		return rc;
	}
// mmap() capture disabled until select() and rest of ioctls are implemented
#if 0
	case VIDIOC_REQBUFS:
	{
		struct v4l2_requestbuffers *rb = arg;

		if (rb->type != V4L2_BUF_TYPE_VIDEO_CAPTURE ||
		    rb->memory != V4L2_MEMORY_MMAP)
			return -EINVAL;

		// FIXME: This shouldn't be hardcoded
		rb->count = OV511_NUMFRAMES;

//		memset(rb->reserved, 0, sizeof(*(rb->reserved)));

		// FIXME: Driver should check that this ioctl was called
		// before allowing buffers queued or queried
		return 0;
	}
	case VIDIOC_STREAMON:
	{
		int type = *((int *)arg);
		int i;

		if (type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;
		
		// FIXME: ISO streaming shouldn't be started until this ioctl
		for (i = 1; i < OV511_NUMFRAMES; i++)
			if (ov->frame[i].grabstate == FRAME_UNUSED)
				ov->frame[i].grabstate = FRAME_READY; // Racy!!

		return 0;
	}
	case VIDIOC_STREAMOFF:
	{
		int type = *((int *)arg);
		int i;

		if (type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		// FIXME: ISO streaming should be stopped by this ioctl
		for (i = 1; i < OV511_NUMFRAMES; i++)
			ov->frame[i].grabstate = FRAME_UNUSED;

		return 0;
	}
#endif
// Incomplete code
#if 0
	case: VIDIOC_QBUF:
	{
		struct v4l2_buffer *buf = arg;
		struct ov511_frame *frame;
		int rc;

		if (buf->index >= OV511_NUMFRAMES) {
			PDEBUG(2, "VIDIOC_QBUF: invalid index: %d", buf->index);
			return -EINVAL;
		}
		frame = &ov->frame[buf->index];

		if (buf->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		switch (frame->grabstate) {
		case FRAME_READY:
		case FRAME_GRABBING:
			buf->flags |= V4L2_BUF_FLAG_MAPPED;
			buf->flags |= V4L2_BUF_FLAG_QUEUED;
			buf->flags = buf->flags &~ V4L2_BUF_FLAG_DONE;
		}

		return 0;
	}
	case: VIDIOC_DQBUF:
	{
		struct v4l2_buffer *buf = arg;

		if (buf->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
			return -EINVAL;

		return 0;
	}
#endif
	case VIDIOC_ENUMSTD:
	case VIDIOC_QUERYSTD:
	case VIDIOC_G_STD:
	case VIDIOC_S_STD:
	{
		// FIXME: Implement decoder support
		return -EINVAL;
	}
#endif
	default:
		PDEBUG(3, "Unsupported IOCtl: 0x%X", cmd);
		return -ENOIOCTLCMD;	/* V4L layer handles this */
	} /* end switch */

	return 0;
}

static int
ov51x_ioctl(struct inode *inode, struct file *file,
            unsigned int cmd, unsigned long arg)
{
	struct video_device *vdev = file->private_data;
	struct usb_ov511 *ov = video_get_drvdata(vdev);
	int rc;

	if (down_interruptible(&ov->lock))
		return -EINTR;

	rc = video_usercopy(inode, file, cmd, arg, ov51x_do_ioctl);

	up(&ov->lock);
	return rc;
}

static ssize_t
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 8)
ov51x_read(struct file *file, char __user *buf, size_t cnt, loff_t *ppos)
#else
ov51x_read(struct file *file, char *buf, size_t cnt, loff_t *ppos)
#endif
{
	struct video_device *vdev = file->private_data;
	int noblock = file->f_flags&O_NONBLOCK;
	unsigned long count = cnt;
	struct usb_ov511 *ov = video_get_drvdata(vdev);
	int i, rc = 0, frmx = -1;
	struct ov511_frame *frame;

	if (down_interruptible(&ov->lock))
		return -EINTR;

	PDEBUG(4, "%ld bytes, noblock=%d", count, noblock);

	if (!vdev || !buf) {
		rc = -EFAULT;
		goto error;
	}

	if (!ov->dev) {
		rc = -ENODEV;
		goto error;
	}

// FIXME: Only supports two frames
	/* See if a frame is completed, then use it. */
	if (ov->frame[0].grabstate >= FRAME_DONE)	/* _DONE or _ERROR */
		frmx = 0;
	else if (ov->frame[1].grabstate >= FRAME_DONE)/* _DONE or _ERROR */
		frmx = 1;

	/* If nonblocking we return immediately */
	if (noblock && (frmx == -1)) {
		rc = -EAGAIN;
		goto error;
	}

	/* If no FRAME_DONE, look for a FRAME_GRABBING state. */
	/* See if a frame is in process (grabbing), then use it. */
	if (frmx == -1) {
		if (ov->frame[0].grabstate == FRAME_GRABBING)
			frmx = 0;
		else if (ov->frame[1].grabstate == FRAME_GRABBING)
			frmx = 1;
	}

	/* If no frame is active, start one. */
	if (frmx == -1) {
		if ((rc = ov51x_new_frame(ov, frmx = 0))) {
			err("read: ov51x_new_frame error");
			goto error;
		}
	}

	frame = &ov->frame[frmx];

restart:
	/* Wait while we're grabbing the image */
	PDEBUG(4, "Waiting image grabbing");
	rc = wait_event_interruptible(frame->wq,
		(frame->grabstate == FRAME_DONE)
		|| (frame->grabstate == FRAME_ERROR)
		|| !ov->present);

	if (rc)
		goto error;

	if (!ov->present) {
		rc = -ENODEV;
		goto error;
	}

	PDEBUG(4, "Got image, frame->grabstate = %d", frame->grabstate);
	PDEBUG(4, "bytes_recvd = %d", frame->bytes_recvd);

	if (frame->grabstate == FRAME_ERROR) {
		frame->bytes_read = 0;
		err("** ick! ** Errored frame %d", ov->curframe);
		if (ov51x_new_frame(ov, frmx)) {
			err("read: ov51x_new_frame error");
			goto error;
		}
		goto restart;
	}


	/* Repeat until we get a snapshot frame */
	if (ov->snap_enabled)
		PDEBUG(4, "Waiting snapshot frame");
	if (ov->snap_enabled && !frame->snapshot) {
		frame->bytes_read = 0;
		if ((rc = ov51x_new_frame(ov, frmx))) {
			err("read: ov51x_new_frame error");
			goto error;
		}
		goto restart;
	}

	/* Clear the snapshot */
	if (ov->snap_enabled && frame->snapshot) {
		frame->snapshot = 0;
		ov51x_clear_snapshot(ov);
	}

	/* Decompression, format conversion, etc... */
	ov51x_postprocess(ov, frame);

	PDEBUG(4, "frmx=%d, bytes_read=%ld, length=%ld", frmx,
		frame->bytes_read,
		get_frame_length(frame));

	/* copy bytes to user space; we allow for partials reads */
//	if ((count + frame->bytes_read)
//	    > get_frame_length((struct ov511_frame *)frame))
//		count = frame->scanlength - frame->bytes_read;

	/* FIXME - count hardwired to be one frame... */
	count = get_frame_length(frame);

	PDEBUG(4, "Copy to user space: %ld bytes", count);
	if ((i = copy_to_user(buf, frame->data + frame->bytes_read, count))) {
		PDEBUG(4, "Copy failed! %d bytes not copied", i);
		rc = -EFAULT;
		goto error;
	}

	frame->bytes_read += count;
	PDEBUG(4, "{copy} count used=%ld, new bytes_read=%ld",
		count, frame->bytes_read);

	/* If all data have been read... */
	if (frame->bytes_read
	    >= get_frame_length(frame)) {
		frame->bytes_read = 0;

// FIXME: Only supports two frames
		/* Mark it as available to be used again. */
		ov->frame[frmx].grabstate = FRAME_UNUSED;
		if ((rc = ov51x_new_frame(ov, !frmx))) {
			err("ov51x_new_frame returned error");
			goto error;
		}
	}

	PDEBUG(4, "read finished, returning %ld (sweet)", count);

	up(&ov->lock);
	return count;

error:
	up(&ov->lock);
	return rc;
}

static int
ov51x_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct video_device *vdev = file->private_data;
	unsigned long start = vma->vm_start;
	unsigned long size  = vma->vm_end - vma->vm_start;
	struct usb_ov511 *ov = video_get_drvdata(vdev);
	unsigned long page, pos;

	PDEBUG(4, "mmap: %ld (%lX) bytes", size, size);

	if (size > (((OV511_NUMFRAMES
	              * MAX_DATA_SIZE(ov->maxwidth, ov->maxheight)
	              + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1))))
		return -EINVAL;

	if (down_interruptible(&ov->lock))
		return -EINTR;

	if (!ov->dev) {
		up(&ov->lock);
		return -ENODEV;
	}

	pos = (unsigned long)ov->fbuf;
	while (size > 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 10)
		page = vmalloc_to_pfn((void *)pos);
		if (remap_pfn_range(vma, start, page, PAGE_SIZE, PAGE_SHARED)) {
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 3) || defined(RH9_REMAP)
		page = kvirt_to_pa(pos);
		if (remap_page_range(vma, start, page, PAGE_SIZE,
				     PAGE_SHARED)) {
#else
		page = kvirt_to_pa(pos);
		if (remap_page_range(start, page, PAGE_SIZE, PAGE_SHARED)) {
#endif
			up(&ov->lock);
			return -EAGAIN;
		}
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	up(&ov->lock);
	return 0;
}

static struct file_operations ov511_fops = {
	.owner =	THIS_MODULE,
	.open =		ov51x_open,
	.release =	ov51x_release,
	.read =		ov51x_read,
	.mmap =		ov51x_mmap,
	.ioctl =	ov51x_ioctl,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 16)
	.compat_ioctl =	v4l_compat_ioctl32,
#endif
	.llseek =	no_llseek,
};

static struct video_device vdev_template = {
	.owner =	THIS_MODULE,
	.name =		"OV511 USB Camera",
	.type =		VID_TYPE_CAPTURE,
	.hardware =	VID_HARDWARE_OV511,
	.fops =		&ov511_fops,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	.release =	video_device_release,
#endif
	.minor =	-1,
};

#if defined(CONFIG_VIDEO_PROC_FS)
static int
ov51x_control_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		    unsigned long ularg)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 5)
	struct proc_dir_entry *pde = PDE(inode);
#else
	struct proc_dir_entry *pde = inode->u.generic_ip;
#endif
	struct usb_ov511 *ov;
	void *arg = (void *) ularg;
	int rc;

	if (!pde)
		return -ENOENT;

	ov = pde->data;
	if (!ov)
		return -ENODEV;

	if (!ov->dev)
		return -EIO;

	/* Should we pass through standard V4L IOCTLs? */

	switch (cmd) {
	case OV511IOC_GINTVER:
	{
		return OV511_INTERFACE_VER;
	}
	case OV511IOC_GUSHORT:
	{
		struct ov511_ushort_opt opt;
		int v = -1;

		if (copy_from_user(&opt, arg, sizeof(opt)))
			return -EFAULT;

		switch (opt.optnum) {
		case OV511_USOPT_BRIGHT:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_BRIGHT, &v);
			if (rc)
				return rc;
			break;
		case OV511_USOPT_SAT:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_SAT, &v);
			if (rc)
				return rc;
			break;
		case OV511_USOPT_HUE:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_HUE, &v);
			if (rc)
				return rc;
			break;
		case OV511_USOPT_CONTRAST:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_CONT, &v);
			if (rc)
				return rc;
			break;
		default:
			err("Invalid get short option number");
			return -EINVAL;
		}

		opt.val = v;
		if (copy_to_user(arg, &opt, sizeof(opt)))
			return -EFAULT;

		return 0;
	}
	case OV511IOC_SUSHORT:
	{
		struct ov511_ushort_opt opt;
		int v;

		if (copy_from_user(&opt, arg, sizeof(opt)))
			return -EFAULT;

		v = opt.val;
		switch (opt.optnum) {
		case OV511_USOPT_BRIGHT:
			return sensor_set_control(ov, OVCAMCHIP_CID_BRIGHT, v);
		case OV511_USOPT_SAT:
			return sensor_set_control(ov, OVCAMCHIP_CID_SAT, v);
		case OV511_USOPT_HUE:
			return sensor_set_control(ov, OVCAMCHIP_CID_HUE, v);
		case OV511_USOPT_CONTRAST:
			return sensor_set_control(ov, OVCAMCHIP_CID_CONT, v);
		default:
			err("Invalid set short option number");
			return -EINVAL;
		}
	}
	case OV511IOC_GUINT:
	{
		struct ov511_uint_opt opt;

		if (copy_from_user(&opt, arg, sizeof(opt)))
			return -EFAULT;

		switch (opt.optnum) {
		case OV511_UIOPT_POWER_FREQ:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_FREQ,
			    &opt.val);
			if (rc < 0)
				return rc;
			break;
		case OV511_UIOPT_BFILTER:
			rc = sensor_get_control(ov, OVCAMCHIP_CID_BANDFILT,
			    &opt.val);
			if (rc < 0)
				return rc;
			break;
		case OV511_UIOPT_LED:
			opt.val = ov->led_policy;
			break;
		case OV511_UIOPT_DEBUG:
			opt.val = debug;
			break;
		case OV511_UIOPT_COMPRESS:
			opt.val = ov->compress;
			break;
		default:
			err("Invalid get int option number");
			return -EINVAL;
		}

		if (copy_to_user(arg, &opt, sizeof(opt)))
			return -EFAULT;

		return 0;
	}
	case OV511IOC_SUINT:
	{
		struct ov511_uint_opt opt;

		if (copy_from_user(&opt, arg, sizeof(opt)))
			return -EFAULT;

		switch (opt.optnum) {
		case OV511_UIOPT_POWER_FREQ:
			rc = sensor_set_light_freq(ov, opt.val);
			if (rc < 0)
				return rc;
			break;
		case OV511_UIOPT_BFILTER:
			rc = sensor_set_control(ov, OVCAMCHIP_CID_BANDFILT,
			    opt.val);
			if (rc < 0)
				return rc;
			break;
		case OV511_UIOPT_LED:
			if (opt.val <= 2) {
				ov->led_policy = opt.val;
				if (ov->led_policy == LED_OFF)
					ov51x_led_control(ov, 0);
				else if (ov->led_policy == LED_ON)
					ov51x_led_control(ov, 1);
			} else {
				return -EINVAL;
			}
			break;
		case OV511_UIOPT_DEBUG:
			if (opt.val <= 5)
				debug = opt.val;
			else
				return -EINVAL;
			break;
		case OV511_UIOPT_COMPRESS:
			ov->compress = opt.val;
			if (ov->compress) {
				if (ov->bclass == BCL_OV511)
					ov511_init_compression(ov);
				else if (ov->bclass == BCL_OV518)
					ov518_init_compression(ov);
			}
			break;
		default:
			err("Invalid get int option number");
			return -EINVAL;
		}

		return 0;
	}
	default:
		return -ENOTTY;
	} /* end switch */

	return 0;
}
#endif

/****************************************************************************
 *
 * OV511 and sensor configuration
 *
 ***************************************************************************/

/* This is called when an OV7610, OV7620, or OV76BE is detected. */
static int
ov7xx0_configure(struct usb_ov511 *ov)
{
	ov->internal_client.addr = OV7xx0_SID;

	ov->maxwidth = 640;
	ov->maxheight = 480;

	if (ov->bclass == BCL_OV518) {
		/* The OV518 cannot go as low as the sensor can */
		ov->minwidth = 160;
		ov->minheight = 120;
	} else {
		ov->minwidth = 64;
		ov->minheight = 48;
	}

	return camchip_init_settings(ov);
}

/* This is called when an OV6620, OV6630, OV6630AE, or OV6630AF is detected. */
static int
ov6xx0_configure(struct usb_ov511 *ov)
{
	ov->internal_client.addr = OV6xx0_SID;

	ov->maxwidth = 352;
	ov->maxheight = 288;

	if (ov->bclass == BCL_OV518) {
		/* The OV518 cannot go as low as the sensor can */
		ov->minwidth = 160;
		ov->minheight = 120;
	} else {
		ov->minwidth = 64;
		ov->minheight = 48;
	}

	return camchip_init_settings(ov);
}

/* This is called when an SAA7111A video decoder is detected. */
static int
saa7111a_configure(struct usb_ov511 *ov)
{
	ov->internal_client.addr = SAA7111A_SID;

	/* 640x480 not supported with PAL */
	if (ov->pal) {
		ov->maxwidth = 320;
		ov->maxheight = 240;		/* Even field only */
	} else {
		ov->maxwidth = 640;
		ov->maxheight = 480;		/* Even/Odd fields */
	}

	ov->minwidth = 320;
	ov->minheight = 240;		/* Even field only */

	ov->has_decoder = 1;
	ov->num_inputs = 8;
	ov->norm = VIDEO_MODE_AUTO;
	ov->stop_during_set = 0;	/* Decoder guarantees stable image */

	// FIXME: Fix this for OV518(+)
	/* Latch to negative edge of clock. Otherwise, we get incorrect
	 * colors and jitter in the digital signal. */
	if (ov->bclass == BCL_OV511)
		reg_w(ov, 0x11, 0x00);
	else
		warn("SAA7111A not yet supported with OV518/OV518+");

	ov->sensor = SEN_SAA7111A;
	return 0;
}

/* This initializes the OV511/OV511+ and the sensor */
static int
ov511_configure(struct usb_ov511 *ov)
{
	static struct regval regvals_init_511[] = {
		{ R51x_SYS_RESET,	0x7f, 0xff },
	 	{ R51x_SYS_INIT,	0x01, 0xff },
	 	{ R51x_SYS_RESET,	0x7f, 0xff },
		{ R51x_SYS_INIT,	0x01, 0xff },
		{ R51x_SYS_RESET,	0x3f, 0xff },
		{ R51x_SYS_INIT,	0x01, 0xff },
		{ R51x_SYS_RESET,	0x3d, 0xff },
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	static struct regval regvals_norm_511[] = {
		{ R511_DRAM_FLOW_CTL, 	0x01, 0xff },
		{ R51x_SYS_SNAP,	0x00, 0xff },
		{ R51x_SYS_SNAP,	0x02, 0xff },
		{ R51x_SYS_SNAP,	0x00, 0xff },
		{ R511_FIFO_OPTS,	0x1f, 0xff },
		{ R511_COMP_EN,		0x00, 0xff },
		{ R511_COMP_LUT_EN,	0x03, 0xff },
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	/* The OV511+ has undocumented bits in the flow control register.
	 * Setting it to 0xff fixes the corruption with moving objects. */
	static struct regval regvals_norm_511_plus[] = {
		{ R511_DRAM_FLOW_CTL,	0xff, 0xff },
		{ R51x_SYS_SNAP,	0x00, 0xff },
		{ R51x_SYS_SNAP,	0x02, 0xff },
		{ R51x_SYS_SNAP,	0x00, 0xff },
		{ R511_FIFO_OPTS,	0xff, 0xff },
		{ R511_COMP_EN,		0x00, 0xff },
		{ R511_COMP_LUT_EN,	0x03, 0xff },
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	PDEBUG(4, "");

	ov->customid = reg_r(ov, R511_SYS_CUST_ID);
	if (ov->customid < 0) {
		err("Unable to read camera bridge registers");
		goto error;
	}

	ov->desc = symbolic(camlist, ov->customid);
	info("model: %s (id=%d)", ov->desc, ov->customid);

	if (0 == strcmp(ov->desc, NOT_DEFINED_STR)) {
		err("Camera type (%d) not recognized", ov->customid);
		err("Please notify " EMAIL " of the name,");
		err("manufacturer, model, and this number of your camera.");
		err("Also include the output of the detection process.");
	} 

	/* Some devices have special properties that are determined from the
	 * custom ID. Overrides for some of these features are required since
	 * IDs sometimes are used without OV's permission */
	switch (ov->customid) {
	case 6:                      /* USB Life TV (NTSC) */
		ov->tuner_type = 8;          /* Temic 4036FY5 3X 1981 */
		break;
	case 22:
		ov->tuner_type = 29;         /* LG TPI8PSB12 */
		ov->pal = 1;
		break;
	case 38:
		warn("WARNING: Tuner type for this device is unknown");
		ov->pal = 1;
		break;
	case 70:                     /* USB Life TV (PAL/SECAM) */
		ov->tuner_type = 3;          /* Philips FI1216MF */
		ov->pal = 1;
		break;
	case 150:
		ov->tuner_type = 28;         /* LG TPI8PSB02 */
		ov->pal = 1;
		break;
	case 193:                    /* Interbiometrics Securecams */
	case 194:
	case 195:
	case 196:
		ov->sensor_mono = 1; /* OV7120 (monochrome) sensor */
		ov->clockdiv = 2;    /* This provides the best results */
		break;
	}

	if (write_regvals(ov, regvals_init_511)) goto error;

	if (ov->led_policy == LED_OFF || ov->led_policy == LED_AUTO)
		ov51x_led_control(ov, 0);

	if (ov->bridge == BRG_OV511) {
		if (write_regvals(ov, regvals_norm_511)) goto error;
	} else if (ov->bridge == BRG_OV511PLUS) {
		if (write_regvals(ov, regvals_norm_511_plus)) goto error;
	} else {
		err("Invalid bridge");
	}

	reg_w(ov, R51x_I2C_CLOCK, i2c_clockdiv);

	OV511_SET_DECOMP_OPS(ov);

	if (ov511_init_compression(ov)) goto error;

	ov->packet_numbering = 1;
	ov511_set_packet_size(ov, 0);

	ov->snap_enabled = snapshot;

	ov51x_init_i2c(ov);

	return 0;

error:
	err("OV511 Config failed");
	return -EBUSY;
}

/* This initializes the OV518/OV518+ and the sensor */
static int
ov518_configure(struct usb_ov511 *ov)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 7)
	struct usb_interface *ifp;
	struct usb_host_interface *alt;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	struct usb_interface *ifp = ov->dev->config[0].interface[0];
#else
	struct usb_interface *ifp = &ov->dev->config[0].interface[0];
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 7)
	__u16 mxps;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 45)
	__u16 mxps = ifp->altsetting[7].endpoint[0].desc.wMaxPacketSize;
#else
	__u16 mxps = ifp->altsetting[7].endpoint[0].wMaxPacketSize;
#endif

	/* For 518 and 518+ */
	static struct regval regvals_init_518[] = {
		{ R51x_SYS_RESET,	0x40, 0xff },
	 	{ R51x_SYS_INIT,	0xe1, 0xff },
	 	{ R51x_SYS_RESET,	0x3e, 0xff },
		{ R51x_SYS_INIT,	0xe1, 0xff },
		{ R51x_SYS_RESET,	0x00, 0xff },
		{ R51x_SYS_INIT,	0xe1, 0xff },
		{ 0x46,			0x00, 0xff }, 
		{ 0x5d,			0x03, 0xff },
		/* Set LED GPIO pin to output mode */
		{ 0x57,			0x00, 0x02 }, 
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	static struct regval regvals_norm_518[] = {
		{ R51x_SYS_SNAP,	0x02, 0xff }, /* Reset */
		{ R51x_SYS_SNAP,	0x01, 0xff }, /* Enable */
		{ 0x31, 		0x0f, 0xff },
		{ 0x5d,			0x03, 0xff },
		{ 0x24,			0x9f, 0xff },
		{ 0x25,			0x90, 0xff },
		{ 0x20,			0x00, 0xff },
		{ 0x51,			0x04, 0xff },
		{ 0x71,			0x19, 0xff },
		{ 0x2f,			0x80, 0xff },
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	static struct regval regvals_norm_518_plus[] = {
		{ R51x_SYS_SNAP,	0x02, 0xff }, /* Reset */
		{ R51x_SYS_SNAP,	0x01, 0xff }, /* Enable */
		{ 0x31, 		0x0f, 0xff },
		{ 0x5d,			0x03, 0xff },
		{ 0x24,			0x9f, 0xff },
		{ 0x25,			0x90, 0xff },
		{ 0x20,			0x60, 0xff },
		{ 0x21,			0x1f, 0xff },
		{ 0x51,			0x02, 0xff },
		{ 0x71,			0x19, 0xff },
		{ 0x40,			0xff, 0xff },
		{ 0x41,			0x42, 0xff },
		{ 0x33,			0x04, 0xff },
		{ 0x21,			0x19, 0xff },
		{ 0x3f,			0x10, 0xff },
		{ 0x2f,			0x80, 0xff },
		{ 0x00,			0x00, 0x00 }, /* end */
	};

	PDEBUG(4, "");

	/* First 5 bits of custom ID reg are a revision ID on OV518 */
	info("Device revision %d", 0x1F & reg_r(ov, R511_SYS_CUST_ID));

	/* Give it the default description */
	ov->desc = symbolic(camlist, 0);

	if (write_regvals(ov, regvals_init_518)) goto error;

	/* LED is off by default with OV518; have to explicitly turn it on */
	if (ov->led_policy == LED_OFF || ov->led_policy == LED_AUTO)
		ov51x_led_control(ov, 0);
	else
		ov51x_led_control(ov, 1);

	/* Don't require compression if dumppix is enabled; otherwise it's
	 * required. OV518 has no uncompressed mode, to save RAM. */
	if (!dumppix && !ov->compress) {
		ov->compress = 1;
		warn("Compression required with OV518...enabling");
	}

	/* The OV518/OV518+ code can't handle anything but YUV420P yet */
	if (force_palette == 0)		/* Allow user to override */
		force_palette = VIDEO_PALETTE_YUV420P;

	if (ov->bridge == BRG_OV518) {
		if (write_regvals(ov, regvals_norm_518)) goto error;
	} else if (ov->bridge == BRG_OV518PLUS) {
		if (write_regvals(ov, regvals_norm_518_plus)) goto error;
	} else {
		err("Invalid bridge");
	}

	OV518_SET_DECOMP_OPS(ov);

	if (ov518_init_compression(ov)) goto error;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 7)
	ifp = usb_ifnum_to_if(ov->dev, 0);
	if (ifp) {
		alt = usb_altnum_to_altsetting(ifp, 7);
		if (alt) {
#  if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 11)
			mxps = le16_to_cpu(alt->endpoint[0].desc.wMaxPacketSize);
#  else
			mxps = alt->endpoint[0].desc.wMaxPacketSize;
#  endif
		} else {
			err("Failed to determine MxPS");
			goto error;
		}
	} else {
		goto error;
	}
#endif

	/* Some chips have packet numbering by default, some don't */
	if (mxps == 897)
		ov->packet_numbering = 1;
	else
		ov->packet_numbering = 0;

	ov518_set_packet_size(ov, 0);

	ov->snap_enabled = snapshot;

	ov51x_init_i2c(ov);

	return 0;

error:
	err("OV518 Config failed");
	return -EBUSY;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
/****************************************************************************
 *  sysfs
 ***************************************************************************/

static inline struct usb_ov511 *cd_to_ov(struct class_device *cd)
{
	struct video_device *vdev = to_video_device(cd);
	return video_get_drvdata(vdev);
}

static ssize_t show_custom_id(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	return sprintf(buf, "%d\n", ov->customid);
} 
static CLASS_DEVICE_ATTR(custom_id, S_IRUGO, show_custom_id, NULL);

static ssize_t show_model(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	return sprintf(buf, "%s\n", ov->desc);
} 
static CLASS_DEVICE_ATTR(model, S_IRUGO, show_model, NULL);

static ssize_t show_bridge(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	return sprintf(buf, "%s\n", symbolic(brglist, ov->bridge));
} 
static CLASS_DEVICE_ATTR(bridge, S_IRUGO, show_bridge, NULL);

static ssize_t show_sensor(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	return sprintf(buf, "%s\n", symbolic(senlist, ov->sensor));
} 
static CLASS_DEVICE_ATTR(sensor, S_IRUGO, show_sensor, NULL);

static ssize_t show_brightness(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	int x;

	if (!ov->dev)
		return -ENODEV;
	sensor_get_control(ov, OVCAMCHIP_CID_BRIGHT, &x);
	return sprintf(buf, "%d\n", x);
} 
static CLASS_DEVICE_ATTR(brightness, S_IRUGO, show_brightness, NULL);

static ssize_t show_saturation(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	int x;

	if (!ov->dev)
		return -ENODEV;
	sensor_get_control(ov, OVCAMCHIP_CID_SAT, &x);
	return sprintf(buf, "%d\n", x);
} 
static CLASS_DEVICE_ATTR(saturation, S_IRUGO, show_saturation, NULL);

static ssize_t show_contrast(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	int x;

	if (!ov->dev)
		return -ENODEV;
	sensor_get_control(ov, OVCAMCHIP_CID_CONT, &x);
	return sprintf(buf, "%d\n", x);
} 
static CLASS_DEVICE_ATTR(contrast, S_IRUGO, show_contrast, NULL);

static ssize_t show_hue(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	int x;

	if (!ov->dev)
		return -ENODEV;
	sensor_get_control(ov, OVCAMCHIP_CID_HUE, &x);
	return sprintf(buf, "%d\n", x);
} 
static CLASS_DEVICE_ATTR(hue, S_IRUGO, show_hue, NULL);

static ssize_t show_exposure(struct class_device *cd, char *buf)
{
	struct usb_ov511 *ov = cd_to_ov(cd);
	int x;

	if (!ov->dev)
		return -ENODEV;
	sensor_get_control(ov, OVCAMCHIP_CID_EXP, &x);
	return sprintf(buf, "%d\n", x);
} 
static CLASS_DEVICE_ATTR(exposure, S_IRUGO, show_exposure, NULL);

static void ov_create_sysfs(struct video_device *vdev)
{
	video_device_create_file(vdev, &class_device_attr_custom_id);
	video_device_create_file(vdev, &class_device_attr_model);
	video_device_create_file(vdev, &class_device_attr_bridge);
	video_device_create_file(vdev, &class_device_attr_sensor);
	video_device_create_file(vdev, &class_device_attr_brightness);
	video_device_create_file(vdev, &class_device_attr_saturation);
	video_device_create_file(vdev, &class_device_attr_contrast);
	video_device_create_file(vdev, &class_device_attr_hue);
	video_device_create_file(vdev, &class_device_attr_exposure);
}
#endif

/****************************************************************************
 *
 *  USB routines
 *
 ***************************************************************************/

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
static int
ov51x_probe(struct usb_interface *intf, const struct usb_device_id *id)
{
	struct usb_device *dev = interface_to_usbdev(intf);
#else
static void *
ov51x_probe(struct usb_device *dev, unsigned int ifnum,
	    const struct usb_device_id *id)
{
#endif
	struct usb_interface_descriptor *idesc;
	struct usb_ov511 *ov;
	int i;
	u16 vendor, product;

	PDEBUG(1, "probing for device...");

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1)
		return -ENODEV;
#else
	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1)
		return NULL;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 7)
	idesc = &intf->cur_altsetting->desc;
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
	idesc = &intf->altsetting[0].desc;
#else
	idesc = &dev->actconfig->interface[ifnum].altsetting[0];
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 11)
	vendor = le16_to_cpu(dev->descriptor.idVendor);
	product = le16_to_cpu(dev->descriptor.idProduct);
#else
	vendor = dev->descriptor.idVendor;
	product = dev->descriptor.idProduct;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
	if (idesc->bInterfaceClass != 0xFF)
		return -ENODEV;
	if (idesc->bInterfaceSubClass != 0x00)
		return -ENODEV;
#else
	if (idesc->bInterfaceClass != 0xFF)
		return NULL;
	if (idesc->bInterfaceSubClass != 0x00)
		return NULL;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 20)
	/* Since code below may sleep, we use this as a lock */
	MOD_INC_USE_COUNT;
#endif

	if ((ov = kmalloc(sizeof(*ov), GFP_KERNEL)) == NULL) {
		err("couldn't kmalloc ov struct");
		goto error_out;
	}

	memset(ov, 0, sizeof(*ov));

	ov->dev = dev;
	ov->iface = idesc->bInterfaceNumber;
	ov->led_policy = led;
	ov->compress = compress;
	ov->clockdiv = clockdiv;
	ov->num_inputs = 1;	   /* Video decoder init functs. change this */
	ov->stop_during_set = !fastset;
	ov->tuner_type = tuner;
	ov->sensor = CC_UNKNOWN;

	switch (product) {
	case PROD_OV511:
		ov->bridge = BRG_OV511;
		ov->bclass = BCL_OV511;
		break;
	case PROD_OV511PLUS:
		ov->bridge = BRG_OV511PLUS;
		ov->bclass = BCL_OV511;
		break;
	case PROD_OV518:
		ov->bridge = BRG_OV518;
		ov->bclass = BCL_OV518;
		break;
	case PROD_OV518PLUS:
		ov->bridge = BRG_OV518PLUS;
		ov->bclass = BCL_OV518;
		break;
	case PROD_ME2CAM:
		if (vendor != VEND_MATTEL)
			goto error;
		ov->bridge = BRG_OV511PLUS;
		ov->bclass = BCL_OV511;
		break;
	default:
		err("Unknown product ID 0x%04x", product);
		goto error;
	}

	info("USB %s video device found", symbolic(brglist, ov->bridge));

	init_waitqueue_head(&ov->wq);

	init_MUTEX(&ov->lock);	/* to 1 == available */
	init_MUTEX(&ov->buf_lock);
	init_MUTEX(&ov->cbuf_lock);

	ov->buf_state = BUF_NOT_ALLOCATED;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 20)
	if (usb_make_path(dev, ov->usb_path, OV511_USB_PATH_LEN) < 0) {
		err("usb_make_path error");
		goto error;
	}
#else
	snprintf(ov->usb_path, OV511_USB_PATH_LEN, "dev:%d/bus:%d",
		dev->bus->busnum, dev->devnum);
#endif

	/* Allocate control transfer buffer. */
	/* Must be kmalloc()'ed, for DMA compatibility */
	ov->cbuf = kmalloc(OV511_CBUF_SIZE, GFP_KERNEL);
	if (!ov->cbuf)
		goto error;

	if (ov->bclass == BCL_OV518) {
		if (ov518_configure(ov) < 0)
			goto error;
	} else {
		if (ov511_configure(ov) < 0)
			goto error;
	}

	for (i = 0; i < OV511_NUMFRAMES; i++) {
		ov->frame[i].framenum = i;
		init_waitqueue_head(&ov->frame[i].wq);
	}

	for (i = 0; i < OV511_NUMSBUF; i++) {
		ov->sbuf[i].ov = ov;
		spin_lock_init(&ov->sbuf[i].lock);
		ov->sbuf[i].n = i;
	}

#ifdef OV511_DEBUG
	if (dump_bridge) {
		if (ov->bclass == BCL_OV511)
			ov511_dump_regs(ov);
		else
			ov518_dump_regs(ov);
	}
#endif

	ov->vdev = video_device_alloc();
	if (!ov->vdev)
		goto error;

	memcpy(ov->vdev, &vdev_template, sizeof(*ov->vdev));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	ov->vdev->dev = &dev->dev;
#endif
	video_set_drvdata(ov->vdev, ov);

	for (i = 0; i < OV511_MAX_DEV_HINTS; i++) {
		if (device_hint[i].cid != ov->customid)
			break;

		if (video_register_device(ov->vdev, VFL_TYPE_GRABBER,
			device_hint[i].minor) >= 0) {
			break;
		}
	}

	if (ov->vdev->minor == -1) {
		for (i = 0; i < OV511_MAX_UNIT_VIDEO; i++) {
			/* Minor 0 cannot be specified;
			 * assume user wants autodetect */
			if (unit_video[i] == 0)
				break;

			if (video_register_device(ov->vdev, VFL_TYPE_GRABBER,
				unit_video[i]) >= 0) {
				break;
			}
		}
	}

	/* Use the next available one */
	if ((ov->vdev->minor == -1) &&
	    video_register_device(ov->vdev, VFL_TYPE_GRABBER, -1) < 0) {
		err("video_register_device failed");
		goto error;
	}

	info("Device at %s registered to minor %d", ov->usb_path,
	     ov->vdev->minor);

	/* Update I2C adapter name with minor # */
	sprintf(ov->i2c_adap.name, "OV51x #%d", ov->vdev->minor);

	create_proc_ov511_cam(ov);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	usb_set_intfdata(intf, ov);
	ov_create_sysfs(ov->vdev);
#endif

	ov->present = 1;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 20)
	MOD_DEC_USE_COUNT;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	return 0;
#else
	return ov;
#endif

error:
	if (ov->vdev) {
		if (-1 == ov->vdev->minor)
			video_device_release(ov->vdev);
		else
			video_unregister_device(ov->vdev);
		ov->vdev = NULL;
	}

	down(&ov->cbuf_lock);
	kfree(ov->cbuf);
	ov->cbuf = NULL;
	up(&ov->cbuf_lock);

	kfree(ov);
	ov = NULL;

error_out:
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 20)
	MOD_DEC_USE_COUNT;
#endif
	err("Camera initialization failed");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
	return -EIO;
#else
	return NULL;
#endif
}


static void
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
ov51x_disconnect(struct usb_interface *intf)
{
	struct usb_ov511 *ov = usb_get_intfdata(intf);
#else
ov51x_disconnect(struct usb_device *dev, void *ptr)
{
	struct usb_ov511 *ov = (struct usb_ov511 *) ptr;
#endif
	int n;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 20)
	MOD_INC_USE_COUNT;
#endif

	PDEBUG(3, "");

	ov->present = 0;

	/* Ensure that no synchronous control requests
	 * are active before disconnect() returns */
	down(&ov->cbuf_lock);
	kfree(ov->cbuf);
	ov->cbuf = NULL;
	up(&ov->cbuf_lock);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 36)
	dev_set_drvdata(&intf->dev, NULL);
#endif
	if (!ov)
		return;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0)
	if (ov->vdev)
		video_unregister_device(ov->vdev);
#else
	video_unregister_device(ov->vdev);
	if (ov->user)
		PDEBUG(3, "Device open...deferring video_unregister_device");
#endif

	for (n = 0; n < OV511_NUMFRAMES; n++)
		ov->frame[n].grabstate = FRAME_ERROR;

	ov->curframe = -1;

	/* This will cause the process to request another frame */
	for (n = 0; n < OV511_NUMFRAMES; n++)
		wake_up_interruptible(&ov->frame[n].wq);

	wake_up_interruptible(&ov->wq);

	/* Can't take lock earlier since it would deadlock
	 * read() or ioctl() if open and sleeping */
	down(&ov->lock);

	ov->streaming = 0;
	ov51x_unlink_isoc(ov);

        destroy_proc_ov511_cam(ov);

	i2c_del_adapter(&ov->i2c_adap);

	ov->dev = NULL;

	up(&ov->lock);

	down(&ov_free_lock);
	if (ov && !ov->user) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0)
		if (ov->vdev)
			video_device_release(ov->vdev);
#endif
		kfree(ov);
		ov = NULL;
	}
	up(&ov_free_lock);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 20)
	MOD_DEC_USE_COUNT;
#endif
	PDEBUG(3, "Disconnect complete");
}

static struct usb_driver ov511_driver = {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 20)) && \
    (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16))
	.owner =	THIS_MODULE,
#endif
	.name =		"ov511",
	.id_table =	device_table,
	.probe =	ov51x_probe,
	.disconnect =	ov51x_disconnect
};

/****************************************************************************
 *
 *  Module routines
 *
 ***************************************************************************/

static int
ov511_atoi(char *s, int dflt)
{
	int k = 0;

	if (*s == '\0' || *s < '0' || *s > '9')
		return dflt;

	while (*s != '\0' && *s >= '0' && *s <= '9') {
		k = 10 * k + (*s - '0');
		s++;
	}

	return k;
}

/* Hints are of the form:  c<custom_id>:<minor>[,c<custom_id>:<minor>]... */
static int
ov511_parse_device_hints(void)
{
	int n;

	for (n = 0; n < OV511_MAX_DEV_HINTS; n++) {
		char *s, *colon, *c;

		s = dev_hint[n];
		if (s != NULL && *s != '\0') {
			/* parse string: chop at 'c' & ':' */
			colon = c = s;
			while (*c != '\0' && *c != 'c')
				c++;
			while (*colon != '\0' && *colon != ':')
				colon++;

			/* Few sanity checks */
			if (*c != '\0' && c > colon) {
				err("Bad dev_hint: 'c' must be before colon");
				return -EINVAL;
			}

			if (*colon == '\0') {
				err("Bad dev_hint: no colon");
				return -EINVAL;
			}

			/* Parse arg values */
			device_hint[n].minor = ov511_atoi(colon + 1, -1);
			if (device_hint[n].minor == -1) {
				err("Bad dev_hint: no minor number");
				return -EINVAL;
			}

			device_hint[n].cid = ov511_atoi(c + 1, -1);
			if (device_hint[n].cid == -1) {
				err("Bad dev_hint: no custom id");
				return -EINVAL;
			}

			PDEBUG(5, "device_hint[%d]:", n);
			PDEBUG(5, "  custom_id: %d", device_hint[n].cid);
			PDEBUG(5, "  minor    : %d", device_hint[n].minor);
		} else {
			/* Mark fields as unused */
			device_hint[n].cid = -1;
			device_hint[n].minor = -1;
		}
	} /* ..for MAX_DEV_HINTS */

	return 0;
}

static int __init
usb_ov511_init(void)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 0)
	EXPORT_NO_SYMBOLS;
#endif

#ifdef OV511_DEBUG
	ov511_debug = debug;
#endif

	init_MUTEX(&ov_free_lock);

	if (ov511_parse_device_hints() < 0)
		return -EINVAL;

        proc_ov511_create();

	if (usb_register(&ov511_driver) < 0)
		return -ENODEV;

#if defined(HAVE_V4L2)
	if (v4l2)
		info(DRIVER_VERSION " : " DRIVER_DESC " (V4L2 enabled)");
	else
		info(DRIVER_VERSION " : " DRIVER_DESC " (V4L2 disabled)");
#else
		info(DRIVER_VERSION " : " DRIVER_DESC);
#endif
	return 0;
}

static void __exit
usb_ov511_exit(void)
{
	usb_deregister(&ov511_driver);
	info("driver deregistered");

        proc_ov511_destroy();
}

module_init(usb_ov511_init);
module_exit(usb_ov511_exit);
