// OpenSG NURBS example: testSurfaceTextureRender.cpp
//
// Shows how to texture a NURBS surface

#include <OSGGLUT.h>
#include <OSGConfig.h>
#include <OSGSimpleSceneManager.h>
#include <OSGSimpleMaterial.h>
#include <OSGTextureChunk.h>
#include <OSGImage.h>
#include <OSGGLUTWindow.h>
#include <OSGSolidBackground.h>
#include <OSGSurface.h>

OSG_USING_NAMESPACE

SimpleSceneManager *mgr;
SimpleMaterialPtr  gpcl_defaultmat;
ImagePtr           gpcl_image;
Real32             g_error;
SurfacePtr         gpcl_surface;

// redraw the window
void display(void)
{      
    // render
    
    mgr->redraw();

    // all done, swap  (better not GV)
    //glutSwapBuffers();
}

// react to size changes
void reshape(int w, int h)
{
    mgr->resize(w,h);
    glutPostRedisplay();
}

// react to mouse button presses
void mouse(int button, int state, int x, int y)
{
    if (state)
        mgr->mouseButtonRelease(button, x, y);
    else
        mgr->mouseButtonPress(button, x, y);
        
    glutPostRedisplay();
}

// react to mouse motions with pressed buttons
void motion(int x, int y)
{
    mgr->mouseMove(x, y);
    glutPostRedisplay();
}

// react to keys
void keyboard(unsigned char k, int, int)
{
    switch(k)
    {
    case 27:    exit(1);
    case 'y':   
    case 'z':
                glPolygonMode( GL_FRONT_AND_BACK, GL_POINT);
                std::cerr << "PolygonMode: Point." << std::endl;
                break;
    case 'x':   glPolygonMode( GL_FRONT_AND_BACK, GL_LINE);
                std::cerr << "PolygonMode: Line." << std::endl;
                break;
    case 'c':   glPolygonMode( GL_FRONT_AND_BACK, GL_FILL);
                std::cerr << "PolygonMode: Fill." << std::endl;
                break;
    case 'f':   g_error *= 2;
                std::cerr << "Error: " << g_error << std::endl;
                beginEditCP( gpcl_surface, Surface::ErrorFieldMask );
                {
                    gpcl_surface->setError( g_error );
                }
                endEditCP(   gpcl_surface, Surface::ErrorFieldMask );
                break;
    case 'g':   g_error /= 2;
                std::cerr << "Error: " << g_error << std::endl;
                beginEditCP( gpcl_surface, Surface::ErrorFieldMask );
                {
                    gpcl_surface->setError( g_error );
                }
                endEditCP(   gpcl_surface, Surface::ErrorFieldMask );
                break;
    }
}

void setupDefaultTexture( void );

void setupDefaultMaterial( void )
{
    gpcl_defaultmat = SimpleMaterial::create();
	addRefCP( gpcl_defaultmat );
    beginEditCP(gpcl_defaultmat);
	{
		gpcl_defaultmat->setDiffuse( Color3f(1.0,0.0,0.0) ); // RED
		gpcl_defaultmat->setAmbient( Color3f(0.2,0.2,0.2) );
		gpcl_defaultmat->setEmission( Color3f(0.02,0.02,0.02) );
		gpcl_defaultmat->setSpecular( Color3f(0.78,0.78,0.78) );
		gpcl_defaultmat->setShininess( 128 );
        TextureChunkPtr xchunk = TextureChunk::create();
        addRefCP( xchunk );
        beginEditCP( xchunk );
        {
        	xchunk->setImage( gpcl_image );
        	xchunk->setMinFilter( GL_LINEAR );
        	xchunk->setMagFilter( GL_LINEAR );
        	xchunk->setWrapS( GL_CLAMP );
        	xchunk->setWrapT( GL_CLAMP );
        	xchunk->setEnvMode( GL_REPLACE );
        	xchunk->setInternalFormat( GL_RGB16_EXT );
        }
        endEditCP( xchunk );
        std::cerr<<"Adding texture chunk to default material..."<<std::endl;
        gpcl_defaultmat->addChunk( xchunk );
	}
    endEditCP(gpcl_defaultmat);
}






NodePtr makeScene( void )
{
    setupDefaultTexture();
    setupDefaultMaterial();

    NodePtr root = Node::create();
	addRefCP( root );
    SurfacePtr surface = Surface::create();
    GeoPositions3fPtr cps = GeoPositions3f::create();
    beginEditCP( cps, GeoPositions3f::GeoPropDataFieldMask );
    {
        // control points should always be 3D for the time being,
        // rational support will be added later
        cps->clear();
        cps->push_back( Pnt3f(  1,  1,  0 ));
        cps->push_back( Pnt3f(  1,  0,  1 ));
        cps->push_back( Pnt3f(  1, -1,  0 ));
        cps->push_back( Pnt3f(  0,  1, -1 ));
        cps->push_back( Pnt3f(  0,  0,  0 ));
        cps->push_back( Pnt3f(  0, -1, -1 ));
        cps->push_back( Pnt3f( -1,  1,  0 ));
        cps->push_back( Pnt3f( -1,  0,  1 ));
        cps->push_back( Pnt3f( -1, -1,  0 ));
    }
    endEditCP( cps, GeoPositions3f::GeoPropDataFieldMask );
    GeoTexCoords2fPtr texcps = GeoTexCoords2f::create();
    beginEditCP( texcps, GeoTexCoords2f::GeoPropDataFieldMask  );
    {
        // texture coordinates for the control points, always 2D
        texcps->clear();
        texcps->push_back( Vec2f( 0.0006078289457852043, 0.03890105253025306) );
        texcps->push_back( Vec2f( 0.0003687335542147959, 0.2993150529249993) );
        texcps->push_back( Vec2f( 0.000607828945785204,  0.5597290533197458) );
        texcps->push_back( Vec2f( 0.004676797701953115,  0.02359894746974695) );
        texcps->push_back( Vec2f( 0.004676797701953117,  0.2993150529249994) );
        texcps->push_back( Vec2f( 0.004676797701953115,  0.575031158380252) );
        texcps->push_back( Vec2f( 0.008745766458121028,  0.03890105253025303) );
        texcps->push_back( Vec2f( 0.008984861849691437,  0.2993150529249993) );
        texcps->push_back( Vec2f( 0.008745766458121028,  0.5597290533197458) );
    }
    endEditCP( texcps, GeoTexCoords2f::GeoPropDataFieldMask );
    beginEditCP( surface );
    {
        // we have no trimming, so just remove any curves that might be present
        // (Note that a dummy trimming will be added to the internal representation)
        surface->removeCurves();

        // set up dimensions and knot vectors:
        surface->setDimU( 2 );
        surface->setDimV( 2 );
        surface->getKnotsU().clear();
        surface->getKnotsU().push_back( 0 );
        surface->getKnotsU().push_back( 0 );
        surface->getKnotsU().push_back( 0 );
        surface->getKnotsU().push_back( 1 );
        surface->getKnotsU().push_back( 1 );
        surface->getKnotsU().push_back( 1 );
        surface->getKnotsV().clear();
        surface->getKnotsV().push_back( 0 );
        surface->getKnotsV().push_back( 0 );
        surface->getKnotsV().push_back( 0 );
        surface->getKnotsV().push_back( 1 );
        surface->getKnotsV().push_back( 1 );
        surface->getKnotsV().push_back( 1 );
        
        // set control points and texture control points
        surface->setControlPoints( cps );
        surface->setTextureControlPoints( texcps );
        
        // set error
        surface->setError( g_error );
        
        // and finally set the material
        surface->setMaterial( gpcl_defaultmat );
    }
    endEditCP( surface );
    beginEditCP( root );
    {
        root->setCore( surface );
    }
    endEditCP( root );
    gpcl_surface = surface;

    return root;
}

int main(int argc, char **argv)
{
    g_error = 0.001;

    if ( argc == 2 )
    {
        g_error = atof( argv[1] );
    }
    if ( g_error < 0.001 )
    {
        g_error = 0.001;
    }
    
    osgInit(argc,argv);
    // GLUT init
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);

    int winid = glutCreateWindow("OpenSG");
    
    glutReshapeFunc(reshape);
    glutDisplayFunc(display);
    glutIdleFunc(display);
    glutMouseFunc(mouse);
    glutMotionFunc(motion);
    glutKeyboardFunc(keyboard);

    GLUTWindowPtr gwin = GLUTWindow::create();
	beginEditCP(gwin);
	{
	    gwin->setId(winid);
	}
    endEditCP(gwin);
    gwin->init();

    // create the scene
    NodePtr scene;
    scene = makeScene( );
    
    if ( scene == NullFC )
    {
        std::cerr<<"makeScene returned NullFC, exiting..."<<std::endl;
        return -1;
    }

    // create the SimpleSceneManager helper
    mgr = new SimpleSceneManager;

    // create the window and initial camera/viewport
    mgr->setWindow( gwin );
    // tell the manager what to manage
    mgr->setRoot  ( scene );
    
    // show the whole scene
    mgr->showAll();
    mgr->redraw();
	SolidBackgroundPtr bgr = SolidBackground::create();
    beginEditCP( bgr );
    bgr->setColor( Color3f( 1.0, 1.0, 1.0 ));
    endEditCP( bgr );
    mgr->getWindow()->getPort(0)->setBackground( bgr );
    
    // GLUT main loop
    glutMainLoop();

    return 0;
}

// texture data, from here there's nothing interesting. :)

static unsigned char texturedata[49152] = {
199, 199, 53, 204, 204, 65, 209, 209, 
81, 213, 213, 100, 215, 215, 121, 215, 
215, 140, 212, 212, 161, 207, 207, 178, 
201, 201, 193, 199, 199, 199, 199, 53, 
53, 203, 63, 49, 208, 77, 44, 212, 
93, 40, 214, 111, 38, 215, 129, 37, 
214, 148, 38, 211, 165, 41, 206, 181, 
46, 201, 193, 51, 198, 199, 54, 199, 
199, 53, 203, 203, 63, 208, 208, 77, 
212, 212, 93, 214, 214, 111, 215, 215, 
129, 214, 214, 148, 211, 211, 165, 206, 
206, 181, 201, 201, 193, 198, 198, 199, 
199, 53, 53, 203, 49, 63, 208, 44, 
77, 212, 40, 93, 214, 38, 111, 215, 
37, 129, 214, 38, 148, 211, 41, 165, 
206, 46, 181, 201, 51, 193, 198, 54, 
199, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
204, 187, 48, 211, 192, 60, 218, 197, 
77, 223, 201, 97, 226, 202, 120, 225, 
202, 142, 221, 199, 164, 215, 195, 183, 
207, 190, 198, 204, 187, 204, 204, 48, 
65, 210, 59, 61, 216, 73, 56, 222, 
90, 53, 225, 110, 50, 226, 129, 49, 
224, 150, 51, 220, 169, 54, 214, 185, 
58, 207, 198, 63, 203, 204, 65, 204, 
187, 48, 210, 191, 59, 216, 196, 73, 
222, 199, 90, 225, 202, 110, 226, 203, 
129, 224, 201, 150, 220, 198, 169, 214, 
194, 185, 207, 189, 198, 203, 187, 204, 
203, 63, 49, 209, 59, 59, 215, 55, 
73, 221, 51, 90, 224, 48, 110, 225, 
48, 129, 223, 49, 150, 219, 52, 169, 
213, 56, 185, 206, 61, 198, 203, 64, 
204, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
209, 171, 43, 218, 175, 55, 226, 180, 
72, 234, 183, 94, 237, 185, 119, 236, 
184, 144, 231, 182, 168, 223, 178, 187, 
214, 173, 203, 209, 171, 209, 209, 43, 
81, 217, 54, 77, 225, 68, 73, 232, 
86, 70, 236, 108, 68, 237, 129, 67, 
235, 152, 68, 229, 173, 71, 222, 190, 
75, 214, 203, 79, 209, 210, 81, 209, 
171, 43, 217, 175, 54, 225, 179, 68, 
232, 182, 86, 236, 184, 108, 237, 185, 
129, 235, 184, 152, 229, 181, 173, 222, 
177, 190, 214, 173, 203, 209, 171, 210, 
208, 77, 44, 215, 73, 55, 223, 69, 
69, 229, 66, 87, 234, 63, 108, 235, 
63, 129, 233, 64, 152, 227, 67, 172, 
220, 71, 189, 212, 75, 202, 208, 77, 
208, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
213, 152, 39, 223, 155, 51, 234, 158, 
69, 242, 160, 92, 247, 162, 119, 246, 
161, 145, 239, 159, 171, 229, 157, 191, 
219, 154, 206, 213, 152, 213, 213, 39, 
100, 222, 50, 97, 231, 65, 95, 240, 
84, 92, 246, 107, 91, 247, 130, 90, 
244, 154, 91, 237, 176, 93, 228, 194, 
96, 218, 207, 98, 213, 214, 100, 213, 
152, 39, 222, 155, 50, 231, 157, 65, 
240, 160, 84, 246, 161, 107, 247, 162, 
130, 244, 161, 154, 237, 159, 176, 228, 
156, 194, 218, 154, 207, 213, 152, 214, 
212, 93, 40, 221, 90, 51, 229, 87, 
66, 237, 84, 84, 243, 83, 107, 244, 
82, 130, 241, 83, 154, 235, 85, 175, 
226, 88, 192, 217, 92, 206, 212, 93, 
212, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
215, 131, 37, 226, 132, 50, 237, 133, 
67, 247, 133, 90, 252, 134, 118, 251, 
133, 146, 243, 133, 172, 233, 132, 193, 
221, 132, 208, 215, 131, 215, 215, 37, 
121, 225, 48, 120, 235, 63, 120, 244, 
82, 119, 250, 106, 119, 252, 130, 118, 
249, 156, 119, 241, 178, 119, 231, 195, 
120, 221, 209, 120, 215, 216, 121, 215, 
131, 37, 225, 132, 48, 235, 132, 63, 
244, 133, 82, 250, 133, 106, 252, 134, 
130, 249, 133, 156, 241, 133, 178, 231, 
132, 195, 221, 132, 209, 215, 131, 216, 
214, 111, 38, 224, 110, 48, 234, 108, 
63, 243, 107, 83, 249, 106, 106, 251, 
106, 130, 247, 106, 155, 240, 108, 177, 
230, 109, 195, 220, 111, 208, 214, 112, 
215, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
215, 112, 37, 225, 110, 50, 236, 108, 
68, 246, 107, 91, 251, 106, 119, 249, 
106, 146, 242, 108, 172, 232, 109, 192, 
221, 111, 208, 215, 112, 215, 215, 37, 
140, 224, 48, 142, 234, 63, 143, 243, 
83, 145, 249, 106, 146, 251, 130, 146, 
247, 155, 145, 240, 177, 144, 230, 195, 
143, 220, 208, 141, 214, 215, 140, 215, 
112, 37, 224, 110, 48, 234, 109, 63, 
243, 107, 83, 249, 106, 106, 251, 106, 
130, 247, 107, 155, 240, 108, 177, 230, 
109, 195, 220, 111, 208, 214, 112, 215, 
215, 129, 37, 225, 129, 48, 235, 129, 
63, 244, 130, 82, 251, 130, 106, 252, 
130, 130, 249, 130, 156, 241, 130, 178, 
231, 129, 196, 221, 129, 209, 215, 129, 
216, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
212, 91, 40, 221, 88, 53, 231, 84, 
70, 239, 81, 93, 243, 80, 119, 242, 
80, 144, 236, 83, 169, 227, 86, 190, 
217, 90, 205, 212, 91, 212, 212, 40, 
161, 220, 51, 164, 229, 66, 167, 237, 
85, 170, 242, 107, 172, 243, 130, 172, 
240, 154, 171, 234, 175, 169, 226, 192, 
166, 217, 205, 162, 211, 212, 160, 212, 
91, 40, 220, 88, 51, 229, 85, 66, 
237, 82, 85, 242, 80, 107, 243, 80, 
130, 240, 81, 154, 234, 83, 175, 226, 
86, 192, 217, 90, 205, 211, 92, 212, 
214, 148, 38, 223, 150, 49, 233, 152, 
64, 241, 154, 83, 247, 155, 106, 249, 
156, 130, 245, 155, 155, 238, 153, 177, 
229, 151, 194, 219, 149, 207, 213, 148, 
214, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
207, 74, 45, 215, 69, 57, 223, 65, 
74, 229, 61, 95, 233, 59, 120, 232, 
60, 143, 227, 62, 166, 220, 67, 185, 
211, 71, 201, 207, 74, 207, 207, 45, 
178, 214, 56, 182, 221, 70, 186, 227, 
88, 190, 232, 109, 192, 233, 129, 193, 
230, 151, 192, 225, 171, 189, 218, 188, 
185, 211, 201, 181, 207, 207, 178, 207, 
74, 45, 214, 70, 56, 221, 66, 70, 
227, 62, 88, 232, 60, 109, 233, 59, 
129, 230, 60, 151, 225, 63, 171, 218, 
67, 188, 211, 71, 201, 207, 74, 207, 
211, 165, 41, 219, 169, 52, 227, 172, 
67, 235, 175, 85, 240, 177, 108, 241, 
178, 130, 238, 177, 153, 232, 174, 174, 
224, 171, 191, 215, 167, 204, 210, 165, 
211, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
201, 59, 51, 207, 54, 62, 214, 49, 
79, 219, 46, 98, 221, 44, 120, 221, 
44, 141, 217, 47, 162, 211, 51, 181, 
205, 57, 195, 201, 59, 201, 201, 51, 
193, 207, 61, 197, 213, 75, 202, 217, 
91, 205, 221, 111, 208, 221, 129, 208, 
220, 149, 207, 216, 167, 204, 210, 183, 
200, 204, 196, 195, 201, 202, 193, 201, 
59, 51, 207, 55, 61, 213, 50, 75, 
217, 47, 91, 221, 44, 111, 221, 44, 
129, 220, 45, 149, 216, 48, 167, 210, 
52, 183, 204, 57, 196, 201, 59, 202, 
206, 181, 46, 213, 185, 56, 220, 189, 
71, 226, 192, 88, 230, 195, 109, 231, 
196, 129, 229, 194, 151, 224, 191, 171, 
217, 187, 187, 210, 183, 200, 206, 180, 
207, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
199, 53, 53, 204, 48, 65, 209, 43, 
81, 213, 39, 100, 215, 37, 121, 215, 
37, 140, 212, 40, 161, 207, 45, 178, 
201, 51, 193, 199, 53, 199, 199, 53, 
199, 203, 63, 203, 208, 77, 208, 212, 
93, 212, 214, 111, 214, 215, 129, 215, 
214, 148, 214, 211, 165, 211, 206, 181, 
206, 201, 193, 201, 198, 199, 198, 199, 
53, 53, 203, 49, 63, 208, 44, 77, 
212, 40, 93, 214, 38, 111, 215, 37, 
129, 214, 38, 148, 211, 41, 165, 206, 
46, 181, 201, 51, 193, 198, 54, 199, 
201, 193, 51, 206, 198, 61, 212, 202, 
75, 217, 206, 92, 220, 208, 111, 221, 
209, 129, 219, 207, 149, 215, 204, 167, 
210, 200, 183, 204, 196, 196, 201, 193, 
202, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
198, 199, 54, 203, 204, 64, 208, 208, 
77, 212, 212, 93, 214, 215, 112, 215, 
216, 129, 213, 214, 148, 210, 211, 165, 
206, 207, 180, 201, 202, 193, 198, 199, 
199, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
203, 203, 203, 203, 203, 203, 203, 203, 
};

void setupDefaultTexture( void )
{
    gpcl_image = Image::create();
    gpcl_image->set(Image::OSG_RGB_PF, 1024, 16, 1, 1, 1, 0, texturedata);
}
