/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// RaStore.cpp: implementation of the RaStore class.
//
//////////////////////////////////////////////////////////////////////

#include "RaStore.h"
#include <PKI_PKCS12.h>
#include <PKI_P7B.h>
#include "svintl.h"
#include <openssl/ocsp.h>

#include <vector>
using namespace std;


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


RaStore::RaStore(const mString & EntityName, ENGINE * e):NewPKIStore(EntityName, e)
{
	m_AclValidator = NULL;
}

RaStore::~RaStore()
{
}

bool RaStore::CreateTables(const SQL_Connection * DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {RASTORE_CREATE_1, RASTORE_CREATE_2, NULL};

	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}


void RaStore::SetMinPasswdLen(unsigned long MinPasswdLen)
{
	ConfAccessLock.LockWrite();
	m_MinPasswdLen = MinPasswdLen;
	ConfAccessLock.UnlockWrite();
}

void RaStore::set_Policies(const HashTable_String & Policies)
{
	ConfAccessLock.LockWrite();
	m_Policies = Policies;
	ConfAccessLock.UnlockWrite();
}

bool RaStore::SetGroups(const mVector<UsersGroup> & groups)
{
	ConfAccessLock.LockWrite();
	m_groups = groups;
	ConfAccessLock.UnlockWrite();
	return true;
}


bool RaStore::DeleteProfile(const PKI_CERT &UserCert, long ProfileId, bool CheckOwnership)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mVector<NewpkiProfileDatasCert> Certs;
	size_t i;

	if(CheckOwnership)
	{
		if(!GetProfileAccessCheck(UserCert, ProfileId, sql))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		// We also need to make sure there are no valid certificates
		// for this profile
		if(!GetCerts(ProfileId, Certs))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		for(i=0; i<Certs.size(); i++)
		{
			if(Certs[i].get_state() != NEWPKI_PROFILE_CERT_STATE_REVOKED &&
				Certs[i].get_state() != NEWPKI_PROFILE_CERT_STATE_ERROR)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				ERR_add_error_data(1, _sv("\nAll certificates associated with this profile \nmust be revoked or in error state"));
				return false;
			}
		}
	}
	// Delete profile
	if(req.sprintf(RASTORE_DELETE_PROFILE, ProfileId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(CheckOwnership)
	{
		// Delete all associated certificates
		for(i=0; i<Certs.size(); i++)
		{
			if(req.sprintf(RASTORE_DELETE_CERT, Certs[i].get_id()) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			if(!sql.Execute(req))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	return true;
}

bool RaStore::GetCertReqTransactionId(unsigned long CertReqId, Asn1OctetString & transactionId)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	
	if(req.sprintf(RASTORE_GET_CERT, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!sql.Value(0, "ee_tid", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!StringtoTransactionID(req, transactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::SetCertReqTransactionId(unsigned long CertReqId, const Asn1OctetString & transactionId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString tid;
	
	if(!transactionIDtoString(transactionId, tid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RASTORE_SET_CERT_TID, tid.c_str(), CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::GetProfileTransactionId(const PKI_CERT & UserCert, unsigned long ProfileId, Asn1OctetString & transactionId)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString data;

	if(!GetProfileAccessCheck(UserCert, ProfileId, sql))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// The profile id
	if(!sql.Value(0, "ee_tid", data))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!StringtoTransactionID(data, transactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::InsertProfileFromEE(const X509_NAME * dn, const mString & ee_id, const Asn1OctetString & transactionId, bool admin_validation, unsigned long ownergroup, long & ProfileId)
{
	NewpkiProfile Profile;

	if(!Profile.set_dn(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_eeValidation(0);
	Profile.set_ldapUid("");
	Profile.set_ownerGroupSerial(ownergroup);

	if(!DoInsertProfile(PKI_CERT::EmptyInstance, Profile, admin_validation, ee_id, transactionId, ProfileId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool RaStore::InsertProfile(const PKI_CERT & UserCert, const NewpkiProfile & Profile, long & ProfileId)
{
	if(!DoInsertProfile(UserCert, Profile, false, "", Asn1OctetString::EmptyInstance, ProfileId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::DoInsertProfile(const PKI_CERT & UserCert, const NewpkiProfile & Profile, 
							bool admin_validation, 
							const mString & ee_id, 
							const Asn1OctetString & transactionId, 
							long & ProfileId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strHash;
	mString strDN;
	mString strUID;
	mString strId;
	bool Exists;
	mString pem_datas;
	mString pem_signature;
	char * dn;
	unsigned long profile_id;
	time_t startTime;
	HashTable_Dn Dn;
	int status;


	if(transactionId && !transactionIDtoString(transactionId, strId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Dn.From_X509_NAME(Profile.get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Profile.get_eeValidation())
	{
		//We only check the policy when the DN isn't sent to the EE
		ConfAccessLock.LockRead();
		if(!Dn.ValidateAgainstPolicy(m_Policies))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
		ConfAccessLock.UnlockRead();
	}
	else
	{
		// Make sure we have an email address for this profile
		// so the EE can send the email
		if(Dn.SeekEntryName("emailAddress", -1) == HASHTABLE_NOT_FOUND)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_DN_FIELD_UNPRESENT);
			ERR_set_error_data("emailAddress", ERR_TXT_STRING);
			return false;
		}
	}


	// Search if the profile is unique
	if(!X509_NAMEtoHash(Profile.get_dn(), strHash))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	profile_id = 0;
	if(!ProfileExists(strHash, Exists, profile_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(Exists)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_UNIQUE_PROFILE);
		if(strHash.sprintf("%ld", profile_id) > 0)
			ERR_add_error_data(1, strHash.c_str());
		return false;
	}

	// Sign the profile
	if(!SignProfile(Profile, pem_signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Profile.to_PEM(pem_datas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	dn = X509_NAME_oneline((X509_NAME*)Profile.get_dn(), NULL, 0);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	strDN = sql.FormatString(dn);
	free(dn);

	if(Profile.get_ldapUid().size())
		strUID = sql.FormatString(Profile.get_ldapUid());
	else
		strUID = "";

	unsigned long owner_serial;
	int owner_type;

	owner_serial = Profile.get_ownerGroupSerial();
	if(owner_serial)
	{
		owner_type = OWNER_TYPE_GROUP;
		if(!IsValidGroupId(owner_serial))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_UNKNOWN);
			return false;
		}
	}
	else
	{
		owner_serial = UserCert.GetSerial();
		if(!owner_serial)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		owner_type = OWNER_TYPE_USER;
	}

	if(admin_validation)
		status = NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION;
	else
		status = Profile.get_eeValidation()?NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION:NEWPKI_PROFILE_STATE_VALIDATED;

	if(req.sprintf(RASTORE_INSERT_PROFILE, ee_id.c_str(), strId.c_str(), 
											owner_type, owner_serial, 
											strHash.c_str(), strDN.c_str(), 
											strUID.c_str(), status, 
											pem_datas.c_str(), 
											pem_signature.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ProfileId = sql.GetLastID();

	// Every 5000 profiles we request the table to be optimized
	if( (ProfileId % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing profiles table..."));
		if(!sql.OptimizeTable(RASTORE_PROFILES_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize profiles table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized profiles table in %ld secondes"), time(NULL) - startTime);
		}
	}

	return true;
}

bool RaStore::ProfileExists(const mString & dn_hash, bool & exists, unsigned long & ProfileId)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	long NumRows;
	mString strpid;

	if(ProfileId)
	{
		if(strpid.sprintf(" AND profile_id != '%ld'", ProfileId) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}

	if(req.sprintf(RASTORE_SEARCH_PROFILE_DN, dn_hash.c_str(), strpid.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(NumRows)
	{
		if(!sql.Value(0, "profile_id", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		ProfileId = req.c_ulng();
		exists = true;
	}
	else
		exists = false;

	return true;
}


bool RaStore::EnumProfiles(const PKI_CERT &UserCert, mVector<NewpkiProfileDatas> & Profiles, 
						long index, long num, long state, const mString & filter, bool AllProfiles)
{
	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString Where;
	mString tmpDatas;
	size_t v_index, i, j;

	// PKI Admins can see all the profile
	if(!AllProfiles && !m_AclValidator->IsPkiAdministrator(UserCert))
	{
		if(Where.sprintf("WHERE (owner_type=%d AND owner_serial=%ld)", OWNER_TYPE_USER, UserCert.GetSerial()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}

		// Create the WHERE syntax according to user's groups
		ConfAccessLock.LockRead();

		for(i=0; i<m_groups.size(); i++)
		{
			for(j=0; j<m_groups[i].get_usersSerial().size() ; j++)
			{
				// Add a WHERE entry
				if(m_groups[i].get_usersSerial()[j] == UserCert.GetSerial())
				{
					if(tmpDatas.sprintf(" OR (owner_type=%d AND owner_serial=%ld)", OWNER_TYPE_GROUP, m_groups[i].get_serial()) <= 0)
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
						return false;
					}
					Where += tmpDatas;
					break;
				}
			}
		}
		ConfAccessLock.UnlockRead();
	}

	if(state)
	{
		if(!Where.size())
			Where = "WHERE ";
		else
			Where += " AND ";
		req.sprintf("%d", state);
		Where += "STATE = '";
		Where += req;
		Where += "'";
	}
	if(filter.size())
	{
		if(!Where.size())
			Where = "WHERE ";
		else
			Where += " AND ";
		Where += "LOWER(dn) LIKE '%%";
		Where += sql.FormatString(filter).c_str();
		Where += "%%'";
	}

	if(req.sprintf(RASTORE_GET_PROFILES, Where.c_str(), index, num) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	v_index = 0;
	for(i=0; (long)i<NumRows; i++)
	{
		Profiles.insert(Profiles.begin()+v_index);
		
		if(!SqlToNewpkiProfileDatas(sql, i, Profiles[v_index]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		//Get the certs associated with the profile !
		if(!GetCerts(Profiles[v_index].get_id(), Profiles[v_index].get_certs()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		Profiles[v_index].set_isOK();
		v_index++;
	}
	
	return true;
}

bool RaStore::InsertCertReq(const mString & CaName, unsigned long ProfileId, int Type, const PKI_CSR & pkcs10, const RaStorePrivdatas & PrivDatas, const ASN1_BIT_STRING * flags, const mString & admin_mail, const Asn1OctetString & transactionId, unsigned long & CertReqId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strFlags;
	mString strTid;
	Asn1EncryptSign crypt;
	mString pem_datas;
	mString strPrivDatas;
	time_t startTime;

	if(flags && !ASN1_BIT_STRINGtoString(flags, strFlags))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(transactionId.get_BufferLen() && !transactionIDtoString(transactionId, strTid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(PrivDatas)
	{
		if(!PrivDatas.to_SignEncrypt(crypt, m_EntityCert.GetPrivateKey().GetRsaKey(), m_EntityCert.GetPrivateKey().GetRsaKey(), EVP_sha1(), EVP_des_ede3_cbc()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!crypt.to_PEM(strPrivDatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		crypt.Clear();
	}
	else
	{
		strPrivDatas = "";
	}

	if(req.sprintf(RASTORE_INSERT_CERT, ProfileId, strTid.c_str(), CaName.c_str(), pkcs10.GetPemCSR().c_str(), strPrivDatas.c_str(), Type, NEWPKI_PROFILE_CERT_STATE_WAITING, strFlags.c_str(), admin_mail.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	CertReqId = sql.GetLastID();
	if(CertReqId == (unsigned long)-1)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Every 5000 certificates we request the table to be optimized
	if( (CertReqId % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing certificates table..."));
		if(!sql.OptimizeTable(RASTORE_CERTS_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize certificates table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized certificates table in %ld secondes"), time(NULL) - startTime);
		}
	}
	return true;
}

bool RaStore::GetCerts(unsigned long ProfileId, mVector<NewpkiProfileDatasCert> & Certs)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString err;
	mString req;
	long NumRows;
	long i;
	PKI_CERT ttCert;

	if(req.sprintf(RASTORE_GET_CERTS, ProfileId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	for(i=0; i<NumRows; i++)
	{
		Certs.insert(Certs.begin() + i);
		ERR_clear_error();
		if(!Sql2Cert(&sql, Certs[i], i) || !Certs[i])
		{
			Certs[i].set_id(0);
			Certs[i].set_flags(0);
			Certs[i].set_state(NEWPKI_PROFILE_CERT_STATE_ERROR);
			Certs[i].set_type(NEWPKI_PROFILE_CERT_TYPE_PKCS12);
			Certs[i].set_caName(_sv("Unknown"));
			ERR_to_mstring(Certs[i].get_error());
		}
		Certs[i].set_isOK();
	}
	
	return true;
}

bool RaStore::SqlToNewpkiProfileDatas(SQL & sql, long pos, NewpkiProfileDatas & Profile)
{
	mString tmpDatas;

	// The profile id
	if(!sql.Value(pos, "profile_id", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_id(tmpDatas.c_ulng());

	// The ee id
	if(!sql.Value(pos, "ee_id", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Profile.set_eeId(tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// The state
	if(!sql.Value(pos, "state", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_state(tmpDatas.c_ulng());

	//The real profile
	if(!sql.Value(pos, "datas", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// We need to reconvert if from the PEM
	if(!Profile.get_profile().from_PEM(tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_isOK();

	return true;
}

bool RaStore::GetProfile(const PKI_CERT & UserCert, unsigned long ProfileId, NewpkiProfileDatas & Profile)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);

	if(!GetProfileAccessCheck(UserCert, ProfileId, sql))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!SqlToNewpkiProfileDatas(sql, 0, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::GetProfileByCertId(unsigned long CertId, NewpkiProfileDatas & Profile)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;

	if(req.sprintf(RASTORE_GET_PROFILE_BY_CERT, CertId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!SqlToNewpkiProfileDatas(sql, 0, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::GetEeProfile(const mString & ee_name, unsigned long ProfileId, NewpkiProfileDatas & Profile)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString tmpDatas;
	mString req;
	long NumRows;
	mString eeid;

	if(eeid.sprintf("%ld:%s", ProfileId, ee_name.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(req.sprintf(RASTORE_GET_EE_PROFILE, eeid.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!SqlToNewpkiProfileDatas(sql, 0, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool RaStore::DeleteCertReq(unsigned long CertReqId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	// Delete the cert
	if(req.sprintf(RASTORE_DELETE_CERT, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool RaStore::InsertResponseCa(const mString & ca_name, 
							   unsigned long cert_id, 
							   const NewpkiResponse & resp)
{
	int Status = NEWPKI_PROFILE_CERT_STATE_ERROR;
	mString strError;
	mString strSqlError;
	mString strPrivDatas;
	mString x509;
	mString p7b;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	mString tid;
	mString currStatus;
	mString AdminMail;
	unsigned long serial = 0;
	PKI_CRL Crl;
	ASN1_BIT_STRING * flags;
	bool SendCertificate;
	bool SendPkcs12Pwd;
	bool NotifyAdmin;
	bool GenPkcs12Pwd;
	LOG_MESSAGE_TYPE LogMessage;
	LOG_MESSAGE_STATUS LogStatus = (LOG_MESSAGE_STATUS)0;
	NewpkiProfile Profile;
	HashTable_Dn ProfileDn;
	MailInfo CertificateMail;
	MailInfo PasswordMail;

	if(req.sprintf(RASTORE_GET_CERT_WITH_PROFILE, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.Value(0,"id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert_id = req.c_ulng();
	if(!cert_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.Value(0,"serial", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	serial = req.c_ulng();

	if(!sql.Value(0,"status", currStatus))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.Value(0,"x509", x509))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.Value(0,"p7b", p7b))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.Value(0,"priv_datas", strPrivDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!sql.Value(0,"flags", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	flags = StringtoASN1_BIT_STRING(req);
	if(!flags)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	NotifyAdmin = (ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)flags, REQUEST_CERT_FLAG_SEND_ADMIN_NOTIFICATION) == 1);
	SendCertificate = (ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)flags, REQUEST_CERT_FLAG_SEND_CERT_TO_USER) == 1);
	GenPkcs12Pwd = (ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)flags, REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD) == 1);
	SendPkcs12Pwd = (ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)flags, REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER) || GenPkcs12Pwd);
	ASN1_BIT_STRING_free(flags);


	if(!sql.Value(0,"admin_mail", AdminMail))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!sql.Value(0, "datas", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// We need to reconvert if from the PEM
	if(!Profile.from_PEM(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!ProfileDn.From_X509_NAME(Profile.get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	switch(resp.get_type())
	{
		case NEWPKI_RESPONSE_TYPE_CERT:
			if(currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING)
			{
				return true;
			}

			// Make sure we're talking about the same ID
			if(cert_id != resp.get_certResponse().get_id())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			
			if(!InsertResponseCert(resp.get_certResponse(), ca_name, Crl, LogMessage, LogStatus, Status, strError, ProfileDn, strPrivDatas, x509, p7b, serial, CertificateMail, PasswordMail, SendCertificate, SendPkcs12Pwd, GenPkcs12Pwd))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case NEWPKI_RESPONSE_TYPE_REV:
			if(currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_REV)
			{
				return true;
			}

			// Make sure we're talking about the same ID
			if(cert_id != resp.get_revResponse().get_id())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}

			SendCertificate = false;
			GenPkcs12Pwd = false;
			SendPkcs12Pwd = false;
			LogMessage = LOG_MESSAGE_TYPE_CERT_REV_RESP;

			if(resp.get_revResponse().get_lastCrl())
			{
				if(!(Crl = resp.get_revResponse().get_lastCrl()))
					ERR_clear_error();
			}

			// Did everything go right for the rev ?
			switch(resp.get_revResponse().get_status())
			{
				case REV_RESPONSE_OK:
					Status = NEWPKI_PROFILE_CERT_STATE_REVOKED;
					LogStatus = LOG_STATUS_TYPE_SUCCESS;
					strPrivDatas = "";
					break;
				
				case REV_RESPONSE_ERROR:
					LogStatus = LOG_STATUS_TYPE_FAILURE;
					Status = CaStatus2StoreStatus(resp.get_revResponse().get_certStatus(), 
													NEWPKI_PROFILE_CERT_STATE_ACTIVE);
					ERROR_ENTRIES_to_string(resp.get_revResponse().get_errors(), strError);
					break;

				default:
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
					break;
			}
			break;
		case NEWPKI_RESPONSE_TYPE_SUSP:
			if(currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP)
			{
				return true;
			}

			// Make sure we're talking about the same ID
			if(cert_id != resp.get_suspResponse().get_id())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}

			SendCertificate = false;
			GenPkcs12Pwd = false;
			SendPkcs12Pwd = false;
			LogMessage = LOG_MESSAGE_TYPE_CERT_SUSP_RESP;

			if(resp.get_suspResponse().get_lastCrl())
			{
				if(!(Crl = resp.get_suspResponse().get_lastCrl()))
					ERR_clear_error();
			}

			// Did everything go right for the suspension ?
			switch(resp.get_suspResponse().get_status())
			{
				case SUSP_RESPONSE_OK:
					Status = NEWPKI_PROFILE_CERT_STATE_SUSPENDED;
					LogStatus = LOG_STATUS_TYPE_SUCCESS;
					break;
				
				case SUSP_RESPONSE_ERROR:
					LogStatus = LOG_STATUS_TYPE_FAILURE;
					Status = CaStatus2StoreStatus(resp.get_suspResponse().get_certStatus(), 
													NEWPKI_PROFILE_CERT_STATE_ACTIVE);
					ERROR_ENTRIES_to_string(resp.get_suspResponse().get_errors(), strError);
					break;

				default:
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
					break;
			}
			break;
		case NEWPKI_RESPONSE_TYPE_UNSUSP:
			if(currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP)
			{
				return true;
			}

			// Make sure we're talking about the same ID
			if(cert_id != resp.get_unsuspResponse().get_id())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}

			SendCertificate = false;
			GenPkcs12Pwd = false;
			SendPkcs12Pwd = false;
			LogMessage = LOG_MESSAGE_TYPE_CERT_UNSUSP_RESP;

			if(resp.get_unsuspResponse().get_lastCrl())
			{
				if(!(Crl = resp.get_unsuspResponse().get_lastCrl()))
					ERR_clear_error();
			}

			// Did everything go right for the suspension ?
			switch(resp.get_unsuspResponse().get_status())
			{
				case UNSUSP_RESPONSE_OK:
					Status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;
					LogStatus = LOG_STATUS_TYPE_SUCCESS;
					break;
				
				case UNSUSP_RESPONSE_ERROR:
					LogStatus = LOG_STATUS_TYPE_FAILURE;
					Status = CaStatus2StoreStatus(resp.get_unsuspResponse().get_certStatus(), 
													NEWPKI_PROFILE_CERT_STATE_SUSPENDED);
					ERROR_ENTRIES_to_string(resp.get_unsuspResponse().get_errors(), strError);
					break;

				default:
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
					break;
			}
			break;
		case NEWPKI_RESPONSE_TYPE_ERR:
			if(currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING && 
				currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_ACTIVE && 
				currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP && 
				currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP && 
				currStatus.c_int() != NEWPKI_PROFILE_CERT_STATE_WAITING_REV)
			{
				return true;
			}
			SendCertificate = false;
			GenPkcs12Pwd = false;
			SendPkcs12Pwd = false;
			Status = NEWPKI_PROFILE_CERT_STATE_ERROR;
			LogMessage = LOG_MESSAGE_TYPE_CERT_SIG_RESP;
			strPrivDatas = "";
			ERROR_ENTRIES_to_string(resp.get_errors(), strError);
			break;
		
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
			break;
	}

	// Let's do some verification before sending the mails
	if(SendPkcs12Pwd)
	{
		// TODO On ne peut pas laisser cette rponse
		// nous tre repropose indfiniment, il faut demander
		// sa rvocation
		if(!PasswordMail || !PasswordMail.get_MailTo().c_str())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(SendCertificate)
	{
		// TODO On ne peut pas laisser cette rponse
		// nous tre repropose indfiniment, il faut demander
		// sa rvocation
		if(!CertificateMail || !CertificateMail.get_MailTo().c_str() || !CertificateMail.get_Attach().get_BufferLen())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(NotifyAdmin)
	{
		if(!AdminMail.size())
		{
			// TODO Ajouter une entre de log
			NotifyAdmin = false;
		}
	}

	// Are we supposed to send the PKCS#12 password to the user ?
	if(SendPkcs12Pwd)
	{
		// TODO On ne peut pas laisser cette rponse
		// nous tre repropos indfiniment, il faut demander
		// se rvocation
		if(!SendMail(m_EntityCert.GetStringName(), PasswordMail, false))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}


	// Are we supposed to send the certificate to the user ?
	if(SendCertificate)
	{
		// TODO On ne peut pas laisser cette rponse
		// nous tre repropos indfiniment, il faut demander
		// se rvocation
		if(!SendMail(m_EntityCert.GetStringName(), CertificateMail, false))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	// Are we supposed to notify the admin ?
	if(NotifyAdmin)
	{
		PasswordMail.Clear();
		PasswordMail.set_MailTo(AdminMail);
		PasswordMail.set_Subject(_sv("Received response"));
		if(req.sprintf(_sv("Received response for request %ld, from ca \"%s\"."), cert_id, ca_name.c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		PasswordMail.set_Body(req);
		PasswordMail.set_SignMail(true);

		if(!SendMail(m_EntityCert.GetStringName(), PasswordMail, false))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}



	// Set the response in the database
	if(strError.size())
		strSqlError = sql.FormatString(strError);

	if(req.sprintf(RASTORE_SET_CERT_RESP, Status, x509.c_str(), p7b.c_str(), strSqlError.c_str(), strPrivDatas.c_str(), serial, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(Crl)
		UpdateFromCRL(ca_name, Crl);


	if(LogMessage)
	{
		switch(LogStatus)
		{
			case LOG_STATUS_TYPE_SUCCESS:
				m_Logging->LogMessage(LogStatus, LogMessage, 0, ca_name.c_str(), cert_id);
				break;
			case LOG_STATUS_TYPE_FAILURE:
				m_Logging->LogMessage(LogStatus, LogMessage, 0, ca_name.c_str(), cert_id, NULL, strError.c_str());
				break;
			default:
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
		}
	}

	return true;
}

bool RaStore::InsertResponseEe(const mString & ee_name, unsigned long profile_id, const NewpkiResponse & resp)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	int status;

	if(req.sprintf(RASTORE_GET_PROFILE, profile_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!sql.Value(0, "profile_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	profile_id = req.c_ulng();
	if(!profile_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!sql.Value(0, "state", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	status = req.c_int();

	// Make sure we're talking about the same ID
	if(profile_id != resp.get_eeValResponse().get_raId())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	// We ignore the response if DN has already been validated
	if(status != NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION)
	{
		return true;
	}

	// Let's update the profile with the new DN
	if(!ChangeProfileDN(PKI_CERT::EmptyInstance, profile_id, resp.get_eeValResponse().get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


bool RaStore::InsertResponse(const mString & sender_name, unsigned long priv_attr, const NewpkiResponse & resp)
{
	// We don't have anything to do with this response
	if(resp.get_type() == NEWPKI_RESPONSE_TYPE_EE_DN_REMOVE ||
		resp.get_type() == NEWPKI_RESPONSE_TYPE_EE_UPDATE_CERT_STATUS ||
		resp.get_type() == NEWPKI_RESPONSE_TYPE_EE_CERT_PUBLISH)
		return true;

	if(resp.get_type() == NEWPKI_RESPONSE_TYPE_EE_VAL)
	{
		return InsertResponseEe(sender_name, priv_attr, resp);
	}
	else
	{
		return InsertResponseCa(sender_name, priv_attr, resp);
	}
}

bool RaStore::GetCertReq(unsigned long cert_id, NewpkiProfileDatasCert & Cert)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	
	if(req.sprintf(RASTORE_GET_CERT, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!Sql2Cert(&sql, Cert, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::Sql2Cert(SQL *sql, NewpkiProfileDatasCert & Cert, int index)
{
	mString value;
	ASN1_BIT_STRING * flags;

	Cert.Clear();

	if(!sql->Value(index,"id", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_id(value.c_ulng());

	if(!sql->Value(index,"ca_name", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_caName(value);

	if(!sql->Value(index,"status", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_state(value.c_ulng());


	if(!sql->Value(index,"type", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_type(value.c_ulng());

	
	
	if(!sql->Value(index,"flags", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	flags = StringtoASN1_BIT_STRING(value);
	if(!flags)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Cert.set_flags(flags))
	{
		ASN1_BIT_STRING_free(flags);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ASN1_BIT_STRING_free(flags);

	if(!sql->Value(index,"admin_mail", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_adminMail(value);

	// Are we in state of having the PKCS#12 ?
	if(Cert.get_state() == NEWPKI_PROFILE_CERT_STATE_ACTIVE && 
		Cert.get_type() == NEWPKI_PROFILE_CERT_TYPE_PKCS12)
	{
		// Do we have the PKCS#12 ?
		if(!sql->Value(index,"priv_datas", value))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(value.size())
		{
			if(!Cert.get_p12().Load(value, NULL))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	
	
	if(!sql->Value(index,"x509", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
	{
		if(!Cert.get_cert().SetCert(value.c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!sql->Value(index,"p7b", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
	{
		if(!Cert.get_p7b().Load(value.c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!sql->Value(index,"error", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
		Cert.set_error(value);

	Cert.set_isOK();
	return true;
}

bool RaStore::MarkCertificate(const PKI_CERT & UserCert, bool CheckOwner, unsigned long id, int NewStatus, unsigned long & serial, mString & ca_name, mString & ldap_uid)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	unsigned long owner_serial;
	unsigned long owner_type;
	PKI_CERT Cert;
	NewpkiProfile Profile;

	if(req.sprintf(RASTORE_GET_CERT_4_REV, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Check the current status of the certificate
	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(NewStatus)
	{
		case NEWPKI_PROFILE_CERT_STATE_WAITING_REV:
			if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_ACTIVE)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				return false;
			}
			break;
		case NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP:
			if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_ACTIVE)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				return false;
			}
			break;
		case NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP:
			if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_SUSPENDED)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				return false;
			}
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			return false;
	}
	if(CheckOwner)
	{
		if(!sql.Value(0, "owner_type", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		owner_type = req.c_ulng();

		if(!sql.Value(0, "owner_serial", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		owner_serial = req.c_ulng();
			
		if(!CanUserAccessProfile(UserCert, owner_serial, owner_type))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			return false;
		}
	}
	// Get the serial of the x509
	if(!sql.Value(0, "x509", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Cert.SetCert(req.c_str()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	serial = Cert.GetSerial();
	
	// Get the ca_name
	if(!sql.Value(0, "ca_name", ca_name))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Get the LDAP RDN
	if(!sql.Value(0, "datas", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// We need to reconvert if from the PEM
	if(!Profile.from_PEM(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ldap_uid = Profile.get_ldapUid();
	Profile.Clear();


	if(req.sprintf(RASTORE_SET_CERT_STATE, NewStatus, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}



bool RaStore::MarkCertificateForRevocation(const PKI_CERT & UserCert, bool CheckOwner, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid)
{
	if(!MarkCertificate(UserCert, CheckOwner, id, 
						NEWPKI_PROFILE_CERT_STATE_WAITING_REV, 
						serial, ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::MarkCertificateForSuspension(const PKI_CERT & UserCert, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid)
{
	if(!MarkCertificate(UserCert, true, id, NEWPKI_PROFILE_CERT_STATE_WAITING_SUSP, 
						serial, ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::MarkCertificateForUnsuspension(const PKI_CERT & UserCert, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid)
{
	if(!MarkCertificate(UserCert, true, id, NEWPKI_PROFILE_CERT_STATE_WAITING_UNSUSP, 
						serial, ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::CanUserAccessProfile(const PKI_CERT & UserCert, unsigned long owner_serial, unsigned long owner_type)
{
	size_t i, j;

	if(m_AclValidator->IsPkiAdministrator(UserCert))
	{
		return true;
	}

	if(owner_type == OWNER_TYPE_USER)
	{
		if(owner_serial == UserCert.GetSerial())
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	ConfAccessLock.LockRead();
	for(i=0; i < m_groups.size(); i++)
	{
		// is this the group ?
		if(m_groups[i].get_serial() == owner_serial)
		{
			for(j=0; j<m_groups[i].get_usersSerial().size(); j++)
			{
				// is this the user ?
				if(m_groups[i].get_usersSerial()[j] == UserCert.GetSerial())
				{
					ConfAccessLock.UnlockRead();
					return true;
				}
			}
			ConfAccessLock.UnlockRead();
			return false;
		}
	}
	ConfAccessLock.UnlockRead();
	return false;
}

bool RaStore::IsValidGroupId(unsigned long group_id)
{
	size_t i;

	ConfAccessLock.LockRead();
	for(i=0; i < m_groups.size(); i++)
	{
		// is this the group ?
		if(m_groups[i].get_serial() == group_id)
		{
			ConfAccessLock.UnlockRead();
			return true;
		}
	}
	ConfAccessLock.UnlockRead();
	return false;
}

bool RaStore::DeletePKCS12(const PKI_CERT &UserCert, unsigned long id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(!GetCertificateAccessCheck(UserCert, id, sql))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RASTORE_DELETE_PKCS12, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool RaStore::UpdateFromCRL(const mString & ca_name, const PKI_CRL &Crl)
{
	mString req;
	size_t i;
	const REVOCATION_INFO * currRev;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString supsList;
	mString tmpString;

	if(!Crl.RevokedCertsCount())
	{
		return true;
	}

	// Update the certificates that were unsuspended directly on the CA
	for(i=0; i<Crl.RevokedCertsCount(); i++)
	{
		currRev = Crl.GetRevokedCert(i);
		if(!currRev)
			continue;
		if(currRev->reason != OCSP_REVOKED_STATUS_CERTIFICATEHOLD)
			continue;
		if(supsList.size())
			tmpString.sprintf(", %ld", currRev->serial);
		else
			tmpString.sprintf("%ld", currRev->serial);

		supsList += tmpString;
	}
	if(supsList.size())
	{
		if(req.sprintf(RASTORE_UPDATE_FROM_CRL_UNSUSP, NEWPKI_PROFILE_CERT_STATE_ACTIVE, 
						ca_name.c_str(), NEWPKI_PROFILE_CERT_STATE_SUSPENDED, 
						supsList.c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		// There is no suspended cert, we mark as active all suspended certs
		if(req.sprintf(RASTORE_UPDATE_NO_SUSP, NEWPKI_PROFILE_CERT_STATE_ACTIVE, 
						ca_name.c_str(), NEWPKI_PROFILE_CERT_STATE_SUSPENDED) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	// Update the certificates that were revoked or suspended
	for(i=0; i<Crl.RevokedCertsCount(); i++)
	{
		currRev = Crl.GetRevokedCert(i);
		if(!currRev)
			continue;

		if(currRev->reason != OCSP_REVOKED_STATUS_CERTIFICATEHOLD)
		{
			if(req.sprintf(RASTORE_UPDATE_FROM_CRL_REV, NEWPKI_PROFILE_CERT_STATE_REVOKED, ca_name.c_str(), currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}
		else
		{
			if(req.sprintf(RASTORE_UPDATE_FROM_CRL_SUSP, NEWPKI_PROFILE_CERT_STATE_SUSPENDED, ca_name.c_str(), currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}



bool RaStore::OnNewProfile(const NewpkiProfile &newProfile)
{
	PKI_CERT EmptyCert;
	long id;
	return InsertProfile(EmptyCert, newProfile, id);
}

bool RaStore::GetKnownUIDs(mVector<mString> & KnownUIDs)
{
	mString req;
	long i;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	long NumRows;

	if(req.sprintf(RASTORE_GET_KNOWN_RDNS) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	for(i=0; i<NumRows; i++)
	{
		// The profile id
		if(!sql.Value(i, "ldap_uid", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		KnownUIDs.push_back(req);
	}
	
	return true;
}

void RaStore::GeneratePassword(mString &strP12Password)
{
	unsigned long MinPwdLen;
	char * AllowedChars = {"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890{}()[],.:;!?"};
	size_t AllowedCharsLen = strlen(AllowedChars);
	unsigned long i;
	char tt[2];
	size_t j;

	ConfAccessLock.LockRead();
	MinPwdLen = m_MinPasswdLen;
	ConfAccessLock.UnlockRead();

	strP12Password = "";
	srand(time(NULL));
	tt[1] = '\0';
	for(i=0; i<MinPwdLen; i++)
	{
		do
		{
			j = rand();
		}
		while(j > AllowedCharsLen);

		tt[0] = AllowedChars[j];
		strP12Password += tt;
	}
}


void RaStore::SetAclValidator(const X509_ACL_Validator *AclValidator)
{
	m_AclValidator = AclValidator;
}

bool RaStore::InsertResponseCert(const NewpkiCertResponse & cert_response, const mString & ca_name, PKI_CRL & Crl, LOG_MESSAGE_TYPE & LogMessage, LOG_MESSAGE_STATUS & LogStatus, int & Status, mString & strError, const HashTable_Dn & ProfileDn, mString & strPrivDatas, mString & x509, mString & p7b, unsigned long & serial, MailInfo & CertificateMail, MailInfo & PasswordMail, bool & SendCertificate, bool & SendPkcs12Pwd, bool & GenPkcs12Pwd)
{
	mString strP12Password;
	mString UserMail;
	mString MailBody;
	mString Id;
	mString AttachName;
	mString cn;

	PKI_PKCS12 p12;	
	PKI_RSA	PrivKey;
	PKI_P7B p7;
	HashTable_String ParentCerts;
	size_t i;
	int pos;
	RaStorePrivdatas PrivDatas;
	Asn1EncryptSign crypt;

	LogMessage = LOG_MESSAGE_TYPE_CERT_SIG_RESP;

	if(cert_response.get_lastCrl())
	{
		if(!(Crl = cert_response.get_lastCrl()) )
			ERR_clear_error();
	}

	// Is it an error or the certificate ?
	switch(cert_response.get_status())
	{
		// A certificate
		case CERT_RESPONSE_OK:
			Status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;
			LogStatus = LOG_STATUS_TYPE_SUCCESS;

			x509 = cert_response.get_certificate().GetCertPEM();
			serial = cert_response.get_certificate().GetSerial();

			// Get the user's mail address
			pos = cert_response.get_certificate().GetCertDN().SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND);
			if(pos != HASHTABLE_NOT_FOUND)
				UserMail = cert_response.get_certificate().GetCertDN().Get(pos);
			if(!UserMail.size())
			{
				pos = ProfileDn.SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND);
				if(pos != HASHTABLE_NOT_FOUND)
					UserMail = ProfileDn.Get(pos);
			}

			// Get the user's cn
			pos = cert_response.get_certificate().GetCertDN().SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
			if(pos != HASHTABLE_NOT_FOUND)
				cn = cert_response.get_certificate().GetCertDN().Get(pos);
			if(!cn.size())
			{
				pos = ProfileDn.SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
				if(pos != HASHTABLE_NOT_FOUND)
					cn = ProfileDn.Get(pos);
			}
			if(!cn.size())
				cn = _sv("unknown");

			/********************************************
			 *            Generate the PKCS7            *
			 ********************************************/
			p7.Clear();
			p7.AddCert(cert_response.get_certificate().GetX509());
			for(i=0; i<cert_response.get_parentcerts().size(); i++)
			{
				p7.AddCert(cert_response.get_parentcerts()[i].GetX509());
			}
			if(!p7.Generate())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			p7b = p7.GetPemP7B();
			p7.Clear();



			
			// Are we supposed to generate a PKCS12 ?
			if(strPrivDatas.size())
			{
				/*********************************************
				 *            Generate the PKCS12            *
				 *********************************************/
				if(!crypt.from_PEM(strPrivDatas.c_str()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(!PrivDatas.from_SignEncrypt(crypt, m_EntityCert.GetPrivateKey().GetRsaKey(), m_EntityCert.GetPrivateKey().GetRsaKey()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}

				// Load the private datas into a more explicit way

				// Are we supposed to generate the PKCS#12 password ?
				if(GenPkcs12Pwd)
				{
					// Generate it
					GeneratePassword(strP12Password);
				}
				else
				{
					// Get it from the private datas
					strP12Password = PrivDatas.get_passwd();
				}
				if(!strP12Password.size())
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}

				if(SendPkcs12Pwd)
				{
					if(MailBody.sprintf(_sv("For: %s\nFrom: %s\nThis is your certificate's password:\n\n%s"), 
							cert_response.get_certificate().GetStringName(), 
							ca_name.c_str(), 
							strP12Password.c_str()) > 0)
					{
						PasswordMail.set_MailTo(UserMail);
						PasswordMail.set_Subject(_sv("Your certificate's password"));
						PasswordMail.set_Body(MailBody);
						PasswordMail.set_SignMail(true);
					}
				}

				if(!PrivKey.SetKey(PrivDatas.get_privkey()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				PrivDatas.Clear();

				// Ok, let's generate the PKCS12
				p12.SetEndUserCert(cert_response.get_certificate());
				p12.SetEndUserKey(PrivKey);

				ParentCerts.AllowDuplicateNames();

				for(i=0; i<cert_response.get_parentcerts().size(); i++)
				{
					Id = i;
					ParentCerts.Add(Id.c_str(), cert_response.get_parentcerts()[i].GetCertPEM().c_str());
				}
				p12.SetParentCerts(ParentCerts);

				if(!p12.Generate(strP12Password.c_str()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				// The P12 is saved in DB
				strPrivDatas = p12.GetPemPKCS12();
				p12.Clear();

				if(SendCertificate)
				{
					// Save the P12 in DER
					if(!strPrivDatas.ToDER(CertificateMail.get_Attach()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					if(AttachName.sprintf("%s.p12", cn.c_str()) > 0)
					{
						CertificateMail.set_AttachName(AttachName);
						CertificateMail.set_AttachType("application/x-pkcs12");
					}
				}
			}
			else
			{
				if(SendCertificate)
				{
					// Save the p7b in DER
					if(!p7b.ToDER(CertificateMail.get_Attach()))
					{
						NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
						return false;
					}
					if(AttachName.sprintf("%s.p7b", cn.c_str()) > 0)
					{
						CertificateMail.set_AttachName(AttachName);
						CertificateMail.set_AttachType("application/x-pkcs7-certificates");
					}
				}
			}

			// Generate the whole email
			if(SendCertificate)
			{
				if(MailBody.sprintf(_sv("For: %s\nFrom: %s\nYour certificate has been generated."), 
					cert_response.get_certificate().GetStringName(), 
					ca_name.c_str()) >= 0)
				{
					CertificateMail.set_MailTo(UserMail);
					CertificateMail.set_Subject(_sv("Your certificate"));
					CertificateMail.set_Body(MailBody);
					CertificateMail.set_SignMail(true);
				}
			}
			break;
		
		// An error
		case CERT_RESPONSE_ERROR:
			Status = NEWPKI_PROFILE_CERT_STATE_ERROR;
			LogStatus = LOG_STATUS_TYPE_FAILURE;
			SendCertificate = false;
			GenPkcs12Pwd = false;
			SendPkcs12Pwd = false;
			strPrivDatas = "";
			ERROR_ENTRIES_to_string(cert_response.get_errors(), strError);
			break;

		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
			break;
	}

	return true;
}

bool RaStore::ChangeProfileUID(const PKI_CERT &UserCert, unsigned long id, const mString & uid)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString pem_signature;
	mString pem_profile;
	NewpkiProfileDatas Profile;

	if(!GetProfile(UserCert, id, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Changing the owner
	Profile.get_profile().set_ldapUid(uid);

	// Signing the profile
	if(!SignProfile(Profile.get_profile(), pem_signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Profile.get_profile().to_PEM(pem_profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RASTORE_SET_UID, uid.c_str(), pem_profile.c_str(), pem_signature.c_str(), id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::AcceptProfile(const PKI_CERT & UserCert, unsigned long id)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString pem_signature;
	mString pem_profile;
	NewpkiProfileDatas Profile;

	if(!GetProfile(UserCert, id, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(Profile.get_state() != NEWPKI_PROFILE_STATE_WAITING_FOR_ADMIN_VALIDATION)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(req.sprintf(RASTORE_SET_STATE, NEWPKI_PROFILE_STATE_VALIDATED, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::ChangeProfileOwner(const PKI_CERT &UserCert, unsigned long id, unsigned long group_id)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString pem_signature;
	mString pem_profile;
	NewpkiProfileDatas Profile;

	if(!GetProfile(UserCert, id, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!group_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!IsValidGroupId(group_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_GROUP_UNKNOWN);
		return false;
	}

	// Changing the owner
	Profile.get_profile().set_ownerGroupSerial(group_id);

	// Signing the profile
	if(!SignProfile(Profile.get_profile(), pem_signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Profile.get_profile().to_PEM(pem_profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RASTORE_SET_OWNER, OWNER_TYPE_GROUP, group_id, pem_profile.c_str(), pem_signature.c_str(), id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::ChangeProfileDN(const PKI_CERT &UserCert, unsigned long id, const X509_NAME *dn)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString pem_signature;
	mString pem_profile;
	NewpkiProfileDatas Profile;
	char * ldn;
	mString strDN;
	mString strHash;
	bool Exists;
	unsigned long profile_id;
	HashTable_Dn hDn;

	// Get the profile
	if(!GetProfile(UserCert, id, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// The profile cannot be changed by an admin
	// when we're waiting for an EE validation !
	if(UserCert && Profile.get_state() == NEWPKI_PROFILE_STATE_WAITING_FOR_VALIDATION)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("Waiting for EE validation"));
		return false;
	}

	// Search if the profile is unique
	if(!X509_NAMEtoHash(dn, strHash))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	profile_id = id;
	if(!ProfileExists(strHash, Exists, profile_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(Exists)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_UNIQUE_PROFILE);
		strHash.sprintf("%ld", profile_id);
		ERR_add_error_data(1, strHash.c_str());
		return false;
	}

	// We now make sure the DN respects the policies
	if(!hDn.From_X509_NAME(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.LockRead();
	if(!hDn.ValidateAgainstPolicy(m_Policies))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	// Changing the DN
	if(!Profile.get_profile().set_dn(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ldn = X509_NAME_oneline((X509_NAME*)dn, NULL, 0);
	if(!ldn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	strDN = sql.FormatString(ldn);
	free(ldn);


	// Signing the profile
	if(!SignProfile(Profile.get_profile(), pem_signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Profile.get_profile().to_PEM(pem_profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(RASTORE_SET_DN, strHash.c_str(), 
									strDN.c_str(), 
									pem_profile.c_str(), 
									pem_signature.c_str(), 
									NEWPKI_PROFILE_STATE_VALIDATED, 
									id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::GetProfileAccessCheck(const PKI_CERT &UserCert, unsigned long ProfileId, SQL & sql)
{
	mString req;
	long NumRows;
	unsigned long owner_serial;
	unsigned long owner_type;

	if(req.sprintf(RASTORE_GET_PROFILE, ProfileId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!sql.Value(0, "owner_type", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_type = req.c_ulng();

	if(!sql.Value(0, "owner_serial", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_serial = req.c_ulng();
		
	if(UserCert && !CanUserAccessProfile(UserCert, owner_serial, owner_type))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	return true;
}


bool RaStore::GetCertificateAccessCheck(const PKI_CERT &UserCert, unsigned long CertId, SQL & sql)
{
	mString req;
	long NumRows;
	unsigned long owner_serial;
	unsigned long owner_type;

	if(req.sprintf(RASTORE_GET_CERT_OWNER, CertId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Check the current status of the certificate
	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_ACTIVE)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!sql.Value(0, "owner_type", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_type = req.c_ulng();

	if(!sql.Value(0, "owner_serial", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_serial = req.c_ulng();
		
	if(!CanUserAccessProfile(UserCert, owner_serial, owner_type))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	return true;
}

bool RaStore::RequestCertificate(const PKI_CERT & UserCert, const NewpkiRequestCert & Request, PKI_CSR & Csr, mString & ldap_uid, unsigned long & CertReqId)
{
	PKI_RSA privKey;
	NewpkiProfileDatas profile;
	HashTable_Dn Dn;
	RaStorePrivdatas PrivDatas;
	mString admin_mail;
	long pos;

	// Get admin mail
	pos = UserCert.GetCertDN().SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND);
	if(pos != HASHTABLE_NOT_FOUND)
	{
		admin_mail = UserCert.GetCertDN().Get(pos);
	}
	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_SEND_ADMIN_NOTIFICATION))
	{
		if(!admin_mail.size())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_set_error_data(_sv("emailAddress not found in admin certificate"), ERR_TXT_STRING);
			return false;
		}
		if(!MailerIsUp())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_MAIL_SERVER_NOT_SET);
			return false;
		}
	}
	
	if((ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_SEND_CERT_TO_USER) ||
		ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD) ||
		ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER)) &&
		!MailerIsUp())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MAIL_SERVER_NOT_SET);
		return false;
	}

	
	//Let's get the profile
	if(!GetProfile(UserCert, Request.get_profileId(), profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Is the profile ready for signature ?
	if(profile.get_state() != NEWPKI_PROFILE_STATE_VALIDATED)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Are we supposed to send the certificate to the user ?
	// REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD means as well that 
	// the password is automatically sent to the user !
	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_SEND_CERT_TO_USER) ||
		ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD) ||
		ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER))
	{
		// Does the user's DN have a emailAddress field ?
		if(!Dn.From_X509_NAME(profile.get_profile().get_dn()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(Dn.SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND) == HASHTABLE_NOT_FOUND)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_set_error_data(_sv("emailAddress not found in user's DN"), ERR_TXT_STRING);
			return false;
		}
	}


	switch(Request.get_request().get_type())
	{
		case REQUEST_CERT_BODY_PKCS10:
			if( !(Csr = Request.get_request().get_pkcs10()) )
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;

		case REQUEST_CERT_BODY_PKCS12:
			// Let's generate the PKCS#10
			if(!Dn.From_X509_NAME(profile.get_profile().get_dn()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!privKey.GenerateKey(Request.get_request().get_p12Privdatas().get_keylen(), m_Engine))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Csr.GenerateCSR(Dn, privKey))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			// Copying the password
			if(!ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)Request.get_flags(), REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD))
			{
				PrivDatas.set_passwd(Request.get_request().get_p12Privdatas().get_passwd());
			}
			// Copying the private key 
			if(!PrivDatas.set_privkey(privKey.GetRSA()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			PrivDatas.set_isOK();
			break;
	}


	// We get the LDAP UID
	ldap_uid = profile.get_profile().get_ldapUid();


	// Insert the request in the store, and get the id
	if(!InsertCertReq(Request.get_caName(), Request.get_profileId(), Request.get_type(), Csr, PrivDatas, Request.get_flags(), admin_mail, Asn1OctetString::EmptyInstance, CertReqId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::RequestCertificateForEE(const Asn1OctetString & transactionId, const mString & ee_name, const NewpkiEeRequestCert & Request, PKI_CSR & Csr, unsigned long & CertReqId)
{
	PKI_RSA privKey;
	NewpkiProfileDatas profile;
	HashTable_Dn Dn;
	RaStorePrivdatas PrivDatas;
	
	//Let's get the profile
	if(!GetEeProfile(ee_name, Request.get_profileId(), profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Is the profile ready for signature ?
	if(profile.get_state() != NEWPKI_PROFILE_STATE_VALIDATED)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	switch(Request.get_request().get_type())
	{
		case REQUEST_CERT_BODY_PKCS10:
			if( !(Csr = Request.get_request().get_pkcs10()) )
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;

		case REQUEST_CERT_BODY_PKCS12:
			// Let's generate the PKCS#10
			if(!Dn.From_X509_NAME(profile.get_profile().get_dn()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!privKey.GenerateKey(Request.get_request().get_p12Privdatas().get_keylen(), m_Engine))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Csr.GenerateCSR(Dn, privKey))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			PrivDatas.set_passwd(Request.get_request().get_p12Privdatas().get_passwd());
			// Copying the private key 
			if(!PrivDatas.set_privkey(privKey.GetRSA()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			PrivDatas.set_isOK();
			break;
	}
	// Insert the request in the store, and get the id
	if(!InsertCertReq(Request.get_caName(), profile.get_id(), Request.get_type(), Csr, PrivDatas, NULL, "", transactionId, CertReqId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::UnmarkCertificateForRevocation(unsigned long id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(RASTORE_SET_CERT_STATE, NEWPKI_PROFILE_CERT_STATE_ACTIVE, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::UnmarkCertificateForSuspension(unsigned long id)
{
	if(!UnmarkCertificateForRevocation(id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool RaStore::UnmarkCertificateForUnsuspension(unsigned long id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(RASTORE_SET_CERT_STATE, NEWPKI_PROFILE_CERT_STATE_SUSPENDED, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


bool RaStore::SignProfile(const NewpkiProfile & Profile, mString & pem_signature)
{
	RaStoreSig signature;
	NEWPKI_PROFILE * lProfile;
	
	// Sign the profile
	lProfile=NULL;
	if(!Profile.give_Datas(&lProfile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		if(lProfile)
			ASN1_item_free((ASN1_VALUE*)lProfile, NewpkiProfile::get_ASN1_ITEM());
		return false;
	}
	if(!NEWPKI_PROFILE_sign(lProfile, signature, (EVP_PKEY*)m_EntityCert.GetPrivateKey().GetRsaKey()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)lProfile, NewpkiProfile::get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lProfile, NewpkiProfile::get_ASN1_ITEM());
	
	
	//Get the signature into pem
	if(!signature.to_PEM(pem_signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool RaStore::DeleteRequest(const PKI_CERT &UserCert, long RequestId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	unsigned long owner_serial;
	unsigned long owner_type;
	PKI_CERT Cert;
	NewpkiProfile Profile;


	if(req.sprintf(RASTORE_GET_CERT_4_REV, RequestId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Check the current status of the certificate
	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_ERROR)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!sql.Value(0, "owner_type", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_type = req.c_ulng();

	if(!sql.Value(0, "owner_serial", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	owner_serial = req.c_ulng();
		
	if(!CanUserAccessProfile(UserCert, owner_serial, owner_type))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Delete request
	if(req.sprintf(RASTORE_DELETE_CERT, RequestId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

