/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma indent

using System
using System.IO
using System.Text

namespace Nemerle.IO
  public class PipeReader : TextReader
    input_reader : TextReader
    filter : string -> string
    mutable input_ptr : int
    mutable line : string = ""
    
    /// [filter] is called for each line of the input.
    public this (input_reader : TextReader, filter : string -> string)
      this.filter = filter
      this.input_reader = input_reader

    fill_buffer () : void
      input_ptr = 0
      def orig = input_reader.ReadLine ()
      if (orig == null)
        line = ""
      else
        line = filter (orig + "\n")
        when (line == "")
          fill_buffer ()

    protected override Dispose (disposing : bool) : void
      base.Dispose (disposing)
      (input_reader : IDisposable).Dispose ()

    public override Peek () : int
      when (line.Length <= input_ptr)
        fill_buffer ()
      if (line.Length <= input_ptr) -1
      else line [input_ptr] :> int

    public override Read () : int
      def ch = Peek ()
      when (ch != -1) input_ptr++
      ch

    public override ReadToEnd () : string
      def ch = Read ()
      if (ch == -1)
        null
      else
        def buf = StringBuilder ()
        _ = buf.Append (ch :> char)
        def loop ()
          match (Read ())
            | -1 => buf.ToString ()
            | ch => _ = buf.Append (ch :> char); loop ()
        loop ()

    public override ReadLine () : string
      def ch = Read ()
      if (ch == -1)
        null
      else
        def buf = StringBuilder ()
        _ = buf.Append (ch :> char)
        def loop ()
          def ch = Read ()
          if (ch == '\r' :> int)
            loop ()
          else if (ch == '\n' :> int || ch == -1)
            buf.ToString ()
          else
            _ = buf.Append (ch :> char)
            loop ()
        loop ()
