/* Copyright (C) 2000-2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MYODBCUtil.h"

/*!
    \note   - unixODBC does not have a header section in ini files.
            - On XP, the "32 bit" parts are secretly stripped away
            inside SQLGetPrivateProfileString() somewhere.
*/            
#define MYODBCUTIL_ODBCINI_HEADER_SECTION "ODBC Data Sources"

/*!
    \brief  Read data source from system information.

    \param  pDataSource Data Source struct.
    \param  pszDSN The data source name to lookup.

    \note   This will not replace existing values in pDataSource.

            On XP we use hard-coded driver name and driver
            file name (if not given) as it seems MS is in the middle
            of changing odbc system info (probably to better support
            64 bit).
*/            
BOOL MYODBCUtilReadDataSource( MYODBCUTIL_DATASOURCE *pDataSource, LPCSTR pszDSN )
{
    char    szEntryNames[SQL_MAX_DSN_LENGTH * MYODBCUTIL_MAX_DSN_NAMES];
    char *  pszEntryName;
    char    szValue[4096];
    int     nChars  = 0;
#if defined(WIN32)
    UWORD   nMode   = ODBC_BOTH_DSN;

    if ( !SQLGetConfigMode( &nMode ) )
    {
        fprintf( stderr, "[%s][%d][ERROR] SQLGetConfigMode failed!\n", __FILE__, __LINE__ );
        return FALSE;
    }
#endif

    if ( !pszDSN || !(*pszDSN) )
        return TRUE;

    *szEntryNames = '\0';

#if defined(__APPLE__) && 0
    /*!
        \note   OSX

                SQLGetPrivateProfileString is the proper call and is available - but
                at this time it appears utterly broken. So we call an alternative
                instead. 
    */
    if ( ( nChars = GetPrivateProfileString( pszDSN, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "odbc.ini" ) ) < 1 )
#elif defined(__APPLE__)
    if ( ( nChars = SQLGetPrivateProfileString( pszDSN, "", "", szEntryNames, sizeof( szEntryNames ) - 1, "odbc.ini" ) ) < 1 )
#else
    if ( ( nChars = SQLGetPrivateProfileString( pszDSN, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "ODBC.INI" ) ) < 1 )
#endif
    {
        return FALSE;
    }

#if defined(WIN32)
    {
        int     nLen    = strlen( szEntryNames );

        /*!
            \note   XP

                    Bug in SQLGetPrivateProfileString when mode is ODBC_BOTH_DSN and we are looking for a system
                    DSN. In this case SQLGetPrivateProfileString will find the system dsn but return a corrupt
                    list of attributes. 

                    A corrupt list of attributes can be identified because the first attribute (if any) will be
                    followed by more than one '\0'.

                    The solution is to detect this condition and set mode to ODBC_SYSTEM_DSN and try again. We 
                    also ensure we reset the mode when done - regardless of outcome.
        */                
        if ( nMode == ODBC_BOTH_DSN && nLen < nChars && szEntryNames[nLen + 1 ] == '\0' )
        {
            *szEntryNames = '\0';
            if ( !SQLSetConfigMode( ODBC_SYSTEM_DSN ) )
                fprintf( stderr, "[%s][%d][ERROR] SQLSetConfigMode failed!\n", __FILE__, __LINE__ );

            if ( ( nChars = SQLGetPrivateProfileString( pszDSN, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "ODBC.INI" ) ) < 1 )
                return FALSE;
        }
    }
#endif

    /*!
        Looks like things are going to work. Lets set DSN.
    */        
    if ( !pDataSource->pszDSN )
        pDataSource->pszDSN = _global_strdup( pszDSN ) ;

    /*!
        Scan result and return TRUE if we find a match.
    */
    pszEntryName = szEntryNames;
    while ( *pszEntryName )
    {
        *szValue = '\0';
#if defined(__APPLE__) && 0
        if ( GetPrivateProfileString( pszDSN, pszEntryName, NULL, szValue, sizeof( szValue ) - 1, "odbc.ini" ) > 0 )
#elif defined(__APPLE__)
        if ( SQLGetPrivateProfileString( pszDSN, pszEntryName, "", szValue, sizeof( szValue ) - 1, "odbc.ini" ) > 0 )
#elif defined(WIN32)
        if ( SQLGetPrivateProfileString( pszDSN, pszEntryName, NULL, szValue, sizeof( szValue ) - 1, "ODBC.INI" ) > 0 )
#else
        if ( SQLGetPrivateProfileString( pszDSN, pszEntryName, "", szValue, sizeof( szValue ) - 1, "ODBC.INI" ) > 0 )
#endif
        {
            /*!
                ODBC RULE
                (SQLDriverConnect)

                The driver uses any information it retrieves from the system information 
                to augment the information passed to it in the connection string. If the 
                information in the system information duplicates information in the 
                connection string, the driver uses the information in the connection 
                string.
            */    
            if ( strcasecmp( pszEntryName, "DATABASE" ) == 0 || strcasecmp( pszEntryName, "DB" ) == 0 )
            {
                if ( !pDataSource->pszDATABASE )
                    pDataSource->pszDATABASE = _global_strdup( szValue ) ;
            }
            else if ( strcasecmp( pszEntryName, "DESCRIPTION" ) == 0 || strcasecmp( pszEntryName, "DESC" ) == 0 )
            {
                if ( !pDataSource->pszDESCRIPTION )
                    pDataSource->pszDESCRIPTION = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "DRIVER" ) == 0 )
            {
#if defined(WIN32)
                if ( !pDataSource->pszDriverFileName )
                    pDataSource->pszDriverFileName = _global_strdup( szValue );
#else
                if ( *szValue == '/' )
                {
                    if ( !pDataSource->pszDriverFileName )
                        pDataSource->pszDriverFileName = _global_strdup( szValue );
                }
                else
                {
                    if ( !pDataSource->pszDRIVER )
                        pDataSource->pszDRIVER = _global_strdup( szValue );
                }
#endif
            }
            else if ( strcasecmp( pszEntryName, "OPTION" ) == 0 )
            {    
                if ( !pDataSource->pszOPTION )
                    pDataSource->pszOPTION = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "PWD" ) == 0 || strcasecmp( pszEntryName, "PASSWORD" ) == 0 )
            {    
                if ( !pDataSource->pszPASSWORD )
                    pDataSource->pszPASSWORD = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "PORT" ) == 0 )
            {    
                if ( !pDataSource->pszPORT )
                    pDataSource->pszPORT = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "SERVER" ) == 0 )
            {    
                if ( !pDataSource->pszSERVER )
                    pDataSource->pszSERVER = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "SOCKET" ) == 0 )
            {    
                if ( !pDataSource->pszSOCKET )
                    pDataSource->pszSOCKET = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "STMT" ) == 0 )
            {    
                if ( !pDataSource->pszSTMT )
                    pDataSource->pszSTMT = _global_strdup( szValue );
            }
            else if ( strcasecmp( pszEntryName, "UID" ) == 0 || strcasecmp( pszEntryName, "USER" ) == 0 )
            {    
                if ( !pDataSource->pszUSER )
                    pDataSource->pszUSER = _global_strdup( szValue );
            }
            else
            {
                /* What the ? */
                fprintf( stderr, "[%s][%d][ERROR] Unknown attribute (%s).\n", __FILE__, __LINE__, pszEntryName );
            }
        }
        else
        {
            /*  
                At least two diff Users stated they did not want this message and I think we can do with out
                this one so lets silently move on.

                see; CSC 3985, email 52189

                fprintf( stderr, "[%s][%d][WARNING] Failed to get value for attribute (%s).\n", __FILE__, __LINE__, pszEntryName );
            */    
        }

        pszEntryName += strlen( pszEntryName ) + 1;
    } /* while */


    /* try harder to get the friendly driver name */
    if ( !pDataSource->pszDRIVER )
    {
#if defined(__APPLE__) && 0
        if ( GetPrivateProfileString( MYODBCUTIL_ODBCINI_HEADER_SECTION, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "odbc.ini" ) < 1 )
#elif defined(__APPLE__)
        if ( SQLGetPrivateProfileString( MYODBCUTIL_ODBCINI_HEADER_SECTION, "", "", szEntryNames, sizeof( szEntryNames ) - 1, "odbc.ini" ) < 1 )
#else
        if ( SQLGetPrivateProfileString( MYODBCUTIL_ODBCINI_HEADER_SECTION, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "ODBC.INI" ) < 1 )
#endif
        {
            return FALSE;
        }
    
        pszEntryName = szEntryNames;
        while ( *pszEntryName )
        {
            *szValue = '\0';
#if defined(__APPLE__) && 0
            if ( GetPrivateProfileString( MYODBCUTIL_ODBCINI_HEADER_SECTION, pszEntryName, NULL, szValue, sizeof( szValue ) - 1, "odbc.ini" ) > 0 )
#else
            if ( SQLGetPrivateProfileString( MYODBCUTIL_ODBCINI_HEADER_SECTION, pszEntryName, NULL, szValue, sizeof( szValue ) - 1, "ODBC.INI" ) > 0 )
#endif
            {
                if ( strcasecmp( pszEntryName, pszDSN ) == 0 )
                {
                    pDataSource->pszDRIVER = _global_strdup( szValue );
                }
            }
            pszEntryName += strlen( pszEntryName ) + 1;
        }
    }

#if defined(WIN32)
    /*!
        XP

        Some people have reported that the friendly driver name is never found so...
    */
    if ( !pDataSource->pszDRIVER )
        pDataSource->pszDRIVER = MYODBCINST_DRIVER_NAME;
#endif

#if defined(WIN32)
    /* ALWAYS restore mode */
    SQLSetConfigMode( nMode );
#endif

    return TRUE;
}

