/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.14 2004/08/27 18:32:52 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mdatatypes.h"
#include "mconfig.h"
#include "datatype.h"

/* begin of Mailhist */

int mdata_Mailhist_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.hist->data.mail.x, #x);

	WR(incoming_mails, ld);
	WR(outgoing_mails, ld);
	WR(incoming_bytes, ld);
	WR(outgoing_bytes, ld);
#undef WR
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.hist->x, #x);
	WR(year, d);
	WR(month, d);
	WR(week, d);
	WR(days_passed, d);
#undef WR
	return 0;
}

int mdata_Mailhist_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_MAILHIST) return -1;
	
	free(data->data.hist);

	return 0;
}

int mdata_Mailhist_setdata(mdata *data, const char *str,
			   long incoming_mails,
			   long outgoing_mails,
			   long incoming_bytes,
			   long outgoing_bytes,
			   unsigned int year,
			   unsigned int month,
			   unsigned int week,
			   unsigned int days_passed
			   ) {

	data->key = str;
	assert(data->key);

	if (data->type == M_DATA_TYPE_UNSET)
		data->type		= M_DATA_TYPE_MAILHIST;

	data->data.hist->data.mail.incoming_mails	= incoming_mails;
	data->data.hist->data.mail.outgoing_mails	= outgoing_mails;

	data->data.hist->data.mail.incoming_bytes	= incoming_bytes;
	data->data.hist->data.mail.outgoing_bytes	= outgoing_bytes;

	data->data.hist->year  = year;
	data->data.hist->month = month;
	data->data.hist->week  = week;
	data->data.hist->days_passed = days_passed;

	data->data.hist->type = M_HIST_TYPE_MAIL;

	return 0;
}

int mdata_Mailhist_from_xml(void *user_data, int tagtype, const xmlChar *value, const xmlChar **attrs) {
	int i;
	const mdata_values data_values[] = {
		{ "incoming_mails", M_DATA_FIELDTYPE_LONG },
		{ "outgoing_mails", M_DATA_FIELDTYPE_LONG },
		{ "incoming_bytes", M_DATA_FIELDTYPE_LONG },
		{ "outgoing_bytes", M_DATA_FIELDTYPE_LONG },
		{ "year", M_DATA_FIELDTYPE_LONG },
		{ "month", M_DATA_FIELDTYPE_LONG },
		{ "week", M_DATA_FIELDTYPE_LONG },
		{ "days_passed", M_DATA_FIELDTYPE_LONG },

		{ NULL, M_DATA_FIELDTYPE_UNSET }
	};

	mstate_stack *m = user_data;

//	M_WP();

	switch(tagtype) {
	case M_TAG_BEGIN:
		for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
			;

		if (data_values[i].string == NULL) {
			M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "unknown tag '%s'\n",
				 value);
			return -1;
		}
#define SET_DATA(x) \
	m->st[m->st_depth].data = &(((mdata *)(m->st[m->st_depth-1].data))->x);
		switch(i) {
		case 0: SET_DATA(data.hist->data.mail.incoming_mails); break;
		case 1: SET_DATA(data.hist->data.mail.outgoing_mails); break;
		case 2: SET_DATA(data.hist->data.mail.incoming_bytes); break;
		case 3: SET_DATA(data.hist->data.mail.outgoing_bytes); break;
		case 4: SET_DATA(data.hist->year); break;
		case 5: SET_DATA(data.hist->month); break;
		case 6: SET_DATA(data.hist->week); break;
		case 7: SET_DATA(data.hist->days_passed); break;
		default:
			return -1;
		}
#undef SET_DATA
		/* the same for all 4 values */
		m->st[m->st_depth].function = mdata_insert_value;
		m->st[m->st_depth].type = data_values[i].type;

		break;
	case M_TAG_END:
		((mdata *)(m->st[m->st_depth-1].data))->type = M_DATA_TYPE_MAILHIST;
		((mdata *)(m->st[m->st_depth-1].data))->data.hist->type = M_HIST_TYPE_MAIL;
		switch(m->st[m->st_depth-2].type) {
		case M_DATA_FIELDTYPE_LIST: {
			mlist *h = m->st[m->st_depth-2].data;
			mdata *data = m->st[m->st_depth-1].data;

			mlist_insert_sorted(h, data);
			break;
		}
		default:
			M_WP();
		}

		break;
	case M_TAG_TEXT:
		M_WP();
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n",
			 tagtype);
		return -1;
	}

	return 0;
}



int mdata_Mailhist_append(mdata *dst, mdata *src) {
	fprintf(stderr, "%s.%d: (mailhist) abstract function 'append' !!\n", __FILE__, __LINE__);

	return 0;
}

mdata *mdata_Mailhist_init() {
	mdata *data = mdata_init();
	assert(data);

	data->key		= NULL;
	data->type		= M_DATA_TYPE_MAILHIST;
	
	data->data.hist         = malloc(sizeof(data_History));
	memset(data->data.hist, 0, sizeof(data_History));

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
	data->data.hist->data.mail.incoming_mails	= 0;
	data->data.hist->data.mail.outgoing_mails	= 0;

	data->data.hist->data.mail.incoming_bytes	= 0;
	data->data.hist->data.mail.outgoing_bytes	= 0;

	data->data.hist->year	= 0;
	data->data.hist->month	= 0;
	data->data.hist->week	= 0;

	data->data.hist->days_passed = 0;
	data->data.hist->data.web.xfersize = 0;

	data->data.hist->type = M_HIST_TYPE_MAIL;

	return data;
}

mdata *mdata_Mailhist_copy(mdata *src) {
	mdata *data = mdata_Mailhist_init();
	assert(data);

	fprintf(stderr, "%s.%d: (mailhist) abstract function 'copy' !!\n", __FILE__, __LINE__);

/*
	mdata_Mailhist_setdata(data, src->key, src->data.mailhist.count, src->data.mailhist.grouped);
*/
	return data;
}

mdata *mdata_Mailhist_create(const char *str,
			     long incoming_mails,
			     long outgoing_mails,
			     long incoming_bytes,
			     long outgoing_bytes,
			     unsigned int year,
			     unsigned int month,
			     unsigned int week,
			     unsigned int days_passed) {
	mdata *data = mdata_Mailhist_init();
	assert(data);

	mdata_Mailhist_setdata(data, str,
			       incoming_mails,
			       outgoing_mails,
			       incoming_bytes,
			       outgoing_bytes,
			       year,
			       month,
			       week,
			       days_passed);

	return data;
}

mdata  *mdata_Mailhist_create_by_state(mstate *state) {
	char buf[7];

	long incoming_mails;
	long outgoing_mails;
	long incoming_bytes;
	long outgoing_bytes;
#if 0
	unsigned int year;
	unsigned int month;
	unsigned int week;
	unsigned int days_passed;
#endif

	int last_day = 0;

	struct tm tm;
	mstate_mail *stamail;
	time_t timestamp;

	mdata *data = mdata_Mailhist_init();
	assert(data);

	if (state == NULL) return NULL;

	timestamp = state->timestamp;

	tm = *localtime(&(state->timestamp));

	sprintf(buf, "%04d%02d", state->year, state->month);

	outgoing_mails = incoming_mails = 0;
	outgoing_bytes = incoming_bytes = 0;

	if (state->ext && state->ext_type == M_STATE_TYPE_MAIL) {
		int i;
		stamail = state->ext;

		for ( i = 0; i < 31; i++) {
			incoming_mails += stamail->days[i].incoming_mails;
			outgoing_mails += stamail->days[i].outgoing_mails;
			incoming_bytes += stamail->days[i].incoming_bytes;
			outgoing_bytes += stamail->days[i].outgoing_bytes;
			if (stamail->days[i].incoming_mails) last_day = i+1;
		}
	}
	mdata_Mailhist_setdata(data, strdup(buf),
			       incoming_mails,
			       outgoing_mails,
			       incoming_bytes,
			       outgoing_bytes,
			       state->year,
			       state->month,
			       state->week,
			       last_day
			       );

	return data;
}

/* end of Mailhist */
