/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <time.h>
/* other libs */
#include <db.h>
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
/* local */
#include <libbtutil/const.h>
#include <libbtutil/util.h>
#include <libbttracker/types/btt_peer.h>

const btt_peer new_btt_peer = {
    BT_EMPTY_PEERID,            /* peerid */
    BT_EMPTY_INFOHASH,          /* infohash */
    "",                         /* ua */
    "",                         /* event */

    0,                          /* flags */

    { AF_INET, 0, { 0 } },      /* address */
    { AF_INET, 0, { 0 } },      /* real_address */

    0, 0,                       /* first_t, last_t */
    0, 0,                       /* first_serve_t, last_serve_t */
    0,                          /* complete_t */
    0,                          /* return_interval */
    0,                          /* hits */
    0,                          /* serves */
    -1,                         /* num_want */
    0,                          /* num_got */

    0, 0, 0, 0                  /* announce_bytes, uploaded, downloaded, left */
};

/* bt_peer object used in iterators for the details webpage */
const btt_peer web_btt_peer = {
    BT_EMPTY_PEERID,            /* peerid */
    BT_EMPTY_INFOHASH,          /* infohash */
    "",                         /* ua */
    "",                         /* event */

    BTT_PEER_HTML,              /* flags */

    { AF_INET, 1, { 1 } },      /* address */
    { AF_INET, 1, { 1 } },      /* real_address */

    0, 0,                       /* first_t, last_t */
    0, 0,                       /* first_serve_t, last_serve_t */
    0,                          /* complete_t */
    0,                          /* return_interval */
    0,                          /* hits */
    0,                          /* serves */
    -1,                         /* num_want */
    0,                          /* num_got */

    0, 0, 0, 0                  /* announce_bytes, uploaded, downloaded, left */
};

const btt_peer_flag btt_peer_flags[] = {
    { BTT_PEER_SHIELD,          "Shielded",             "Shield"        },
    { BTT_PEER_COMPACT,         "Send Compact",         "Compact"       },
    { BTT_PEER_NOPEERID,        "Send No Peer ID",      "NoPeerID"      },
    { BTT_PEER_SHUNNED,         "Shunned",              "Shunned"       },
    { BTT_PEER_PASSIVE,         "Passive",              "Passive"       },
    { BTT_PEER_HTML,            "HTML",                 "HTML"          },
    { 0,                        NULL,                   NULL            }
};

int btt_peer_crossref(DB* dbp, const DBT* pkey, const DBT* pdata, DBT* skey) {
    memset(skey, 0, sizeof(DBT));
    skey->data = ((btt_peer*)pdata->data)->infohash;
    skey->size = sizeof((btt_peer*)pdata->data)->infohash;
    return (0);
}

int btt_peer_req2struct(apr_pool_t* p, const char* args, btt_peer* peer) {
    char* cp;
    char* aargs;
    unsigned char gotpid, gothash;
    bt_form_pair fp;
    struct in_addr ipconv;
    int conv;
    time_t now = time(NULL);
 
    aargs = apr_pstrdup(p, args);
    cp = aargs;

    gotpid = 0;
    gothash = 0;
 
    while(cp) {
        fp = bt_next_form_pair_unesc(cp);
        if((cp = fp.eop))
            cp++;
   
        if(!strcmp("info_hash", fp.key)) {
            if(fp.value_len == BT_INFOHASH_LEN) {
                gothash++;
                memcpy(peer->infohash, fp.value, BT_INFOHASH_LEN);
            } else {
                fprintf(
                    stderr,
                    "bt_peer_req2struct(): got an invalid infohash "
                    "of length %"APR_SIZE_T_FMT"\n", fp.value_len
                );
                fflush(stderr);
                return 0;
            }
        } else if(!strcmp("peer_id", fp.key)) {
            if(fp.value_len == BT_PEERID_LEN) {
                gotpid++;
                memcpy(peer->peerid, fp.value, BT_PEERID_LEN);
            } else {
                fprintf(
                    stderr,
                    "bt_peer_req2struct(): got an invalid peerid "
                    "of length %"APR_SIZE_T_FMT"\n", fp.value_len
                );
                fflush(stderr);
                return 0;
            }
        } else if(!strcmp("ip", fp.key)) {
            if(inet_aton(fp.value, &ipconv)) {
                peer->address.sin_family = AF_INET;
                peer->address.sin_addr = ipconv;
            } else {
                fprintf(stderr, "Got an invalid IP of \"%s\"\n", fp.value);
                fflush(stderr);
                return 0;
            }
        } else if(!strcmp("port", fp.key)) {
            if((conv = atoi(fp.value))) {
                peer->address.sin_family = AF_INET;
                peer->address.sin_port = htons(conv);
            } else {
                fprintf(
                    stderr,
                    "bt_peer_req2struct(): got an invalid port of \"%s\"\n",
                    fp.value
                );
                fflush(stderr);
                return 0;
            }
        } else if(!strcmp("numwant", fp.key))
            peer->num_want = (int32_t)atoll(fp.value);
        else if(!strcmp("uploaded", fp.key))
            peer->uploaded = (bt_off_t)atoll(fp.value);
        else if(!strcmp("downloaded", fp.key))
            peer->downloaded = (bt_size_t)atoll(fp.value);
        else if(!strcmp("left", fp.key))
            peer->left = (bt_size_t)atoll(fp.value);
        else if(!strcmp("event", fp.key)) {
            if(fp.value_len > BT_EVENT_LEN - 1) {
                fprintf(
                    stderr,
                    "bt_peer_req2struct(): Got an invalid event "
                    "of length %"APR_SIZE_T_FMT"\n", fp.value_len
                );
                fflush(stderr);
                return 0;
            } else if(fp.value_len) {
                if(
                    (!strcmp("started", fp.value)) ||
                    (!strcmp("completed", fp.value)) ||
                    (!strcmp("stopped", fp.value))
                ) {
                    strcpy(peer->event, fp.value);
                } else {
                    fprintf(
                        stderr,
                        "bt_peer_req2struct(): Got an invalid/unknown "
                        "event of \"%s\"\n", fp.value
                    );
                    fflush(stderr);
                    return 0;
                }
            }
        } else if(!strcmp("shield", fp.key)) {
            /* special libbtt extension; send "shield=1" to shield core seeds
             * when there's other seeders available
             */
            if(fp.value_len == 1 && fp.value[0] == '1')
                peer->flags = peer->flags | BTT_PEER_SHIELD;
        } else if(!strcmp("compact", fp.key)) {
            if(fp.value_len == 1 && fp.value[0] == '1')
                peer->flags = peer->flags | BTT_PEER_COMPACT;
        } else if(!strcmp("no_peer_id", fp.key)) {
            if(fp.value_len == 1 && fp.value[0] == '1')
                peer->flags = peer->flags | BTT_PEER_NOPEERID;
        }
    }
 
    if(!gotpid) {
        fprintf(stderr, "bt_peer_req2struct(): Did not get a peer_id.\n");
        fflush(stderr);
        return 0;
    }
 
    if(!gothash) {
        fprintf(stderr, "bt_peer_req2struct(): Did not get an infohash.\n");
        fflush(stderr);
        return 0;
    }
 
    if(!peer->address.sin_port) {
        fprintf(stderr, "bt_peer_req2struct(): Did not get a port.\n");
        fflush(stderr);
        return 0;
    }

    peer->first_t = now;
    peer->last_t = now;
 
    return 1;
}

static char* btt_peer_ip(apr_pool_t* p, btt_peer* peer) {
    char* ip;
    if(!(ip = inet_ntoa(peer->address.sin_addr)))
        return NULL;
    return apr_pstrdup(p, ip);
}

/* sets "result" to the string, returns length of the string */
int btt_peer2compact(apr_pool_t* p, btt_peer* peer, char** result) {
    char* rv = NULL;
    uint32_t ip = peer->address.sin_addr.s_addr;

    if(ip) {
        uint16_t port = peer->address.sin_port;
        rv = apr_palloc(p, sizeof(ip) + sizeof(port));
        memcpy(rv, &ip, sizeof(ip) + sizeof(port));
        memcpy(&(rv[sizeof(ip)]), &port, sizeof(port));
        *result = rv;
        return sizeof(ip) + sizeof(port);
    } else {
        return 0;
    }
}

/* sets "result" to the string, returns length of the string */
apr_size_t btt_peer2nopeerid(apr_pool_t* p, btt_peer* peer, char **result) {
    char *rv = NULL;
    char *ip;
    apr_size_t ip_len;
    apr_size_t len;
    
    if(!(ip = btt_peer_ip(p, peer)))
        return 0;

    ip_len = (apr_size_t) strlen(ip);
    
    rv = apr_psprintf(
        p, "d2:ip%"APR_SIZE_T_FMT":%s4:porti%uee", ip_len, ip,
        ntohs(peer->address.sin_port)
    );
    
    len = (apr_size_t) strlen(rv);
    *result = rv;
    return len;
}

/* sets "result" to the string, returns length of the string */
apr_size_t btt_peer2bencode(apr_pool_t* p, btt_peer* peer, char **result) {
    char *rv_s = NULL;
    char *ip;
    char ip_len_s[20];
    size_t ip_len_s_len;
    apr_size_t rv;
    apr_size_t ip_len;

    if(!(ip = btt_peer_ip(p, peer)))
        return 0;

    ip_len = (apr_size_t)strlen(ip);

    sprintf(ip_len_s, "%"APR_SIZE_T_FMT, ip_len);
    ip_len_s_len = strlen(ip_len_s);
    
    rv_s = apr_psprintf(
        p,
        "d2:ip%s:%s7:peer id20:xxxxxxxxxxxxxxxxxxxx4:porti%uee",
        ip_len_s, ip, ntohs(peer->address.sin_port)
    );
    
    rv = (apr_size_t)strlen(rv_s);
    memcpy(rv_s + 17 + ip_len_s_len + ip_len, peer->peerid, BT_PEERID_LEN);

    *result = rv_s;
    return rv;
}

int btt_peer2tr(apr_pool_t*p, btt_peer* peer, char** result) {
    const char* tr_class;
    char* rv;
    /* even though the flags array is a constant, 
     * gcc can't figure out that flags will always be initialized in this
     * function so I gotta set it here */

    char* flags = NULL; 
    int f;
    time_t now = time(NULL);
 
    if(peer->uploaded) {
        if(peer->left)
            tr_class = "peer";
        else if(peer->flags & BTT_PEER_SHIELD)
            tr_class = "shield";
        else
            tr_class = "seed";
    } else
        tr_class = "leech";
  
    if(peer->flags) {
        /* if theres only one flag we don't have to allocate an extra string */
        for(f=0;btt_peer_flags[f].flag;f++) {
            if(peer->flags & btt_peer_flags[f].flag) {
                if(flags)
                    flags = apr_pstrcat(
                        p, flags, ", ", btt_peer_flags[f].config_name, NULL
                    );
                else
                    flags = btt_peer_flags[f].config_name;
            }
        }
    } else {
        flags = "";
    }

    /* tr_class, peerid, ua, uploaded, downloaded, left,
     * startup time, last announce, announce bytes, last serve, flags
     */
    rv = apr_psprintf(
        p,
        "<TR CLASS=\"%s\"><TD><CODE>%s</CODE></TD><TD NOWRAP>%s</TD>"
        "<TD CLASS=\"numeric\">%s</TD><TD CLASS=\"numeric\">%s</TD>"
        "<TD CLASS=\"numeric\">%s</TD>"
        "<TD CLASS=\"numeric\">"BT_TIME_T_FMT"s</TD>"
        "<TD CLASS=\"numeric\">"BT_TIME_T_FMT"s</TD>"
        "<TD CLASS=\"numeric\">%s</TD>"
        "<TD CLASS=\"numeric\">"BT_TIME_T_FMT"s</TD>"
        "<TD><CODE>%s</CODE></TD></TR>\n",
        tr_class, bt_str_peerid(p, peer->peerid), peer->ua,
        bt_nice_size(p, peer->uploaded), bt_nice_size(p, peer->downloaded),
        bt_nice_size(p, peer->left), now - peer->first_t, now - peer->last_t,
        bt_nice_size(p, peer->announce_bytes),
        now - peer->last_serve_t, flags
    );
 
    *result = rv;
    return strlen(rv);
}
