/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Copyright (C) 2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Jacob Berkman         <jacob@ximian.com>
 *          Hans Petter Jansson   <hpj@ximian.com>
 *          Carlos Garnacho Parro <carlosg@gnome.org>
 */

#include <config.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>

#ifdef ENABLE_GNOME
#include <libgnomeui/libgnomeui.h>
#include <libgnome/gnome-program.h>
#include <libgnome/gnome-help.h>
#else
#include <stdlib.h>
#endif

#include <string.h>
#include "gst-tool.h"
#include "gst-dialog.h"

enum {
	PLATFORM_LIST_COL_LOGO,
	PLATFORM_LIST_COL_NAME,
	PLATFORM_LIST_COL_ID,
	PLATFORM_LIST_COL_LAST
};

static void  gst_tool_class_init   (GstToolClass *class);
static void  gst_tool_init         (GstTool      *tool);
static void  gst_tool_finalize     (GObject      *object);

static GObject* gst_tool_constructor (GType                  type,
				      guint                  n_construct_properties,
				      GObjectConstructParam *construct_params);
static void  gst_tool_set_property (GObject      *object,
				    guint         prop_id,
				    const GValue *value,
				    GParamSpec   *pspec);

static void gst_tool_impl_close    (GstTool *tool);

enum {
	PROP_0,
	PROP_NAME,
	PROP_TITLE,
	PROP_ICON
};

G_DEFINE_ABSTRACT_TYPE (GstTool, gst_tool, G_TYPE_OBJECT);

static void
gst_tool_class_init (GstToolClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);

	object_class->set_property = gst_tool_set_property;
	object_class->constructor  = gst_tool_constructor;
	object_class->finalize     = gst_tool_finalize;

	class->close = gst_tool_impl_close;
	class->update_gui = NULL;
	class->update_config = NULL;

	g_object_class_install_property (object_class,
					 PROP_NAME,
					 g_param_spec_string ("name",
							      "name",
							      "Tool name",
							      NULL,
							      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (object_class,
					 PROP_TITLE,
					 g_param_spec_string ("title",
							      "title",
							      "Tool title",
							      NULL,
							      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (object_class,
					 PROP_ICON,
					 g_param_spec_string ("icon",
							      "icon",
							      "Tool icon",
							      NULL,
							      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

static gboolean
report_window_close_cb (GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gtk_widget_hide (widget);
	return TRUE;
}

static GladeXML*
gst_tool_load_glade_common (GstTool *tool, const gchar *widget)
{
	GladeXML *xml;
	
	g_return_val_if_fail (tool != NULL, NULL);
	g_return_val_if_fail (GST_IS_TOOL (tool), NULL);
	g_return_val_if_fail (tool->glade_common_path != NULL, NULL);

	xml = glade_xml_new (tool->glade_common_path, widget, NULL);

	if (!xml) {
		g_error ("Could not load %s\n", tool->glade_common_path);
	}

	return xml;
}

DBusGProxy *
activate_system_daemon ()
{
    GError *error = NULL;
    DBusGProxy *proxy;
    DBusGConnection *connection;

    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
    if (connection == NULL)
    {
	g_warning("Failed to get dbus connection\n");
	g_error_free (error);
	exit(-1);
    }
    proxy = dbus_g_proxy_new_for_name (connection,
	"org.moblin.SystemDaemon",
	"/org/moblin/SystemDaemon",
	"org.moblin.SystemDaemon");

    if (proxy == NULL)
    {
	g_warning("Failed to establish DBUS connection\n");
	exit(-1);
    }
    return proxy;
}

static void
gst_tool_init (GstTool *tool)
{
	GdkPixbuf *pixbuf;
	GladeXML  *xml;

	tool->icon_theme = gtk_icon_theme_get_default ();
	tool->glade_common_path  = MA_GLADE_DIR "/common.glade";
	tool->proxy = activate_system_daemon ();
	tool->gconf_client = gconf_client_get_default ();

	xml = gst_tool_load_glade_common (tool, "report_window");

	tool->report_window = glade_xml_get_widget (xml, "report_window");
	tool->report_label = glade_xml_get_widget (xml, "report_label");
	tool->report_progress = glade_xml_get_widget (xml, "report_progress");
	tool->report_pixmap = glade_xml_get_widget (xml, "report_pixmap");
	g_signal_connect (G_OBJECT (tool->report_window), "delete_event",
			  G_CALLBACK (report_window_close_cb), tool);

	pixbuf = gtk_icon_theme_load_icon (tool->icon_theme, "gnome-system-config", 48, 0, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (tool->report_pixmap), pixbuf);

	if (pixbuf)
		gdk_pixbuf_unref (pixbuf);

	g_object_unref (xml);
}

static GObject*
gst_tool_constructor (GType                  type,
		      guint                  n_construct_properties,
		      GObjectConstructParam *construct_params)
{
	GObject *object;
	GstTool *tool;
	gchar *widget_name;
	GError *error;

	object = (* G_OBJECT_CLASS (gst_tool_parent_class)->constructor) (type,
									  n_construct_properties,
									  construct_params);
	tool = GST_TOOL (object);

	if (tool->name) {
		tool->glade_path = g_strdup_printf (MA_GLADE_DIR "/%s.glade", tool->name);

		widget_name = g_strdup_printf ("%s_admin", tool->name);
		tool->main_dialog = gst_dialog_new (tool, widget_name, tool->title);
		g_free (widget_name);
	}
	return object;
}

static void
gst_tool_set_property (GObject      *object,
		       guint         prop_id,
		       const GValue *value,
		       GParamSpec   *pspec)
{
	GstTool *tool = GST_TOOL (object);

	switch (prop_id) {
	case PROP_NAME:
		tool->name = g_value_dup_string (value);
		break;
	case PROP_TITLE:
		tool->title = g_value_dup_string (value);
		break;
	case PROP_ICON:
		tool->icon = g_value_dup_string (value);
		break;
	}
}

static void
gst_tool_finalize (GObject *object)
{
#if 0
	GstTool *tool = GST_TOOL (object);

	g_free (tool->name);
	tool->name = NULL;
	g_free (tool->title);
	tool->title = NULL;
	g_free (tool->icon);
	tool->icon = NULL;
	g_free (tool->glade_path);
	tool->glade_path = NULL;
	if (tool->proxy)
	{
	    g_object_unref (tool->proxy);
	    tool->proxy = NULL;
	}
	if (tool->main_dialog)
		gtk_widget_destroy (GTK_WIDGET (tool->main_dialog));
	tool->main_dialog = NULL;

	if (tool->report_window)
		gtk_widget_destroy (tool->report_window);
	tool->report_window = NULL;

	if (tool->gconf_client)
		g_object_unref (tool->gconf_client);
	tool->gconf_client = NULL;
#endif

	(* G_OBJECT_CLASS (gst_tool_parent_class)->finalize) (object);
}

static void
gst_tool_impl_close (GstTool *tool)
{
	gtk_widget_hide (GTK_WIDGET (tool->main_dialog));

	/* process necessary events to hide the dialog */
	while (gtk_events_pending ())
		gtk_main_iteration ();

	/* process pending async requests */
/*	g_object_unref (tool);*/
	gtk_main_quit ();
}

void
gst_tool_update_gui (GstTool *tool)
{
	g_return_if_fail (GST_IS_TOOL (tool));

	if (GST_TOOL_GET_CLASS (tool)->update_gui)
		(* GST_TOOL_GET_CLASS (tool)->update_gui) (tool);
}

void
gst_tool_update_config (GstTool *tool)
{
	g_return_if_fail (GST_IS_TOOL (tool));

	if (GST_TOOL_GET_CLASS (tool)->update_config)
		(* GST_TOOL_GET_CLASS (tool)->update_config) (tool);
}

void
gst_tool_close (GstTool *tool)
{
GError *error = NULL;

	g_return_if_fail (GST_IS_TOOL (tool));
	if (GST_TOOL_GET_CLASS (tool)->close)
		(* GST_TOOL_GET_CLASS (tool)->close) (tool);
}

gboolean
gst_tool_is_authenticated (GstTool *tool)
{
	/* FIXME */
	return TRUE;
}

void
gst_init_tool (const gchar *app_name, int argc, char *argv [], GOptionEntry *entries)
{
	GOptionContext *context;

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	if (entries) {
		context = g_option_context_new (NULL);
		g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
		g_option_context_add_group (context, gtk_get_option_group (TRUE));
		g_option_context_parse (context, &argc, &argv, NULL);
		g_option_context_free (context);
	}

	gtk_init (&argc, &argv);	
}
