/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//*****************************************************************************
//
// Class:			IPCBuffer
//
// Description:		IPCBuffer is a class that provides a section of named shared 
//                  memory, to be shared between processes. The bIsOwner 
//                  parameter of the ctor determines whether or not the object is
//                  the one that creates the shared memory (server), or just uses
//                  it (client) after it was created. Use isOk() and lastError() 
//                  after construction to check for errors.
//                  The object may also me secure, in which case only the owner of the
//                  buffer can change it. To make the object secure, set the 
//                  bSecure parameter of the ctor to true.
//*****************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "IPCBuffer.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <memory.h>
#include <errno.h>

#define FAIL -1

const unsigned long IPCBuffer::IPC_BUF_ACCESS_READ = 1;
const unsigned long IPCBuffer::IPC_BUF_ACCESS_WRITE = 2;
const char IPCBuffer::PROJ = '#'; //Pre-defined value for proj parameter of ftok()
const unsigned int IPCBuffer::SUCCESS = 0;

//default ctor: must use init() afterwards
IPCBuffer::IPCBuffer()
{
	reset();
}

void IPCBuffer::reset()
{
	m_pName = NULL;
	m_bIsOwner = false;
	m_pData = NULL;
	m_nSize = 0;
	m_nAccess = 0;
	m_shmid = FAIL;
	m_nLastError = SUCCESS;
	m_shmget_flag = 0;
	m_shmat_flag = 0;
	m_key = 0;
	memcpy(m_marker, "INTELAMT_________________________________", SHM_MARKER_RESERVED_BYTES);
}

/*
ctor: initialize this IPCBuffer.
arguments:
    pName -    Pointer to name of shared memory, to be used by all processes that
               access it. The pointed name must be available until the object is
               destroyed.
    bIsOwner - determines whether this object should try to create the shared
               memory, or just open it when it already exists.
    nSize    - required buffer size to allocate.
    bSecure  - Should this buffer be secure, meaning that the OS will enforce
               that only the owner can change it.
    nAccess  - required access to shared memory. Can be IPC_BUF_ACCESS_READ,
               IPC_BUF_ACCESS_WRITE, or a combination of both with bitwise OR.
*/
IPCBuffer::IPCBuffer(const char* pName, bool bIsOwner, unsigned int nSize, bool bSecure,
		     unsigned long nAccess /*= IPC_BUF_ACCESS_READ | IPC_BUF_ACCESS_WRITE*/)
{
	init(pName, bIsOwner, nSize, bSecure, nAccess);
}

//initialize this buffer after it was created.
bool IPCBuffer::init(const char* pName, bool bIsOwner, unsigned int nSize, bool bSecure,
		  unsigned long nAccess /*= IPC_BUF_ACCESS_READ | IPC_BUF_ACCESS_WRITE*/)
{
	char *pmp;
	int i;
	int open_flag = 0;
	int open_mode = 0;

	close();

	reset();

	m_pName = pName;
	m_bIsOwner = bIsOwner;
	m_nSize = nSize;
	m_nAccess = nAccess;

	if (m_bIsOwner) {
		//Create file (to be used for the shared memory NAME ONLY).
		open_flag = O_CREAT | O_TRUNC | O_WRONLY;

		//give read+write permission to owner, read only to group and others.
		open_mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;

		//Create the shared memory, give read+write permission to owner, read only
		//to group and others.
		m_shmget_flag = IPC_CREAT | IPC_EXCL | open_mode;

		if (!bSecure) {
			//If the buffer is not secure, group and others can write to it
			m_shmget_flag |= (S_IWGRP | S_IWOTH);
		}
	} else {
		open_flag = O_RDONLY; //Open existing file
		if (nAccess == IPC_BUF_ACCESS_READ) {
			m_shmat_flag = SHM_RDONLY; //Read only access.
		}
	}

	int fd = open(m_pName, open_flag, open_mode);
	if (fd == FAIL) {
		m_nLastError = errno;
		return false;
	}
	::close(fd);

	pmp = strrchr(m_pName, '/');
	if (NULL == pmp) {
		pmp = (char *)m_pName;
	}
	for (i = 0; ((pmp) && (pmp[i]) && (i < SHM_MARKER_RESERVED_BYTES)); i++) {
		m_marker[i] = pmp[i];
	}

	m_key = ftok(m_pName, PROJ);
	if (m_key == FAIL) {
		m_nLastError = errno;
		return false;
	}

	return shmConnect();
}

//connect to shared memory
bool IPCBuffer::shmConnect()
{
	if (m_pData) {
		//already connected
		return true;
	}

	m_shmid = shmget(m_key, m_nSize, m_shmget_flag);
	if (m_shmid == FAIL) {
		m_nLastError = errno;
		if ((m_bIsOwner) && (EEXIST == m_nLastError)) {
			m_shmid = shmget(m_key, m_nSize, m_shmget_flag & ~(IPC_CREAT | IPC_EXCL));
			if (m_shmid == FAIL) {
				return false;
			}

			m_pData = (char*)shmat(m_shmid, NULL, m_shmat_flag);
			if (m_pData == (char*)FAIL) {
				m_pData = NULL;
				m_shmid = FAIL;
				return false;
			}

			if (0 != memcmp(m_pData, m_marker, SHM_MARKER_RESERVED_BYTES)) {
				shmdt(m_pData);
				m_pData = NULL;
				m_shmid = FAIL;
				return false;
			}
			
			m_nLastError = 0;
		} else {
			return false;
		}
	} else {
		m_pData = (char*)shmat(m_shmid, NULL, m_shmat_flag);
		if (m_pData == (char*)FAIL) {
			m_pData = NULL;
			m_nLastError = errno;
			return false;
		}

		if (m_bIsOwner) {
			memcpy(m_pData, m_marker, SHM_MARKER_RESERVED_BYTES);
		} else if (0 != memcmp(m_pData, m_marker, SHM_MARKER_RESERVED_BYTES)) {
			shmdt(m_pData);
			m_pData = NULL;
			m_shmid = FAIL;
			m_nLastError = ENOENT;
			return false;
		}
	}

	return true;
}

//disconnect from shared memory
void IPCBuffer::shmDisconnect()
{
	if (m_pData) {
		//Detach shared memory
		shmdt(m_pData);
		m_pData = NULL;
	}
}

//free resources allocated by IPC Buffer
void IPCBuffer::close()
{
	shmDisconnect();

	if (m_bIsOwner)
	{
		if (m_shmid != FAIL)
		{
			//Owner must remove the shared memory when done with it.
			shmctl(m_shmid, IPC_RMID, NULL);
			m_shmid = FAIL;
		}

		remove(m_pName);
	}
}

//dtor
IPCBuffer::~IPCBuffer()
{
	close();
}

/*
Safely read data from the buffer.
Arguments:
	nOffset - offset (in bytes) in IPCBuffer to start reading from
	pDest - destination buffer that will hold the data we read
	nSize - number of bytes to read
Return value:
	true on success, false on failure.
*/
bool IPCBuffer::read(unsigned int nOffset, void *pDest, unsigned int nSize) const
{
	if (! (m_nAccess & IPC_BUF_ACCESS_READ)) {
		return false; //read not allowed
	}

	if (!isOk() || (m_pData == NULL)) {
		return false;
	}

	if ((nOffset + SHM_MARKER_RESERVED_BYTES + nSize > m_nSize)
	    || (nOffset + SHM_MARKER_RESERVED_BYTES + nSize < nSize)
	    || (nOffset + SHM_MARKER_RESERVED_BYTES + nSize < nOffset)
	    || (pDest == NULL))
		return false;

	memcpy(pDest, m_pData + SHM_MARKER_RESERVED_BYTES + nOffset, nSize);
	return true;
}

/*
Safely write data to the buffer.
Arguments:
	nOffset - offset (in bytes) in IPCBuffer to start writing to.
	pSource - Source buffer that holds the data to write.
	nSize - number of bytes to write.
Return value:
	true on success, false on failure.
*/
bool IPCBuffer::write(unsigned int nOffset, const void *pSource, unsigned int nSize)
{
	if (! (m_nAccess & IPC_BUF_ACCESS_WRITE)) {
		return false; //write not allowed
	}

	if (!isOk() || (m_pData == NULL)) {
		return false;
	}

	if ((nOffset + SHM_MARKER_RESERVED_BYTES + nSize > m_nSize)
	    || (nOffset + SHM_MARKER_RESERVED_BYTES + nSize < nSize)
	    || (nOffset + SHM_MARKER_RESERVED_BYTES + nSize < nOffset)
	    || (pSource == NULL)) {
		return false;
	}

	memcpy(m_pData + SHM_MARKER_RESERVED_BYTES + nOffset, pSource, nSize);
	return true;
}
