/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//*****************************************************************************
//
// Class:			IPCAMTStatus
//
// Description:	    IPCAMTStatus is an object that holds the status of AMT, and
//                  this status can be shared across processes. When creating the
//                  object, you must specify whether you are the server or client
//                  of this status - being the server means you create the status
//                  and only write to it, and being a client means you connect
//                  to an existing status and only read from it.
//                  After initialization, you should use isOk() and lastError()
//                  to check the error status of the object.
//*****************************************************************************
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "IPCAMTStatus.h"

#ifdef _WIN32
	const char* IPCAMTStatus::BUF_NAME = "Global\\AMTStatus";
#else
	const char* IPCAMTStatus::BUF_NAME = "/usr/share/AMTStatus";
#endif

const unsigned int IPCAMTStatus::BUF_SIZE = 512;
const unsigned long IPCAMTStatus::TYPE_ENABLEMENT_ONLY = 0;


//default ctor: must use init() afterwards
IPCAMTStatus::IPCAMTStatus()
{
	m_bWriteOk = true;
	m_bServer = false;
#ifndef _WIN32
	m_shmReconnect = false;
#endif
}

/*
ctor: initialize this object.
Arguments:
	bServer - specify whether this object should create the status, or just
	          try to open an existing one.*/
IPCAMTStatus::IPCAMTStatus(bool bServer)
{
	init(bServer);
}

bool IPCAMTStatus::init(bool bServer)
{
	m_bServer = bServer;

	//Initializing the IPC buffer
	if (!shmInit()) {
		return false;
	}

	m_bWriteOk = true;
	if (m_bServer) {
		//If we are the server, we need to prepare the status struct in the shared
		//memory.
		Status status;
		status.header.dwType = TYPE_ENABLEMENT_ONLY;
		status.header.dwBodyLength = sizeof(status.body.enablementOnly);
		status.body.enablementOnly.enablement = STATUS_UNKNOWN;
		m_bWriteOk = writeStatus(status);
		if (!m_bWriteOk) {
			return false;
		}
#ifndef _WIN32
		if (m_shmReconnect) {
			//reconnect is for clients only
			m_shmReconnect = false;
		}
#endif
	}

#ifndef _WIN32
	if (m_shmReconnect) {
		m_IPCBuffer.shmDisconnect();
	}
#endif

	return true;
}

//Initialize the IPC buffer
bool IPCAMTStatus::shmInit()
{
	unsigned long nAccessType = IPCBuffer::IPC_BUF_ACCESS_READ;
	//Server can read/write, client can only read.
	if (m_bServer) {
		nAccessType |= IPCBuffer::IPC_BUF_ACCESS_WRITE;
	}

	//Initializing the IPC buffer
	if (!m_IPCBuffer.init(BUF_NAME, m_bServer, BUF_SIZE, true, nAccessType)) {
		return false;
	}

	return true;
}

void IPCAMTStatus::close()
{
	if (m_bServer) {
		_SetEnablement(STATUS_UNKNOWN);
	}
	m_IPCBuffer.close();
}

/*return the error code of the last error*/
unsigned long IPCAMTStatus::lastError() const
{
	return m_IPCBuffer.lastError();
}

/*returns true if this object is ok (no errors).*/
bool IPCAMTStatus::isOk() const
{
	return m_IPCBuffer.isOk() && m_bWriteOk;
}

#ifndef _WIN32
/*set option - reconnect for each shared memory access */
void IPCAMTStatus::setShmReconnect(bool reconnect)
{
	if (!m_bServer) {
		m_shmReconnect = reconnect;
		if (!m_shmReconnect) {
			m_IPCBuffer.shmConnect();
		}
	}
}
#endif

/*
Set the status of this object.
Arguments:
	status - the status to set
Return value:
	true on success, false on failure.*/
bool IPCAMTStatus::SetEnablement(Enablement enablement)
{
#ifndef _WIN32
	if (m_shmReconnect) {
		if (!m_IPCBuffer.shmConnect()) {
			if (!shmInit()) {
				return false;
			}
		}
	}
#endif

	bool result = _SetEnablement(enablement);

#ifndef _WIN32
	if (m_shmReconnect) {
		m_IPCBuffer.shmDisconnect();
	}
#endif

	return result;
}

/*
Set the status of this object.
Arguments:
	status - the status to set
Return value:
	true on success, false on failure.*/
bool IPCAMTStatus::_SetEnablement(Enablement enablement)
{
	Status status;

	if (!readStatus(status)) {
		return false;
	}

	switch (status.header.dwType)
	{
		case TYPE_ENABLEMENT_ONLY:
			status.body.enablementOnly.enablement = enablement;
			break;
		default: //unknown type
			return false;
	}
	if (!writeStatus(status)) {
		return false;
	}

	return true;
}

/*
Get the status of this object.
Return value:
	STATUS_DISABLED or STATUS_ENABLED on success, STATUS_UNKNOWN on failure.
*/
IPCAMTStatus::Enablement IPCAMTStatus::GetEnablement()
{
#ifndef _WIN32
	if (m_shmReconnect) {
		if (!m_IPCBuffer.shmConnect()) {
			if (!shmInit()) {
				return STATUS_UNKNOWN;
			}
		}
	}
#endif

	Enablement result = _GetEnablement();

#ifndef _WIN32
	if (m_shmReconnect) {
		m_IPCBuffer.shmDisconnect();
	}
#endif

	return result;
}

/*
Get the status of this object.
Return value:
	STATUS_DISABLED or STATUS_ENABLED on success, STATUS_UNKNOWN on failure.
*/
IPCAMTStatus::Enablement IPCAMTStatus::_GetEnablement() const
{
	Status status;

	if (!readStatus(status)) {
		return STATUS_UNKNOWN;
	}

	switch (status.header.dwType) {
		case TYPE_ENABLEMENT_ONLY:
			return status.body.enablementOnly.enablement;
		default: //unknown type
			return STATUS_UNKNOWN;
	}
}

bool IPCAMTStatus::writeStatus(const Status& status)
{
	unsigned int nOffset = 0;

	//write header
	if (!m_IPCBuffer.write(nOffset, &status.header, sizeof(status.header))) {
		return false;
	}
	nOffset += sizeof(status.header);
	//write body
	if (!m_IPCBuffer.write(nOffset, &status.body, status.header.dwBodyLength)) {
		return false;
	}

	return true;
}

bool IPCAMTStatus::readStatus(Status& status) const
{
	unsigned int nOffset = 0;

	//read header
	if (!m_IPCBuffer.read(nOffset, &status.header, sizeof(status.header))) {
		return false;
	}
	nOffset += sizeof(status.header);
	//read body
	if (!m_IPCBuffer.read(nOffset, &status.body, status.header.dwBodyLength)) {
		return false;
	}

	return true;
}
