/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#include "AMTStatusStateMachine.h"
#include "IMEGuiNotifications.h"
#include <assert.h>

//*******************************************************************************
//
// Class:	    AMTStatusStateMachine
//
// Description:	    This class handles all the logic of the MEStatus application,
//                  and it communicates with the outside world through two interfaces:
//                  IMEGuiCommands, which it inherits from, and IMEGuiCommands, which
//                  it points to. The idea is for the user of this class to treat it 
//                  as an IMEGuiCommands object. The user would pass it a pointer to an
//                  IMEGuiNotifications object which would receive notifications that 
//                  are caused by activating commands from IMEGuiCommands.
//
//********************************************************************************

AMTStatusStateMachine::AMTStatusStateMachine()
{
	m_State = STATE_INITIAL_UNKNOWN;
	m_nSetEnablementCount = 0;
	m_pMEGuiNotifications = NULL;
}

AMTStatusStateMachine::~AMTStatusStateMachine()
{
}

bool AMTStatusStateMachine::Init(IMEGuiNotifications *pMEGuiNotifications)
{
	m_Config.Init();
	m_nSetEnablementCount = 0;
	if (pMEGuiNotifications == NULL)
  	        return false;
	m_pMEGuiNotifications = pMEGuiNotifications;
	return true;
}

void AMTStatusStateMachine::SetEnablement(IPCAMTStatus::Enablement enablement)
{
	//Save enablement to configuration for next time we run.
	m_Config.SetAMTLastEnabled(enablement == IPCAMTStatus::STATUS_ENABLED);

	//In any state, we always notify the GUI of the enablement change.
	m_pMEGuiNotifications->OnAMTChangeEnablement(enablement);

	//Handle the new enablement according to our current state:
	switch (m_State)
	{
		case STATE_INITIAL_UNKNOWN:
			m_State = SetEnablementOnInitialUnknownState(enablement);
			break;
		case STATE_ENABLED:
			m_State = SetEnablementOnEnabledState(enablement);
			break;
		case STATE_DISABLED:
			m_State = SetEnablementOnDisabledState(enablement);
			break;
		case STATE_UNKNOWN:
			m_State = SetEnablementOnUnknownState(enablement);
			break;
	}
}

void AMTStatusStateMachine::SetPopupForCurrentUser(bool bPopupForCurrentUser)
{
	m_Config.SetPopupForCurrentUser(bPopupForCurrentUser);
}

bool AMTStatusStateMachine::GetPopupForCurrentUser() const
{
	return m_Config.PopupForCurrentUser();
}

std::string AMTStatusStateMachine::GetOEMUrl() const
{
	return m_Config.GetOemUrl();
}

std::string AMTStatusStateMachine::GetDefaultOEMUrl() const
{
	return m_Config.GetDefaultOemUrl();
}

unsigned long AMTStatusStateMachine::GetCheckAMTStatusTimeout() const
{
        return m_Config.CheckAMTStatusTimeout();
}


AMTStatusStateMachine::State AMTStatusStateMachine::GetStateByEnablement(IPCAMTStatus::Enablement enablement)
{
	switch (enablement)
	{
		case IPCAMTStatus::STATUS_ENABLED:
			return STATE_ENABLED;

		case IPCAMTStatus::STATUS_DISABLED:
			return STATE_DISABLED;

		case IPCAMTStatus::STATUS_UNKNOWN:
		default:
			return STATE_UNKNOWN;
	}
}

AMTStatusStateMachine::State AMTStatusStateMachine::SetEnablementOnInitialUnknownState(IPCAMTStatus::Enablement enablement)
{
	switch (enablement)
	{
		case IPCAMTStatus::STATUS_ENABLED:
			if (m_Config.PopupForCurrentUser() || (!m_Config.AMTLastEnabled()))
			{
				/*If we are configured to popup for current user, OR last time we ran,
				  AMT was disabled, we show the popup.*/
				m_pMEGuiNotifications->OnAMTEnablementPopup(IPCAMTStatus::STATUS_ENABLED);
			}
			return STATE_ENABLED;
		
		case IPCAMTStatus::STATUS_DISABLED:
			if (m_Config.PopupForCurrentUser())
				m_pMEGuiNotifications->OnAMTEnablementPopup(IPCAMTStatus::STATUS_DISABLED);
			return STATE_DISABLED;
		
		default:
		case IPCAMTStatus::STATUS_UNKNOWN:
		        if (++m_nSetEnablementCount == m_Config.AMTStatusInitMaxRetries())
			{
			        /*We were set to UNKNOWN status too many times - we already notified the 
				  GUI that the status is unknown, and now we tell the GUI to popup with 
				  this message.*/
				m_pMEGuiNotifications->OnAMTEnablementPopup(IPCAMTStatus::STATUS_UNKNOWN);

				//We change to UNKNOWN state so that this message will not popup again.
				return STATE_UNKNOWN; 
			}
			return STATE_INITIAL_UNKNOWN;
	}
}

AMTStatusStateMachine::State AMTStatusStateMachine::SetEnablementOnEnabledState(IPCAMTStatus::Enablement enablement)
{
	return GetStateByEnablement(enablement);
}

AMTStatusStateMachine::State AMTStatusStateMachine::SetEnablementOnDisabledState(IPCAMTStatus::Enablement enablement)
{
	switch (enablement)
	{
		case IPCAMTStatus::STATUS_ENABLED:
			//We always popup when moving from disabled to enabled.
		        m_pMEGuiNotifications->OnAMTEnablementPopup(IPCAMTStatus::STATUS_ENABLED);
			return STATE_ENABLED;
		default:
			return GetStateByEnablement(enablement);
	}
}

AMTStatusStateMachine::State AMTStatusStateMachine::SetEnablementOnUnknownState(IPCAMTStatus::Enablement enablement)
{
	return GetStateByEnablement(enablement);
}
