#!/usr/bin/env python
#-*- coding: utf-8 -*-
# memaker.py - A gtk based avatar creation program.
# Copyright 2007 Jason Brower encompass@gmail.com, Christopher Denter (motor AT the-space-station DOT com), Aaron Spike
# Copyright 2007 Matthew Brennan Jones <mattjones@workhorsy.org>
# Copyright 2007 Og Maciel
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. See the file LICENSE.
# If not, see <http://www.gnu.org/licenses/>.

import sys , os, fileinput, shutil, rsvg
from motor import *
import Image, ImageChops
import pygtk
import gtk
import gtk.glade
import cairo
import StringIO
import pickle

class ImageLoader():
    def __init__(self, width, height, bgcolor, scale_method = gtk.gdk.INTERP_BILINEAR):
        self._width = width
        self._height = height
        self._bgcolor = bgcolor
        self._scale_method = scale_method
        self._cache_dir = os.path.abspath(os.path.expanduser('~/.MeMaker/cache/')) + '/'
        self._cache_file = self._cache_dir + 'thumbnails.cache'
        
        self.open()
    
    def get_unique_file_part_path(self, file_name):
        file_name = os.path.abspath(file_name)
        return file_name[len(self._cache_dir)+1: len(file_name)] + ".png"
    
    def get_thumb_nail_path(self, file_name):
        file_name = os.path.abspath(file_name)
        unique_file_path = self.get_unique_file_part_path(file_name)
        thumb_nail_file_name = self._cache_dir + unique_file_path
        return thumb_nail_file_name
    
    def is_in_cache(self, file_name):
        file_name = os.path.abspath(file_name)
        thumb_nail_file_name = self.get_thumb_nail_path(file_name)
        
        is_cached = \
        self._cache.has_key(file_name) and \
        self._cache[file_name] == os.stat(file_name)[8] and \
        os.path.isfile(thumb_nail_file_name)
        
        return is_cached
    
    def cache_reason(self, file_name):
        file_name = os.path.abspath(file_name)
        thumb_nail_file_name = self.get_thumb_nail_path(file_name)
        
        if not self._cache.has_key(file_name): return "Not in hash "
        if not self._cache[file_name] == os.stat(file_name)[8]: return "Timestamp updated"
        if not os.path.isfile(thumb_nail_file_name): return "Thumbnail missing"
        
        return "None"
    
    def clear_cache(self):
        self._cache = {}
        
        for root, dirs, files in os.walk(self._cache_dir, topdown=False):
            for name in files:
                os.remove(os.path.join(root, name))
            for name in dirs:
                os.rmdir(os.path.join(root, name))
    
    def open(self):
        # Set the cache as an empty hash
        self._cache = {}
        
        # Just return if there is no cache file
        if not os.path.isfile(self._cache_file):
            return
        
        # Load the cache file, if there is an error, throw away the cache file
        pickle_file = None
        try:
            try:
                pickle_file = open(self._cache_file, 'r')
                self._cache = pickle.load(pickle_file)
            except Exception:
                if pickle_file != None: pickle_file.close()
                pickle_file = None
                os.remove(self._cache_file)
        finally:
            if pickle_file != None: pickle_file.close()
    
    def close(self):
        pickle_file = open(self._cache_file, 'w')
        pickle.dump(self._cache, pickle_file)
        pickle_file.close()
        self._cache = {}
    
    def load_cached_image(self, file_name):
        """
        Works just like load_image, but will cache images and return cached ones if the 
        original has not changed. Returns a Gdk Pixbuf
        """
        file_name = os.path.abspath(file_name)
        pixbuf = None

        # If the image is not cached, or the timestamp is outdated, or the thumbnail is missing, cache it
        if self.is_in_cache(file_name) == False:
            self._cache[file_name] = os.stat(file_name)[8]
            image = self.load_image(file_name)
            unique_file_path = file_name[len(self._cache_dir)+1: len(file_name)]
            
            dir_to_create = self._cache_dir
            path_fragments = unique_file_path.split('/')
            while len(path_fragments) > 1:
                dir_to_create += path_fragments.pop(0) + '/'
                if os.path.isdir(dir_to_create) == False:
                    os.mkdir(dir_to_create)
            
            thumb_nail_file_name = self._cache_dir + unique_file_path
            image.save(thumb_nail_file_name + ".png", "png")
        
        # Load the image from the cache here
        cached_file = self._cache_dir + file_name[len(self._cache_dir)+1: len(file_name)] + ".png"
        pixbuf = gtk.gdk.pixbuf_new_from_file(cached_file)
            
        return pixbuf
        
    def load_image(self, file_name):
        """
        Loads an image from file, crops and scales it, and returns it as a PIL Image
        """
        file_name = os.path.abspath(file_name)
        
        # Load the image and make sure it is atleast 5 
        # times as big as the crop area will be.
        pixbuf = gtk.gdk.pixbuf_new_from_file(file_name)
        w = pixbuf.get_width()
        h = pixbuf.get_height()

        # NOTE: Since we have no vector to determine just how mutch bigger to
        # make the image, just make sure that if it is less than 5 times bigger
        # than cropped image will be, reload it to be 5 times as big.
        if w < self._width * 5.0 or h < self._height * 5.0:
            pixbuf = gtk.gdk.pixbuf_new_from_file_at_size(file_name, self._width * 5, self._height * 5)
            w = pixbuf.get_width()
            h = pixbuf.get_height()

        # Convert the pixbuf into an PIL Image
        image = Image.frombuffer("RGBA", [w, h], pixbuf.get_pixels_array(), "raw", "RGBA", 0, 1)
        if image.mode != "RGBA": image = image.convert("RGBA")

        # Crop the image to trim off the empty space
        bbox = image.getbbox()
        if bbox: image = image.crop(bbox)

        # Calculate how much to move the image to be in the center
        image.thumbnail((self._width, self._height), Image.ANTIALIAS)
        width_expand = 0
        height_expand = 0
        if image.size[0] < self._width:
            width_expand = int((self._width - image.size[0]) / 2.0)
        if image.size[1] < self._height:
            height_expand = int((self._height - image.size[1]) / 2.0)

        # Expand the background of the image to be sqaure
        other = Image.new("RGBA", (self._width, self._height))
        other.paste(image, (width_expand, height_expand))
        image = other
        return image
        
        # FIXME: I would like to return a Pixbuf here, but when we convert the Image to Pixbuf, we loose alpha
        ## Convert the Image to io string
        #image_as_file = StringIO.StringIO()
        #image.save(image_as_file, "ppm")
        #raw_image = image_as_file.getvalue()
        #image_as_file.close()

        ## Convert the io string to a pixbuf and resize it
        #pixbuf_loader = gtk.gdk.PixbufLoader("pnm")
        #pixbuf_loader.write(raw_image, len(raw_image))
        #pixbuf = pixbuf_loader.get_pixbuf()
        #pixbuf_loader.close()

        #return pixbuf

