""" The resource plugin. """

# Standard library imports.
import logging
from sets import Set

# Enthought library imports.
from enthought.envisage import Plugin
from enthought.envisage.core.core_plugin import CorePlugin
from enthought.naming.api import Context

# Local imports.
from resource_manager import ResourceManager


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ResourcePlugin(Plugin):
    """ The resource plugin. """

    # Services that we create.
    IRESOURCE_MANAGER = 'enthought.envisage.resource.IResourceManager'

    #### 'ResourcePlugin' interface ###########################################

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin. """

        # Get a reference to the type manager service.
        type_manager = self.get_service(CorePlugin.ITYPE_MANAGER)

        # Create and register the resource manager as a service.
        resource_manager = self._create_resource_manager(type_manager)
        self.register_service(self.IRESOURCE_MANAGER, resource_manager)

        # Add any contributed resource views.
        self._add_resource_views(resource_manager)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_resource_manager(self, type_manager):
        """ Creates and populates the resource manager. """

        # Create a brand spanking new resource manager.
        resource_manager = ResourceManager(type_manager=type_manager)

        # Resource type extensions.
        logger.debug("Parsing resource type contributions to resource manager")
        for class_name in self._get_ordered_resource_classes():
            # Import the class that implements the resource type.
            logger.debug('  Adding resource type: %s', class_name)
            klass = self.import_symbol(class_name)

            # Create the resource type and add it to the resource manager.
            resource_type = klass(id=class_name)
            resource_manager.add_resource_type(resource_type)

        return resource_manager

    def _add_resource_views(self, resource_manager):
        """ Adds any contributed resource views. """

        # Plugin definition imports.
        from resource_plugin_definition import ResourceView

        # Get all resource view extensions.
        extensions = self.get_extensions(ResourceView)

        for extension in extensions:
            resource_type = resource_manager.lookup(extension.resource_type)
            if resource_type is not None:
                # fixme: Lazy load!
                klass = self.application.import_symbol(extension.class_name)
                resource_type.views[extension.id] = klass

        return

    def _get_ordered_resource_classes(self):
        """ Get all of the resource types in order.

        The list that is returned is a list of class names which satisfies
        all of the 'precedes' constraints of the ResourceTypes that are
        registered.

        """

        # Plugin definition imports.
        from resource_plugin_definition import ResourceManager

        # All of the resource types, unsorted.
        resource_types = []

        # Get all the resource types that we know about.
        extensions = self.get_extensions(ResourceManager)
        for extension in extensions:
            resource_types.extend(extension.resource_types)

        # Check for duplicate definitions.
        self._validate_resource_uniqueness( resource_types )

        # Order the list according to the 'precedes' clauses etc.
        deferred = []
        result = []
        while len(resource_types) > 0:
            for resource_type in resource_types:
                for dependant in resource_type.precedes:
                    # If we haven't already ordered a dependant, then we defer.
                    if not dependant in result:
                        deferred.append(resource_type)
                        break

                else:
                    # All dependants were present, put this one at the end.
                    result.append(resource_type.class_name)

            if len(resource_types) == len(deferred):
                # We were unable to clear any dependancies, so we have a cycle.
                resource_names = [(obj.class_name, obj.precedes) for obj in deferred]
                raise 'Cyclic or unresolvable dependancy in resource types: %s' % resource_names

            resource_types = deferred
            deferred = []

        result.reverse()
        return result

    def _validate_resource_uniqueness(self, resource_types):
        """ Raise KeyError if all resources in resource_type are not unique.
        """
        unique = {}
        for resource in resource_types:
            unique[resource.class_name] = \
                    unique.setdefault(resource.class_name, 0) + 1

        non_unique = [ key for key in unique.keys() if unique[key] > 1 ]

        if len(non_unique) > 0:
            raise KeyError, "Duplicate definition of ResourceType(s) %s" \
                            % non_unique

        return

#### EOF ######################################################################
