#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005-2007 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

"""
The plugin definition for the single project plugin.

"""

# Standard imports
import os

# Enthought library imports
from enthought.envisage import ExtensionPoint, PluginDefinition
from enthought.envisage.action.action_plugin_definition import \
    Action, ActionSet, Group, Location, Menu
from enthought.envisage.core.core_plugin_definition import Preferences, \
    Runnable
from enthought.envisage.resource.resource_plugin_definition import \
    ResourceManager, ResourceType
from enthought.envisage.workbench.workbench_plugin_definition import \
    Perspective, View, Workbench
from enthought.envisage.workbench.action.action_plugin_definition import \
    WorkbenchActionSet
from enthought.envisage.workbench.preference.preference_plugin_definition \
     import Page, PreferencePages

from enthought.traits.api import Int, Str


##############################################################################
# Constants
##############################################################################

# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = 'enthought.envisage.single_project'

# The ID used for actions, groups, and menus that apply to the project view
# control when nothing is selected.
NO_SELECTION_MENU_ID = '%s.no_selection_menu' % ID

# The ID used for actions, groups, and menus that apply to projects within
# the project view.
PROJECT_MENU_ID = '%s.project_menu' % ID

# This plugin's UI service referenced within our action contributions.
IPROJECT_UI = '%s.UiService' % ID


##############################################################################
# Extension points.
##############################################################################

class FactoryDefinition(ExtensionPoint):
    """
    A project factory definition.

    An instance of the specified class is used to open and/or create new
    projects.

    The extension with the highest priority wins!  In the event of a tie,
    the first instance wins.

    """

    # The name of the class that implements the factory.
    class_name = Str

    # The priority of this factory
    priority = Int


class SingleProjectActionSet(ActionSet):
    """
    Action and menu definitions for the project view.

    """

    # A mapping from human-readable root names to globally unique Ids.
    aliases = {
        'NoSelectionMenu' : NO_SELECTION_MENU_ID,
        'ProjectMenu' : PROJECT_MENU_ID,
        }


class SyncProjectSelection(ExtensionPoint):
    """
    An extension point for all specified traits to be synced with the
    current project's selection.

    """

    # The location of the object to be synced
    uol = Str

    # The name of the trait to be synced
    name = Str


class UIServiceFactory(ExtensionPoint):
    """
    The factory used to create the UI service.

    The extension with the highest priority wins!  In the event of a tie,
    the first instance wins.#

    """

    # The name of the class that implements the factory.
    class_name = Str

    # The priority of this factory
    priority = Int


##############################################################################
# Extensions.
##############################################################################

#### Action classes used in multiple action sets #############################

class NewProjectAction(Action):
    description = 'Create a project'
    id = 'NewProject'
    image = 'new_project'
    method_name = 'create'
    name = '&New...'
    object = 'service://' + IPROJECT_UI
    tooltip = 'Create a project'

class OpenProjectAction(Action):
    description = 'Open an existing project'
    id = 'OpenProject'
    image = 'open_project'
    method_name = 'open'
    name = '&Open...'
    object = 'service://' + IPROJECT_UI
    tooltip = 'Open a project'

class SaveProjectAction(Action):
    class_name = ID + '.action.save_action.SaveAction'
    description = 'Save the current project'
    enabled = False
    id = 'SaveProject'
    image = 'save_project'
    lazy_load = False
    name = '&Save'
    tooltip = 'Save this project'

class SaveAsProjectAction(Action):
    class_name = ID + '.action.save_as_action.SaveAsAction'
    description = 'Save the current project to a different location'
    enabled = False
    id = 'SaveAsProject'
    image = 'save_as_project'
    lazy_load = False
    name = 'Save &As...'
    tooltip = 'Save this project to a different location'

class CloseProjectAction(Action):
    class_name = ID + '.action.close_action.CloseAction'
    description = 'Close the current project'
    enabled = False
    id = 'CloseProject'
    image = 'close_project'
    lazy_load = False
    name = '&Close'
    tooltip = 'Close this project'


#### Factory Definitions #####################################################

factory_definition = FactoryDefinition(
    class_name = ID + '.project_factory.ProjectFactory',
    priority = 0,
    )

ui_service_factory = UIServiceFactory(
    class_name = ID + '.ui_service_factory.UIServiceFactory',
    priority = 0,
    )


#### Resources ###############################################################

resource_manager = ResourceManager(
    resource_types = [
        ResourceType(
            class_name = ID + '.project_resource_type.ProjectResourceType',
            precedes   = [
                'enthought.envisage.resource.instance_resource_type.' + \
                    'InstanceResourceType',
                ]
            ),
        ],
    )


#### Runnables ##############################################################

# A runnable that opens the first workbench window.
runnable = Runnable(
    class_name = ID + '.project_runnable.ProjectRunnable'
    )


#### Single Project Actions #################################################

single_project_action_set = SingleProjectActionSet(
    id = ID + '.single_project_action_set',
    name = 'SingleProjectActionSet',

    groups = [
        Group(
            id = 'OpenGroup',
            location = Location(path='NoSelectionMenu')
            ),
        Group(
            id = 'SaveGroup',
            location = Location(path='NoSelectionMenu', after='OpenGroup')
            ),
        Group(
            id = 'CloseGroup',
            location = Location(path='NoSelectionMenu', after='SaveGroup')
            ),
        Group(
            id = 'SaveGroup',
            location = Location(path='ProjectMenu')
            ),
        Group(
            id = 'CloseGroup',
            location = Location(path='ProjectMenu', after='SaveGroup')
            ),
        ],

    actions = [
        NewProjectAction(
            locations = [
                Location(path='NoSelectionMenu/OpenGroup'),
                ],
            ),
        OpenProjectAction(
            locations = [
                Location(path='NoSelectionMenu/OpenGroup', after='NewProject'),
                ],
            ),
        SaveProjectAction(
            locations = [
                Location(path='NoSelectionMenu/SaveGroup'),
                # FIXME: UOL actions don't seem to work on the ProjectMenu
                # because the events don't have the application trait set.
                Location(path='ProjectMenu/SaveGroup'),
                ],
            ),
        SaveAsProjectAction(
            locations = [
                Location(path='NoSelectionMenu/SaveGroup', after='SaveProject'),
                # FIXME: UOL actions don't seem to work on the ProjectMenu
                # because the events don't have the application trait set.
                Location(path='ProjectMenu/SaveGroup', after='SaveProject'),
                ],
            ),
        CloseProjectAction(
            locations = [
                Location(path='NoSelectionMenu/CloseGroup'),
                # FIXME: UOL actions don't seem to work on the ProjectMenu
                # because the events don't have the application trait set.
                Location(path='ProjectMenu/CloseGroup'),
                ],
            ),
        ],
    )

#### Single Project Preferences ##############################################

PATH_PREFERENCE_PAGE= \
    ID + '.default_path_preference_page.DefaultPathPreferencePage'


preferences = Preferences(
    defaults = {
        # Preferred path for saving/opening projects
        'preferred_path': os.path.expanduser('~'),
    }
)

preference_pages = PreferencePages(pages = [
                        Page(id=PATH_PREFERENCE_PAGE,
                             class_name=PATH_PREFERENCE_PAGE,
                             name='Directory',
                             category='')
                        ])

#### Workbench Perspectives and Views ########################################

workbench = Workbench(
    perspectives = [
        Perspective(
            id = ID + '.perspective.project',
            name = 'Project',
            contents = [
                Perspective.Item(
                    id = ID + '.view.project_view.ProjectView',
                    position = 'left',
                    width = 0.25,
                    ),
                Perspective.Item(
                    id = 'enthought.plugins.python_shell.view.PythonShellView',
                    position = 'bottom',
                    width = 0.75,
                    ),
                ]
            ),
        ],

    views = [
        View(
            id         = ID + '.view.project_view.ProjectView',
            class_name = ID + '.view.project_view.ProjectView',
            image      = 'images/view.png',
            name       = 'Current Project',
            position   = 'left',
            ),
        ],
    )


#### Workbench Window Actions ################################################

workbench_action_set = WorkbenchActionSet(
    id   = ID + '.ProjectActionSet',
    name = 'Project',

    groups = [
        Group(
            id = 'PerspectiveGroup',
            location = Location(path='ToolBar')
            ),
        Group(
            id = 'ProjectGroup',
            location = Location(path='ToolBar', after='PerspectiveGroup')
            ),
        Group(
            id = 'ProjectGroup',
            location = Location(path='MenuBar/FileMenu', before='ExitGroup')
            ),
        ],

    menus = [
        Menu(
            groups = [
                Group(id='OpenGroup'),
                Group(id='SaveGroup'),
                Group(id='CloseGroup'),
                ],
            id = 'ProjectMenu',
            location = Location(path='MenuBar/FileMenu/ProjectGroup'),
            name = '&Project',
            ),
        ],

    actions = [
        NewProjectAction(
            locations = [
                Location(path='MenuBar/FileMenu/ProjectMenu/OpenGroup'),
                Location(path='ToolBar/ProjectGroup')
                ],
            ),
        OpenProjectAction(
            locations = [
                Location(path='MenuBar/FileMenu/ProjectMenu/OpenGroup',
                    after='NewProject'),
                Location(path='ToolBar/ProjectGroup', after='NewProject')
                ],
            ),
        SaveProjectAction(
            locations = [
                Location(path='MenuBar/FileMenu/ProjectMenu/SaveGroup'),
                Location(path='ToolBar/ProjectGroup', after='OpenProject')
                ],
            ),
        SaveAsProjectAction(
            locations = [
                Location(path='MenuBar/FileMenu/ProjectMenu/SaveGroup',
                    after='SaveProject'),
                Location(path='ToolBar/ProjectGroup', after='SaveProject')
                ],
            ),
        CloseProjectAction(
            locations = [
                Location(path='MenuBar/FileMenu/ProjectMenu/CloseGroup'),
                Location(path='ToolBar/ProjectGroup', after='SaveAsProject')
                ],
            ),
        Action(
            class_name = ID + '.action.switch_to_action.SwitchToAction',
            description = 'View the current project in the Project perspective',
            id = 'SwitchToProject',
            image = 'switch_project',
            locations = [
                Location( path = 'ToolBar/PerspectiveGroup' )
                ],
            name = 'Switch To &Project',
            tooltip = 'Go to the Project perspective',
            ),
        ],
    )


##############################################################################
# The plugin definition.
##############################################################################

class ProjectPluginDefinition(PluginDefinition):
    # The plugin's globally unique identifier.
    id = ID

    # The name of the class that implements the plugin.
    class_name = ID + '.plugin_implementation.PluginImplementation'

    # General information about the plugin.
    name = 'Single Project Plugin'
    version = '1.0.5'
    provider_name = 'Enthought Inc'
    provider_url = 'www.enthought.com'
    autostart = True

    # The Id's of the plugins that this plugin requires.
    requires = [
        'enthought.envisage.resource',
        'enthought.envisage.workbench',
        'enthought.plugins.python_shell',
        ]

    # The extension points offered by this plugin.
    extension_points = [
        FactoryDefinition,
        SingleProjectActionSet,
        SyncProjectSelection,
        UIServiceFactory,
        ]

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = [
        factory_definition,
        preferences,
        preference_pages,
        resource_manager,
        runnable,
        single_project_action_set,
        ui_service_factory,
        workbench,
        workbench_action_set,
        ]


#### EOF #####################################################################
