""" A tree view of a resource naming system. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage.resource.api import ResourceManager
from enthought.naming.ui.api import NamingTree
from enthought.pyface.action.api import ActionController
from enthought.traits.api import Bool, Instance, Property

# Local imports.
from resource_tree_model import ResourceTreeModel


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ResourceTree(NamingTree):
    """ A tree view of a resource naming system. """

    #### 'Tree' interface #####################################################

    # The model that provides the data for the tree.
    model = Instance(ResourceTreeModel)

    # Selection mode.
    selection_mode = 'extended'

    #### 'ResourceTree' interface #############################################

    # An optional action controller.
    controller = Instance(ActionController)

    # Should the tree be read-only (i.e., no context menus or label editing
    # etc)?
    read_only = Bool(False)

    # The resource manager (for convenience, this just delegates to the model).
    resource_manager = Property(Instance(ResourceManager))

    ###########################################################################
    # 'Tree' interface.
    ###########################################################################

    #### Trait initializers ###################################################

    def _model_default(self):
        """ Initializes the model trait. """

        return ResourceTreeModel()

    ###########################################################################
    # 'NodeTree' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_resource_manager(self):
        """ Returns the resource manager. """

        return self.model.resource_manager

    def _set_resource_manager(self, resource_manager):
        """ Sets the resource manager. """

        self.model.resource_manager = resource_manager

        return

    #### Protected methods ####################################################

    def _perform_default_action(self, action, obj):
        """ Perform the default action. """

        # If no one has specified a different controller, use ourself as the
        # controller.
        if self.controller is not None:
            controller = self.controller
            controller.node = obj
        else:
            controller = self

        controller.perform(action, self._create_action_event(obj))

        return


    ###########################################################################
    # 'Tree' interface.
    ###########################################################################

    #### Trait event handlers #################################################

    def _node_right_clicked_changed(self, (obj, point)):
        """ Called when the right mouse button is clicked on the tree. """

        if not self.read_only:
            # Add the node that the right-click occurred on to the selection.
            self.select(obj)

            # fixme: Ugly! The intent is to save the object that we were
            # clicked on so that we can add it to the action event in our
            # role as an action controller.
            self._context = obj

            # Ask the model for the node's context menu.
            menu_manager = self.model.get_context_menu(obj)
            if menu_manager is not None:
                self._popup_menu(menu_manager, obj, point)

        return

    ###########################################################################
    # 'ActionController' interface.
    ###########################################################################

    def add_to_menu(self, menu_item):
        """ Adds a menu item to a menu bar. """

        pass

    def add_to_toolbar(self, toolvar_item):
        """ Adds a tool bar item to a tool bar. """

        pass

    def can_add_to_menu(self, action):
        """ Returns True iff an action can be added to the menu. """

        return True

    def perform(self, action, event):
        """ Performs an action. """

        # fixme: We need a more formal API for adding stuff to events!
        event.widget = self
        event.node   = self._context

        # Call the action's perform method.
        #
        # NOTE: If the only argument to the action's perform method is 'self'
        # then the action is using the DEPRECATED interface.
        import inspect

        args, varargs, varkw, defaults = inspect.getargspec(action.perform)
        if len(args) == 1:
            logger.warn('Action [%s] - deprecated "perform" interface', action)
            action.perform()

        else:
            action.perform(event)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _popup_menu(self, menu_manager, obj, point):
        """ Popup the menu described by the menu manager. """

        # fixme: This is a bit hacky. It allows a controller to be used to
        # perform all actions on context menus in the tree.
        if self.controller is not None:
            controller = self.controller
            controller.node = obj

        else:
            controller = self

        # Create the actual menu control.
        menu = menu_manager.create_menu(self.control, controller)
        if not menu.is_empty():
            # Show the menu. If an action is selected it will be performed
            # *before* this call returns.
            menu.show(*point)

            # This gives the actions in the menu manager a chance to cleanup
            # any event listeners etc.
            menu_manager.destroy()

        return

    def _controller_changed(self, old, new):
        """ Called whenever the controller for this resource tree changes. """

        # Ensure that the previous controller doesn't still think we are its
        # tree.
        if old is not None:
            if hasattr(old, 'tree') and old.tree == self:
                old.tree = None

        # Let the new controller know we are now its tree
        if new is not None:
            new.tree = self

        return

##### EOF #####################################################################
