/**
 * sampletest.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1-> Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * 2-> Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "sampletest.h"
#include "message.h"

using namespace MateEdit;
int handle_undo(Message * undo_message, Document & doc, QPtrList<Message> & pending_list, int local_pc);
int handle_do(Message * message, Document & doc, TextOwner & owners, int local_pc, MessageQueue * messageQueue, int server_state);

EqualityTest::EqualityTest()
    : Tester()
{
}

void EqualityTest::allTests()
{
    MessageCoords m_0_0(0, 0);
    MessageCoords m_0_1(0, 1);
    MessageCoords m_0_2(0, 2);
    MessageCoords m_1_0(1, 0);
    MessageCoords m_1_1(1, 1);
    MessageCoords m_1_2(1, 2);
    MessageCoords m_1_4(1, 4);
    MessageCoords m_1_5(1, 5);
    MessageCoords m_1_6(1, 6);
    MessageCoords m_2_4(2, 4);
    MessageCoords m_5_4(5, 4);
    MessageCoords m_5_5(5, 5);

    CHECKNORESULT( m_1_1, m_1_1 );
    XFAILNORESULT( m_1_0, m_1_1 );
    XFAILNORESULT( m_1_0, m_0_1 );
    XFAILNORESULT( m_0_1, m_1_1 );
    XFAILNORESULT( m_1_1, m_1_0 );
}

CoordsShiftDownTest::CoordsShiftDownTest()
    : Tester()
{
}

void CoordsShiftDownTest::allTests()
{
    MessageCoords m_0_0(0, 0);
    MessageCoords m_0_1(0, 1);
    MessageCoords m_0_2(0, 2);
    MessageCoords m_0_3(0, 3);
    MessageCoords m_1_0(1, 0);
    MessageCoords m_1_1(1, 1);
    MessageCoords m_1_2(1, 2);
    MessageCoords m_1_3(1, 3);
    MessageCoords m_1_4(1, 4);
    MessageCoords m_1_5(1, 5);
    MessageCoords m_1_6(1, 6);
    MessageCoords m_2_4(2, 4);
    MessageCoords m_5_4(5, 4);
    MessageCoords m_5_5(5, 5);

    m_1_0.shiftDown( m_0_0, m_0_1 );
    CHECKNORESULT( m_1_0, MessageCoords(1,0) );
    m_1_0.setCoords(1,0);

    m_1_6.shiftDown( m_1_0, m_1_5 );
    CHECKNORESULT( m_1_6, m_1_1 );
    m_1_6.setCoords(1,6);

    m_1_4.shiftDown( m_1_0, m_1_1 );
    CHECKNORESULT( m_1_4, m_1_3 );
    m_1_4.setCoords(1,4);

    m_1_4.shiftDown( m_0_0, m_1_1 );
    CHECKNORESULT( m_1_4, m_0_3 );
    m_1_4.setCoords(1,4);

    m_1_0.shiftDown( m_1_1, m_1_2 );
    CHECKNORESULT( m_1_0, MessageCoords(1,0) );
    m_1_0.setCoords(1,0);

    m_1_5.shiftDown( m_1_2, m_1_4 );
    CHECKNORESULT( m_1_5, m_1_3 );
    m_1_5.setCoords(1,5);

    m_2_4.shiftDown( m_0_2, m_1_2 );
    CHECKNORESULT( m_2_4, m_1_4 );
    m_2_4.setCoords(2,4);
}

CoordsShiftUpTest::CoordsShiftUpTest() : Tester()
{
}

void CoordsShiftUpTest::allTests()
{
    MessageCoords m_0_0(0, 0);
    MessageCoords m_0_1(0, 1);
    MessageCoords m_0_2(0, 2);
    MessageCoords m_0_3(0, 3);
    MessageCoords m_1_0(1, 0);
    MessageCoords m_1_1(1, 1);
    MessageCoords m_1_2(1, 2);
    MessageCoords m_1_3(1, 3);
    MessageCoords m_1_4(1, 4);
    MessageCoords m_1_5(1, 5);
    MessageCoords m_1_6(1, 6);
    MessageCoords m_2_3(2, 3);
    MessageCoords m_2_4(2, 4);
    MessageCoords m_5_4(5, 4);
    MessageCoords m_5_5(5, 5);

    m_1_0.shiftUp( m_0_0, m_0_1 );
    CHECKNORESULT( m_1_0, MessageCoords(1,0) );
    m_1_0.setCoords(1,0);

    m_1_0.shiftUp( m_1_0, m_1_5 );
    CHECKNORESULT( m_1_0, m_1_5 );
    m_1_0.setCoords(1,0);

    m_1_2.shiftUp( m_1_0, m_1_1 );
    CHECKNORESULT( m_1_2, m_1_3 );
    m_1_2.setCoords(1,2);

    m_1_2.shiftUp( m_0_0, m_1_1 );
    CHECKNORESULT( m_1_2, m_2_3 );
    m_1_2.setCoords(1,2);

    m_1_0.shiftUp( m_1_1, m_1_2 );
    CHECKNORESULT( m_1_0, MessageCoords(1,0) );
    m_1_0.setCoords(1,0);

    m_1_2.shiftUp( m_1_2, m_1_4 );
    CHECKNORESULT( m_1_2, m_1_4 );
    m_1_2.setCoords(1,2);
}

InsertInsertTest1::InsertInsertTest1() : Tester()
{
}

void InsertInsertTest1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    MessageCoords coordsA1( 0, 3 );
    MessageCoords coordsB1from( 0, 3 );
    MessageCoords coordsB1to( 0, 6 );

    Message * messageA1 = new Message (Message::Insert, 1, 0, 0, "abc");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 2, 0, "def");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "abc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 1);
    CHECK( rtc, 0 );
    CHECK( messageA1->to( 1 ), coordsA1 );


    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );
    CHECK( messageB1->from( 1 ), coordsB1from );
    CHECK( messageB1->to( 1 ), coordsB1to );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 3, 2 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "def";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4, 3 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertInsert1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}


InsertInsertTest2::InsertInsertTest2() : Tester()
{
}

void InsertInsertTest2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    MessageCoords coordsA1from( 0, 0 );
    MessageCoords coordsA1to( 0, 1 );
    MessageCoords coordsA2from( 0, 0 );
    MessageCoords coordsA2to( 0, 1 );
    MessageCoords coordsB1from( 0, 2 );
    MessageCoords coordsB1to( 0, 3 );

    Message * messageA1 = new Message (Message::Insert, 1, 0, 0, "b");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,1);
    Message * messageA2 = new Message (Message::Insert, 1, 0, 0, "c");
    messageA2->setFrom(0,0);
    messageA2->setTo(0,1);
    Message * messageB1 = new Message (Message::Insert, 2, 3, 0, "a");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,1);

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "b";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 1);
    CHECK( rtc, 0 );
    CHECK( messageA1->from( 1 ), coordsA1from );
    CHECK( messageA1->to( 1 ), coordsA1to );

    rtc = handle_do(messageA2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA2->type(), Message::Insert );
    expected = "cb";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA2, doc, owners, messageQueue.localPc(), &messageQueue, 2);
    CHECK( rtc, 0 );
    CHECK( messageA2->to( 1 ), coordsA2to );
    CHECK( messageA2->from( 1 ), coordsA2from );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "cba";
    result = doc.at(0);
    CHECK( result, expected );

    CHECK( messageB1->from( 1 ), coordsB1from );
    CHECK( messageB1->to( 1 ), coordsB1to );

    Message * undo_message = messageQueue.undoMessage( messageA2->id(), messageA2->sender(), pending_list, 4 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "ba";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "a";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertInsert2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}


InsertInsertTest3::InsertInsertTest3() : Tester()
{
}

void InsertInsertTest3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    MessageCoords coordsA1from( 0, 0 );
    MessageCoords coordsA1to( 0, 1 );
    MessageCoords coordsB1from( 0, 1 );
    MessageCoords coordsB1to( 0, 2 );
    MessageCoords coordsB2from( 0, 1 );
    MessageCoords coordsB2to( 0, 2 );

    Message * messageA1 = new Message (Message::Insert, 1, 0, 0, "a");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,1);
    Message * messageB1 = new Message (Message::Insert, 2, 2, 0, "b");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,1);
    Message * messageB2 = new Message (Message::Insert, 2, 3, 0, "c");
    messageB2->setFrom(0,0);
    messageB2->setTo(0,1);

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageA1->from( 1 ), coordsA1from );
    CHECK( messageA1->to( 1 ), coordsA1to );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 1 ), messageB1->to( 1 ), messageB1->sender(), messageB1->text() );

    CHECK( messageB1->from( 1 ), coordsB1from );
    CHECK( messageB1->to( 1 ), coordsB1to );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );

    rtc = messageQueue.processMessage( messageB2 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    doc.insert( messageB2->from( 1 ), messageB2->to( 1 ), messageB2->sender(), messageB2->text() );

    CHECK( messageB2->from( 1 ), coordsB2from );
    CHECK( messageB2->to( 1 ), coordsB2to );
    CHECK( (Message::Type)messageB2->type(), Message::Insert );

    expected = "acb";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "cb";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "b";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertInsert3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

InsertDeleteTest1::InsertDeleteTest1() : Tester()
{
}

void InsertDeleteTest1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 3 );
    MessageCoords coordsA1to( 0, 4 );
    MessageCoords coordsB1from( 0, 1 );
    MessageCoords coordsB1toA( 0, 6 );
    MessageCoords coordsB1toB( 0, 5 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);
    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "h");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,5);


    messageQueue.processMessage( messageC1 );


    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );


    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    if( (Message::Type)messageA1->type() == Message::Insert)
    {
        doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );
    }

    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( messageB1->to( 1 ), coordsB1toA );
    CHECK( (Message::Type)messageA1->type(), Message::NoOp );

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abchdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertDelete1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

InsertDeleteTest2::InsertDeleteTest2() : Tester()
{
}

void InsertDeleteTest2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 2 );
    MessageCoords coordsA1to( 0, 3 );
    MessageCoords coordsB1from( 0, 1 );
    MessageCoords coordsB1toA( 0, 2 );
    MessageCoords coordsB1toB( 0, 2 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "h");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,2);


    messageQueue.processMessage( messageC1 );


    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );


    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( messageB1->to( 1 ), coordsB1toA );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );

    expected = "achdef";
    result = doc.at(0);
    CHECK( result, expected );


    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abchdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertDelete2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

InsertDeleteTest3::InsertDeleteTest3() : Tester()
{
}

void InsertDeleteTest3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 3 );
    MessageCoords coordsA1to( 0, 4 );
    MessageCoords coordsB1from( 0, 4 );
    MessageCoords coordsB1toA( 0, 5 );
    MessageCoords coordsB1toB( 0, 5 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "h");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,4);
    messageB1->setTo(0,5);


    messageQueue.processMessage( messageC1 );


    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );


    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( messageB1->to( 1 ), coordsB1toA );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );

    expected = "abchdf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abchdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertDelete3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

InsertDeleteTest4::InsertDeleteTest4() : Tester()
{
}

void InsertDeleteTest4::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 3 );
    MessageCoords coordsA1to( 0, 4 );
    MessageCoords coordsA2from( 0, 1 );
    MessageCoords coordsA2to( 0, 2 );
    MessageCoords coordsB1from( 0, 1 );
    MessageCoords coordsB1to( 0, 5 );
    MessageCoords coordsB2from( 0, 2 );
    MessageCoords coordsB2to( 0, 3 );
    MessageCoords coordsB3from( 0, 1 );
    MessageCoords coordsB3to( 0, 2 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 5, 1, "H");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,4);
    Message * messageA2 = new Message (Message::Insert, 1, 6, 1, "2");
    messageA2->setFrom(0,1);
    messageA2->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,5);
    Message * messageB2 = new Message (Message::Insert, 2, 0, 1, "4");
    messageB2->setFrom(0,2);
    messageB2->setTo(0,3);
    Message * messageB3 = new Message (Message::Insert, 2, 0, 1, "3");
    messageB3->setFrom(0,1);
    messageB3->setTo(0,2);


    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1to );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );

    rtc = messageQueue.processMessage( messageB2 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB2->from( 2 ), messageB2->to( 2 ), messageB2->sender(), messageB2->text() );

    messageB2->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB2 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageB2->from( 2 ), coordsB2from );
    CHECK( messageB2->to( 2 ), coordsB2to );
    CHECK( (Message::Type)messageB2->type(), Message::Insert );

    rtc = messageQueue.processMessage( messageB3 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
    messageB3->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB3 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    doc.insert( messageB3->from( 2 ), messageB3->to( 2 ), messageB3->sender(), messageB3->text() );

    CHECK( messageB3->from( 2 ), coordsB3from );
    CHECK( messageB3->to( 2 ), coordsB3to );
    CHECK( (Message::Type)messageB3->type(), Message::Insert );


    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( (Message::Type)messageA1->type(), Message::NoOp );

    rtc = messageQueue.processMessage( messageA2 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    doc.insert( messageA2->from( 2 ), messageA2->to( 2 ), messageA2->sender(), messageA2->text() );

    CHECK( messageA2->from( 2 ), coordsA2from );
    CHECK( messageA2->to( 2 ), coordsA2to );
    CHECK( (Message::Type)messageA2->type(), Message::Insert );

    expected = "a23f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list, 7 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list, 8 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 9 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2bcHdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA2->id(), messageA2->sender(), pending_list, 10 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcHdef";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 11 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 12 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "InsertDelete4 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}


DeleteDeleteTest1::DeleteDeleteTest1() : Tester()
{
}

void DeleteDeleteTest1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 1 );
    MessageCoords coordsA1to( 0, 3 );
    MessageCoords coordsB1from( 0, 2 );
    MessageCoords coordsB1toA( 0, 4 );
    MessageCoords coordsB1toB( 0, 4 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Delete, 1, 2, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);


//    messageQueue.processMessage( messageC1 );

//    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );
    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Insert );
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );
    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
#endif
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );

//    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Delete );
    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
#endif
#if 0
    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
#endif
    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 2);
    CHECK( rtc, 0 );

    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( messageB1->to( 1 ), coordsB1toA );
    CHECK( (Message::Type)messageA1->type(), Message::Delete );

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DeleteDeleteTest1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

DeleteDeleteTest2::DeleteDeleteTest2() : Tester()
{
}

void DeleteDeleteTest2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    MessageCoords coordsA1from( 0, 1 );
    MessageCoords coordsA1to( 0, 4 );
    MessageCoords coordsB1from( 0, 4 );
    MessageCoords coordsB1toA( 0, 6 );
    MessageCoords coordsB1toB( 0, 6 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,4);
    messageB1->setTo(0,6);


    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 2 ), messageC1->to( 2 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );


    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
    messageA1->setText( text );

    CHECK( messageA1->from( 2 ), coordsA1from );
    CHECK( messageA1->to( 2 ), coordsA1to );
    CHECK( messageB1->from( 2 ), coordsB1from );
    CHECK( messageB1->to( 2 ), coordsB1toB );
    CHECK( messageB1->to( 1 ), coordsB1toA );
    CHECK( (Message::Type)messageA1->type(), Message::Delete );

    expected = "a";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DeleteDeleteTest2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

DeleteDeleteTest3::DeleteDeleteTest3() : Tester()
{
}

void DeleteDeleteTest3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;
    int rtc;
    QString text;
    QString expected;
    QString result;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(4);
    MessageCoords coordsA1from( 0, 1 );
    MessageCoords coordsA1to( 0, 4 );
    MessageCoords coordsB1from( 0, 3 );
    MessageCoords coordsB1to( 0, 5 );
    MessageCoords coordsD1from( 0, 2 );
    MessageCoords coordsD1to( 0, 3 );

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Delete, 1, 4, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);
    Message * messageD1 = new Message (Message::Insert, 4, 0, 1, "h");
    messageD1->setFrom(0,2);
    messageD1->setTo(0,3);


    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageD1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }
    messageD1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageD1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }
    CHECK( messageD1->from( 4 ), coordsD1from );
    CHECK( messageD1->to( 4 ), coordsD1to );


    doc.insert( messageD1->from( 4 ), messageD1->to( 4 ), messageD1->sender(), messageD1->text() );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.remove( messageB1->from( 4 ), messageB1->to( 4 ) );

    CHECK( messageB1->from( 4 ), coordsB1from );
    CHECK( messageB1->to( 4 ), coordsB1to );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.remove( messageA1->from( 4 ), messageA1->to( 4 ) );

    CHECK( messageA1->from( 4 ), coordsA1from );
    CHECK( messageA1->to( 4 ), coordsA1to );
    CHECK( (Message::Type)messageA1->type(), Message::Delete );


    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageD1->id(), messageD1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 7 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 8 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DeleteDeleteTest3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD1_1::TestDD1_1() : Tester()
{
}

void TestDD1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 0, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 1 ), messageA1->to( 1 ) );
    messageA1->setText( text );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "adef";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );





    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

     undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

     handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

     expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD4_1::TestDD4_1() : Tester()
{
}

void TestDD4_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
    messageA1->setText( text );

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "adef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD4 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD1_2::TestDD1_2() : Tester()
{
}

void TestDD1_2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 0, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,4);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 1 ), messageA1->to( 1 ) );
    messageA1->setText( text );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "acdef";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );





    expected = "acdf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD1_2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD4_2::TestDD4_2() : Tester()
{
}

void TestDD4_2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,4);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcdf";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
    messageA1->setText( text );





    expected = "acdf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "acdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD4_2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD2_1::TestDD2_1() : Tester()
{
}

void TestDD2_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 0, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 1 ), messageA1->to( 1 ) );
    messageA1->setText( text );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );





    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD3_1::TestDD3_1() : Tester()
{
}

void TestDD3_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
    messageA1->setText( text );





    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD5_1::TestDD5_1() : Tester()
{
}

void TestDD5_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 0, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 1 ), messageA1->to( 1 ) );
    messageA1->setText( text );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );





    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD5 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD6_1::TestDD6_1() : Tester()
{
}

void TestDD6_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }



    CHECK( (Message::Type)messageA1->type(), Message::NoOp );
    if((Message::Type)messageA1->type() == Message::Delete)
    {
        text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
        messageB1->setText( text );
    }





    expected = "af";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD6 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD7_1::TestDD7_1() : Tester()
{
}

void TestDD7_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 0, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 1 ), messageA1->to( 1 ) );
    messageA1->setText( text );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );
    //CHECK( (Message::Type)messageA1->type(), Message::Insert );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    if((Message::Type)messageB1->type() == Message::Delete)
    {
        text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
        messageB1->setText( text );
    }



    CHECK( (Message::Type)messageB1->type(), Message::NoOp );

    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD7 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDD8_1::TestDD8_1() : Tester()
{
}

void TestDD8_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Delete, 1, 3, 1);
    messageA1->setFrom(0,2);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    //    CHECK( messageA1->to( 1 ), coordsA1 );



    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageA1->from( 2 ), messageA1->to( 2 ) );
    messageA1->setText( text );





    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DD8 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII1_1::TestII1_1() : Tester()
{
}

void TestII1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "2");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Insert, 2, 3, 1, "4");
    messageB1->setFrom(0,3);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "a2bcdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 1 ), messageB1->to( 1 ), messageB1->sender(), messageB1->text() );





    expected = "a2bc4def";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abc4def";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII4_1::TestII4_1() : Tester()
{
}

void TestII4_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "2");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 1, "4");
    messageB1->setFrom(0,3);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 2 ), messageB1->to( 2 ), messageB1->sender(), messageB1->text() );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abc4def";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );





    expected = "a2bc4def";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2bcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II4 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII2_1::TestII2_1() : Tester()
{
}

void TestII2_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "234");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Insert, 2, 3, 1, "567");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "a234bcdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 1 ), messageB1->to( 1 ), messageB1->sender(), messageB1->text() );





    expected = "a234b567cdef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab567cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII3_1::TestII3_1() : Tester()
{
}

void TestII3_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "234");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 1, "567");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 2 ), messageB1->to( 2 ), messageB1->sender(), messageB1->text() );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab567cdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );





    expected = "a234b567cdef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a234bcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}


TestII5_1::TestII5_1() : Tester()
{
}

void TestII5_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Insert, 2, 3, 1, "3456");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abc12def";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 1 ), messageB1->to( 1 ), messageB1->sender(), messageB1->text() );





    expected = "ab3456c12def";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab3456cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II5 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII6_1::TestII6_1() : Tester()
{
}

void TestII6_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 1, "3456");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 2 ), messageB1->to( 2 ), messageB1->sender(), messageB1->text() );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab3456cdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );





    expected = "ab3456c12def";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abc12def";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II6 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}


TestII7_1::TestII7_1() : Tester()
{
}

void TestII7_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Insert, 2, 3, 1, "3456");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 1 ), messageB1->to( 1 ), messageB1->sender(), messageB1->text() );





    expected = "ab123456cdef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab3456cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II7 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestII8_1::TestII8_1() : Tester()
{
}

void TestII8_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);



    int rtc;
    QString expected;
    QString result;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 1, "3456");
    messageB1->setFrom(0,2);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageB1->from( 2 ), messageB1->to( 2 ), messageB1->sender(), messageB1->text() );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab3456cdef";
    result = doc.at(0);
    CHECK( result, expected );





    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );





    expected = "ab123456cdef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    QString text = "II8 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID1_1::TestID1_1() : Tester()
{
}

void TestID1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,5);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ab12cde";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcde";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI4_1::TestDI4_1() : Tester()
{
}

void TestDI4_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,5);
    messageB1->setTo(0,6);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcde";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ab12cde";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI4 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID2_1::TestID2_1() : Tester()
{
}

void TestID2_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ab12cf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI3_1::TestDI3_1() : Tester()
{
}

void TestDI3_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ab12cf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID3_1::TestID3_1() : Tester()
{
}

void TestID3_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,4);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcd12ef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI2_1::TestDI2_1() : Tester()
{
}

void TestDI2_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,4);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    if(messageA1->type() == Message::Insert)
    {
        doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );
    }
    CHECK( (Message::Type)messageA1->type(), Message::NoOp );

    expected = "abcf";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcd12ef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID4_1::TestID4_1() : Tester()
{
}

void TestID4_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,4);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,3);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcd12ef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ad12ef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "adef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID4 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI1_1::TestDI1_1() : Tester()
{
}

void TestDI1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,4);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,3);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "adef";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ad12ef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcd12ef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID5_1::TestID5_1() : Tester()
{
}

void TestID5_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abc12def";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID5 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI6_1::TestDI6_1() : Tester()
{
}

void TestDI6_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,3);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    if(messageA1->type() == Message::Insert)
    {
        doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );
    }
    CHECK( (Message::Type)messageA1->type(), Message::NoOp );

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abc12def";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI6 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID6_1::TestID6_1() : Tester()
{
}

void TestID6_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "1234");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab1234cdef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ab1234cef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID6 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI5_1::TestDI5_1() : Tester()
{
}

void TestDI5_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "1234");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,6);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,3);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abcef";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ab1234cef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab1234cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI5 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID8_1::TestID8_1() : Tester()
{
}

void TestID8_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ab12f";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID8 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI7_1::TestDI7_1() : Tester()
{
}

void TestDI7_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "12");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,4);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,5);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abf";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ab12f";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab12cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI7 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestID7_1::TestID7_1() : Tester()
{
}

void TestID7_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "123");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 3, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 1 ), messageA1->to( 1 ), messageA1->sender(), messageA1->text() );

    messageA1->setServerState( 1 );
    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "ab123cdef";
    result = doc.at(0);
    CHECK( result, expected );


    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 1 ), messageB1->to( 1 ) );
    messageB1->setText( text );

    expected = "ab123ef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "ID7 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

TestDI8_1::TestDI8_1() : Tester()
{
}

void TestDI8_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);

    int rtc;
    QString expected;
    QString result;
    QString text;

    Message * messageC1 = new Message (Message::Insert, 3, 1, 0, "abcdef");
    messageC1->setFrom(0,0);
    messageC1->setTo(0,6);

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "123");
    messageA1->setFrom(0,2);
    messageA1->setTo(0,5);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,2);
    messageB1->setTo(0,4);

    messageQueue.processMessage( messageC1 );

    doc.insert( messageC1->from( 4 ), messageC1->to( 4 ), messageC1->sender(), messageC1->text() );

    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    text = doc.remove( messageB1->from( 2 ), messageB1->to( 2 ) );
    messageB1->setText( text );

    messageB1->setServerState( 2 );
    rtc = messageQueue.processMessage( messageB1 );
    if (rtc < 0)
    {
        CHECK( rtc, -1 );
    }

    expected = "abef";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = messageQueue.processMessage( messageA1 );
    if (rtc < 0)
    {
        CHECK( rtc, 0 );
    }

    doc.insert( messageA1->from( 2 ), messageA1->to( 2 ), messageA1->sender(), messageA1->text() );

    expected = "ab123ef";
    result = doc.at(0);
    CHECK( result, expected );

    Message * undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list, 4 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ab123cdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageA1->id(), messageA1->sender(), pending_list, 5 );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "abcdef";
    result = doc.at(0);
    CHECK( result, expected );

    undo_message = messageQueue.undoMessage( messageC1->id(), messageC1->sender(), pending_list, 6 );
    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    text = "DI8 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest1_1::ChalksTest1_1() : Tester()
{
}

void ChalksTest1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 0, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 1, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message  (Message::Delete, 2, 4, 2);
    messageB2->setFrom(0,1);
    messageB2->setTo(0,3);
    Message * messageC1 = new Message (Message::Insert, 3, 3, 1, "c");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "ABCBCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 2);
    CHECK( rtc, 0 );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Insert );
    expected = "ABCBCcD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::Delete );
    CHECK( rtc, 0 );
    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest1_1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest1_2::ChalksTest1_2() : Tester()
{
}

void ChalksTest1_2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 2, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message  (Message::Delete, 2, 0, 2);
    messageB2->setFrom(0,1);
    messageB2->setTo(0,3);
    Message * messageC1 = new Message (Message::Insert, 3, 3, 1, "c");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "BCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 1);
    CHECK( rtc, 0 );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABCBCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::Delete );
    CHECK( rtc, 0 );
    expected = "ABCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Insert );
    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 4);
    CHECK( rtc, 0 );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest1_2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest1_3::ChalksTest1_3() : Tester()
{
}

void ChalksTest1_3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(3);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 2, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 1, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message  (Message::Delete, 2, 4, 2);
    messageB2->setFrom(0,1);
    messageB2->setTo(0,3);
    Message * messageC1 = new Message (Message::Insert, 3, 0, 1, "c");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "BCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Insert );
    expected = "BCcD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABCBCcD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 3);
    CHECK( rtc, 0 );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::Delete );
    CHECK( rtc, 0 );
    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest1_3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest2_1::ChalksTest2_1() : Tester()
{
}

void ChalksTest2_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 0, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 1, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message  (Message::Insert, 2, 4, 2, "c");
    messageB2->setFrom(0,5);
    messageB2->setTo(0,6);
    Message * messageC1 = new Message (Message::Delete, 3, 3, 1);
    messageC1->setFrom(0,0);
    messageC1->setTo(0,3);

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    expected = "ABCBCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 2);
    CHECK( rtc, 0 );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Delete );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::NoOp );
    CHECK( rtc, 0 );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest2_1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest2_2::ChalksTest2_2() : Tester()
{
}

void ChalksTest2_2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    int rtc;
    QString text;

    Message * messageA1 = new Message (Message::Insert, 1, 2, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 0, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message  (Message::Insert, 2, 0, 2, "c");
    messageB2->setFrom(0,5);
    messageB2->setTo(0,6);
    Message * messageC1 = new Message (Message::Delete, 3, 3, 1);
    messageC1->setFrom(0,0);
    messageC1->setTo(0,3);



    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    QString expected = "BCD";
    QString result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 1);
    CHECK( rtc, 0 );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABCBCD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::Insert );
    CHECK( rtc, 0 );
    expected = "ABCBCcD";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Delete );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 4);
    CHECK( rtc, 0 );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest2_2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ChalksTest2_3::ChalksTest2_3() : Tester()
{
}

void ChalksTest2_3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(3);
    int rtc;

    QString text;

    Message * messageA1 = new Message (Message::Insert, 1, 2, 0, "ABC");
    messageA1->setFrom(0,0);
    messageA1->setTo(0,3);
    Message * messageB1 = new Message (Message::Insert, 2, 1, 0, "BCD");
    messageB1->setFrom(0,0);
    messageB1->setTo(0,3);
    Message * messageB2 = new Message (Message::Insert, 2, 4, 2, "c");
    messageB2->setFrom(0,5);
    messageB2->setTo(0,6);
    Message * messageC1 = new Message  (Message::Delete, 3, 0, 1);
    messageC1->setFrom(0,0);
    messageC1->setTo(0,3);


    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Insert );
    QString expected = "BCD";
    QString result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Delete );
    expected = "";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 3);
    CHECK( rtc, 0 );

    rtc = handle_do(messageB2, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( (Message::Type)messageB2->type(), Message::NoOp );
    CHECK( rtc, 0 );
    expected = "ABC";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ChalksTest2_3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ImineEtAlTest1_1::ImineEtAlTest1_1() : Tester()
{
}

void ImineEtAlTest1_1::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(1);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 0, 1, "x");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 2, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,2);
    Message * messageC1 = new Message (Message::Insert, 3, 3, 1, "y");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);
    Message * messageD1 = new Message (Message::Insert, 4, 1, 0, "abc");
    messageD1->setFrom(0,0);
    messageD1->setTo(0,3);

    rtc = handle_do(messageD1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageD1->type(), Message::Insert );
    expected = "abc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "axbc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );
    expected = "axc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::NoOp );
    expected = "axc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 4);
    CHECK( rtc, 0 );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ImineEtAlTest1_1 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ImineEtAlTest1_2::ImineEtAlTest1_2() : Tester()
{
}

void ImineEtAlTest1_2::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(2);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 3, 1, "x");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 0, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,2);
    Message * messageC1 = new Message (Message::Insert, 3, 4, 1, "y");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);
    Message * messageD1 = new Message (Message::Insert, 4, 1, 0, "abc");
    messageD1->setFrom(0,0);
    messageD1->setTo(0,3);

    rtc = handle_do(messageD1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageD1->type(), Message::Insert );
    expected = "abc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );
    expected = "ac";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 2);
    CHECK( rtc, 0 );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::NoOp );
    expected = "ac";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "axc";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ImineEtAlTest1_2 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

ImineEtAlTest1_3::ImineEtAlTest1_3() : Tester()
{
}

void ImineEtAlTest1_3::allTests()
{
    Document doc(0);
    TextOwner owners(1);

    QPtrList<Message> pending_list;

    MessageQueue messageQueue;
    messageQueue.setLocalPc(3);
    int rtc;
    QString text;
    QString expected;
    QString result;

    Message * messageA1 = new Message (Message::Insert, 1, 4, 1, "x");
    messageA1->setFrom(0,1);
    messageA1->setTo(0,2);
    Message * messageB1 = new Message (Message::Delete, 2, 2, 1);
    messageB1->setFrom(0,1);
    messageB1->setTo(0,2);
    Message * messageC1 = new Message (Message::Insert, 3, 0, 1, "y");
    messageC1->setFrom(0,2);
    messageC1->setTo(0,3);
    Message * messageD1 = new Message (Message::Insert, 4, 1, 0, "abc");
    messageD1->setFrom(0,0);
    messageD1->setTo(0,3);

    rtc = handle_do(messageD1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageD1->type(), Message::Insert );
    expected = "abc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageC1->type(), Message::Insert );
    expected = "abyc";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageB1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageB1->type(), Message::Delete );
    expected = "ac";
    result = doc.at(0);
    CHECK( result, expected );

    rtc = handle_do(messageC1, doc, owners, messageQueue.localPc(), &messageQueue, 3);
    CHECK( rtc, 0 );

    rtc = handle_do(messageA1, doc, owners, messageQueue.localPc(), &messageQueue, 0);
    CHECK( rtc, 0 );
    CHECK( (Message::Type)messageA1->type(), Message::Insert );
    expected = "axc";
    result = doc.at(0);
    CHECK( result, expected );

#if 0
    pending_list.clear();
    Message * undo_message = messageQueue.undoMessage( messageB3->id(), messageB3->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f4";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB2->id(), messageB2->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "a2f";
    result = doc.at(0);
    CHECK( result, expected );

    pending_list.clear();
    undo_message = messageQueue.undoMessage( messageB1->id(), messageB1->sender(), pending_list );

    handle_undo(undo_message, doc, pending_list, messageQueue.localPc());

    expected = "ABCcD";
    result = doc.at(0);
    CHECK( result, expected );
#endif

    text = "ImineEtAlTest1_3 Expected: >" + expected + "< Result: >" + result + "<";
    messageQueue.debug(text);
}

MessageCoordsShiftUpTest::MessageCoordsShiftUpTest() : Tester()
{
}

void MessageCoordsShiftUpTest::allTests()
{
    MessageCoords coords3_3(3,3);
    MessageCoords cmp_coords3_3(3,3);
    MessageCoords cmp_coords5_4(5,4);
    MessageCoords cmp_coords3_4(3,4);
    MessageCoords cmp_coords3_8(3,8);
    MessageCoords cmp_coords5_0(5,0);

    coords3_3.shiftUp ( MessageCoords(1,0), MessageCoords(1,1) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftUp ( MessageCoords(1,0), MessageCoords(3,1) );
    CHECK( coords3_3, cmp_coords5_4 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftUp ( MessageCoords(3,1), MessageCoords(3,2) );
    CHECK( coords3_3, cmp_coords3_4 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftUp ( MessageCoords(3,1), MessageCoords(3,6) );
    CHECK( coords3_3, cmp_coords3_8 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftUp ( MessageCoords(3,3), MessageCoords(5,0) );
    CHECK( coords3_3, cmp_coords5_0 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftUp ( MessageCoords(3,4), MessageCoords(5,0) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftUp ( MessageCoords(4,0), MessageCoords(4,1) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftUp ( MessageCoords(4,1), MessageCoords(5,1) );
    CHECK( coords3_3, cmp_coords3_3 );
}

MessageCoordsShiftDownTest::MessageCoordsShiftDownTest() : Tester()
{
}

void MessageCoordsShiftDownTest::allTests()
{
    MessageCoords coords3_3(3,3);
    MessageCoords cmp_coords3_3(3,3);
    MessageCoords cmp_coords2_3(2,3);
    MessageCoords cmp_coords1_3(1,3);
    MessageCoords cmp_coords1_7(1,7);
    MessageCoords cmp_coords1_2(1,2);
    MessageCoords cmp_coords1_1(1,1);
    MessageCoords cmp_coords3_1(3,1);
    MessageCoords cmp_coords3_2(3,2);

    coords3_3.shiftDown ( MessageCoords(1,0), MessageCoords(1,1) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftDown ( MessageCoords(1,0), MessageCoords(2,0) );
    CHECK( coords3_3, cmp_coords2_3 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,0), MessageCoords(2,1) );
    CHECK( coords3_3, cmp_coords2_3 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,0), MessageCoords(3,0) );
    CHECK( coords3_3, cmp_coords1_3 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,5), MessageCoords(3,1) );
    CHECK( coords3_3, cmp_coords1_7 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,0), MessageCoords(3,1) );
    CHECK( coords3_3, cmp_coords1_2 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,1), MessageCoords(3,1) );
    CHECK( coords3_3, cmp_coords1_3 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(1,1), MessageCoords(3,4) );
    CHECK( coords3_3, cmp_coords1_1 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(3,1), MessageCoords(5,2) );
    CHECK( coords3_3, cmp_coords3_1 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(3,0), MessageCoords(3,1) );
    CHECK( coords3_3, cmp_coords3_2 );
    coords3_3.setCoords(3,3);

    coords3_3.shiftDown ( MessageCoords(3,4), MessageCoords(5,0) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftDown ( MessageCoords(4,0), MessageCoords(4,1) );
    CHECK( coords3_3, cmp_coords3_3 );

    coords3_3.shiftDown ( MessageCoords(4,1), MessageCoords(5,1) );
    CHECK( coords3_3, cmp_coords3_3 );
}

DocumentTest1::DocumentTest1() : Tester()
{
}

void DocumentTest1::allTests()
{
    Document doc(0);

    QString text("abcdefg");
    MessageCoords start(0,0);
    MessageCoords end(0,7);

    doc.insert( start, end, 1, text);

    QString result = doc.at(0);
    CHECK( result, text);

    start.setCoords(0,2);
    end.setCoords(0,5);

    QString expected("cde");
    result = doc.remove( start, end );
    CHECK( result, expected);
}

DocumentTest2::DocumentTest2() : Tester()
{
}

void DocumentTest2::allTests()
{
    Document doc(0);

    QString text("abcdefg\n1234\n3456");
    MessageCoords start(0,0);
    MessageCoords end(2,2);

    doc.insert( start, end, 1, text);

    QString expected = "abcdefg";
    QString result = doc.at(0);
    CHECK( result, expected );

    expected = "1234";
    result = doc.at(1);
    CHECK( result, expected );

    expected = "3456";
    result = doc.at(2);
    CHECK( result, expected );

    text = "\n\n";
    start.setCoords(1,4);
    end.setCoords(3,0);

    doc.insert ( start, end, 1, text );

    expected = "abcdefg";
    result = doc.at(0);
    CHECK( result, expected );

    expected = "1234";
    result = doc.at(1);
    CHECK( result, expected );

    expected = "";
    result = doc.at(2);
    CHECK( result.isEmpty(), expected.isEmpty() );

    expected = "";
    result = doc.at(3);
    CHECK( result.isEmpty(), expected.isEmpty() );

    expected = "3456";
    result = doc.at(4);
    CHECK( result, expected );

    expected = "45";
    start.setCoords(4,1);
    end.setCoords(4,3);
    result = doc.remove( start, end );
    CHECK( result, expected );

    expected = "\n\n";
    start.setCoords(1,4);
    end.setCoords(3,0);
    result = doc.remove( start, end );
    CHECK( result, expected );

    expected = "abcdefg\n1234";
    start.setCoords(0,0);
    end.setCoords(1,4);
    result = doc.remove( start, end );
    CHECK( result, expected );

    expected = "";
    result = doc.at(0);
    CHECK( result.isEmpty(), expected.isEmpty() );

    expected = "36";
    result = doc.at(1);
    CHECK( result, expected );
}

int handle_do(Message * message, Document & doc, TextOwner & owners, int local_pc, MessageQueue * messageQueue, int server_state)
{
    int rtc;
    if ( message != NULL )
    {
        if( server_state > 0)
        {
            if( message->serverState() == 0)
            {
                message->setServerState( server_state );
            }
            else
            {
                return -1;
            }
        }

        rtc = messageQueue->processMessage( message );
        if (rtc < 0)
        {
            if( server_state > 0 )
            {
                rtc = 0;
            }
        }

        if( server_state == 0 )
        {
            if ( message->type() == Message::Insert )
            {
                doc.insert( message->from( local_pc ), message->to( local_pc ), local_pc, message->text() );
                owners.insert( message->from( local_pc ), message->to( local_pc ), local_pc, message->text() );
            }
            if ( message->type() == Message::Delete )
            {
                QString text = doc.remove( message->from( local_pc ), message->to( local_pc ) );
                message->setText( text );
                owners.remove( message->from( local_pc ), message->to( local_pc ) );
            }
        }
    }
    return rtc;
}


int handle_undo(Message * undo_message, Document & doc, QPtrList<Message> & pending_list, int local_pc)
{
    if ( undo_message != NULL )
    {
        if ( undo_message->type() == Message::Insert )
        {
            doc.insert( undo_message->from( local_pc ), undo_message->to( local_pc ), local_pc, undo_message->text() );
        }
        if ( undo_message->type() == Message::Delete )
        {
            QString text = doc.remove( undo_message->from( local_pc ), undo_message->to( local_pc ) );
            undo_message->setText( text );
        }
        QPtrListIterator<Message> pending_it( pending_list );
        Message * pending_message;
        for ( pending_message = pending_it.toFirst(); pending_message; ++pending_it, pending_message = pending_it.current() )
        {
            if ( pending_message->type() == Message::Insert )
            {
                doc.insert( pending_message->from( local_pc ), pending_message->to( local_pc ), local_pc, pending_message->text() );
            }
            if ( pending_message->type() == Message::Delete )
            {
                QString text = doc.remove( pending_message->from( local_pc ), pending_message->to( local_pc ) );
                pending_message->setText( text );
            }
        }
    }
    pending_list.clear();
    return 0;
}
