/***************************************************************************
 *   Copyright (C) 2004-2005 by Andreas Ramm                               *
 *   psychobrain@gmx.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qstring.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qsocket.h>
#include <qptrlist.h>
#include <qtimer.h>

#include <kmessagebox.h>
#include <klocale.h>

#include <qbuffer.h>
#include "message.h"

namespace MateEdit
{

    MessageCoords::MessageCoords()
    {
        m_para = 0;
        m_idx = 0;
    }

    MessageCoords::MessageCoords( unsigned int a, unsigned int b )
    {
        m_para = a;
        m_idx = b;
    }

    MessageCoords::MessageCoords( const MessageCoords & coords  )
    {
        m_para = coords.para();
        m_idx = coords.idx();
    }


    void MessageCoords::setCoords( unsigned int a, unsigned int b )
    {
        m_para = a;
        m_idx = b;
    }

    void MessageCoords::setCoords( const MessageCoords & coords  )
    {
        m_para = coords.para();
        m_idx = coords.idx();
    }

    bool MessageCoords::operator==( const MessageCoords &b ) const
    {
        return m_para == b.m_para && m_idx == b.m_idx;
    }

    bool MessageCoords::operator!=( const MessageCoords &b ) const
    {
        return m_para != b.m_para || m_idx != b.m_idx;
    }

    bool MessageCoords::operator<=( const MessageCoords &b ) const
    {
        return m_para < b.m_para || ( m_para == b.m_para && m_idx <= b.m_idx );
    }

    bool MessageCoords::operator<( const MessageCoords &b ) const
    {
        return m_para < b.m_para || ( m_para == b.m_para && m_idx < b.m_idx );
    }

    bool MessageCoords::operator>=( const MessageCoords &b ) const
    {
        return m_para > b.m_para || ( m_para == b.m_para && m_idx >= b.m_idx );
    }

    bool MessageCoords::operator>( const MessageCoords &b ) const
    {
        return m_para > b.m_para || ( m_para == b.m_para && m_idx > b.m_idx );
    }

    int MessageCoords::shiftUp( const MessageCoords &b, const MessageCoords &c )
    {
        if ( b > c )
        {
            return -2;
        }
        if ( operator< (b) )
        {
            return -1;
        }
        if ( m_para > c.m_para)
        {
            m_para += c.m_para - b.m_para;
        }
        else
        {
            if ( ( m_para == b.m_para ) && ( b.m_para == c.m_para ) && ( m_idx >= b.m_idx ) )
            {
                m_idx += c.m_idx - b.m_idx;
            }
            else if ( ( m_para == b.m_para ) && ( m_para < c.m_para ) )
            {
                m_idx -= b.m_idx;
            }
            else if ( ( m_para > b.m_para ) && ( m_para == c.m_para ) )
            {
                m_idx += c.m_idx;
            }
            m_para += c.m_para - b.m_para;
        }
        return 0;
    }

    int MessageCoords::shiftDown( const MessageCoords &b, const MessageCoords &c )
    {
        if ( b > c )
        {
            return -2;
        }
        if ( operator< (b) )
        {
            return -1;
        }
        if ( m_para > c.m_para)
        {
            m_para -= c.m_para - b.m_para;
        }
        else
        {
            if ( ( m_para == b.m_para ) && ( b.m_para == c.m_para ) && ( m_idx > c.m_idx ) )
            {
                m_idx -= c.m_idx - b.m_idx;
                m_para -= c.m_para - b.m_para;
            }
            else if ( ( operator>= (b) ) && ( operator<= (c) ) )
            {
                m_idx = b.m_idx;
                m_para = b.m_para;
            }
            else if ( ( m_para > b.m_para ) &&  ( m_para == c.m_para ) && ( m_idx >= c.m_idx ) )
            {
                m_idx += b.m_idx - c.m_idx;
                m_para -= c.m_para - b.m_para;
            }
        }
        return 0;
    }

    unsigned int MessageCoords::para() const
    {
        return m_para;
    }

    unsigned int MessageCoords::idx() const
    {
        return m_idx;
    }

    MessageCoords & Message::from( unsigned int sender )
    {
        return m_fromCoords[sender];
    }

    MessageCoords & Message::originalFrom()
    {
        return m_originalFromCoords;
    }

    MessageCoords & Message::to( unsigned int sender )
    {
        return m_toCoords[sender];
    }


    #ifndef QT_NO_DATASTREAM

    QDataStream & operator<<( QDataStream & ostream, const MessageCoords & b )
    {
        return ostream << (Q_INT32)b.para() << (Q_INT32)b.idx();
    }

    QDataStream & operator>>( QDataStream & istream, MessageCoords & b )
    {
        Q_INT32 i1, i2;
        istream >> i1 >> i2;
        b.setCoords( i1, i2 );
        return istream;
    }
    #endif

    QTextStream & operator<<( QTextStream & ostream, const MessageCoords & b )
    {
        return ostream << (Q_INT32)b.para() << ":" << (Q_INT32)b.idx();
    }

    Message::Message()
    {
        m_type = Undefined;
        m_original_type = Undefined;
        m_sender = 0;
        m_state.server = 0;
        m_state.client = 0;
        m_id = 0;
        QString message = QString::null;
        m_originalFromCoordsSet = false;
        m_image = NULL;
        m_affected_list = NULL;
        m_undone = false;
    }

    Message::Message( Type type, unsigned int sender, unsigned int server_state, unsigned int client_state, const QString & buffer )
    {
        this->m_type = type;
        this->m_original_type = type;
        this->m_sender = sender;
        this->m_state.server = server_state;
        this->m_state.client = client_state;
        this->m_message = buffer;
        this->m_message.replace(QChar ('\r') ,QChar('\n'));
        m_originalFromCoordsSet = false;
        this->m_image = NULL;
        m_id = 0;
        m_affected_list = NULL;
        m_undone = false;
    }

    Message::~Message()
    {
        delete m_image;
    }

    unsigned int Message::type()
    {
        return m_type;
    }

    unsigned int Message::sender()
    {
        return m_sender;
    }

    unsigned int Message::serverState()
    {
        return m_state.server;
    }

    unsigned int Message::clientState()
    {
        return m_state.client;
    }

    QString & Message::text()
    {
        return m_message;
    }

    QImage & Message::image ()
    {
        return *m_image;
    }

    unsigned int Message::id ()
    {
        return m_id;
    }

    unsigned int Message::parentId ()
    {
        return m_parent_id;
    }

    bool Message::undone ()
    {
        return m_undone;
    }

    void Message::setType( Type type )
    {
        if ( m_original_type == Undefined )
        {
            m_original_type = m_type;
        }
        m_type = type;
    }

    void Message::setSender( unsigned int new_sender )
    {
        m_sender = new_sender;
    }

    void Message::setServerState( unsigned int new_server_state )
    {
        m_state.server = new_server_state;
    }

    void Message::setClientState( unsigned int new_client_state )
    {
        m_state.client = new_client_state;
    }

    void Message::setText( QString & new_message )
    {
        m_message = new_message;
        this->m_message.replace(QChar ('\r') ,QChar('\n'));
    }

    void Message::setId ( unsigned int id )
    {
        m_id = id;
    }

    void Message::setImage ( const QImage & new_image )
    {
        delete m_image;
        m_image = new QImage (new_image);
    }

    void Message::setFrom( unsigned int fromPara, unsigned int fromIdx )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].setCoords( fromPara, fromIdx );
        }
        if ( ! m_originalFromCoordsSet )
        {
            m_originalFromCoords.setCoords( fromPara, fromIdx );
            m_originalFromCoordsSet = true;
        }

    }

    void Message::setFrom( const MessageCoords & coords )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].setCoords( coords );
        }
        if ( ! m_originalFromCoordsSet )
        {
            m_originalFromCoords.setCoords( coords );
            m_originalFromCoordsSet = true;
        }

    }

    void Message::setTo( unsigned int toPara, unsigned int toIdx )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_toCoords[i].setCoords( toPara, toIdx );
        }
    }

    void Message::setTo( const MessageCoords & coords )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_toCoords[i].setCoords( coords );
        }
    }


    void Message::shiftUpFrom ( const MessageCoords &b, const MessageCoords &c )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].shiftUp ( b , c );
        }
    }

    void Message::shiftUpFrom ( unsigned int sender, const MessageCoords &b, const MessageCoords &c )
    {
        m_fromCoords[sender].shiftUp ( b , c );
    }

    void Message::shiftUpTo ( const MessageCoords &b, const MessageCoords &c )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_toCoords[i].shiftUp ( b , c );
        }
    }

    void Message::shiftUpTo ( unsigned int sender, const MessageCoords &b, const MessageCoords &c )
    {
        m_toCoords[sender].shiftUp ( b , c );
    }


    void Message::shiftDownFrom ( const MessageCoords &b, const MessageCoords &c )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].shiftDown ( b , c );
        }
    }

    void Message::shiftDownFrom ( unsigned int sender, const MessageCoords &b, const MessageCoords &c )
    {
        m_fromCoords[sender].shiftDown ( b , c );
    }

    void Message::shiftDownTo ( const MessageCoords &b, const MessageCoords &c )
    {
        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_toCoords[i].shiftDown ( b , c );
        }
    }

    void Message::shiftDownTo ( unsigned int sender, const MessageCoords &b, const MessageCoords &c )
    {
        m_toCoords[sender].shiftDown ( b , c );
    }

    void Message::addAffected ( Message * message )
    {
        if ( ! m_affected_list )
        {
            m_affected_list = new QPtrList<Message>;
        }
        m_affected_list->append( message );
    }

    QPtrList<Message> * Message::affectedList()
    {
        return m_affected_list;
    }

    void Message::createUndo ( Message * message, int local_pc )
    {
        if ( message->m_original_type == Insert || (message->m_original_type == Undefined && message->m_type == Insert) )
        {
            m_original_type = Delete;
            m_type = Delete;
        }
        else if ( message->m_original_type == Delete || (message->m_original_type == Undefined && message->m_type == Delete) )
        {
            m_original_type = Insert;
            m_type = Insert;
        }
        if ( message->m_type == NoOp )
        {
            message->setType( NoOp );
        }

        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].setCoords( message->m_fromCoords[i] );
        }
        m_originalFromCoords.setCoords( message->m_originalFromCoords );
        m_originalFromCoordsSet = true;

        for ( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_toCoords[i].setCoords( message-> m_toCoords[i] );
        }

        m_message = message->m_message;

        m_sender = message->m_sender;

        m_parent_id = message->m_id;

        message->m_undone = true;
    }

    void Message::send( KNetwork::KActiveSocketBase * socket )
    {
        QBuffer buffer;
        QDataStream stream(&buffer);
        buffer.open(IO_WriteOnly);
        stream << m_type << m_fromCoords[0] << m_toCoords[0] << m_sender << m_state.server << m_state.client;

        if ( m_type == Insert || m_type == Init || m_type == InitUserData || m_type == InfoMessage || m_type == ChatMessage )
        {
            stream << m_message;
        }

        if ( m_type == InitUserData )
        {
            stream << *m_image;
        }

        if ( m_type == Undo )
        {
            stream << m_id;
        }

        stream.setDevice( socket );
        stream << buffer.buffer().size();
        socket->writeBlock(buffer.buffer().data(), buffer.buffer().size());
        socket->flush();
    };

    void Message::read( QBuffer * buffer  )
    {
        QDataStream stream ( buffer );
        buffer->open(IO_ReadOnly);
        // To Do: Change this to be architecture independent
        MessageCoords a;

        stream >> m_type >> m_fromCoords[0] >> m_toCoords[0] >> m_sender >> m_state.server >> m_state.client;

        m_original_type = m_type;

        for ( int i = 1; i < MATEEDIT_MAX_USERS; i++ )
        {
            m_fromCoords[i].setCoords( m_fromCoords[0] );
            m_toCoords[i].setCoords( m_toCoords[0] );
        }

        if ( m_type == Insert || m_type == Init || m_type == InitUserData || m_type == InfoMessage || m_type == ChatMessage )
        {
            stream >> m_message;
        }

        if ( m_type == InitUserData )
        {
            QImage image;
            stream >> image;
            this->setImage( image );
        }

        if ( m_type == Undo )
        {
            stream >> m_id;
        }
    };

    void Message::restoreType()
    {
        m_type = m_original_type;
    };

    void Message::debug( const char *file, int line, const char * comment = "")
    {
        QFile f("debugoutput.txt");
        f.open(IO_WriteOnly | IO_Append);

        QTextStream stream( &f );

        stream << ">-------- " << comment << endl;

        stream << file << ":" << line << endl;

        stream << m_sender << "-" << m_state.server << "-" << m_state.client << "-" << m_type << "| 1 - " << m_fromCoords[1] << "--" << m_toCoords[1] << "| 2 - " << m_fromCoords[2] << "--" << m_toCoords[2] << "| 3 - " << m_fromCoords[3] << "--" << m_toCoords[3] << "| 4 - " << m_fromCoords[4] << "--" << m_toCoords[4] << "| 5 - " << m_fromCoords[5] << "--" << m_toCoords[5] << "|"  << m_id  << "|"  << m_parent_id  << "|"  << m_undone << endl;

//        if ( m_type == Insert || m_type == Init || m_type == InitUserData || m_type == InfoMessage || m_type == ChatMessage || m_type == Delete )
        {
            stream << "<<" << m_message << ">>" << endl;
        }
        stream << "--------<" <<endl;

        f.close();
    }



    MessageQueue::MessageQueue()
    : m_client_it(MATEEDIT_MAX_USERS)
    {
        m_message_list = new QPtrList<Message> ;
        m_message_list->setAutoDelete( true );
        m_pending_message_list = new QPtrList<Message> ;
        m_shifted_ops = new QPtrList<Shift>;
//        m_client_it
        m_current_message_ids = new unsigned int[MATEEDIT_MAX_USERS];
        for(int i = 0; i < MATEEDIT_MAX_USERS; ++i)
        {
            QPtrListIterator<Message> * it = new QPtrListIterator<Message>( *m_message_list ) ;
            m_client_it.insert( i, it );
            m_current_message_ids[i] = 0;
        }
    }

    MessageQueue::~MessageQueue()
    {
//        delete[] m_client_it;
        delete m_pending_message_list ;
        delete m_message_list;
        delete m_current_message_ids;
        delete m_shifted_ops;
    }

    void MessageQueue::appendProcessed( const Message * message )
    {
        m_message_list->append( message );
    };

    void MessageQueue::clearProcessed()
    {
        m_message_list->clear();
    };

    void MessageQueue::setLocalPc(Q_UINT32 pc)
    {
        m_local_pc = pc;
    };

    Q_UINT32 MessageQueue::localPc()
    {
        return m_local_pc;
    };

    void MessageQueue::debug(QString text)
    {
        QFile f("debugoutput.txt");
        f.open(IO_WriteOnly | IO_Append);

        QTextStream stream( &f );

        stream << "Start: " << text <<endl;

        f.close();

        QPtrListIterator<Message> process_it( *m_message_list );

        for ( Message * queue_message = (Message * )process_it; queue_message; queue_message = (Message * )++process_it )
        {
            queue_message->debug(__FILE__, __LINE__);
        }

        f.open(IO_WriteOnly | IO_Append);

        QPtrListIterator<Shift> it( *m_shifted_ops );
        Shift * shift;
        while ( (shift = it.current()) != NULL )
        {
            stream <<
            "type" <<
            shift->type() << " : " <<
            "queueSender" <<
            shift->queueSender() << " : " <<
            "queueId" <<
            shift->queueId() << " : " <<
            "serverSender" <<
            shift->serverSender() << " : " <<
            "serverId" <<
            shift->serverId() << " : " <<
            "serverFrom" <<
            shift->serverFrom() << " : " <<
            "index" <<
            shift->index() << " : " <<
            "nChars" <<
            shift->nChars() << " : " <<
            "offset" <<
            shift->offset() << " : " << endl;
        ++it;
        }

        stream << "End: " << text <<endl;

        f.close();
    }

    Message * MessageQueue::findMessage( unsigned int id, unsigned int sender )
    {
        Message * message;

        if ( sender >= MATEEDIT_MAX_USERS )
        {
            return NULL;
        }

        QPtrListIterator<Message> process_it( *m_message_list );

        while ( ( (message = process_it.current()) != NULL ) && ( (message->id() != id ) || (message->sender() != sender ) ) )
        {
            ++process_it;
        }

        return message;
    }

    Message * MessageQueue::undoMessage( unsigned int id, unsigned int sender, QPtrList<Message> & pending_list, int server_state, int client_state )
    {
        int rtc = 0;
        Message * message;
        Message * queue_message;
        Message * undo_message = NULL;

        if ( sender >= MATEEDIT_MAX_USERS )
        {
            return NULL;
        }

        QPtrListIterator<Message> process_it( *m_message_list );

        while ( ( (message = process_it.current()) != NULL ) && ( (message->id() != id ) || (message->sender() != sender ) ) )
        {
            ++process_it;
        }

        QPtrListIterator<Message> store_it( *m_message_list );
        store_it = process_it;

        if ( message == NULL )
        {
            return NULL;
        }

        undo_message = new Message();

        if ( undo_message == NULL )
        {
            return NULL;
        }

        undo_message->createUndo( message, m_local_pc );
        undo_message->setServerState( server_state );

    //    message->debug(__FILE__, __LINE__);

        for ( queue_message = (Message * )++process_it; queue_message; queue_message = (Message * )++process_it )
        {
            if ( ( undo_message->type() != Message::NoOp && queue_message->type() != Message::NoOp) || ( undo_message->type() != Message::NoOp && queue_message->type() == Message::NoOp && queue_message->undone()) )
            {
                unsigned int type = queue_message->type();
                queue_message->restoreType();
                queue_message->debug(__FILE__, __LINE__,"Queue message undo");
                undo_message->debug(__FILE__, __LINE__, "Undo message before transformation");
                rtc = transformMessage ( *queue_message, *undo_message, false, true );
                undo_message->debug(__FILE__, __LINE__, "Undo message after transformation");
                queue_message->setType((Message::Type)type);
            }
        }

        if ( message->affectedList() && !message->affectedList()->isEmpty())
        {
            QPtrListIterator<Message> affected_it( *message->affectedList() );
            Message * affected_message;
            for ( affected_message = affected_it.toFirst(); affected_message; ++affected_it, affected_message = affected_it.current() )
            {
                pending_list.append( affected_message );
                // We only have NoOp's so we start after the one that cause it
                process_it = store_it;
                for ( queue_message = (Message * )++process_it; queue_message; queue_message = (Message * )++process_it )
                {
                    affected_message->restoreType();
                    if ( ( affected_message->type() != Message::NoOp ) && ( affected_message != queue_message ) && ( message->sender() != queue_message->sender() ) )
                    {
                        queue_message->debug(__FILE__, __LINE__,"Queue message affected");
                        affected_message->debug(__FILE__, __LINE__, "Affected message before transformation");
                        rtc = transformMessage ( *queue_message, *affected_message, true );
                        affected_message->debug(__FILE__, __LINE__, "Affected message after transformation");
                    }
                }
                affected_message->debug(__FILE__, __LINE__, "Affected message done");
            }
        }
        message->setType( Message::NoOp );
        appendProcessed( undo_message );
        return undo_message;
    }


    int MessageQueue::processMessage( Message * message )
    {
        int rtc = 0;
        Message * queue_message;

        if ( message == NULL )
        {
            return -3;
        }

        if ( message->sender() >= MATEEDIT_MAX_USERS )
        {
            message->setType( Message::Undefined );
            return -2;
        }

        if ( message->sender() == m_local_pc && message->serverState() != 0 ) //This message is already applied but server state needs to be updated
        {
            QPtrListIterator<Message> process_it( *m_message_list );

            if ( m_client_it[ message->sender() ]->current() == NULL )
            {
                m_client_it[ message->sender() ]->toFirst();
            }
            process_it = *(m_client_it[ message->sender() ]);

            queue_message = (Message * )process_it;
            while ( queue_message && queue_message->serverState() != 0 )
            {
                ++process_it;
                queue_message = (Message * )process_it;
            }
            if ( queue_message )
            {
                queue_message->setServerState( message->serverState() );
            }
            return -1;
        }

        if ( message->id() == 0 )
        {
            ++m_current_message_ids[message->sender()];
            message->setId( m_current_message_ids[message->sender()] );
        }
        bool advance_iterator = true;



        QPtrListIterator<Message> process_it( *m_message_list );

        if ( m_client_it[ message->sender() ]->current() == NULL )
        {
            m_client_it[ message->sender() ]->toFirst();
        }
        process_it = *(m_client_it[ message->sender() ]);
        QPtrListIterator<Message> & current_iterator = *(m_client_it[ message->sender() ]);

        message->debug(__FILE__, __LINE__, "Server message");

        QPtrList<Message> to_end_of_list;
        for ( queue_message = (Message * )process_it; queue_message; queue_message = (Message * )++process_it )
        {
            if ( ( message->type() != Message::NoOp )
                   && ( ( queue_message->clientState() >= message->clientState() ) || ( queue_message->serverState() == 0 ) || (queue_message->serverState() > message->clientState()) ) )
            {
                advance_iterator = false;
                if( queue_message->serverState() == 0 && queue_message->clientState() > message->clientState() )
                {
                    Message * taken_msg = m_message_list->take();
                    to_end_of_list.append(taken_msg);
                    taken_msg->debug(__FILE__, __LINE__,"Taken message");
                }
                else
                {
                    queue_message->debug(__FILE__, __LINE__,"Queue message");
                    message->debug(__FILE__, __LINE__, "Server message before transformation");
                    rtc = transformMessage ( *queue_message, *message );
                    message->debug(__FILE__, __LINE__, "Server message after transformation");
                }
            }
            else
            {
                queue_message->debug(__FILE__, __LINE__,"Skipped message");
                if ( advance_iterator )
                {
                    if ( ! current_iterator.atLast() )
                    {
                        ++current_iterator;
                    }
                }
            }
        }

        appendProcessed( message );

        Message * transfer_msg;
        while( transfer_msg = to_end_of_list.take() )
        {
            if( transfer_msg->type() != Message::NoOp )
            {
                transfer_msg->debug(__FILE__, __LINE__,"Appended queue message");
                message->debug(__FILE__, __LINE__, "Server message before transformation");
                rtc = transformMessage ( *transfer_msg, *message );
                m_message_list->append(transfer_msg);
                message->debug(__FILE__, __LINE__, "Server message after transformation");
            }
        }
        return rtc;
    }

    //If you are reading this you should be contacting me directly and I can give you more information on how this works. :) Hopefully soon, I will integrate that information here though.
    int MessageQueue::transformMessage ( Message & queueMessage, Message & serverMessage, bool undo, bool ignore_ps )
    {
        unsigned int queue_message_sender = queueMessage.sender();
        unsigned int queue_message_type   = queueMessage.type();
        MessageCoords & queue_message_from = queueMessage.from( serverMessage.sender() );
        MessageCoords & queue_message_to = queueMessage.to( serverMessage.sender() );

        unsigned int server_message_sender = serverMessage.sender();
        unsigned int server_message_type   = serverMessage.type();
        MessageCoords & server_message_from = serverMessage.from( serverMessage.sender() );
        MessageCoords & server_message_to = serverMessage.to( serverMessage.sender() );


        if ( ( queue_message_sender == server_message_sender ) && ( ! undo ) && ( ! ignore_ps ) )    /* Messages from same client already have relevant transformations applied */
        {
            return 0;
        }
        if ( queue_message_type == Message::Insert && server_message_type == Message::Insert )
        {
            if ( queue_message_from < server_message_from )    //New insert position behind queued one? // 1,2,6
            {
                QPtrListIterator<Shift> it( *m_shifted_ops );
                Shift * shift;
                while ( ( (shift = it.current()) != NULL ) && ((shift->queueSender() != queueMessage.sender() || 0 != queueMessage.id())|| (shift->serverSender() != serverMessage.sender() || shift->serverId() != serverMessage.parentId() ) ) )
                {
                    qDebug("shift->queueId() %i:%i != queueMessage.id() %i:%i ) && (shift->serverId() %i:%i != serverMessage.parentId() %i:%i", shift->queueSender(), shift->queueId(), queueMessage.sender(), queueMessage.id(), shift->serverSender(), shift->serverId(), serverMessage.sender(), serverMessage.parentId() );
                    ++it;
                }
                qDebug("after shift");

                if( ignore_ps && shift && shift->type() == Shift::Q_LEFT_OF_S )
                {
                    qDebug("II1,2,6 if");
                }
                else
                {
                    serverMessage.shiftUpFrom( queue_message_from , queue_message_to );
                    serverMessage.shiftUpTo( queue_message_from , queue_message_to );
                }
                serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( ( ( queue_message_from >= server_message_from ) && ( queueMessage.originalFrom() <= server_message_from )  ) && ( queue_message_sender > server_message_sender ) )    //New insert position the same as queued one? Take "hierarchy" into account // 7, 8
            {
                if( ignore_ps && queueMessage.id() && serverMessage.id() )
                {
                    serverMessage.shiftUpFrom( queue_message_from , queue_message_to );
                    serverMessage.shiftUpTo( queue_message_from , queue_message_to );
                }
            }
            else if ( ( queue_message_from >= server_message_from ) && ( queueMessage.originalFrom() <= server_message_from ) ) // 7,8
            {
                QPtrListIterator<Shift> it( *m_shifted_ops );
                Shift * shift;
                while ( ( (shift = it.current()) != NULL ) && ((shift->queueSender() != queueMessage.sender() || 0 != queueMessage.id())|| (shift->serverSender() != serverMessage.sender() || shift->serverId() != serverMessage.parentId() ) ) )
                {
                    qDebug("shift->queueId() %i:%i != queueMessage.id() %i:%i ) && (shift->serverId() %i:%i != serverMessage.parentId() %i:%i", shift->queueSender(), shift->queueId(), queueMessage.sender(), queueMessage.id(), shift->serverSender(), shift->serverId(), serverMessage.sender(), serverMessage.parentId() );
                    ++it;
                }
                if( ignore_ps && serverMessage.id() == 0 && queueMessage.id() == 0 && shift && shift->type() == Shift::Q_LEFT_OF_S )
                {
                }
                else
                {
                    serverMessage.shiftUpFrom( queue_message_from , queue_message_to );
                    serverMessage.shiftUpTo( queue_message_from , queue_message_to );
                }
                serverMessage.debug(__FILE__, __LINE__);
            }
            else // 3,4,5
            {
                // no change
                QPtrListIterator<Shift> it( *m_shifted_ops );
                Shift * shift;
                while ( ( (shift = it.current()) != NULL ) && ((shift->serverSender() != queueMessage.sender() || 0 != queueMessage.id())|| (shift->queueSender() != serverMessage.sender() || shift->queueId() != serverMessage.parentId() ) ) )
                {
                    qDebug("shift->queueId() %i:%i != queueMessage.id() %i:%i ) && (shift->serverId() %i:%i != serverMessage.parentId() %i:%i", shift->serverSender(), 0, queueMessage.sender(), queueMessage.id(), shift->queueSender(), shift->queueId(), serverMessage.sender(), serverMessage.parentId() );
                    ++it;
                }
                qDebug("after shift");

                if( ignore_ps && shift && shift->type() == Shift::Q_LEFT_OF_S )
                {
                    qDebug("II1,2,6 if");
                }
                if(queueMessage.serverState() == 0)
                {
                    queueMessage.shiftUpFrom( server_message_from , server_message_to );
                    queueMessage.shiftUpTo( server_message_from , server_message_to );
                    queueMessage.debug(__FILE__, __LINE__);
                }
            }
        }
        else if ( queue_message_type == Message::Insert && server_message_type == Message::Delete )
        {
            if ( ( server_message_from < queue_message_from ) && ( server_message_to >= queue_message_from ) ) //3,5
            {
                if( ! ignore_ps || server_message_to > queue_message_from)
                {
                    serverMessage.shiftUpTo( server_message_sender, queue_message_from , queue_message_to );
                    if ( server_message_sender != m_local_pc )
                    {
                        serverMessage.shiftUpTo( m_local_pc, queue_message_from , queue_message_to );
                    }
                }
                serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( server_message_to < queue_message_from ) //4
            {
                //no change
            }
            else if ( server_message_from >= queue_message_from ) //1,2,6,7,8
            {
                if( queueMessage.id() != 0
                 || serverMessage.id() != 0
                 || (server_message_from < queue_message_to && serverMessage.originalFrom() > queue_message_to ) //)
                    || (queueMessage.parentId() == 0 && serverMessage.parentId() != 0) ) //InsertDelete breaks here FIXME
                {
                    serverMessage.shiftUpFrom( queue_message_from , queue_message_to );
                    serverMessage.shiftUpTo( queue_message_from , queue_message_to );
                    serverMessage.debug(__FILE__, __LINE__);
                }
            }
        }
        else if ( queue_message_type == Message::Delete && server_message_type == Message::Insert )
        {
            if ( server_message_from > queue_message_from && server_message_from > queue_message_to ) //1
            {
                serverMessage.shiftDownFrom( queue_message_from , queue_message_to );
                serverMessage.shiftDownTo( queue_message_from , queue_message_to );
    //            serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( ( server_message_from > queue_message_from && server_message_from <= queue_message_to )
                   || ( server_message_from >= queue_message_from && server_message_from <= queue_message_to && ignore_ps ) )//2,6
            {
                if( ignore_ps )
                {
                    QPtrListIterator<Shift> it( *m_shifted_ops );
                    Shift * shift;
                    while ( ( (shift = it.current()) != NULL ) && ((shift->queueSender() != queueMessage.sender() || shift->queueId() != queueMessage.id())|| (shift->serverSender() != serverMessage.sender() || shift->serverId() != serverMessage.parentId() ) ) )
                    {
                        qDebug("shift->queueId() %i:%i != queueMessage.id() %i:%i ) && (shift->serverId() %i:%i != serverMessage.parentId() %i:%i", shift->queueSender(), shift->queueId(), queueMessage.sender(), queueMessage.id(), shift->serverSender(), shift->serverId(), serverMessage.sender(), serverMessage.parentId() );
                        ++it;
                    }
                    qDebug("after shift");
                    if( shift && shift->type () == Shift::Q_LEFT_OF_S )
                    {
                        qDebug("n chars %i", shift->nChars());
                        QString text = serverMessage.text();
                        MessageCoords index( server_message_from );
                        Message * serverParentMessage = findMessage( serverMessage.parentId(), serverMessage.sender() );
                        MessageCoords serverParentIndex( serverParentMessage->from(serverMessage.sender() ) );
                        int i = 0;
                        QChar text_char;
                        QString new_text;
                        while( index <= queue_message_to )
                        {
                            text_char = text[i];
                            new_text += text_char;
                            if( text_char == '\n' )
                            {
                                index.setCoords( index.para() + 1, index.idx() );
                                serverParentIndex.setCoords( serverParentIndex.para() + 1, serverParentIndex.idx() );
                            }
                            else
                            {
                                index.setCoords( index.para(), index.idx() + 1 );
                                serverParentIndex.setCoords( serverParentIndex.para(), serverParentIndex.idx() + 1 );
                            }
                            ++i;
                        }
                        queueMessage.shiftUpTo( server_message_from, index );
                        QString tmp_string = queueMessage.text() + new_text;
                        queueMessage.setText( tmp_string );
                        serverMessage.setFrom( index );
                        tmp_string = text.mid( i );
                        serverMessage.setText( tmp_string );
                        serverMessage.debug(__FILE__, __LINE__);
                        queueMessage.debug(__FILE__, __LINE__);
                        serverParentMessage->debug(__FILE__, __LINE__);
                        serverParentMessage->setFrom( serverParentIndex );
                        serverParentMessage->setText( tmp_string );
                        serverParentMessage->debug(__FILE__, __LINE__);
                    }
                    if( shift && shift->type () == Shift::S_LEFT_OF_Q )
                    {
                        qDebug("n chars %i", shift->nChars());
                        if( queue_message_from < server_message_from && queue_message_to > server_message_from )
                        {
                            serverMessage.setType( Message::NoOp );  /* trying to insert in deleted area */
                            queueMessage.addAffected( &serverMessage );
                        }
                        else
                        {
                            QString text = serverMessage.text();
                            MessageCoords index( server_message_from );
                            Message * serverParentMessage = findMessage( serverMessage.parentId(), serverMessage.sender() );
                            MessageCoords serverParentIndex( serverParentMessage->from(serverMessage.sender() ) );
                            int i = 0;
                            QChar text_char;
                            QString new_text;
                            while( i < shift->offset() ) //index <= queue_message_to )
                            {
                                text_char = text[i];
                                if( text_char == '\n' )
                                {
                                    index.setCoords( index.para() + 1, index.idx() );
                                    serverParentIndex.setCoords( serverParentIndex.para() + 1, serverParentIndex.idx() );
                                }
                                else
                                {
                                    index.setCoords( index.para(), index.idx() + 1 );
                                    serverParentIndex.setCoords( serverParentIndex.para(), serverParentIndex.idx() + 1 );
                                }
                                ++i;
                            }
                            while( i < ( shift->offset() + shift->nChars() ) )
                            {
                                text_char = text[i];
                                new_text += text_char;
                                ++i;
                            }
                            queueMessage.shiftDownFrom( server_message_from, index );
                            QString tmp_string = new_text + queueMessage.text();
                            queueMessage.setText( tmp_string );
                            serverMessage.setTo( index );
                            tmp_string = text.left( shift->offset() );
                            serverMessage.setText( tmp_string );
                            serverMessage.debug(__FILE__, __LINE__);
                            queueMessage.debug(__FILE__, __LINE__);
                            serverParentMessage->debug(__FILE__, __LINE__);
                            serverParentMessage->setFrom( serverParentIndex );
                            serverParentMessage->setText( tmp_string );
                            serverParentMessage->debug(__FILE__, __LINE__);
                        }
                    }
                    if( shift && shift->type () == Shift::S_LEFT_OF_Q )
                    {
                        // do nothing
                    }
                    else
                    {
                        if( ! serverMessage.id() && server_message_from == queue_message_from )
                        {
                        }
                        else
                        {
                            serverMessage.shiftDownFrom( queue_message_from , queue_message_to );
                            serverMessage.shiftDownTo( queue_message_from , queue_message_to );
                        }
                    }
                }
                else
                {
                    queueMessage.shiftUpTo( server_message_sender, server_message_from, server_message_to );
                    serverMessage.setType( Message::NoOp );  /* redundant operation*/
                    queueMessage.addAffected( &serverMessage );
                }
                queueMessage.debug(__FILE__, __LINE__);
            }
            else
            {
                if(queueMessage.serverState() == 0)
                {
                    queueMessage.shiftUpFrom( server_message_from , server_message_to );
                    queueMessage.shiftUpTo( server_message_from , server_message_to );
                    queueMessage.debug(__FILE__, __LINE__);
                }
            }
        }
        else if ( queue_message_type == Message::Delete && server_message_type == Message::Delete )
        {
            if ( server_message_to <= queue_message_from )  //New delete position in front of queued one? //4
            {
                /* no change */
            }
            else if ( server_message_from > queue_message_to )  //New delete position behind queued one? //1
            {
                serverMessage.shiftDownFrom( queue_message_from , queue_message_to );
                serverMessage.shiftDownTo( queue_message_from , queue_message_to );
    //            serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( ( server_message_from >= queue_message_from ) && ( server_message_to <= queue_message_to ) ) //6,7 //New delete trying to delete already deleted area?
            {
//FIXME         if( ! (ignore_ps && queueMessage.parentId() && serverMessage.parentId() ) )
                {
                    QString text = serverMessage.text();
                    MessageCoords index( server_message_from );
                    int i = 0;
                    QChar text_char;
                    QString new_text;
                    while( index < queue_message_to  && index < server_message_to )
                    {
                        text_char = text[i];
                        new_text += text_char;
                        if( text_char == '\n' )
                        {
                            index.setCoords( index.para() + 1, index.idx() );
                        }
                        else
                        {
                            index.setCoords( index.para(), index.idx() + 1 );
                        }
                        ++i;
                    }

                    qDebug("id %i ", serverMessage.id());
                    if ( i > 0 )
                    {
                        Shift * shift = new Shift ( Shift::Q_LEFT_OF_S, queueMessage.sender(), queueMessage.id(), serverMessage.sender(), serverMessage.id(), server_message_from, index, i);
                        m_shifted_ops->append(shift);
                    }
                    queueMessage.shiftDownTo( server_message_sender, server_message_from, server_message_to );
                    serverMessage.setType( Message::NoOp );  /* redundant operation*/
                    queueMessage.addAffected( &serverMessage );
                    queueMessage.debug(__FILE__, __LINE__);
                }
            }
            else if ( ( server_message_from < queue_message_from ) && ( server_message_to < queue_message_to ) ) //3 //Overlapping deletes; new delete starts in front of queued one?
            {
                qDebug("DD3");
                QString text = serverMessage.text();
                int size = text.length();
                MessageCoords index( queue_message_from );
                int i = 0;
                QChar text_char;
                QString new_text;
                while( index < queue_message_to  && index < server_message_to )
                {
                    text_char = text[size - i - 1];
                    new_text += text_char;
                    if( text_char == '\n' )
                    {
                        index.setCoords( index.para() + 1, index.idx() );
                    }
                    else
                    {
                        index.setCoords( index.para(), index.idx() + 1 );
                    }
                    ++i;
                }

                qDebug("%i", i);
                if ( i > 0 )
                {
                    Shift * shift = new Shift ( Shift::Q_LEFT_OF_S, serverMessage.sender(), serverMessage.id(), queueMessage.sender(), queueMessage.id(), server_message_from, index, i );
                    m_shifted_ops->append(shift);
                    serverMessage.setTo( queue_message_from );
                }

                serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( ( server_message_from > queue_message_from ) && ( server_message_to > queue_message_to ) ) //2 //Overlapping deletes; queued delete starts in front of new one?
            {
                QString text = serverMessage.text();
                MessageCoords index( server_message_from );
                int i = 0;
                QChar text_char;
                QString new_text;
                while( index < queue_message_to  && index < server_message_to )
                {
                    text_char = text[i];
                    new_text += text_char;
                    if( text_char == '\n' )
                    {
                        index.setCoords( index.para() + 1, index.idx() );
                    }
                    else
                    {
                        index.setCoords( index.para(), index.idx() + 1 );
                    }
                    ++i;
                }

                qDebug("%i", i);
                if ( i > 0 )
                {
                    int offset = queueMessage.text().length() - i;
                    Shift * shift = new Shift ( Shift::S_LEFT_OF_Q, serverMessage.sender(), serverMessage.id(), queueMessage.sender(), queueMessage.id(), server_message_from, index, i, offset);
                    m_shifted_ops->append(shift);
                    serverMessage.setFrom( queue_message_to );
                }

                serverMessage.shiftDownFrom( queue_message_from , queue_message_to );
                serverMessage.shiftDownTo( queue_message_from , queue_message_to );
                queueMessage.debug(__FILE__, __LINE__);
                serverMessage.debug(__FILE__, __LINE__);
            }
            else if ( ( server_message_from <= queue_message_from ) && ( server_message_to >= queue_message_to ) )  //New delete overlapping on both sides of queued one? //5 and 8
            {
                QString text = serverMessage.text();
                int size = text.length();
                MessageCoords index( queue_message_from );
                int i = 0;
                QChar text_char;
                QString new_text;
                while( index < queue_message_to  && index < server_message_to )
                {
                    text_char = text[size - i - 1];
                    new_text += text_char;
                    if( text_char == '\n' )
                    {
                        index.setCoords( index.para() + 1, index.idx() );
                    }
                    else
                    {
                        index.setCoords( index.para(), index.idx() + 1 );
                    }
                    ++i;
                }

                qDebug("i %i", i);
                if ( i > 0 )
                {
                    Shift * shift = new Shift ( Shift::S_LEFT_OF_Q, serverMessage.sender(), serverMessage.id(), queueMessage.sender(), queueMessage.id(), server_message_from, index, i );
                    m_shifted_ops->append(shift);
                }
                qDebug("DD5-8");
                if(!ignore_ps)
                {
                    serverMessage.shiftDownTo( queue_message_sender, queue_message_from , queue_message_to );
                    if ( queue_message_sender != m_local_pc )
                    {
                        serverMessage.shiftDownTo( m_local_pc, queue_message_from , queue_message_to );
                    }
                }
                else
                {
                    serverMessage.shiftDownTo( m_local_pc, queueMessage.from( m_local_pc ) , queueMessage.to( m_local_pc ) );
                }
                serverMessage.debug(__FILE__, __LINE__);
            }
        }
        return 0;
    }
    MateEdit::MateEdit( Client * client, QTextEdit * textedit, bool init_app, bool init_user, bool remove_user, bool insert_text, bool delete_text, bool chat_message, bool message )
    {
        m_client = client;

        m_textedit = textedit;

        m_timestamp = 0;

        m_owners = new TextOwner(1);

        m_doc = new Document(0);

        m_syntaxHighlighter = new SyntaxHighlighter( m_textedit, m_owners );

        m_client_socket = new KNetwork::KStreamSocket( QString::null, QString::null, this, "client" );

        users = new QString[MATEEDIT_MAX_USERS];
        users[0] = "Server";
        users[1] = "";
        users[2] = "";
        users[3] = "";
        users[4] = "";
        users[5] = "";
    }
    /**
    * Default Destructor
    */
    MateEdit::~MateEdit()
    {
        delete m_client_socket;
        delete m_owners;
    }


    int MateEdit::connectToServer(const QString & server, int port)
    {
        connect( (QObject *)m_client_socket, SIGNAL( readyRead() ), (QObject *)this, SLOT( readServer() ) );
        connect( (QObject *)m_client_socket, SIGNAL( closed() ), (QObject *)this, SLOT( connectionClosed() ) );
        connect( (QObject *)m_client_socket, SIGNAL( connected(const KResolverEntry&) ), (QObject *)this, SLOT( connectionConnected() ) );
        connect( (QObject *)m_client_socket, SIGNAL( gotError(int) ), (QObject *)this, SLOT( connectionError(int) ) );
        connect( (QObject *)m_client_socket, SIGNAL( stateChanged(int newstate)), (QObject *)this, SLOT( socketStateChanged(int) ) );

        m_client_socket->connect( server, QString::number(port) );

        return 0;
    }

    int MateEdit::insertText( const QString & buffer, const int paraFrom, const int indexFrom, const int paraTo, const int indexTo )
    {
        return insertText( buffer, MessageCoords( paraFrom, indexFrom ), MessageCoords( paraTo, indexTo ) );
    }

    int MateEdit::insertText( const QString & buffer, const MessageCoords & from, const MessageCoords & to )
    {
        if( from == to )
        {
            return -1;
        }
        Message * message = new Message( Message::Insert, m_local_pc, 0, m_timestamp, buffer );
        message->setFrom( from );
        message->setTo( to );
        m_owners->insert( from, to, m_local_pc, buffer );
        m_doc->insert( from, to, m_local_pc, buffer );
        m_messageQueue.processMessage( message );
        message->send( m_client_socket );

        return message->id();
    }

    int MateEdit::deleteText( const QString & buffer, const int paraFrom, const int indexFrom, const int paraTo, const int indexTo )
    {
        return deleteText( buffer, MessageCoords( paraFrom, indexFrom ), MessageCoords( paraTo, indexTo ) );
    }

    int MateEdit::deleteText( const QString & buffer, const MessageCoords & from, const MessageCoords & to )
    {
        if( from == to )
        {
            return -1;
        }
        Message * message = new Message( Message::Delete, m_local_pc, 0, m_timestamp, buffer );
        message->setFrom( from );
        message->setTo( to );
        message->send( m_client_socket );
        m_owners->remove( from, to );
        QString text = m_doc->remove( from, to );
        m_messageQueue.processMessage( message );
        message->setText( text );

        return message->id();
    }

    int MateEdit::undoMessage( int id )
    {
        Message undo_message( Message::Undo, m_local_pc, 0, m_timestamp );
        undo_message.setId( id );

        undo_message.send( m_client_socket );

        return undoMessage( id, m_local_pc, 0, m_timestamp );
    }

    int MateEdit::undoMessage( int id, int sender, int server_state, int client_state )
    {
        QPtrList<Message> pending_list;
        Message * message = m_messageQueue.undoMessage( id, sender, pending_list, server_state, client_state );
        if ( message != NULL )
        {
            if ( message->type() == Message::Insert )
            {
                m_owners->insert( message->from( sender ), message->to( sender ), sender, message->text() );
                m_doc->insert( message->from( sender ), message->to( sender ), sender, message->text() );
                remoteInsertText( message->text(), message->from( sender ), message->to( sender ) );
            }
            if ( message->type() == Message::Delete )
            {
                m_owners->remove( message->from( sender ), message->to( sender ) );
                QString text = m_doc->remove( message->from( sender ), message->to( sender ) );
                message->setText( text );
                remoteDeleteText( message->text(), message->from( sender ), message->to( sender ) );
            }
            QPtrListIterator<Message> pending_it( pending_list );
            Message * pending_message;
            for ( pending_message = pending_it.toFirst(); pending_message; ++pending_it, pending_message = pending_it.current() )
            {
                if ( pending_message->type() == Message::Insert )
                {
                    m_owners->insert( message->from( sender ), message->to( sender ), sender, message->text() );
                    m_doc->insert( pending_message->from( sender ), pending_message->to( sender ), sender, pending_message->text() );
                    remoteInsertText( message->text(), message->from( sender ), message->to( sender ) );
                }
                if ( pending_message->type() == Message::Delete )
                {
                    m_owners->remove( message->from( sender ), message->to( sender ) );
                    QString text = m_doc->remove( pending_message->from( sender ), pending_message->to( sender ) );
                    pending_message->setText( text );
                    remoteDeleteText( message->text(), message->from( sender ), message->to( sender ) );
                }
            }

            return message->id();
        }
        else
        {
            return -1;
        }
    }

    int MateEdit::redoMessage( int id )
    {
        return 0;
    }


    int MateEdit::chat( const QString & buffer)
    {
        Message * message = new Message( Message::ChatMessage, m_local_pc, 0, m_timestamp, buffer );

        message->send( m_client_socket );

        delete message;

        return 0;
    }

    void MateEdit::socketStateChanged(int state)
    {
        qDebug(" MateEdit::socketStateChanged %i", state);
        if(state == KNetwork::KStreamSocket::Idle)
        {
//            remoteSessionClosed("Server shutdown");
        }
    }

    void MateEdit::connectionClosed()
    {
        remoteSessionClosed("Server shutdown");
    }

    void MateEdit::readServer()
    {
        static int size = 0;
        int counter = 0;
        QDataStream stream ( m_client_socket );
        if ( size == 0 )
        {
            stream >> size;
        }
//        qDebug("MateEdit::readServer()");
        //Read messages from server and apply locally
        while ( m_client_socket->bytesAvailable() >= size && size > 0 )
        {
            ++counter;
            int rtc;
            int para;
            int pos;

            Message * message = new Message;

            char *data = new char[size];
            m_client_socket->readBlock(data, size);
            QByteArray bytearray;
            bytearray.assign(data,size);
            QBuffer buffer(bytearray);
            message->read(&buffer);

            m_timestamp = message->serverState();

            switch ( message->type() )
            {
                case Message::Init:
                {
                    m_local_pc = message->sender();
                    m_messageQueue.setLocalPc( m_local_pc );
                    m_syntaxHighlighter->setLocalPc( m_local_pc );
                    QString text = message->text();
                    if ( ! text.isEmpty() )
                    {
                        text.replace(QChar ('\r') ,QChar('\n'));
                        int newlines = text.contains('\n');
                        int to;
                        int pos = text.findRev('\n');
                        if ( pos != -1 )
                        {
                            to = text.length() - pos - 1;
                        }
                        else
                        {
                            to = text.length();
                        }

                        m_owners->insert( MessageCoords( 0, 0 ), MessageCoords( newlines, to ), m_local_pc, text );
                        remoteInsertText( text, MessageCoords( 0, 0 ), MessageCoords( newlines, to ) );
    //                    m_textfield->setText( text );
                    }
                    QString username;
                    QImage image;

                    remoteInitSession( message->sender(), username, image );

                    Message * initmessage = new Message( Message::InitUserData, m_local_pc, 0, m_timestamp, username );

                    initmessage->setImage ( image );

                    initmessage->send( m_client_socket );

                    delete initmessage;
                    break;
                }
                case Message::InitUserData:
                {
                    if ( m_local_pc != message->sender() && users[message->sender()].isEmpty() && ! message->text().isEmpty() )
                    {
                        int user_number = message->sender();

                        QImage image = message->image();

                        users[message->sender()] = message->text();
                        remoteAddUser( user_number, message->text(), image );
                    }
                    break;
                }
                case Message::InfoMessage:
                {
                    if ( m_local_pc != message->sender() && users[message->sender()].isEmpty() && ! message->text().isEmpty() )
                    {
                        remoteMessage ( message->text() );
                    }
                    break;
                }
                case Message::ChatMessage:
                {
                    if ( m_local_pc != message->sender()  && ! message->text().isEmpty() )
                    {
                        QString text = users[message->sender()] + ": " + message->text();
                        remoteChat( text );
                    }
                    break;
                }
                case Message::RemoveUser:
                {
                    int user_number = message->sender();
                    if ( user_number == -1 )
                    {
                        QString text = i18n("Server shutdown");
                        remoteSessionClosed( text );
                    }
                    else
                    {
                        remoteRemoveUser( user_number, users[user_number] );
                    }
                    users[user_number] = "";
                    break;
                }
                case Message::Insert:
                {
                    rtc = m_messageQueue.processMessage( message );
                    if (rtc >= 0)
                    {
                        if ( message->type() != Message::NoOp )
                        {
                            if( ! m_owners->insert( message->from( m_local_pc ), message->to( m_local_pc ), message->sender(), message->text() ) )
                            {
                                qDebug("MateEdit::readServer() Insert m_owners failed");
                            }
                            m_doc->insert( message->from( m_local_pc ), message->to( m_local_pc ), message->sender(), message->text() );
                            remoteInsertText( message->text(), message->from( m_local_pc ), message->to( m_local_pc ) );
                        }
                    }
                    break;
                } //end case Message::Insert
                case Message::Delete:
                {
                    rtc = m_messageQueue.processMessage( message );
                    if (rtc >= 0)
                    {
                        if ( message->type() != Message::NoOp )
                        {
                            m_owners->remove( message->from( m_local_pc ), message->to( m_local_pc ) );
                            QString text = m_doc->remove( message->from( m_local_pc ), message->to( m_local_pc ) );
                            message->setText( text );
                            remoteDeleteText( message->text(), message->from( m_local_pc ), message->to( m_local_pc ) );

                        }
                    }
                    break;
                } //end case Message::Delete:
                case Message::Undo:
                {
                    m_messageQueue.debug("");
                    if ( m_local_pc != message->sender() )
                    {
                        undoMessage( message->id(), message->sender(), message->serverState(), message->clientState() );
                    }
                    else
                    {
                        rtc = m_messageQueue.processMessage( message );
                    }
                    break;
                } //end case Message::Undo:
            } //end switch message->type
            size = 0;
            if ( m_client_socket->bytesAvailable() >= sizeof (int) )
            {
                stream >> size;
            }
        } //end while ( client_socket->bytesAvailable() >= size && size > 0 )
    }

    void MateEdit::remoteInitSession( int user_number, QString & username, QImage & image )
    {
        m_client->mateEditInit( user_number, username, image);
    }

    void MateEdit::remoteInsertText( const QString & buffer, const MessageCoords & from, const MessageCoords & to )
    {
        m_client->mateEditInsertText( buffer, from, to );
    }

    void MateEdit::remoteDeleteText( const QString & buffer, const MessageCoords & from, const MessageCoords & to )
    {
        m_client->mateEditDeleteText( buffer, from, to );
    }

    void MateEdit::remoteChat( const QString & text )
    {
        m_client->mateEditChatMessage( text );
    }

    void MateEdit::remoteMessage ( const QString & text )
    {
        m_client->mateEditMessage( text );
    }

    void MateEdit::remoteSessionClosed ( const QString & text )
    {
        m_client->mateEditSessionClosed( text );
    }

    void MateEdit::remoteAddUser ( int user_number, const QString & name, const QImage & image )
    {
        m_client->mateEditInitUser( user_number, name, image );
    }

    void MateEdit::remoteRemoveUser ( int user_number, const QString & name )
    {
        m_client->mateEditRemoveUser( user_number, name );
    }


    SyntaxHighlighter * MateEdit::syntaxHighlighter()
    {
        return m_syntaxHighlighter;
    }



    TextOwner::TextOwner( int size )
    {
        m_textVector = new QValueVector<lineVector> ( size );
    }

    TextOwner::~TextOwner()
    {
        delete m_textVector;
    }

    int TextOwner::size ()
    {
        return m_textVector->size();
    }

    lineVector TextOwner::operator[] (const unsigned int para)
    {
        return (*m_textVector)[para];
    }

    lineVector TextOwner::at(const unsigned int para)
    {
        return (*m_textVector)[para];
    }

    int TextOwner::at(const unsigned int para, const unsigned int idx)
    {
        return (*m_textVector)[para][idx];
    }

    bool TextOwner::insert( const MessageCoords & from, const MessageCoords & to, int owner, const QString & text )
    {
        bool rtc = true;


        debug( true, from, to );

        if ( to <= from )
        {
            return false;
        }

        if( from.para() > m_textVector->size() )
        {
            return false;
        }

        if ( m_textVector->isEmpty() )
        {
            lineVector * line = new lineVector(0);
            m_textVector->push_back( *line );
        }

        if ( from.para() == to.para() )
        {
            if(from.para() < m_textVector->size() && from.idx() <=(*m_textVector)[from.para()].size())
            {
                if(from.idx() > (*m_textVector)[from.para()].size() -1)
                {
                    for(int i = 0; i < to.idx() - from.idx(); i++)
                    {
                        (*m_textVector)[from.para()].append(owner);
                    }
                }
                else
                {
                    (*m_textVector)[from.para()].insert( &(*m_textVector)[from.para()][from.idx()], to.idx() - from.idx(), owner );
                }
            }
            else
            {
                rtc = false;
            }
        }
        else
        {
            int start;
            int end;
            int i;
            QValueVector<int>::size_type size;
            lineVector * line;

            start = text.find('\n');

            line = new lineVector((const lineVector) (*m_textVector)[from.para()]);     //Create new line as copy of from.m_para

            m_textVector->insert( &(*m_textVector)[from.para() + 1] ,*line );        //insert new line, thus becoming to.m_para

            size = (*m_textVector)[from.para()].size();

            (*m_textVector)[from.para()].erase( &(*m_textVector)[from.para()][from.idx()], &(*m_textVector)[from.para()][size] );    //delete end of line in from.m_para

            (*m_textVector)[from.para() + 1].erase( &(*m_textVector)[from.para() + 1][0], &(*m_textVector)[from.para() + 1][from.idx()] );  //break here //delete beginning of to.m_para, thus ensuring that no parts of to and from overlap

            (*m_textVector)[from.para()].insert( &(*m_textVector)[from.para()][from.idx()], start, owner );         // fill from line

            for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
            {
                line = new lineVector( end - start + 1, owner);

                m_textVector->insert( &(*m_textVector)[from.para() + i] ,*line );

                start = end;
            }
            line = new lineVector( text.length() - start + 1, owner);

            (*m_textVector)[to.para()].insert( &(*m_textVector)[to.para()][0] ,to.idx(), owner );

        }
        debug( true, from, to );
        return rtc;
    }

    void TextOwner::remove( const MessageCoords & from, const MessageCoords & to )
    {
        debug( false, from, to );

        if( from.para() > m_textVector->size() && to.para() > m_textVector->size() )
        {
            return;
        }

        if ( from.para() == to.para() )
        {
            (*m_textVector)[from.para()].erase( &(*m_textVector)[from.para()][from.idx()],  &(*m_textVector)[from.para()][to.idx()] );
        }
        else
        {

            int fromPara = from.para();
            int toPara = to.para();
            int size = (*m_textVector)[from.para()].size();
            (*m_textVector)[fromPara].erase( &(*m_textVector)[fromPara][from.idx()], &(*m_textVector)[fromPara][size] );
            (*m_textVector)[toPara].erase( &(*m_textVector)[toPara][0], &(*m_textVector)[toPara][to.idx()] );

            QValueVector<int>::iterator iteratorLine;

            for( iteratorLine = (*m_textVector)[toPara].begin(); iteratorLine != (*m_textVector)[toPara].end(); ++iteratorLine )
            {
                (*m_textVector)[fromPara].push_back( *iteratorLine );
            }

            if ( fromPara + 1 < toPara )
            {
                m_textVector->erase( &(*m_textVector)[fromPara + 1], &(*m_textVector)[toPara]);
            }
            m_textVector->erase( &(*m_textVector)[fromPara + 1]);

        }
        debug( false, from, to );
    }

    void TextOwner::debug( bool insert, const MessageCoords & from, const MessageCoords & to )
    {
        QValueVector<lineVector>::iterator iteratorTotal;
        QValueVector<int>::iterator iteratorLine;

        QFile f("debugoutput.txt");
        f.open(IO_WriteOnly | IO_Append);

        QTextStream stream( &f );




        stream << "TextOwner-------->" << m_textVector->size() << endl;
        for( iteratorTotal = m_textVector->begin(); iteratorTotal != m_textVector->end(); ++iteratorTotal )
        {
            for( iteratorLine = (*iteratorTotal).begin(); iteratorLine != (*iteratorTotal).end(); ++iteratorLine )
            {
                Q_INT32 x = *iteratorLine;
                stream << x << "|";
            }
            stream << "size: " << (*iteratorTotal).size();
            stream << endl;
        }


        if ( insert == true )
        {
            stream << "Insert" << endl;
        }
        else
        {
            stream << "Delete" << endl;
        }
        stream << from.para() << "," << from.idx() << "||" << to.para() << "," << to.idx() << endl;
        stream << "<--------TextOwner" <<endl;

        f.close();
    }

    SyntaxHighlighter::SyntaxHighlighter ( QTextEdit * textEdit, TextOwner * owners ) : QSyntaxHighlighter::QSyntaxHighlighter( textEdit )
    {
        m_local_pc = 0;
        m_owners = owners;

        colors = new QColor[MATEEDIT_MAX_USERS];
        colors[1] = Qt::darkRed;
        colors[2].setNamedColor("DarkGreen");
        colors[3].setNamedColor("MidnightBlue");
        colors[4] = Qt::darkBlue;
        colors[5].setRgb(20,50,50);

    }

    void SyntaxHighlighter::setLocalPc( int local )
    {
        m_local_pc = local;
    }

    int SyntaxHighlighter::highlightParagraph ( const QString & text, int endStateOfLastPara )
    {
        int owner;
        int i;
        int length;
        int start;
        QColor color;
        Q_UNUSED(text);
        Q_UNUSED(endStateOfLastPara);
        int para = currentParagraph();

        if ( para < 0 || para >= m_owners->size() )
        {
            return 0;
        }
        lineVector line = (*m_owners)[para];

        if (line.isEmpty() )
        {
            owner = 0;
        }
        else
        {
            owner = line[0];
        }

        if ( m_local_pc == owner || owner == 0 )
        {
            color = Qt::black;
        }
        else
        {
            color = colors[owner % MATEEDIT_MAX_USERS];
        }

        start = 0;
        length = line.size();
        for( i = 0 ; i < length ; i++ )
        {
            if ( line[i] != owner )
            {
                setFormat( start, i - start, color);
                start = i;
                owner = line[i];
                if ( m_local_pc == owner || owner == 0 )
                {
                    color = Qt::black;
                }
                else
                {
                    color = colors[owner % MATEEDIT_MAX_USERS];
                }
            }

        }
        setFormat( start, i, color );

        return 0;
    }

    bool Client::mateEditInit( int user_number, QString & username, QImage & image )
    {
        return true;
    }

    bool Client::mateEditInitUser( int user_number, const QString & name, const QImage & image )
    {
        return true;
    }

    bool Client::mateEditRemoveUser( int user_number, const QString & name )
    {
        return true;
    }


    bool Client::mateEditInsertText( const QString & text, const MessageCoords & from, const MessageCoords & to )
    {
        return true;
    }


    bool Client::mateEditDeleteText( const QString & text, const MessageCoords & from, const MessageCoords & to )
    {
        return true;
    }


    bool Client::mateEditMessage( const QString & text )
    {
        return true;
    }


    bool Client::mateEditChatMessage( const QString & text )
    {
        return true;
    }

    bool Client::mateEditSessionClosed( const QString & reason )
    {
        return true;
    }


    Document::Document( int size )
    {
        m_textVector = new QValueVector<QString> ( size );
    }

    Document::~Document()
    {
        delete m_textVector;
    }

    int Document::size ()
    {
        return m_textVector->size();
    }

    QString Document::operator[] (const unsigned int para)
    {
        if(para >= m_textVector->size())
        {
            return "";
        }

        return (*m_textVector)[para];
    }

    QString Document::at(const unsigned int para)
    {
        if( para >= m_textVector->size())
        {
            return "";
        }

        return (*m_textVector)[para];
    }

    QChar Document::at(const unsigned int para, const unsigned int idx)
    {
        if( para >= m_textVector->size())
        {
            return '\0';
        }

        return (*m_textVector)[para][idx];
    }

    void Document::insert( const MessageCoords & from, const MessageCoords & to, int owner, const QString & text )
    {

        debug( true, from, to );

        if( from.para() > m_textVector->size() && to.para() > m_textVector->size() )
        {
            return;
        }

        if ( from.para() == to.para() )
        {
            if ( m_textVector->isEmpty() )
            {
                m_textVector->push_back( text );
            }
            else
            {
                (*m_textVector)[from.para()].insert( from.idx(), text );
            }
        }
        else
        {
            int start;
            int end;
            int i;
//            QValueVector<int>::size_type size;
//            lineVector * line;

            start = text.find('\n');

//            line = new lineVector((const lineVector) (*m_textVector)[from.para()]);     //Create new line as copy of from.m_para

            if ( m_textVector->isEmpty() )
            {
                m_textVector->push_back( QString("") );
            }

            m_textVector->insert( &(*m_textVector)[from.para() + 1] ,(*m_textVector)[from.para()] );        //insert new line, thus becoming to.m_para

            int size = (*m_textVector)[from.para()].length();

            (*m_textVector)[from.para()].remove( from.idx(), size );    //delete end of line in from.m_para

            (*m_textVector)[from.para() + 1].remove( 0, from.idx() );  //break here //delete beginning of to.m_para, thus ensuring that no parts of to and from overlap

            (*m_textVector)[from.para()].append( text.section('\n', 0, 0) );         // fill from line

            for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
            {
                m_textVector->insert( &(*m_textVector)[from.para() + i] ,text.section('\n', i, i) );

                start = end;
            }

            (*m_textVector)[to.para()].prepend( text.section('\n', i, i) );

        }
        debug( true, from, to );
    }

    QString Document::remove( const MessageCoords & from, const MessageCoords & to )
    {
        debug( false, from, to );
        QString deleted_text;

        if( from.para() > m_textVector->size() && to.para() > m_textVector->size() )
        {
            return QString::null;
        }

        if ( from.para() == to.para() )
        {
            deleted_text = (*m_textVector)[from.para()].mid( from.idx(),  to.idx() - from.idx() );
            (*m_textVector)[from.para()].remove( from.idx(),  to.idx() - from.idx() );
        }
        else
        {

            int fromPara = from.para();
            int toPara = to.para();
            int size = (*m_textVector)[from.para()].length();
            deleted_text = (*m_textVector)[from.para()].mid( from.idx(),  size );
            (*m_textVector)[fromPara].remove( from.idx(), size );
            QString deleted_text_end = (*m_textVector)[to.para()].mid( 0,  to.idx() );
            (*m_textVector)[toPara].remove( 0, to.idx() );


            (*m_textVector)[fromPara].append( (*m_textVector)[toPara] );

            if ( fromPara + 1 < toPara )
            {
                deleted_text.append("\n");
                deleted_text.append((*m_textVector)[fromPara + 1]);
                m_textVector->erase( &(*m_textVector)[fromPara + 1], &(*m_textVector)[toPara]);
            }
//            deleted_text.append((*m_textVector)[fromPara + 1]);
            m_textVector->erase( &(*m_textVector)[fromPara + 1]);

            deleted_text.append("\n");
            deleted_text.append(deleted_text_end);

        }
        debug( false, from, to );
        return deleted_text;
    }

    void Document::debug( bool insert, const MessageCoords & from, const MessageCoords & to )
    {
        QValueVector<QString>::iterator iteratorTotal;

        QFile f("debugoutput.txt");
        f.open(IO_WriteOnly | IO_Append);

        QTextStream stream( &f );




        stream << "-------->" <<endl;
        for( iteratorTotal = m_textVector->begin(); iteratorTotal != m_textVector->end(); ++iteratorTotal )
        {
            stream << *iteratorTotal;
        }

        stream << endl;

        if ( insert == true )
        {
            stream << "Insert" << endl;
        }
        else
        {
            stream << "Delete" << endl;
        }
        stream << from.para() << "," << from.idx() << "||" << to.para() << "," << to.idx() << endl;
        stream << "<--------" <<endl;

        f.close();
    }

    Shift::Shift( int type, unsigned int queue_sender, unsigned int queue_id, unsigned int server_sender, unsigned int server_id, MessageCoords & server_from, MessageCoords & index, int n_chars, int offset )
    : m_type(type),
      m_queue_sender(queue_sender),
      m_queue_id(queue_id),
      m_server_sender(server_sender),
      m_server_id(server_id),
      m_server_from(server_from),
      m_index(index),
      m_n_chars(n_chars),
      m_offset(offset)
    {
    }

    unsigned int Shift::queueSender()
    {
        return m_queue_sender;
    }

    int Shift::type()
    {
        return m_type;
    }

    unsigned int Shift::queueId()
    {
        return m_queue_id;
    }

    unsigned int Shift::serverSender()
    {
        return m_server_sender;
    }

    unsigned int Shift::serverId()
    {
        return m_server_id;
    }

    MessageCoords Shift::serverFrom()
    {
        return m_server_from;
    }

    MessageCoords Shift::index()
    {
        return m_index;
    }

    int Shift::nChars()
    {
        return m_n_chars;
    }

    int Shift::offset()
    {
        return m_offset;
    }


}

#include "message.moc"

