#! /usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c)2003, Matthias A. Benkard.

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from core.common import _, path
from core import maxima
from gui import InfoDialog, AlgebraSheet

import gobject
import gtk
import gtk.glade
import logging
import sys

log = logging.getLogger("gui.mainwindow")

class MainWindow:
	def __init__(self):
		self.wTree = gtk.glade.XML(path["glade"], "main_window", "mascyma")
		self.wTree.signal_autoconnect(self)
		self.algebra_sheet_count = 0
		self.plot_sheet_count = 0
		
		# Initialize the tool menu.
		self.toolmenu = self.wTree.get_widget("toolmenu_box")
		self.toolwidth = 0
		for widget in self.toolmenu.get_children():
			widget.connect("size_allocate", self.set_tool_width)
		
		# Start a Maxima thread.
		self.maxima = maxima.MaximaThread(self)
		self.maxima.start()
	
	
	def notify(self, thread, status = None, error = None):
		# This is a kludge to execute self.__notify() in the
		# GTK main loop, which makes a lot of things possible
		# that otherwise would not be (GDK interaction, mainly;
		# opening dialog boxes _is_ a kind of interaction with GDK).
		gtk.idle_add(self.__notify, thread, status, error)
	
	
	def __notify(self, thread, status = None, error = None):
		status_icon = self.wTree.get_widget("status_icon")
		status_bar = self.wTree.get_widget("status_bar")
		gtk.threads_enter()
		if thread.maxima_state == maxima.STARTING:
			status_icon.set_from_stock(gtk.STOCK_EXECUTE, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima is being started."))
		elif thread.maxima_state == maxima.IDLE:
			status_icon.set_from_stock(gtk.STOCK_YES, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima is idle."))
		elif thread.maxima_state == maxima.DEAD:
			status_icon.set_from_stock(gtk.STOCK_ABORT, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima has been killed."))
		elif thread.maxima_state == maxima.NOT_RUNNING:
			status_icon.set_from_stock(gtk.STOCK_NO, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima is not running."))
		elif thread.maxima_state == maxima.PROCESSING:
			status_icon.set_from_stock(gtk.STOCK_EXECUTE, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima is working."))
		else:
			status_icon.set_from_stock(gtk.STOCK_QUESTION, gtk.ICON_SIZE_MENU)
			gtk.Tooltips().set_tip(status_icon, _("Maxima's status cannot be detected."))
		
		if status:
			try:
				status_bar.pop(0)
			except:
				pass
			status_bar.push(0, status)
		gtk.threads_leave()
		if error:
			dlg = gtk.MessageDialog(self.wTree.get_widget("main_window"),
					gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
					_("Unfortunately, an error has occurred in the "
					  "execution of Maxima.\n"
					  "Maybe you have not given the correct path "
					  "to the Maxima executable in Mascyma's "
					  "preferences dialog. If so, please correct this.\n"
					  "If the problem persists, you may try reading the "
					  "manual or the FAQ; you might also want to try "
					  "the mailing-lists and forums (see the Mascyma web "
					  "site), file a bug report or mail the Mascyma "
					  "authors. For contact information, see the about "
					  "box (Help->Info)."))
			gtk.threads_enter()
			result = dlg.run()
			dlg.destroy()
			gtk.threads_leave()
		
		return gtk.FALSE
	
	
	def set_tool_width(self, widget, event):
		"""Sets the toolmenu width according to its children."""
		# It does this because otherwise the toolmenu would be resized
		# whenever the user clicked on a submenu.
		
		if widget.allocation.width > self.toolwidth:
			self.toolwidth = widget.allocation.width
			#for wdg in self.toolmenu.get_children():
			#	wdg.set_size_request(self.toolwidth, wdg.get_size_request[1])
			self.toolmenu.set_size_request(self.toolwidth, self.toolmenu.get_size_request()[1])
			#self.toolmenu.resize()
			log.debug(_("Toolmenu width: %i") % self.toolwidth)
	
	# Menu.
	def on_quit_activate(self, *args):
		self.maxima.maxima_quit()
		gtk.mainquit()
	
	def on_main_window_destroy(self, *args):
		self.maxima.maxima_quit()
		gtk.mainquit()
	
	def on_info_activate(self, *args):
		dlg = InfoDialog()
	
	
	# Tool bar.
	def on_quit_tool_clicked(self, *args):
		self.maxima.maxima_quit()
		gtk.mainquit()
	
	def on_new_sheet_tool_clicked(self, *args):
		self.algebra_sheet_count += 1
		book = self.wTree.get_widget("algebra_notebook")
		
		sheet = AlgebraSheet(self.maxima)
		
		# TODO: Mere text labels are boring. We should make an icon
		# for algebra worksheets and one for plot sheets.
		# FIXME: The gtk-close icon (see below) is much too big.
		# Or rather, the requested size is too large.
		# Damn.
		# set_size_request() remedies this, but I consider it a
		# hack.
		pagelabel = gtk.HBox(2)
		pagelabel.set_homogeneous(False)
		pagelabel.set_spacing(2)
		
		pagelabel.add(gtk.Label(_("Algebra %i") % self.algebra_sheet_count))
		
		# The closing button on the tab label.
		btn = gobject.new(gtk.Button, relief = gtk.RELIEF_NONE, border_width = 0, is_focus = False, can_focus = False)
		img = gobject.new(gtk.Image, stock = gtk.STOCK_CLOSE, icon_size = gtk.ICON_SIZE_MENU, xpad = 0, ypad = 0)
		img.set_size_request(4, 4)
		btn.add(img)
		gtk.Tooltips().set_tip(btn, _("Close sheet"), None)
		btn.show_all()
		
		btn.connect("clicked", lambda x: book.remove_page(book.page_num(sheet)))
		
		pagelabel.add(btn)
		pagelabel.show_all()
		
		book.append_page(sheet, pagelabel)
		book.show_all()
	
	def on_close_sheet_tool_clicked(self, *args):
		book = self.wTree.get_widget("algebra_notebook")
		if not book.get_current_page() == 0:
			book.remove_page(book.get_current_page())
		else:
			log.info(_("Can't close the 'output' sheet."))
			# BTW, does anyone remember what that freaky output sheet
			# is there for in the first place?
	
	
	# Tool menu.
	def show_hide_menu(self, menu_name):
		toolmenu = self.wTree.get_widget("%s_toolmenu" % menu_name)
		icon = self.wTree.get_widget("%s_toolmenu_icon" % menu_name)
		
		if toolmenu.flags() & gtk.VISIBLE:
			log.debug(_("Hiding %s menu.") % menu_name)
			toolmenu.hide_all()
			icon.set_from_stock(gtk.STOCK_ADD, gtk.ICON_SIZE_BUTTON)
		else:
			log.debug(_("Showing %s menu.") % menu_name)
			toolmenu.show_all()
			icon.set_from_stock(gtk.STOCK_REMOVE, gtk.ICON_SIZE_BUTTON)	
	
	def on_equation_toolmenu_button_clicked(self, *args):
		self.show_hide_menu("equation")
	
	def on_function_toolmenu_button_clicked(self, *args):
		self.show_hide_menu("function")
	
	def on_term_toolmenu_button_clicked(self, *args):
		self.show_hide_menu("term")

