/* This file is part of the KDE project
   Copyright (C) 2006 KovoKs <info@kovoks.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qheader.h>
#include <qtimer.h>

#include <klocale.h>
#include <kdebug.h>

#include "datalistview.h"

namespace Mailody {

DataListView::DataListView(QWidget* parent, int data_column, int columns)
    : MailodyBaseListView( parent ), m_size(35)
{
    header()->setResizeEnabled(false);
    setResizeMode(QListView::NoColumn);
    setShowSortIndicator(true);
    setHScrollBarMode(QScrollView::AlwaysOff);
    setSorting(500, false); // no sort by default, but do when clicked..

    m_columns = columns;
    m_datacolumn = data_column;
    for (int i = 0; i < columns; i++)
    {
        addColumn("Column");
        setColumnWidthMode(i,QListView::Manual);
        if (m_datacolumn != i)
            setColumnAlignment(1,Qt::AlignHCenter);
    }
    connect(this, SIGNAL(itemAdded(QListViewItem*)), SLOT(slotResize()));
    connect(this, SIGNAL(itemRemoved(QListViewItem*)), SLOT(slotResize()));
    connect(this, SIGNAL(expanded(QListViewItem*)), SLOT(slotResize()));
    connect(this, SIGNAL(collapsed(QListViewItem*)), SLOT(slotResize()));

    m_clearTimer = new QTimer( this );
    connect( m_clearTimer , SIGNAL(timeout()), SLOT(slotResizeDoIt()) );
}

void DataListView::viewportResizeEvent ( QResizeEvent * )
{
    slotResize();
}

void DataListView::slotResize()
{
    if (m_clearTimer)
        m_clearTimer->start(100, true);
}

void DataListView::slotResizeDoIt()
{
    int extraspace = 4;
    if (verticalScrollBar()->isVisible())
        extraspace = verticalScrollBar()->width()+4;

    blockSignals(true);
    for (int i = 0; i < m_columns; i++)
    {
        if (m_datacolumn == i)
            setColumnWidth(i,width()-((m_columns-1)*m_size)-extraspace);
        else
            setColumnWidth(i,m_size);
    }
    triggerUpdate();
    blockSignals(false);
}

void DataListView::keyPressEvent( QKeyEvent* e )
{
    if (e->key() == Qt::Key_Delete)
        deleteItem();
    else
        KListView::keyPressEvent(e);
}

void DataListView::deleteItem()
{
    QListViewItem* selectedItem = selectedItems( true ).first();
    if (!selectedItem)
        return;

    // Before deleting, save next item, 0 if none.
    QListViewItem* next = selectedItem->nextSibling();

    delete selectedItem;

    // If no next item, pick the last, now the item is deleted
    if (!next)
        next = lastItem();

    // No items in the list.
    if (!next)
        return;

    setSelected(next, true);
    setSelectionAnchor(next);
    ensureItemVisible(next);
    setCurrentItem(next);
}

}

#include "datalistview.moc"
