/* UndoAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.undoredo;

import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.Icon;
import be.ugent.caagt.swirl.actions.SimpleAction;

/**
 * Action which undoes the last change registered with the corresponding
 * undo manager. When the action is enabled, the
 * caption is taken from the current {@link UndoableChange} 
 * in the undo list of the associated {@link UndoManager}.
 */
public class UndoAction extends SimpleAction implements UndoListener {

    //
    private final UndoManager manager;
    
    //
    private final String disabledCaption;

    /**
     * Create an action of this type. When the action is enabled, the
     * the caption is taken from the current {@link UndoableChange} of the
     * associated {@link UndoManager}. . Mnemonic
     * and accelerator key are taken from the description object as defined
     * by the parameters {@code bundle} and {@code key}.
     * @param manager Undo manager to which this action applies
     * @param bundle Resource bundle which contains the description for
     * this action.
     * @param key Key used for the description of this action in the bundle.
     * @param icon Small icon to be used for menu and standard button decoration (or null)
     * @param mediumIcon Medium size icon to be used for tool bar buttons (or null)
     * @param mediumDisabledIcon Medium size icon to be used for a disabled tool bar buttons (or null)
     */
    public UndoAction (UndoManager manager, ResourceBundle bundle, String key, 
		       Icon icon, Icon mediumIcon, Icon mediumDisabledIcon) {
        super(bundle, key, icon, mediumIcon, mediumDisabledIcon);
        this.manager = manager;
        this.disabledCaption = (String)getValue (NAME);
        manager.addUndoListener (this);
        setEnabled (false); // not overridable (PMD)
    }
    
    /**
     * Enables/disables the action and changes the caption accordingly.
     */
    public void setEnabled (boolean newValue) {
        super.setEnabled (newValue);
        String caption = disabledCaption;
        if (newValue) {
            String newCaption = manager.getUndoCaption();
            if (newCaption != null)
                caption = newCaption;
        }
        putValue (NAME, caption);
    }

    /**
     * Undoes the current undoable change.
     */
    public void actionPerformed (ActionEvent e) {
        manager.undoLast ();
    }

    

    /**
     * Enables or disables the button according to whether
     * any changes remain to be undone or not.
     */
    public void undoStateChanged () {
        setEnabled (manager.canUndo ());
    }
    
    
}
